"""Helper class for rendering tool arguments in Qt dialogs."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from dataclasses import dataclass, field
from os import path as os_path
from typing import Callable

# 2. Third party modules
import pandas
from PySide2.QtCore import QObject, Signal
from PySide2.QtWidgets import (QCheckBox, QComboBox, QFileDialog, QHBoxLayout, QLabel,
                               QLineEdit, QPushButton, QSizePolicy, QVBoxLayout, QWidget)

# 3. Aquaveo modules
from xms.guipy import settings
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator
from xms.guipy.widgets.table_with_tool_bar import TableWithToolBar
from xms.tool_core.table_definition import TableDefinition


# 4. Local modules


@dataclass
class UiItem:
    """Data class for storing UI information for an argument."""
    interface_info: dict[str, object]
    value_widget: QWidget
    value_getter: Callable[[], object]
    value_setter: Callable[[object], None]
    label_widget: QLabel | None = None
    widgets: list[QWidget] = field(default_factory=list)
    is_file_argument: bool = False
    file_filter: str = ''
    default_suffix: object | None = None
    select_folder: bool = False
    value_action: object | None = None


class ArgumentLayoutHelper(QObject):
    """Add widgets to layout for tool arguments in Qt dialogs."""

    end_do_value_changed = Signal()
    NONE_SELECTED = '(none selected)'

    def __init__(self, parent_dialog):
        """Initializes the class, sets up the ui.

        Args:
            parent_dialog (QObject): The parent dialog.
        """
        super().__init__()
        self.parent_dialog = parent_dialog
        self.ui_items: dict[str, UiItem] = dict()
        self.ui_items_order = []
        self._last_modified_name = None

    def add_arguments_to_layout(self, layout: QVBoxLayout,
                                interface_items: list[dict[str, object]]) -> None:
        """Add argument objects to the layout.

        Args:
            layout: The layout to append to.
            interface_items: Interface items to include.
        """
        self.ui_items = dict()
        self.ui_items_order = []
        for argument_info in interface_items:
            self.ui_items_order.append(argument_info['name'])
            self.add_argument(layout, argument_info)

    def get_focus_widgets(self) -> tuple[list[QWidget], int]:
        """
        Return a tuple containing a list of focus widgets and the index of the new focus widget.

        Returns:
            A tuple containing:
                - focus_widgets (list[QWidget]): A list of all the focus widgets. If a UiItem has a `value_action`
                    attribute, its value is appended to the list. Otherwise, the `value_widget` attribute is appended to
                    the list.
                - new_focus (int): The index of the new focus widget in the `focus_widgets` list. If `last_focus`
                    attribute is None, then the new focus widget index will be 0. Otherwise, the new focus widget index
                    will be the last focus widget index + 1.
        """
        focus_widgets = []
        last_focus = None
        for name in self.ui_items_order:
            ui_item = self.ui_items[name]
            if ui_item.interface_info['type'] == 'Table':
                # skip tables in tab order
                continue
            if ui_item.value_action:
                focus_widgets.append(ui_item.value_action)
            else:
                focus_widgets.append(ui_item.value_widget)
            if name == self._last_modified_name:
                last_focus = len(focus_widgets) - 1
        new_focus = 0 if last_focus is None else last_focus + 1
        return focus_widgets, new_focus

    def add_argument(self, layout: QVBoxLayout, argument_info: dict[str, object]):  # noqa: C901
        """Add GUI interface for an argument.

        Args:
            layout: The Qt layout to append to.
            argument_info: The argument info.
        """
        argument_type = argument_info['type']
        # widgets = widget_depends[argument_name] if widget_depends and argument_name in widget_depends else []
        widgets = []
        argument_name = str(argument_info['name'])
        label_str = str(argument_info['description'])
        file_filter = str(argument_info.get('file_filter', ''))
        default_suffix = argument_info.get('default_suffix', None)
        select_folder = False

        label_widget = None
        if len(label_str) > 0 and argument_type != 'Boolean':
            label_str = label_str + ':'
            if 'requirement_satisfied' in argument_info:
                label_color = 'black' if argument_info['requirement_satisfied'] else 'brown'
                label_str = f'<font color="{label_color}">* {label_str}</font>'
            if argument_type == 'SaveFile':
                if argument_info.get('file_exists', False):
                    label_str = f'{label_str}  <font color="red">WARNING!  File already exists!</font>'
            label_widget = QLabel(label_str)
            widgets.append(label_widget)
            label_widget.setAccessibleName(argument_name + ' label')
            layout.addWidget(label_widget)

        value_action = None
        if argument_type == 'StringSelector':
            widgets.append(QComboBox())
            layout.addWidget(widgets[-1])
            widgets[-1].addItems(argument_info['choices'])
            widgets[-1].setCurrentText(argument_info['value'])
            widgets[-1].currentIndexChanged.connect(lambda: self.do_value_changed(argument_name))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setCurrentText
            value_getter = widgets[-1].currentText
            is_file_argument = False
        elif argument_type == 'Number':
            widgets.append(QLineEdit())
            layout.addWidget(widgets[-1])
            validator = QxDoubleValidator(parent=self)
            widgets[-1].setValidator(validator)
            widgets[-1].setText(str(argument_info['value']))
            widgets[-1].editingFinished.connect(lambda: self.do_value_changed(argument_name))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = lambda: float(widgets[-1].text())  # noqa: E731
            is_file_argument = False
        elif argument_type == 'Integer':
            widgets.append(QLineEdit())
            layout.addWidget(widgets[-1])
            widgets[-1].setValidator(QxIntValidator())
            widgets[-1].setText(str(argument_info['value']))
            widgets[-1].editingFinished.connect(lambda: self.do_value_changed(argument_name))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = lambda: int(widgets[-1].text())  # noqa: E731
            is_file_argument = False
        elif argument_type == 'SelectFile':
            hor_layout = QHBoxLayout()
            layout.addLayout(hor_layout)
            widgets.append(QPushButton('Select File...'))
            button = widgets[-1]
            hor_layout.addWidget(button)
            button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
            widgets[-1].setAccessibleName(argument_name + '_select')
            # connect to the button click
            widgets.append(QLabel())
            displayed_text = argument_info['value'] if argument_info['value'] else self.NONE_SELECTED
            widgets[-1].setText(displayed_text)
            hor_layout.addWidget(widgets[-1])
            button.clicked.connect(lambda: self.do_open_selector(argument_name, widgets[-1]))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = widgets[-1].text
            value_action = button
            is_file_argument = True
        elif argument_type == 'SaveFile':
            hor_layout = QHBoxLayout()
            layout.addLayout(hor_layout)
            widgets.append(QPushButton('Save As...'))
            button = widgets[-1]
            hor_layout.addWidget(button)
            button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
            widgets[-1].setAccessibleName(argument_name + '_select')
            # connect to the button click
            widgets.append(QLabel())
            hor_layout.addWidget(widgets[-1])
            displayed_text = argument_info['value'] if argument_info['value'] else self.NONE_SELECTED
            widgets[-1].setText(displayed_text)
            button.clicked.connect(lambda: self.do_save_as_selector(argument_name, widgets[-1]))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = widgets[-1].text
            value_action = button
            is_file_argument = True
        elif argument_type == 'SelectFolder':
            hor_layout = QHBoxLayout()
            layout.addLayout(hor_layout)
            widgets.append(QPushButton('Select File...'))
            button = widgets[-1]
            hor_layout.addWidget(button)
            button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
            widgets[-1].setAccessibleName(argument_name + '_select')
            # connect to the button click
            widgets.append(QLabel())
            hor_layout.addWidget(widgets[-1])
            displayed_text = argument_info['value'] if argument_info['value'] else self.NONE_SELECTED
            widgets[-1].setText(displayed_text)
            button.clicked.connect(lambda: self.do_open_selector(argument_name, widgets[-1]))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = widgets[-1].text
            value_action = button
            is_file_argument = True
            select_folder = True
        elif argument_type == 'SaveFolder':
            hor_layout = QHBoxLayout()
            layout.addLayout(hor_layout)
            widgets.append(QPushButton('Save As...'))
            button = widgets[-1]
            hor_layout.addWidget(button)
            button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
            widgets[-1].setAccessibleName(argument_name + '_select')
            # connect to the button click
            widgets.append(QLabel())
            hor_layout.addWidget(widgets[-1])
            displayed_text = argument_info['value'] if argument_info['value'] else self.NONE_SELECTED
            widgets[-1].setText(displayed_text)
            button.clicked.connect(lambda: self.do_save_as_selector(argument_name, widgets[-1]))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = widgets[-1].text
            value_action = button
            is_file_argument = True
            select_folder = True
        elif argument_type == 'String':
            widgets.append(QLineEdit())
            layout.addWidget(widgets[-1])
            widgets[-1].setText(argument_info['value'])
            widgets[-1].editingFinished.connect(lambda: self.do_value_changed(argument_name))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = widgets[-1].text
            is_file_argument = False
        elif argument_type == 'Boolean':
            widgets.append(QCheckBox(label_str))
            layout.addWidget(widgets[-1])
            widgets[-1].setChecked(argument_info['value'])
            widgets[-1].clicked.connect(lambda: self.do_value_changed(argument_name))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setChecked
            value_getter = widgets[-1].isChecked
            is_file_argument = False
        elif argument_type == 'Table':
            widgets.append(TableWithToolBar())
            table_definition = argument_info['table_definition']
            if not isinstance(table_definition, TableDefinition):
                table_def = TableDefinition.from_dict(table_definition)
            else:
                table_def = table_definition
            widgets[-1].setup(table_def, argument_info['value'].copy())
            layout.addWidget(widgets[-1])
            widgets[-1].data_changed.connect(lambda: self.do_value_changed(argument_name))
            widgets[-1].setAccessibleName(argument_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].set_values
            value_getter = widgets[-1].get_values
            is_file_argument = False
        else:
            raise RuntimeError(f'Unsupported argument type: {argument_type}')

        self.ui_items[argument_name] = UiItem(
            interface_info=argument_info,
            value_widget=value_widget,
            value_getter=value_getter,
            value_setter=value_setter,
            label_widget=label_widget,
            widgets=widgets,
            is_file_argument=is_file_argument,
            file_filter=file_filter,
            default_suffix=default_suffix,
            select_folder=select_folder,
            value_action=value_action
        )

    def do_save_as_selector(self, argument_name: str, text_field: QWidget):
        """Display a file save as dialog.

        Args:
            argument_name: Name of the argument.
            text_field: Qt widget holding descriptive text
        """
        self._last_modified_name = argument_name
        ui_item = self.ui_items[argument_name]
        curr_filename = str(ui_item.value_getter())
        path = ''
        if curr_filename:
            path = os_path.dirname(curr_filename)
        if not os_path.exists(path):
            path = settings.get_file_browser_directory()
        file_filter = ui_item.file_filter
        window_text = 'Save Folder' if ui_item.select_folder else 'Save File'
        dlg = QFileDialog(self.parent_dialog, window_text, path, file_filter)
        dlg.selectFile(curr_filename)
        dlg.setLabelText(QFileDialog.Accept, "Save")
        dlg.setFileMode(QFileDialog.AnyFile)
        if ui_item.default_suffix:
            dlg.setDefaultSuffix(ui_item.default_suffix)
        if ui_item.select_folder:
            dlg.setOption(QFileDialog.ShowDirsOnly, on=True)
            dlg.setFileMode(QFileDialog.Directory)
        if dlg.exec():
            new_value = dlg.selectedFiles()[0]
            text_field.setText(new_value)
        self.do_value_changed(argument_name)

    def do_open_selector(self, argument_name, text_field):
        """Display a file open selector dialog.

        Args:
            argument_name: Name of the argument
            text_field (QLineEdit): Qt widget holding descriptive text
        """
        self._last_modified_name = argument_name
        ui_item = self.ui_items[argument_name]
        curr_filename = str(ui_item.value_getter())
        path = ''
        if curr_filename:
            path = os_path.dirname(curr_filename)
        if not os_path.exists(path):
            path = settings.get_file_browser_directory()
        file_filter = ui_item.file_filter
        window_text = 'Select Folder' if ui_item.select_folder else 'Select File'
        dlg = QFileDialog(self.parent_dialog, window_text, path, file_filter)
        dlg.setLabelText(QFileDialog.Accept, "Select")
        if ui_item.select_folder:
            dlg.setOption(QFileDialog.ShowDirsOnly, on=True)
            dlg.setFileMode(QFileDialog.Directory)
        if dlg.exec():
            text_field.setText(dlg.selectedFiles()[0])
        self.do_value_changed(argument_name)

    def on_end_do_value_changed(self):
        """Sends a signal after do_value_changed is called."""
        self.end_do_value_changed.emit()

    def do_value_changed(self, argument_name: str):
        """Get values from widgets and update the argument value.

        Args:
            argument_name: Name of the argument.
        """
        if argument_name is not None:
            self._last_modified_name = argument_name
            ui_item = self.ui_items[argument_name]
            # get the value for this argument and save it in the argument list
            val = ui_item.value_getter()
            if ui_item.is_file_argument:
                if val == self.NONE_SELECTED:
                    val = ''

            if not self.argument_values_equal(argument_name, val):
                ui_item.interface_info['value'] = self.copy_value(val)
                self.on_end_do_value_changed()

    def argument_values_equal(self, argument_name: str, ui_value: object) -> bool:
        """Checks if the original value of an argument matches the provided UI value.

        Args:
            argument_name: The name of the UI argument.
            ui_value: The UI value to compare against the argument's value.

        Returns:
            True if the values are equal, False otherwise.
        """
        ui_item = self.ui_items[argument_name]
        if isinstance(ui_value, pandas.DataFrame):
            return ui_value.equals(ui_item.interface_info['value'])
        return ui_value == ui_item.interface_info['value']

    @staticmethod
    def copy_value(value: object) -> object:
        """Copy object value for interface info.

        Args:
            value: The object to copy.

        Returns:
            The copied value.
        """
        if isinstance(value, pandas.DataFrame):
            return value.copy()
        return copy.copy(value)
