"""A dialog for displaying feature map lines."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from pandas import DataFrame
from rtree import index
import shapely
from shapely.geometry import LineString, Polygon
from shapely.ops import nearest_points

# 3. Aquaveo modules

# 4. Local modules
import xms.tool_xms.algorithms.geom_funcs_with_shapely as gsh


def prune_arc(arc, non_geo_arc, size, left, join_type, num_steps, debug_arcs_dir, mitre_limit=None):
    """Prune the arc.

    Args:
        arc (LineString): LineString to prune
        non_geo_arc (LineString): LineString in non-geographic coords
        size (float): Pruning width
        left (bool): True if left side, False if right
        join_type (string): 'round', 'mitre', 'beveled'
        num_steps (int): number of offsets
        debug_arcs_dir (string): 'out', 'in', or 'orig'
        mitre_limit (float): ratio for very sharp corners, optional
    Returns:
        pruned_arc (LineString): pruned linestring
        debug_arcs (list)
    """
    pruner = PruneArc(arc, non_geo_arc, size, left, join_type, num_steps, debug_arcs_dir, mitre_limit)
    return pruner.do_prune()


class PruneArc():
    """Tool to clean geometry."""

    def __init__(self, native_arc, non_geo_arc, size, left, join_type, num_steps, debug_arcs_dir, mitre_limit=None):
        """Initializes the class."""
        self._native_arc = native_arc
        self._non_geo_arc = non_geo_arc
        self._size = size
        self._is_left = left
        self._join_type = join_type
        self._num_steps = num_steps
        self._debug_arcs_dir = debug_arcs_dir
        self._mitre_limit = mitre_limit
        self._debug_arcs = []

        self._orig_loop_pts = []

    def do_prune(self):
        """Get initial arguments for tool.

        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        if self._size == 0.0:
            return self._native_arc, self._debug_arcs, None

        if self._native_arc.is_ring is True:
            # convex hull
            arcs = self._split_arc_by_convex_hull(self._non_geo_arc)

            arc_1 = self._prune_arc(arcs[0])
            arc_2 = self._prune_arc(arcs[1])

            if not arc_1 or not arc_2:
                return None, self._debug_arcs, 'Error pruning. Try adjusting the "Advanced options."'

            non_geo_pruned = self._combine_pruned_arcs(arc_1, arc_2)
        else:
            non_geo_pruned = self._prune_arc(self._non_geo_arc)

        if non_geo_pruned and non_geo_pruned.is_simple:
            native_pruned = self._non_geo_to_native(non_geo_pruned)
            return native_pruned, self._debug_arcs, None
        else:
            return None, self._debug_arcs, 'Error pruning. Try adjusting the "Advanced options."'

    def _non_geo_to_native(self, non_geo_pruned):
        """Get initial arguments for tool.

        Returns:
               (LineString): Pruned LineString in native coords
        """
        if self._native_arc == self._non_geo_arc:
            return non_geo_pruned

        native_orig = gsh.pts_from_ls(self._native_arc)
        non_geo_orig = gsh.pts_from_ls(self._non_geo_arc)
        non_geo_pruned = gsh.pts_from_ls(non_geo_pruned)

        self._start_index = 0

        return LineString([native_orig[self._find_pt_index_in_list(pt, non_geo_orig)] for pt in non_geo_pruned])

    def _find_pt_index_in_list(self, pt_to_find, pt_list):
        """Get initial arguments for tool.

        Returns:
               (LineString): Pruned LineString in native coords
        """
        try:
            self._start_index = pt_list.index(pt_to_find, self._start_index)
        except ValueError:
            self._start_index = pt_list.index(pt_to_find)
        return self._start_index

    def _split_arc_by_convex_hull(self, arc):
        """Get initial arguments for tool.

        Arguments:
            arc (LineString): arc to split
        Returns:
               (list): split arcs
        """
        convex_hull = arc.convex_hull

        # get the points of the convex hull so that we can determine the two that are the closest to 180 degrees
        convex_pts = gsh.pts_from_shapely_poly(convex_hull)[0]
        last_convex_pt_id = len(convex_pts) - 1
        angles = dict()
        for i in range(1, last_convex_pt_id):
            prev = i - 1
            next = i + 1 if i < last_convex_pt_id else 0
            angles[abs(np.pi - abs(self._angle_between_edges(convex_pts[prev], convex_pts[i], convex_pts[next])))] = i

        angles_list = list(angles.keys())
        angles_list.sort()

        node_1 = angles[angles_list[0]]
        node_2 = angles[angles_list[1]]

        self._orig_loop_pts = gsh.pts_from_ls(arc)
        self._orig_loop_pts.pop(-1)
        last_orig_arc_idx = len(self._orig_loop_pts) - 1
        node_1_orig_idx = self._orig_loop_pts.index(convex_pts[node_1])
        node_2_orig_idx = self._orig_loop_pts.index(convex_pts[node_2])

        # split the arc at these two nodes
        idx = node_1_orig_idx
        pts_1 = [self._orig_loop_pts[idx]]
        while idx != node_2_orig_idx:
            idx = idx + 1 if idx < last_orig_arc_idx else 0
            pts_1.append(self._orig_loop_pts[idx])

        pts_2 = [self._orig_loop_pts[idx]]
        while idx != node_1_orig_idx:
            idx = idx + 1 if idx < last_orig_arc_idx else 0
            pts_2.append(self._orig_loop_pts[idx])

        return [LineString(pts_1), LineString(pts_2)]

    def _combine_pruned_arcs(self, arc_1, arc_2):
        """Get initial arguments for tool.

        Arguments:
            arc_1 (LineString): first pruned arc
            arc_2 (LineString): second pruned arc
        Returns:
            (LineString): LineString of combined pruned arc
        """
        pts_1 = gsh.pts_from_ls(arc_1)
        pts_2 = gsh.pts_from_ls(arc_2)

        if pts_1[-1] == pts_2[0]:
            pts_1.pop()
        if pts_2[-1] == pts_1[0]:
            pts_2.pop()

        # if the original node is still existing (wasn't pruned out), make it the node again
        node_loc = self._orig_loop_pts[0]
        if node_loc in pts_1:
            node_idx = pts_1.index(node_loc)
            combined = pts_1[node_idx:] + pts_2 + pts_1[:node_idx + 1]
        elif node_loc in pts_2:
            node_idx = pts_2.index(node_loc)
            combined = pts_2[node_idx:] + pts_1 + pts_2[:node_idx + 1]
        else:
            combined = pts_1 + pts_2
            combined.append(pts_1[0])

        return LineString(combined)

    def _angle_between_edges(self, pt1, pt2, pt3):
        """Computes the angle between 2 edges.

        Args:
            pt1 (ndarray): The first point.
            pt2 (ndarray): The second point.
            pt3 (ndarray): The third point.

        Returns:
            (float): The angle
        """
        the_angle = 0.0
        dxp = pt1[0] - pt2[0]
        dyp = pt1[1] - pt2[1]
        dxn = pt3[0] - pt2[0]
        dyn = pt3[1] - pt2[1]
        magn = np.sqrt(dxn * dxn + dyn * dyn)
        magp = np.sqrt(dxp * dxp + dyp * dyp)
        if magn != 0.0 and magp != 0.0:
            cos_angle = (dxn * dxp + dyn * dyp) / (magn * magp)
            cos_angle = min(cos_angle, 1.0)
            cos_angle = max(cos_angle, -1.0)
            the_angle = np.arccos(cos_angle)
            the_angle = (2 * np.pi - the_angle) if ((dxp * dyn) - (dxn * dyp)) < 0.0 else the_angle

        return the_angle

    def _prune_arc(self, arc):
        """Get initial arguments for tool.

        Returns:
               (LineString): LineString of pruned arc
        """
        orig_pts = gsh.pts_from_ls(arc)
        if len(orig_pts) < 4:
            return arc

        if self._debug_arcs_dir == 'out':
            self._debug_arcs.append(arc)

        offset_width = (self._size / 2.0) / self._num_steps
        orig_shp_pts = gsh.shapely_pts_from_ls(arc)
        pts = orig_pts.copy()

        # calculate a min seg length so that we don't get a ton of meaningless vertices
        ave_seg_length = arc.length / len(orig_shp_pts)

        min_allowed_length = ave_seg_length * 0.01

        if self._mitre_limit is None:
            self._mitre_limit = 100.0
            # self._mitre_limit = arc.length / (len(orig_pts) - 1)

        # offset in the opposite direction from the prune
        for _ in range(self._num_steps):
            pts = gsh.calculate_arc_offsets(pts, -offset_width if self._is_left else offset_width, self._join_type,
                                            self._mitre_limit)

            if len(pts) < 2:
                return None

            # remove small segments
            pts = gsh.remove_small_segments(pts, min_allowed_length)

            if len(pts) < 2:
                return None

            if len(pts) > 1 and self._debug_arcs_dir == 'out':
                self._debug_arcs.append(LineString(pts))

        if self._debug_arcs_dir == 'in':
            self._debug_arcs.append(LineString(pts))

        # now move it back
        for _ in range(self._num_steps):
            pts = gsh.calculate_arc_offsets(pts, offset_width if self._is_left else -offset_width, self._join_type,
                                            self._mitre_limit)
            if len(pts) > 1:
                # remove small segments
                pts = gsh.remove_small_segments(pts, min_allowed_length)

            if len(pts) > 1 and self._debug_arcs_dir == 'in':
                self._debug_arcs.append(LineString(pts))

        # start with the original points and remove any that are not on the offset arc
        tol = offset_width * 0.1

        last_offset_ls = LineString(pts)

        if last_offset_ls.is_simple and last_offset_ls.is_valid:
            final_sh_pts = self._get_final_pts(orig_shp_pts, last_offset_ls, tol)

            if len(final_sh_pts) == 0:
                final_sh_pts = [orig_shp_pts[0], orig_shp_pts[-1]]

            # the end points get added no matter what
            if final_sh_pts[0] != orig_shp_pts[0]:
                final_sh_pts.insert(0, orig_shp_pts[0])
            if final_sh_pts[-1] != orig_shp_pts[-1]:
                final_sh_pts.append(orig_shp_pts[-1])

            if len(final_sh_pts) > 1:
                if self._check_validity(orig_shp_pts, final_sh_pts):
                    return LineString(final_sh_pts)

        return None

    def _get_final_pts(self, orig_shp_pts, last_offset_ls, tol):
        """Get initial arguments for tool.

        Returns:
               (LineString): LineString of pruned arc, list of debug arcs
        """
        pts = gsh.pts_from_ls(last_offset_ls)
        if len(pts) > 7:
            lines = [LineString(pts[i:i + 5]) for i in range(0, len(pts) - 6, 4)]
            lines.append(LineString(pts[-6:]))
        else:
            lines = [LineString(pts)]
        boxes = [expanded_bounds(line.bounds, tol) for line in lines]

        def generator_func():
            for j, b in enumerate(boxes):
                yield j, b, b
        rtree = index.Index(generator_func())

        final = [sh_pt for sh_pt in orig_shp_pts if self._in_tol(rtree, sh_pt, lines, tol)]
        return final

    def _check_validity(self, orig_shp_pts, final_sh_pts):
        """Get initial arguments for tool.

        Returns:
               (bool): True if valid, False if not
        """
        final_ls = LineString(final_sh_pts)
        num_orig_pts = len(orig_shp_pts)
        final_idx = 3
        c_orig = 1
        p_orig = 0
        while not final_ls.is_simple and len(final_sh_pts) > 3:
            num_final_pts = len(final_sh_pts)
            while c_orig < num_orig_pts and final_idx <= num_final_pts:
                if final_idx == num_final_pts:
                    return False
                c_orig = orig_shp_pts.index(final_sh_pts[final_idx], p_orig)
                if c_orig > p_orig + 1:
                    final_pt_0 = final_sh_pts[final_idx - 3]
                    final_pt_1 = final_sh_pts[final_idx - 2]
                    final_pt_2 = final_sh_pts[final_idx - 1]
                    final_pt_3 = final_sh_pts[final_idx]

                    test_ls = LineString([final_pt_0, final_pt_1, final_pt_2, final_pt_3])
                    if not test_ls.is_simple:
                        del final_sh_pts[final_idx - 1]
                        final_ls = LineString(final_sh_pts)
                        final_idx -= 1
                        c_orig -= 1
                        p_orig -= 1
                        break

                final_idx += 1
                p_orig = c_orig

        if len(final_sh_pts) < 4:
            return False

        final_idx = 1
        num_final_pts = len(final_sh_pts)
        c_orig = 1
        p_orig = 0

        while c_orig < num_orig_pts and final_idx < num_final_pts:
            c_orig = orig_shp_pts.index(final_sh_pts[final_idx], p_orig)
            if c_orig > p_orig + 1:
                final_pt_1 = final_sh_pts[final_idx - 1]
                final_pt_2 = final_sh_pts[final_idx]
                new_seg_length = final_pt_2.distance(final_pt_1)
                if new_seg_length > self._size * 2.0:
                    # does what we are cutting off have a long segment?
                    seg_lengths = [orig_shp_pts[i - 1].distance(orig_shp_pts[i]) for i in range(p_orig, c_orig + 1)]
                    if new_seg_length > max(seg_lengths) * 1.5:
                        # we created an extra long segment and this is a bad prune
                        return False

            final_idx += 1
            p_orig = c_orig

        return final_ls.is_valid

    def _in_tol(self, rtree, pt, lines, tol):
        """Get initial arguments for tool.

        Returns:
               (LineString): LineString of pruned arc, list of debug arcs
        """
        candidates = rtree.intersection(pt.bounds)
        for candidate in candidates:
            if lines[candidate].distance(pt) <= tol:
                return True
        return False


def expanded_bounds(bounds, tol):
    """Get initial arguments for tool.

    Returns:
           (LineString): LineString of pruned arc, list of debug arcs
    """
    return (bounds[0] - tol, bounds[1] - tol, bounds[2] + tol, bounds[3] + tol)


def snap_arcs(primary, secondary, tolerance):
    """Snap the two arcs, only by modifying the primary.

    Args:
        primary (LineString): LineString to be snap
        secondary (LineString): LineString to be snapped to
        tolerance (float): maximum distance for snapping
    Returns:
        snapped_arcs (list): LineString result
    """
    primary_pts = gsh.shapely_pts_from_ls(primary)
    primary_is_loop = primary_pts[0] == primary_pts[-1]

    secondary_end_pts = gsh.shapely_endpoints_from_linestring(secondary)

    # find the closest points to the ends
    orig_num_pts = len(primary_pts)
    to_be_snapped = LineString(primary_pts)
    split_pts = []

    for i, end_pt in enumerate(secondary_end_pts):
        dists = [(p_pt.distance(end_pt), p_pt, idx) for idx, p_pt in enumerate(primary_pts)]
        dists = sorted(dists, key=lambda x: x[0])
        if dists[0][0] <= tolerance:
            dist_along_line = primary.project(dists[0][1])
            split_pts.append((dists[0][1], dists[0][2], dists[0][0], dist_along_line, i))  # point, idx, dist, which end

    if len(split_pts) == 0:
        return None

    snapped_arcs = []
    split_pts = sorted(split_pts, key=lambda x: x[2])

    split_1 = []
    split_2 = []
    snap_location = gsh.shapely_pt_to_loc(secondary_end_pts[split_pts[0][4]])
    if split_pts[0][1] == 0:
        primary_pts[0] = snap_location
        if primary_is_loop:
            primary_pts[-1] = snap_location
        split_2 = primary_pts
    elif split_pts[0][1] == orig_num_pts - 1:
        primary_pts[-1] = snap_location
        split_1 = primary_pts
    else:
        split_1, split_2 = gsh.split_ls_at_pt(to_be_snapped, split_pts[0][0])
        if len(split_1) > 1 and split_1[-1] == split_1[-2]:
            split_1.remove(split_1[-1])
        split_1[-1] = snap_location
        split_2[0] = snap_location

    if len(split_1) > 0:
        snapped_arcs.append(LineString(split_1))
    if len(split_2) > 0:
        snapped_arcs.append(LineString(split_2))

    return snapped_arcs


def collapse_arc(primary, keep_start, connected_arcs_at_start, connected_arcs_at_end):
    """Collapse an arc, keeping the specified node.

    Args:
        primary (LineString): LineString to collapse_
        keep_start (bool): True for keeping the start node, False for the end
        connected_arcs_at_start (list): List of arcs connected at the start node
        connected_arcs_at_end (list): List of arcs connected at the end node
    Returns:
        (list): modified adjacent arcs
    """
    if len(connected_arcs_at_start) == 0 and len(connected_arcs_at_end) == 0:
        return [], None

    arcs_to_adjust = connected_arcs_at_end if keep_start else connected_arcs_at_start

    if len(arcs_to_adjust) == 0:
        return [], None

    end_pts = gsh.endpt_locs_from_ls(primary)
    keep_pt = list(end_pts[0]) if keep_start else list(end_pts[-1])
    delete_pt = list(end_pts[-1]) if keep_start else list(end_pts[0])

    mod_geoms = []
    for arc in arcs_to_adjust:
        arc_pts = gsh.pts_from_ls(arc)
        if arc_pts[0] == delete_pt:
            arc_pts[0] = keep_pt
        elif arc_pts[-1] == delete_pt:
            arc_pts[-1] = keep_pt

        mod_arc = LineString(arc_pts)
        if not mod_arc.is_valid or not mod_arc.is_simple:
            return None, 'Collapse results in an invalid arc.'

        mod_geoms.append(LineString(arc_pts))

    return mod_geoms, None


def curvature_redistribution(arc, max_delta, min_seg_length):
    """Collapse an arc, keeping the specified node.

    Args:
        arc (LineString): LineString to redistribute
        max_delta (float): Maximum delta
        min_seg_length (float): Minimum allowed segment length
    Returns:
        (LineString): redistributed arc
    """
    curve_redister = CurvatureRedistribution(arc, max_delta, min_seg_length)
    arc, msg = curve_redister.do_redist()
    return arc, msg


class CurvatureRedistribution():
    """Tool to clean geometry."""

    def __init__(self, arc, max_delta, min_seg_length):
        """Initializes the class."""
        self._arc = arc
        self._pts = gsh.pts_from_ls_tuple(arc)
        self._max_delta = max_delta * np.pi / 180.0
        self._min_seg_length = min_seg_length
        self._num_verts_to_delete = 0
        self._df = None
        self._msg = None

        self._needs_redist = False
        self._was_modified = False

    def do_redist(self):
        """Redistribution algorithm."""
        if not self._is_arc_legal():
            self._msg = 'Arc does not have enough points for this operation.'
            return None, self._msg

        self._visit_vertices()

        if self._needs_redist is False:
            self._msg = 'Action result: No impact'
            return None, self._msg

        self._delete_flagged_vertices()
        self._recreate_arc()

        return self._arc, self._msg

    def _is_arc_legal(self):
        if self._arc.is_ring is True:
            return len(self._pts) > 4
        return len(self._pts) > 2

    def _visit_vertices(self):
        vtx_dict = {
            'loc': [],  # vertex location
            'prev_loc': [],  # location of previous point
            'next_loc': [],  # location of next point
            'prev_len': [],  # length of previous segment
            'next_len': [],  # length of next segment
            'prev_angle': [],  # angle of previous segment
            'next_angle': [],  # angle of next segment
            'delta': [],  # angle of deflection
            'flag': [],  # valid, smooth, delete
            'delete_metric': []
        }
        num_verts_to_smooth = 0
        prev_vtx = None
        prev_len = None
        prev_angle = None
        num_pts = len(self._pts)
        for idx in range(num_pts):
            cur_vtx = self._pts[idx]
            next_vtx = self._pts[idx + 1] if idx < num_pts - 1 else None
            vtx_dict['loc'].append(cur_vtx)
            vtx_dict['prev_loc'].append(prev_vtx)
            vtx_dict['next_loc'].append(next_vtx)
            vtx_dict['prev_len'].append(prev_len)

            next_len = gsh.distance_between_pts(cur_vtx, next_vtx) if next_vtx else None
            vtx_dict['next_len'].append(next_len)

            vtx_dict['prev_angle'].append(prev_angle)
            next_angle = gsh.get_angle_of_segment(cur_vtx, next_vtx) if next_vtx else None
            vtx_dict['next_angle'].append(next_angle)
            delta = next_angle - prev_angle if (next_angle and prev_angle) else None

            delta = delta - np.pi * 2.0 if (delta and delta > np.pi) else delta
            delta = delta + np.pi * 2.0 if (delta and delta < -np.pi) else delta
            vtx_dict['delta'].append(delta)

            flag = 'valid'
            delete_metric = 0.0
            if delta and abs(delta) > self._max_delta:
                flag, delete_metric = self._evaluate_for_deletion(prev_len, next_len, delta)

            vtx_dict['flag'].append(flag)
            vtx_dict['delete_metric'].append(delete_metric)

            if flag == 'delete':
                self._num_verts_to_delete += 1
            elif flag == 'smooth':
                num_verts_to_smooth += 1

            prev_vtx = self._pts[idx]
            prev_len = next_len
            prev_angle = next_angle

        if self._num_verts_to_delete > 0 or num_verts_to_smooth > 0:
            self._needs_redist = True
            self._df = DataFrame.from_dict(vtx_dict)

    def _delete_flagged_vertices(self):
        while self._num_verts_to_delete > 0:
            # find the vertex with the largest delete metric
            delete_idx = int(self._df[['delete_metric']].idxmax()['delete_metric'])

            valid_to_delete = self._update_verts(delete_idx)
            if valid_to_delete:
                self._df.drop(delete_idx, inplace=True)
            else:
                self._df.at[delete_idx, 'delete_metric'] = 0.0
                self._df.at[delete_idx, 'flag'] = 'cannot_delete'
                self._was_modified = True
            self._num_verts_to_delete -= 1

    def _recreate_arc(self):
        redist_pts = [self._pts[0]]
        cur_pt = self._df.at[0, 'next_loc']
        # remove this first line so that if it is a loop, we don't try to use this point again
        while cur_pt and cur_pt != self._pts[-1]:
            cur_idx = self._df.index[self._df['loc'] == cur_pt][0]
            flag = self._df.at[cur_idx, 'flag']
            if flag == 'smooth':
                smoothed_pts = self._smooth_vertex(cur_idx)
                redist_pts.extend(smoothed_pts)
                self._was_modified = True
            else:
                if flag == 'cannot_delete':
                    self._msg = 'One or more points exceed the max delta but cannot be smoothed.'
                redist_pts.append(cur_pt)
            cur_pt = self._df.at[cur_idx, 'next_loc']
        redist_pts.append(self._pts[-1])

        # only set the arc if it was changed
        num_req_pts = 2 if not self._arc.is_ring else 3
        self._arc = LineString(redist_pts) if (len(redist_pts) >= num_req_pts and self._was_modified) else None
        if self._arc is None:
            no_impact_str = 'Action result: No impact'
            self._msg = f'{self._msg}\n{no_impact_str}' if self._msg is not None else no_impact_str

    def _smooth_vertex(self, idx):
        prev = self._df.at[idx, 'prev_loc']
        prev_len = self._df.at[idx, 'prev_len']
        next = self._df.at[idx, 'next_loc']
        next_len = self._df.at[idx, 'next_len']
        cur = self._df.at[idx, 'loc']
        delta = self._df.at[idx, 'delta']
        gamma = np.pi - delta
        gamma = gamma - (2.0 * np.pi) if gamma > np.pi else gamma
        gamma = gamma + (2.0 * np.pi) if gamma < -np.pi else gamma

        num_segs = np.floor(abs(delta) / (self._max_delta))
        radius = abs(self._min_seg_length / (2.0 * np.sin(delta / (2.0 * num_segs))))
        tl = radius * np.sin(delta / 2.0) / np.sin(gamma / 2.0)

        first = (cur[0] + tl * (prev[0] - cur[0]) / prev_len, cur[1] + tl * (prev[1] - cur[1]) / prev_len, 0.0)
        last = (cur[0] + tl * (next[0] - cur[0]) / next_len, cur[1] + tl * (next[1] - cur[1]) / next_len, 0.0)

        smooth_pts = [last]
        index = num_segs - 1
        while index > 0:
            t_i = index / num_segs
            x = np.square(1 - t_i) * first[0] + (2.0 * t_i * (1 - t_i)) * cur[0] + np.square(t_i) * last[0]
            y = np.square(1 - t_i) * first[1] + (2.0 * t_i * (1 - t_i)) * cur[1] + np.square(t_i) * last[1]
            smooth_pts.append((x, y, 0.0))
            index -= 1
        smooth_pts.append(first)
        return reversed(smooth_pts)

    def _update_verts(self, delete_idx):
        prev_loc = self._df.at[delete_idx, 'prev_loc']
        prev_idx = self._df.index[self._df['loc'] == prev_loc][0]
        prev_prev_angle = self._df.at[prev_idx, 'prev_angle']

        deleting_loc = self._df.at[delete_idx, 'loc']
        # get the "next_loc" by finding the loc that has the deleting loc as the prev_loc
        next_idx = self._df.index[self._df['prev_loc'] == deleting_loc][0]
        next_loc = self._df.at[next_idx, 'loc']
        next_next_angle = self._df.at[next_idx, 'next_angle']

        # angle of the new segment connecting prev and next
        new_angle = gsh.get_angle_of_segment(prev_loc, next_loc)

        # check the deltas to make sure we aren't making things worse
        prev_orig_delta = self._df.at[prev_idx, 'delta']
        prev_new_delta = new_angle - prev_prev_angle

        if prev_new_delta > prev_orig_delta:
            # deleting this will make things worse, so don't do it
            return False

        next_orig_delta = self._df.at[next_idx, 'delta']
        next_new_delta = next_next_angle - new_angle
        if next_new_delta > next_orig_delta:
            # deleting this will make things worse, so don't do it
            return False

        # length of the new segment connecting prev and next
        new_seg_length = gsh.distance_between_pts(prev_loc, next_loc)

        prev_prev_len = self._df.at[prev_idx, 'prev_len']
        prev_next_len = self._df.at[prev_idx, 'next_len']
        ave_len = (prev_prev_len + prev_next_len) / 2.0
        if new_seg_length > ave_len:
            return False

        next_prev_len = self._df.at[next_idx, 'prev_len']
        next_next_len = self._df.at[prev_idx, 'next_len']
        ave_len = (next_prev_len + next_next_len) / 2.0
        if new_seg_length > ave_len:
            return False

        for idx in [prev_idx, next_idx]:
            if idx == prev_idx:
                # values to reset on previous vertex
                self._df.at[prev_idx, 'next_loc'] = next_loc
                self._df.at[prev_idx, 'next_len'] = next_len = new_seg_length
                self._df.at[prev_idx, 'next_angle'] = next_angle = new_angle
                prev_angle = self._df.at[prev_idx, 'prev_angle']
                prev_len = self._df.at[prev_idx, 'prev_len']
            else:
                # reset values on next vertex
                self._df.at[next_idx, 'prev_loc'] = prev_loc
                self._df.at[next_idx, 'prev_len'] = prev_len = new_seg_length
                self._df.at[next_idx, 'prev_angle'] = prev_angle = new_angle
                next_angle = self._df.at[next_idx, 'next_angle']
                next_len = self._df.at[next_idx, 'next_len']

            delta = next_angle - prev_angle if (prev_angle and next_angle) else None
            if delta:
                delta = delta - (2.0 * np.pi) if delta > np.pi else delta
                delta = delta + (2.0 * np.pi) if delta < -np.pi else delta
            self._df.at[idx, 'delta'] = delta

            old_flag = self._df.at[idx, 'flag']
            new_flag = 'valid'
            new_delete_metric = 0.0
            if delta and abs(delta) > self._max_delta:
                new_flag, new_delete_metric = self._evaluate_for_deletion(prev_len, next_len, delta)
            if old_flag != 'delete' and new_flag == 'delete':
                self._num_verts_to_delete += 1
            elif old_flag == 'delete' and new_flag != 'delete':
                self._num_verts_to_delete -= 1
            self._df.at[idx, 'flag'] = new_flag
            self._df.at[idx, 'delete_metric'] = new_delete_metric
        return True

    def _evaluate_for_deletion(self, prev_len, next_len, delta):
        max_transition_length = min(prev_len, next_len) / 2.0
        gamma = np.pi - delta
        gamma = gamma - (2.0 * np.pi) if gamma > np.pi else gamma
        gamma = gamma + (2.0 * np.pi) if gamma < -np.pi else gamma
        max_available_radius = max_transition_length * np.sin(gamma / 2.0) / np.sin(delta / 2.0)
        max_available_arc_length = abs(delta * max_available_radius)
        num_segs = np.floor(abs(delta) / (self._max_delta))
        min_legal_arc_length = self._min_seg_length * num_segs

        # see if this vertex is currently a candidate for deletion
        flag = 'smooth'
        delete_metric = 0.0
        if min_legal_arc_length > max_available_arc_length:
            flag = 'delete'
            delete_metric = min_legal_arc_length - max_available_arc_length

        return flag, delete_metric


class IntersectPolygonsByTolerance():
    """Tool to clean geometry."""

    def __init__(self, is_geo, tolerance):
        """Initializes the class."""
        self._is_geographic = is_geo
        self._tol = tolerance
        self._prune_dist = tolerance

        self._pts_1 = []
        self._pts_2 = []

        self._num_pts_1 = 0
        self._num_pts_2 = 0

        self._polygon_perimeters = []
        self._perimeter = []

        self._all_conns_1 = dict()
        self._all_conns_2 = dict()

        self._changed_by_prune_1 = False
        self._changed_by_prune_2 = False

        self._holes_to_keep = []

        self._changed = False

    def _initialize_poly_pts(self, is_poly_1, poly, non_geo_poly):
        """Get initial arguments for tool.

        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        if is_poly_1:
            self._native_poly_1 = poly
            self._native_pts_1 = gsh.pts_from_shapely_poly(poly)[0]
            self._poly_1 = non_geo_poly if non_geo_poly else poly

            # orig points in non-geographic coordinates
            self._orig_pts_1 = gsh.pts_from_shapely_poly(non_geo_poly)[0]

            # first/last point is repeated
            self._native_pts_1.pop()
        else:
            self._native_poly_2 = poly
            self._native_pts_2 = gsh.pts_from_shapely_poly(poly)[0]
            self._poly_2 = non_geo_poly if non_geo_poly else poly

            # orig points in non-geographic coordinates
            self._orig_pts_2 = gsh.pts_from_shapely_poly(non_geo_poly)[0]

            # first/last point is repeated
            self._native_pts_2.pop()

    def do_intersect(self, poly_1, poly_2, non_geo_poly_1, non_geo_poly_2, is_union):
        """Get initial arguments for tool.

        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        if is_union:
            if poly_1.area < poly_2.area:
                poly_1, poly_2 = poly_2, poly_1
                non_geo_poly_1, non_geo_poly_2 = non_geo_poly_2, non_geo_poly_1

        self._initialize_poly_pts(True, poly_1, non_geo_poly_1)
        self._initialize_poly_pts(False, poly_2, non_geo_poly_2)

        if not self._is_within_tolerance(is_union):
            return None

        self._do_pruning(is_union)

        # do pre-processing stuff
        if not is_union:
            # if it is difference, make sure that pruning didn't make one poly not contain the other
            test_poly_1 = shapely.Polygon(self._pts_1)
            test_poly_2 = shapely.Polygon(self._pts_2)
            if not test_poly_1.contains(test_poly_2):
                self._second_time = True
                self._prune_dist = self._prune_dist / 2.0

                # try pruning by half
                self._initialize_poly_pts(True, poly_1, non_geo_poly_1)
                self._initialize_poly_pts(False, poly_2, non_geo_poly_2)
                self._do_pruning(is_union)

                test_poly_1 = shapely.Polygon(self._pts_1)
                test_poly_2 = shapely.Polygon(self._pts_2)
                if not test_poly_1.contains(test_poly_2):
                    # do no pruning
                    self._prune_dist = 0.0
                    self._do_pruning(is_union)

        self._find_connections()

        # do we have something to work with?
        if len(self._all_conns_1) < 2 or len(self._all_conns_2) < 2:
            return None

        self._create_perimeter(is_union)

        if is_union:
            # all holes are kept, keep it simple by just copying them over - it is not simple for difference
            for poly in [self._native_poly_1, self._native_poly_2]:
                for hole in poly.interiors:
                    self._holes_to_keep.append([(p[0], p[1]) for p in hole.coords])

        return self._create_polygon()

    def _is_within_tolerance(self, is_union):
        """Get initial arguments for tool.

        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        # is any of it within tolerance?
        if is_union:
            dist = self._poly_1.distance(self._poly_2)
        else:
            # is any of it within tolerance? we are checking on outer perimeters
            outer_1 = LineString(self._orig_pts_1[:-1])
            outer_2 = LineString(self._orig_pts_2[:-1])
            nearest = nearest_points(outer_1, outer_2)
            dist = nearest[0].distance(nearest[1])

        self._prune_dist = min(dist, self._tol)
        return dist <= self._tol

    def _do_pruning(self, is_union):
        """Get initial arguments for tool.

        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        # prune first - prune to the right if it is union, prune to the left if by difference (because we are connecting
        # it from the inside) for the bigger polygon only - the smaller polygon is treated the same

        ls_poly_1 = LineString(self._orig_pts_1)
        ls_poly_2 = LineString(self._orig_pts_2)
        self._pts_1 = self._prune_in_tol_subset(ls_poly_2, self._orig_pts_1, self._tol, not is_union)
        self._pts_2 = self._prune_in_tol_subset(ls_poly_1, self._orig_pts_2, self._tol, False)

        self._orig_pts_1.pop()
        self._pts_1.pop()
        self._changed_by_prune_1 = len(self._orig_pts_1) != len(self._pts_1)

        self._orig_pts_2.pop()
        self._pts_2.pop()
        self._changed_by_prune_2 = len(self._orig_pts_2) != len(self._pts_2)

        self._num_pts_1 = len(self._pts_1)
        self._num_pts_2 = len(self._pts_2)

    def _prune_in_tol_subset(self, ls_b, pts_a, tol, prune_left):
        """Get initial arguments for tool.

        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        sh_pts = [gsh.loc_to_shapely_pt(loc) for loc in pts_a]

        subset, start, end = self._find_subset_to_prune(ls_b, sh_pts, tol)
        subset_ls = LineString(subset)

        pruner = PruneArc(subset_ls, subset_ls, self._prune_dist, prune_left, 'mitre', 5, 'None', 100.0)
        pruned_ls = pruner.do_prune()[0]
        if not pruned_ls or not pruned_ls.is_valid:
            pruner = PruneArc(subset_ls, subset_ls, self._prune_dist, prune_left, 'mitre', 10, 'None', 100.0)
            pruned_ls = pruner.do_prune()[0]
            if not pruned_ls or not pruned_ls.is_valid:
                # don't prune
                return sh_pts

        pruned_pts = gsh.shapely_pts_from_ls(pruned_ls)
        if len(pruned_pts) == len(subset):
            # nothing changed with the prune
            return sh_pts

        if start or end:
            if end < start:
                pruned_pts.extend(sh_pts[end:start + 1])
            else:
                pruned_pts.extend(sh_pts[end:])
                pruned_pts.extend(sh_pts[:start + 1])

        # make sure that we didn't cut off the wrong part of the perimeter
        pruned_subset = gsh.shapely_pts_from_ls(pruned_ls)
        # removed_pts = [pt for pt in subset if pt not in pruned_subset]
        removed_pts = list(set(subset) - set(pruned_subset))

        pruned_poly = Polygon(pruned_pts)
        if not pruned_poly.is_valid:
            return sh_pts

        for removed_pt in removed_pts:
            dist_from_poly = pruned_poly.distance(removed_pt)
            if (not prune_left and dist_from_poly > 0.0) or (prune_left and dist_from_poly < 0.0):
                return sh_pts  # this was an invalid prune because it cut off points on the wrong side

        return pruned_pts

    def _find_subset_to_prune(self, poly_b, sh_pts_a, tol):
        """Get initial arguments for tool.

        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        if len(sh_pts_a) < 500:
            return sh_pts_a, None, None

        dists_dict = {pt.distance(poly_b): idx for idx, pt in enumerate(sh_pts_a[:-1])}
        dist_keys = list(dists_dict.keys())
        idx_dict = dict((idx, dist) for dist, idx in dists_dict.items())
        farthest_idx = dists_dict[max(dist_keys)]
        start_subset_idx = None

        cur_idx = farthest_idx + 1 if farthest_idx < len(sh_pts_a) - 2 else 0
        found = False
        while not found and cur_idx != farthest_idx:
            if idx_dict[cur_idx] <= tol:
                start_subset_idx = cur_idx
                found = True
            else:
                cur_idx = cur_idx + 1 if cur_idx < len(sh_pts_a) - 2 else 0

        if start_subset_idx is None:
            return sh_pts_a, None, None

        cur_idx = farthest_idx - 1 if farthest_idx > 0 else len(sh_pts_a) - 2
        found = False
        end_subset_idx = None
        while not found and cur_idx != farthest_idx:
            if idx_dict[cur_idx] <= tol:
                end_subset_idx = cur_idx
                found = True
            else:
                cur_idx = cur_idx - 1 if cur_idx > 0 else len(sh_pts_a) - 2

        if end_subset_idx is None or end_subset_idx == start_subset_idx:
            return sh_pts_a, None, None

        if start_subset_idx < end_subset_idx:
            subset = [sh_pts_a[idx] for idx in range(start_subset_idx, end_subset_idx + 1)]
        else:
            subset = [sh_pts_a[idx] for idx in range(start_subset_idx, len(sh_pts_a))]
            subset.extend([sh_pts_a[idx] for idx in range(0, end_subset_idx + 1)])

        return subset, start_subset_idx, end_subset_idx

    def _find_connections(self):
        """Get initial arguments for tool.

        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        self._pt_1_dists = {pt.distance(self._poly_2): idx for idx, pt in enumerate(self._pts_1)}
        dists = list(self._pt_1_dists.keys())
        self._farthest_pt_1 = self._pt_1_dists[max(dists)]
        self._closest_pt_1 = self._pt_1_dists[min(dists)]

        pts_in_tol_1 = [self._pt_1_dists[dist] for dist in dists if dist <= self._tol]
        ls_poly_1 = LineString(gsh.perim_pts_from_sh_poly(self._poly_1))
        pts_in_tol_2 = [idx for idx, pt in enumerate(self._pts_2) if self._valid_distance(pt, ls_poly_1)]

        if len(pts_in_tol_1) == 0 or len(pts_in_tol_2) == 0:
            return

        for pt_1 in pts_in_tol_1:
            valid_connections = \
                [pt_2 for pt_2 in pts_in_tol_2 if self._valid_intersection(self._pts_1[pt_1], self._pts_2[pt_2])]
            if len(valid_connections) > 0:
                self._all_conns_1[pt_1] = valid_connections

        if len(self._all_conns_1) > 0:
            # we have all connections that are within tolerance - now sort through for what can be used to connect the
            # polys check from the poly 1 side first, and populate connections_from_poly_2
            pt_1_candidates = list(self._all_conns_1.keys())
            pt_1_candidates.sort()
            for pt_1_candidate in pt_1_candidates:
                next_pt_idx = pt_1_candidate + 1 if pt_1_candidate < self._num_pts_1 - 1 else 0
                prev_pt_idx = pt_1_candidate - 1 if pt_1_candidate > 0 else self._num_pts_1 - 1

                # if there aren't adjacent points that are within tolerance, we can't do anything
                if next_pt_idx not in pt_1_candidates and prev_pt_idx not in pt_1_candidates:
                    del self._all_conns_1[pt_1_candidate]
                    continue

                # make a dictionary for the second polygon
                pt_2_candidates = self._all_conns_1[pt_1_candidate]
                for pt_2_candidate in pt_2_candidates:
                    if pt_2_candidate not in self._all_conns_2.keys():
                        self._all_conns_2[pt_2_candidate] = [pt_1_candidate]
                    else:
                        self._all_conns_2[pt_2_candidate].append(pt_1_candidate)

            if len(self._all_conns_1) <= 1 or len(self._all_conns_2) <= 1:
                return

            # both dictionaries have been populated - now make sure that only connections with adjacent points are kept
            self._remove_all_lone_connection_points()

            self._all_pts_in_tol_1 = len(self._all_conns_1.keys()) == len(self._pts_1)
            self._all_pts_in_tol_2 = len(self._all_conns_2.keys()) == len(self._pts_2)

    def _create_perimeter(self, is_union):
        """Get initial arguments for tool.

        Args:
            is_union (bool): True if union, False if difference
        Returns:
               (list): A list of the merged polygons, None if merge did not work
        """
        connection_pts_1 = list(self._all_conns_1.keys())
        # connection_pts_2 = list(self._all_conns_2.keys())  # useful for debugging

        # find where we are going to connect - start with the closest point
        dists = list(self._pt_1_dists.keys())
        found = False
        poly_1_to_poly_2 = None
        poly_1_from_poly_2 = None
        poly_2_from_poly_1 = None
        poly_2_to_poly_1 = None
        while found is False:
            min_dist = min(dists)
            closest_pt_1 = self._pt_1_dists[min_dist]
            if closest_pt_1 not in connection_pts_1:
                dists.remove(min_dist)
                continue

            # is an adjacent pt a valid connection?
            poly_1_to_poly_2 = closest_pt_1
            poly_1_from_poly_2 = closest_pt_1
            pt_before = closest_pt_1 - 1 if closest_pt_1 > 0 else self._num_pts_1 - 1
            while pt_before in connection_pts_1 and pt_before != closest_pt_1:
                poly_1_to_poly_2 = pt_before
                pt_before = pt_before - 1 if pt_before > 0 else self._num_pts_1 - 1

            pt_after = closest_pt_1 + 1 if closest_pt_1 < self._num_pts_1 - 1 else 0
            while pt_after in connection_pts_1 and pt_after != closest_pt_1:
                poly_1_from_poly_2 = pt_after
                pt_after = pt_after + 1 if pt_after < self._num_pts_1 - 1 else 0

            # if poly_1_to_poly_2 == poly_1_from_poly_2:
            #     dists.remove(min_dist)
            #     continue

            conns = self._all_conns_1[poly_1_to_poly_2]
            if is_union:
                poly_2_from_poly_1 = conns[-1]
                pt_after = poly_2_from_poly_1 + 1 if poly_2_from_poly_1 < self._num_pts_2 - 1 else 0
                while pt_after in conns and pt_after != conns[-1]:
                    poly_2_from_poly_1 = pt_after
                    pt_after = pt_after + 1 if pt_after < self._num_pts_2 - 1 else 0
            else:
                poly_2_from_poly_1 = conns[0]
                pt_before = poly_2_from_poly_1 - 1 if poly_2_from_poly_1 > 0 else self._num_pts_2 - 1
                while pt_before in conns and pt_before != conns[0]:
                    poly_2_from_poly_1 = pt_before
                    pt_before = pt_before - 1 if pt_before > 0 else self._num_pts_2 - 1

            conns = self._all_conns_1[poly_1_from_poly_2]
            if is_union:
                poly_2_to_poly_1 = conns[0]
                pt_before = poly_2_to_poly_1 - 1 if poly_2_to_poly_1 > 0 else self._num_pts_2 - 1
                while pt_before in conns and pt_before != conns[0]:
                    poly_2_to_poly_1 = pt_before
                    pt_before = pt_before - 1 if pt_before > 0 else self._num_pts_2 - 1
            else:
                poly_2_to_poly_1 = conns[-1]
                pt_after = poly_2_to_poly_1 + 1 if poly_2_to_poly_1 < self._num_pts_2 - 1 else 0
                while pt_after in conns and pt_after != conns[-1]:
                    poly_2_to_poly_1 = pt_after
                    pt_after = pt_after + 1 if pt_after < self._num_pts_2 - 1 else 0

            found = True
            if poly_2_to_poly_1 == poly_2_from_poly_1:
                if is_union:
                    opts = self._all_conns_1[poly_1_from_poly_2]
                    poly_2_to_poly_1 = opts[-2] if len(opts) > 1 else poly_2_to_poly_1

                    # are they still the same?
                    if poly_2_to_poly_1 == poly_2_from_poly_1:
                        opts = self._all_conns_1[poly_1_to_poly_2]
                        poly_2_from_poly_1 = opts[1] if len(opts) > 1 else poly_2_from_poly_1

                    # valid if they aren't the same now
                    found = poly_2_to_poly_1 != poly_2_from_poly_1 or self._all_pts_in_tol_2
                else:
                    opts = self._all_conns_1[poly_1_to_poly_2]
                    poly_2_from_poly_1 = opts[-2] if len(opts) > 1 else poly_2_from_poly_1

                    # are they still the same?
                    if poly_2_to_poly_1 == poly_2_from_poly_1:
                        opts = self._all_conns_1[poly_1_from_poly_2]
                        poly_2_to_poly_1 = opts[1] if len(opts) > 1 else poly_2_to_poly_1

                    # valid if they aren't the same now
                    found = poly_2_to_poly_1 != poly_1_from_poly_2 or self._all_pts_in_tol_2

            if found is False:
                self._changed = False
                return

        # start from the farthest point on poly 1
        start_end_pt = self._farthest_pt_1

        # here for easier debugging
        used_pts_1 = []
        used_pts_2 = []

        # now add all points from the beginning to this connection
        start_loc = gsh.shapely_pt_to_loc(self._pts_1[start_end_pt])
        start = self._orig_pts_1.index(start_loc)
        end_loc = gsh.shapely_pt_to_loc(self._pts_1[poly_1_to_poly_2])
        end = self._orig_pts_1.index(end_loc)

        if start <= end:
            self._perimeter = [self._native_pts_1[idx] for idx in range(start, end + 1)]
            used_pts_1 = [idx for idx in range(start, end + 1)]
        else:
            self._perimeter = [self._native_pts_1[idx] for idx in range(start, len(self._native_pts_1))]
            used_pts_1 = [idx for idx in range(start, len(self._native_pts_1))]
            self._perimeter.extend([self._native_pts_1[idx] for idx in range(end + 1)])
            used_pts_1.extend([idx for idx in range(end + 1)])

        if poly_2_from_poly_1 == poly_2_to_poly_1 and self._all_pts_in_tol_2:
            start_loc = gsh.shapely_pt_to_loc(self._pts_2[poly_2_from_poly_1])
            start = self._orig_pts_2.index(start_loc)
            self._perimeter.extend([self._native_pts_2[start]])
            used_pts_2 = [start]
        else:
            # now get the points for poly 2
            start_poly_2 = poly_2_from_poly_1

            # get the start/end points for this section
            start_loc = gsh.shapely_pt_to_loc(self._pts_2[start_poly_2])
            start = self._orig_pts_2.index(start_loc)
            end_loc = gsh.shapely_pt_to_loc(self._pts_2[poly_2_to_poly_1])
            end = self._orig_pts_2.index(end_loc)

            if is_union:
                # if this is a union perimeter, we go backwards along this poly
                if start < end:
                    self._perimeter.extend([self._native_pts_2[idx] for idx in range(start, end + 1)])
                    used_pts_2 = [idx for idx in range(start, end + 1)]
                else:
                    self._perimeter.extend([self._native_pts_2[idx] for idx in range(start, len(self._native_pts_2))])
                    used_pts_2 = [idx for idx in range(start, len(self._native_pts_2))]
                    self._perimeter.extend([self._native_pts_2[idx] for idx in range(end + 1)])
                    used_pts_2.extend([idx for idx in range(end + 1)])
            else:
                # if this is a difference perimeter, we go forwards along this poly
                if start >= end:
                    self._perimeter.extend([self._native_pts_2[idx] for idx in range(start, end - 1, -1)])
                    used_pts_2 = [idx for idx in range(start, end - 1, -1)]
                else:
                    self._perimeter.extend([self._native_pts_2[idx] for idx in range(start, -1, -1)])
                    used_pts_2 = [idx for idx in range(start, -1, -1)]
                    self._perimeter.extend(
                        [self._native_pts_2[idx] for idx in range(len(self._native_pts_2) - 1, end - 1, -1)])
                    used_pts_2.extend([idx for idx in range(len(self._native_pts_2) - 1, end - 1, -1)])

        # now go back to where we started
        # get the start/end points for this section
        start_loc = gsh.shapely_pt_to_loc(self._pts_1[poly_1_from_poly_2])
        start = self._orig_pts_1.index(start_loc)
        end_loc = gsh.shapely_pt_to_loc(self._pts_1[start_end_pt])
        end = self._orig_pts_1.index(end_loc)

        if start <= end:
            self._perimeter.extend([self._native_pts_1[idx] for idx in range(start, end + 1)])
            used_pts_1.extend([idx for idx in range(start, end + 1)])
        else:
            self._perimeter.extend([self._native_pts_1[idx] for idx in range(start, len(self._native_pts_1))])
            used_pts_1.extend([idx for idx in range(start, len(self._native_pts_1))])
            self._perimeter.extend([self._native_pts_1[idx] for idx in range(end + 1)])
            used_pts_1.extend([idx for idx in range(end + 1)])

        self._changed = True

    def _create_polygon(self):
        """Get initial arguments for tool.

        Returns:
               (Polygon): The merged polygon
        """
        return Polygon(self._perimeter, self._holes_to_keep) if self._changed else None

    def _valid_intersection(self, pt_1, pt_2):
        """Get initial arguments for tool.

        Returns:
               (bool): Does this connection overlap an original polygon, or is it longer than the tolerance
        """
        distance = pt_1.distance(pt_2)
        if distance > self._tol or distance == 0.0:
            return False

        # we need to check if it intersects the unpruned poly
        index = self._orig_pts_1.index([pt_1.x, pt_1.y, 0.0])
        native_pt_1 = self._native_pts_1[index]

        index = self._orig_pts_2.index([pt_2.x, pt_2.y, 0.0])
        native_pt_2 = self._native_pts_2[index]

        ls_connection = LineString([native_pt_1, native_pt_2])
        if ls_connection.crosses(self._native_poly_1) or ls_connection.crosses(self._native_poly_2):
            return False

        return True

    def _valid_distance(self, pt, poly):
        """Get initial arguments for tool.

        Returns:
               (bool): Does this connection overlap an original polygon, or is it longer than the tolerance
        """
        distance = pt.distance(poly)
        return distance <= self._tol

    def _remove_all_lone_connection_points(self):
        """Get initial arguments for tool."""
        changed_1 = True
        changed_2 = True

        while changed_1 or changed_2:
            changed_1 = self._remove_lone_connection_points(self._all_conns_1, self._all_conns_2, self._num_pts_1)
            changed_2 = self._remove_lone_connection_points(self._all_conns_2, self._all_conns_1, self._num_pts_2)

    def _remove_lone_connection_points(self, dict_a, dict_b, num_pts_a):
        """Get initial arguments for tool.

        Arguments:
            dict_a (dict): Dictionary of polygon we are checking
            dict_b (dict): Dictionary of the other polygon so that we can update when things are deleted from dict_1
            num_pts_a (int): Number of points in set used for dict_a
        Return:
            changed (bool): True if points were deleted
        """
        changed = False
        candidates = list(dict_a.keys())
        for candidate in candidates:
            next_pt_idx = candidate + 1 if candidate < num_pts_a - 1 else 0
            prev_pt_idx = candidate - 1 if candidate > 0 else num_pts_a - 1

            if next_pt_idx not in candidates and prev_pt_idx not in candidates:
                changed = True
                # no adjacent valid points for a connection, so delete this from BOTH dicts
                # get everything that this point connected to on the opposite polygon
                b_idxs = dict_a[candidate]

                # delete from the opposite polygon and connections using this candidate
                for idx in b_idxs:
                    dict_b[idx].remove(candidate)
                    if len(dict_b[idx]) == 0:
                        del dict_b[idx]

                # delete from this dictionary
                del dict_a[candidate]

        return changed
