"""MergeGridsTool class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.constraint import read_grid_from_file
from xms.tool_core import IoDirection, Tool

# 4. Local modules


ARG_INPUT_MESH_TO_RELOAD = 0
ARG_INPUT_MESH_TIMESTAMPS = 1
ARG_INPUT_MESH_NAME = 2
ARG_OUTPUT_RELOADED_MESH = 3


class MeshReloadTool(Tool):
    """Turn on/off process to create mesh backups."""

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Reload from auto mesh backup')

        self._query = Query()
        self._mesh_name_path = {}
        self._mesh_uuid_ts = {}
        self._mesh_ts_filename = {}

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (:obj:`list`): A list of the initial tool arguments.
        """
        available_meshes = self._get_available_items_for_reload('mesh_backup')
        sel_mesh = available_meshes[0]

        arguments = [
            self.string_argument(name='mesh_to_reload', description='Mesh to reload', optional=False,
                                 choices=available_meshes, value=sel_mesh),
            self.string_argument(name='mesh_timestamp', description='Timestamp',
                                 choices=[''], value=''),
            self.string_argument(name='reloaded_mesh_name', description='Reloaded mesh name', value='reloaded mesh'),
            self.grid_argument(name='reloaded_mesh', description='The mesh', hide=True, optional=True,
                               io_direction=IoDirection.OUTPUT)
        ]

        return arguments

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(:obj:`list`): The tool arguments.
        """
        selected_mesh = arguments[ARG_INPUT_MESH_TO_RELOAD].text_value
        if selected_mesh != '' and selected_mesh in self._mesh_name_path.keys():
            arguments[ARG_INPUT_MESH_TIMESTAMPS].hide = False
            arguments[ARG_INPUT_MESH_NAME].hide = False
            backup_dir = self._mesh_name_path[selected_mesh]
            timestamps = self._get_item_timestamps(backup_dir)
            if arguments[ARG_INPUT_MESH_TIMESTAMPS].choices != timestamps:
                arguments[ARG_INPUT_MESH_TIMESTAMPS].choices = timestamps
                if len(timestamps) > 0:
                    arguments[ARG_INPUT_MESH_TIMESTAMPS].value = timestamps[0]
        else:
            arguments[ARG_INPUT_MESH_TIMESTAMPS].hide = True
            arguments[ARG_INPUT_MESH_NAME].hide = True

    def _get_available_items_for_reload(self, type):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            type (:obj:`str`): data type

        Returns:
            available_items (:obj:`list`): list of items available for reload
        """
        available = []
        backup_folder = os.path.join(XmEnv.xms_environ_temp_directory(), type)
        if os.path.isdir(backup_folder):
            for child in os.scandir(backup_folder):
                if child.is_dir():
                    name_file = os.path.join(child.path, 'name.txt')
                    with open(name_file) as f:
                        name = f.readline().strip()
                        available.append(name)
                        self._mesh_name_path[name] = child.path
        else:
            available.append('No available meshes.')
        return available

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (:obj:`list`): The tool arguments.

        Returns:
            (:obj:`dict`): Dictionary of errors for arguments.
        """
        errors = {}
        if arguments[ARG_INPUT_MESH_TO_RELOAD].text_value == 'No available meshes.':
            errors[arguments[ARG_INPUT_MESH_TO_RELOAD].name] = 'No available backups to reload.'
        if arguments[ARG_INPUT_MESH_TIMESTAMPS].text_value == 'No backups available for this mesh.':
            errors[arguments[ARG_INPUT_MESH_TIMESTAMPS].name] = 'No backups available for this mesh.'
        return errors

    def _get_item_timestamps(self, directory):
        """Override to run the tool.

        Args:
            directory (:obj:`str`): directory to find the backup files
            item_path (:obj:`str`): project path for the item
        Return:
            choices (:obj:`list`): list of timestamp strings
        """
        choices = []
        # see if we have backup files
        if os.path.isdir(directory) is True:
            ts_dt_dict = dict()
            for file in os.scandir(directory):
                if file.name != 'name.txt':
                    ts = file.stat(follow_symlinks=False).st_ctime
                    dt = datetime.datetime.fromtimestamp(ts).strftime("%m/%d/%Y, %H:%M:%S")
                    ts_dt_dict[ts] = dt
                    self._mesh_ts_filename[dt] = file.path

            # put these in order
            sorted_times = sorted(ts_dt_dict.keys(), reverse=True)
            for ts in sorted_times:
                choices.append(ts_dt_dict[ts])

        if len(choices) == 0:
            choices.append('No backups available for this mesh.')

        return choices

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (:obj:`list`): The tool arguments.
        """
        selected_ts = arguments[ARG_INPUT_MESH_TIMESTAMPS].value
        file_to_reload = self._mesh_ts_filename[selected_ts]
        reloaded_mesh = read_grid_from_file(file_to_reload)
        reloaded_mesh.uuid = str(uuid.uuid4())
        name = os.path.basename(arguments[ARG_INPUT_MESH_NAME].value)
        self._data_handler.set_output_grid(reloaded_mesh, name, None, force_ugrid=False)
