"""SimComponent class."""
# 1. Standard python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.core.filesystem import filesystem as io_util
from xms.guipy.time_format import XmsTimeFormatter

# 4. Local modules
from xms.tuflowfv.components.tuflowfv_component import TuflowFvComponent
from xms.tuflowfv.data.sim_data import SimData
from xms.tuflowfv.gui.model_control_dialog import ModelControlDialog

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class SimComponent(TuflowFvComponent):
    """A hidden Dynamic Model Interface (DMI) component for the TUFLOWFV model simulation."""

    def __init__(self, main_file):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = SimData(self.main_file)
        self.class_name = 'SimComponent'
        self.module_name = 'xms.tuflowfv.components.sim_component'
        self.tree_commands = [
            ('Model Control...', 'open_model_control'),
        ]  # [(menu_text, menu_method)...]

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (dict): A dictionary with keys being UUIDs as strings representing the objects being linked into
                this component. The values of this dictionary are a list of strings of the parameter names of the
                "takes" from the XML that this is a part of.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        messages = []
        actions = []
        for link_uuid, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param in ['uGrid', 'uGridMesh']:
                    if self.data.info.attrs['domain_uuid']:  # Already have a linked 2DMesh or UGrid
                        params = {'old_uuid': self.data.info.attrs['domain_uuid']}
                        action = ActionRequest(
                            main_file=self.main_file, modality='NO_DIALOG', class_name=self.class_name,
                            module_name=self.module_name, method_name='delete_old_domain', comp_uuid=self.uuid,
                            parameters=params
                        )
                        actions.append(action)
                    self.data.info.attrs['domain_uuid'] = link_uuid
        self.data.commit()
        return messages, actions

    def unlink_event(self, unlinks, lock_state):
        """This will be called when a coverage, or a ugrid, or another component is unlinked from this component.

        Args:
            unlinks (list of str): A list of UUIDs as strings representing the objects being unlinked.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        if self.data.info.attrs['domain_uuid'] in unlinks:
            self.data.info.attrs['domain_uuid'] = ''
        self.data.commit()
        return [], []

    def open_model_control(self, query, params, win_cont):
        """Opens the Model Control dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS.
            params (list): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            tuple(list, list): First element of the tuple being the message, second is ActionRequests to XMS to perform
        """
        time_formatter = XmsTimeFormatter(query.global_time_settings)
        dlg = ModelControlDialog(parent=win_cont, data=self.data, projection=query.display_projection,
                                 tree_node=query.project_tree, sim_uuid=query.parent_item_uuid(),
                                 time_formats=(time_formatter.abs_specifier, time_formatter.qt_abs_specifier))
        if dlg.exec():
            self.data.commit()
        return [], []

    def delete_old_domain(self, query, params):
        """Delete the existing linked domain when another is linked to the simulation.

        Notes:
            This is needed because we want to allow using an SMS 2DMesh or UGrid object as the domain. The xml has both
            a take_mesh2d and take_ugrid parameter with a limit of 1. If we get a link event for the domain and we
            already have one, we need to remove the old because it is of a different type.

        Args:
            query (Query): Object for communicating with XMS.
            params (dict): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        old_uuid = None
        sim_uuid = None
        if params and params[0]:
            old_uuid = params[0].get('old_uuid')
            # Need the UUID of the parent tree item, even though it is the hidden component that actually has the XML
            # take parameter.
            sim_uuid = query.parent_item_uuid()
        if old_uuid and sim_uuid:
            query.unlink_item(taker_uuid=sim_uuid, taken_uuid=old_uuid)
        return [], []

    def copy_external_files(self, new_main_file):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Args:
            new_main_file (str): The location of the new component main file in the package.

        Returns:
            (str): Message on failure, empty string on success
        """
        new_data = SimData(new_main_file)
        new_data.copy_external_files()
        new_data.commit()
        return ''

    def update_proj_dir(self, new_main_file, convert_filepaths):
        """Called when saving a project for the first time or saving a project to a new location.

        Notes:
            All referenced filepaths should be converted to relative from the new project location. If the file path is
            already relative, it is relative to the old project directory. After updating file paths, update the project
            directory in the main file.

        Args:
            new_main_file (str): The location of the new main file.
            convert_filepaths (bool): False if only the project directory should be updated.

        Returns:
            (str): Message on failure, empty string on success
        """
        new_data = SimData(new_main_file)
        if not convert_filepaths:
            # This case is to handle opening a package project for the first time.
            comp_folder = os.path.dirname(self.main_file)
            package_proj_dir = os.path.normpath(os.path.join(comp_folder, '../../..'))
            new_data.info.attrs['proj_dir'] = package_proj_dir
            new_data.commit()  # Save the updated project directory
            return ''
        err_msg = new_data.update_proj_dir()
        # Copy the newly saved file to temp.
        io_util.copyfile(new_main_file, self.main_file)
        return err_msg
