"""BcData class."""
# 1. Standard python modules
import os

# 2. Third party modules
import h5py
import numpy as np
import pkg_resources
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as io_util

# 4. Local modules
from xms.tuflowfv.components.tuflowfv_component import UNINITIALIZED_COMP_ID
from xms.tuflowfv.data.tuflowfv_data import check_for_object_strings_dumb, fix_datetime_format
from xms.tuflowfv.gui import gui_util


# When mapping to a simulation, the new component a copy will be made of the source BC data file named this in the
# mapped component's directory.
BC_MAIN_FILE = 'comp.nc'


"""
Arc curve tables:
HQ - h vs q
Q - time vs q
QH - h vs q
QN - N/A
WL - time vs wl
WLS - time vs wl1, wl2
WL_CURT - time, chainage vs elevation, wl

Point curve tables:
QC - time vs q

Global curve tables:
QG - time vs q/a
"""
# BC type to default CSV column/NetCDF variable name lookups
ARC_BC_TYPES = {
    'HQ': ('H', 'Q'),
    'Q': ('Time', 'Q'),
    'QN': (),
    'WL': ('Time', 'WL'),
    'WLS': ('Time', 'WL_A', 'WL_B'),
    'WL_CURT': ('TIME', 'CHAINAGE', 'ELEVATION', 'WL'),
    'ZG': (),
}
POINT_BC_TYPES = {
    'QC': ('Time', 'Q'),
}
POLYGON_BC_TYPES = {
    'QC_POLY': ('Time', 'Q'),
}
GLOBAL_BC_TYPES = {  # Global BC stored on the simulation, not BC coverage
    'QG': ('Time', 'Q_over_A'),
}
GRIDDED_BC_TYPES = {  # Gridded BC stored on the simulation, not BC coverage
    'MSLP_GRID',
    'W10_GRID',
    'WAVE',
}
CURTAIN_BC_TYPES = {  # Stored with the arcs, but different enough we need to know which arc BCs are curtains
    'WL_CURT'
}
MSLP_BC_TYPES = {
    'WL',
    'WLS',
    'WL_CURT',
}
MAX_NUM_BC_FIELDS = 8  # Most have 1, some have 2, gridded wave has 8
BC_LOCATION_ARC = 0  # Includes curtain BCs, which are time-varying, spatially-varying (along an arc) NetCDF datasets
BC_LOCATION_POINT = 1
BC_LOCATION_GLOBAL = 2  # Time-varying, spatially constant global CSV curves
BC_LOCATION_GRID = 3  # Time-varying, spatially varying global NetCDF datasets
BC_LOCATION_POLY = 4  # QC_Poly added later
GLOBAL_BC_GROUP = 'global_bcs'  # Group path of global BCs, which are stored in a subgroup of the simulation data.
GRIDDED_BC_GROUP = 'gridded_bcs'  # Group path of gridded BCs, which are stored in a subgroup of the simulation data.


def is_x_time(bc_type):
    """Returns True if the x-column of the BC curve is time.

    Args:
        bc_type (str): The BC feature's TUFLOWFV type card

    Returns:
        bool: See description
    """
    # Could be an arc, point, or global BC.
    column_names = ARC_BC_TYPES.get(bc_type, POINT_BC_TYPES.get(bc_type, GLOBAL_BC_TYPES.get(bc_type)))
    if not column_names:
        return False
    # Should return False for curtain arc BCs even though x variable name is 'TIME' because we don't plot it.
    return column_names[0] == 'Time'


def get_default_bc_curve(bc_type):
    """Get an empty, default curve for a BC feature type.

    Args:
        bc_type (str): The BC feature's TUFLOWFV type card

    Returns:
        xr.Dataset: See description
    """
    if bc_type == 'WLS':
        return xr.Dataset(data_vars={
            'Time': xr.DataArray(data=np.array([], dtype=np.float64)),
            'WL_A': xr.DataArray(data=np.array([], dtype=np.float64)),
            'WL_B': xr.DataArray(data=np.array([], dtype=np.float64)),
        })
    else:
        x_col = 'Time' if bc_type != 'HQ' else 'H'
        if bc_type == 'WL':
            y_col = 'WL'
        elif bc_type == 'QG':
            y_col = 'Q_over_A'
        else:
            y_col = 'Q'
        return xr.Dataset(data_vars={
            x_col: xr.DataArray(data=np.array([], dtype=np.float64)),
            y_col: xr.DataArray(data=np.array([], dtype=np.float64)),
        })


def add_default_gridded_bc_data_dict(defaults, dtypes):
    """Add parameters to a default data dict for gridded BC types.

    Args:
        defaults (dict): The default data values dict for the parameters common to all BC locations. Will be updated.
        dtypes (list): The default data dtypes list for the parameters common to all BC locations. Will be updated.
    """
    defaults.update({
        'define_time_units': 0,  # Always defined for arc, point, and global BCs via the curve editor
        'grid_id': UNINITIALIZED_COMP_ID,
        'default3': 0.0,
        'default4': 0.0,
        'default5': 0.0,
        'default6': 0.0,
        'default7': 0.0,
        'default8': 0.0,
        'offset3': 0.0,
        'offset4': 0.0,
        'offset5': 0.0,
        'offset6': 0.0,
        'offset7': 0.0,
        'offset8': 0.0,
        'scale3': 1.0,
        'scale4': 1.0,
        'scale5': 1.0,
        'scale6': 1.0,
        'scale7': 1.0,
        'scale8': 1.0,
        # NetCDF variable names (one per 8 possible fields plus 1 for time)
        'variable5': gui_util.DEFAULT_VALUE_VARIABLE,
        'variable6': gui_util.DEFAULT_VALUE_VARIABLE,
        'variable7': gui_util.DEFAULT_VALUE_VARIABLE,
        'variable8': gui_util.DEFAULT_VALUE_VARIABLE,
        'variable9': gui_util.DEFAULT_VALUE_VARIABLE,
    })
    dtypes.extend([
        np.int32,
        np.int32,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        np.float64,
        object,
        object,
        object,
        object,
        object,
    ])


def get_default_bc_data_dict(gridded):
    """Returns an default data dict and the dtypes for a BC.

    Args:
        gridded (bool): True if a gridded BC, False if point, arc, or global. Adds several parameters that are only
            applicable to gridded BC types.

    Returns:
        tuple(dict, set): The default data and their dtypes
    """
    defaults = {
        'type': 'Monitor',
        'subtype': 1,
        'name': '',
        'friction_slope': 0.001,
        'define_default': 0,
        'default1': 0.0,  # For the majority of BCs, maximum of two fields
        'default2': 0.0,
        'define_offset': 0,
        'offset1': 0.0,
        'offset2': 0.0,
        'define_scale': 0,
        'scale1': 1.0,
        'scale2': 1.0,
        'define_update_dt': 0,
        'update_dt': 0.0,
        'time_units': 'Hours',
        'define_reference_time': 0,
        'use_isodate': 0,
        'reference_time_hours': 0.0,
        'reference_time_date': gui_util.get_tuflow_zero_time(False),
        'include_mslp': 1,
        'define_vertical_distribution': 0,
        'vertical_distribution_type': 'Elevation',
        'vertical_distribution_file': '',
        # Rest only used by gridded and curtain BCs
        'grid_dataset_file': '',
        'variable1': gui_util.DEFAULT_VALUE_VARIABLE,
        'variable2': gui_util.DEFAULT_VALUE_VARIABLE,
        'variable3': gui_util.DEFAULT_VALUE_VARIABLE,
        'variable4': gui_util.DEFAULT_VALUE_VARIABLE,
    }
    dtypes = [
        object,
        np.int32,
        object,
        np.float64,
        np.int32,
        np.float64,
        np.float64,
        np.int32,
        np.float64,
        np.float64,
        np.int32,
        np.float64,
        np.float64,
        np.int32,
        np.float64,
        object,
        np.int32,
        np.int32,
        np.float64,
        object,
        np.int32,
        np.int32,
        object,
        object,
        object,
        object,
        object,
        object,
        object,
    ]
    if gridded:
        add_default_gridded_bc_data_dict(defaults, dtypes)
    return defaults, dtypes


def get_default_bc_data(fill, gridded):
    """Returns an default data dict for a BC point or arc.

    Args:
        fill (bool): True if Dataset should be initialized with an default row, False if it should be empty
        gridded (bool): True if a gridded BC, False if point, arc, or global. Adds several parameters that are only
            applicable to gridded BC types.

    Returns:
        dict: The default data
    """
    defaults, dtypes = get_default_bc_data_dict(gridded)
    return {
        variable: ('comp_id', np.array([value] if fill else [], dtype=dtype))
        for dtype, (variable, value) in zip(dtypes, defaults.items())
    }


BC_STRING_VARIABLES = {  # Variables that have a str dtype, have to make fixed size before serializing to NetCDF
    'type',
    'name',
    'time_units',
    'reference_time_date',
    'vertical_distribution_type',
    'vertical_distribution_file',
    'grid_dataset_file',
    'variable1',
    'variable2',
    'variable3',
    'variable4',
    'variable5',
    'variable6',
    'variable7',
    'variable8',
    'variable9',
}


class BcData(XarrayBase):
    """Manages data file for the boundary conditions coverage hidden component."""

    def __init__(self, data_file, group_path=''):
        """Initializes the data class.

        Args:
            data_file (str): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
            group_path (str): Name of the group in the main file to save to
        """
        # Initialize member variables before calling super so they are available for commit() call
        self._filename = data_file
        self._group_path = group_path
        self._info = None
        self._globals = None
        self._bcs = None
        self._points = None
        self._polygons = None
        self._curves = {}  # {comp_id: {bc_type: xr.Dataset}}
        # Create the default file before calling super because we have our own attributes to write.
        self._get_default_datasets(data_file)
        super().__init__(data_file)
        self._info = None  # Reread this from the file because our info may be in a subgroup
        _ = self.info

    def _get_default_datasets(self, data_file):
        """Create default datasets if needed.

        Args:
            data_file (str): Name of the data file. If it doesn't exist, it will be created.
        """
        exists = os.path.isfile(data_file)
        if exists and self._group_path:  # Need to check if the subgroup exists not just the file.
            with h5py.File(data_file, 'r') as f:
                if self._group_path not in f:
                    exists = False

        if not exists:
            info_attrs = {
                'FILE_TYPE': 'TUFLOWFV_BC',
                'VERSION': pkg_resources.get_distribution('xmstuflowfv').version,
                'display_uuid': '',
                'point_display_uuid': '',
                'poly_display_uuid': '',
                'cov_uuid': '',
                'next_comp_id': 0,
                'proj_dir': '',
            }
            self._info = xr.Dataset(attrs=info_attrs)

            global_attrs = {
                'export_format': '2dm',
            }
            self._globals = xr.Dataset(attrs=global_attrs)

            bc_table = get_default_bc_data(fill=False, gridded=self._group_path == GRIDDED_BC_GROUP)
            coords = {'comp_id': np.array([], np.int64)}
            self._bcs = xr.Dataset(data_vars=bc_table, coords=coords)

            points_table = get_default_bc_data(fill=False, gridded=False)
            self._points = xr.Dataset(data_vars=points_table, coords=coords)

            polygon_table = get_default_bc_data(fill=False, gridded=False)
            self._polygons = xr.Dataset(data_vars=polygon_table, coords=coords)
            self.commit()
        else:
            self._migrate_data()

    def _migrate_data(self):
        """Check for missing data when opening an existing file."""
        if self.bcs is not None:
            for i in range(self.bcs.sizes['comp_id']):
                self.bcs.reference_time_date[i] = fix_datetime_format(self.bcs.reference_time_date[i].item())
            if 'name' not in self.bcs:
                self.bcs['name'] = (['comp_id'], np.full(self.bcs.sizes['comp_id'], '', dtype=object))
        if self.points is not None:
            for i in range(self.points.sizes['comp_id']):
                self.points.reference_time_date[i] = fix_datetime_format(self.points.reference_time_date[i].item())
            if 'name' not in self.points:
                self.points['name'] = (['comp_id'], np.full(self.points.sizes['comp_id'], '', dtype=object))

    def _get_bc_curve_from_file(self, comp_id, bc_type, default):
        """Get the times series curve for a BC point(s)/arc(s).

        Args:
            comp_id (int): Component id of the BC feature to retrieve curve for
            bc_type (str): The TUFLOWFV BC type card of the feature
            default (bool): If True, create a default empty curve if the curve does not exist for the given component
                id and BC type

        Returns:
            Union[Dataset, None]: The BC curve xarray Dataset or None if not found.
        """
        group_name = f'{self._group_path}/' if self._group_path else ''
        dset_path = f'{group_name}bc_curves/{comp_id}/{bc_type}'
        bc_curve = self.get_dataset(dset_path, False)
        if default and bc_curve is None:
            bc_curve = get_default_bc_curve(bc_type)
        return bc_curve

    def _copy_curves(self, from_comp_id, to_comp_id, bc_location):
        """Copy the BC curves from one feature to another.

        Args:
            from_comp_id (int): Component id of the BC feature to copy curves from
            to_comp_id (int): Component id of the BC feature to copy curves to
            bc_location (int): Location of the BC, one of BC_LOCATION_* constants
        """
        if bc_location == BC_LOCATION_POINT:
            bc_types = POINT_BC_TYPES
        elif bc_location == BC_LOCATION_ARC:
            bc_types = ARC_BC_TYPES
        elif bc_location == BC_LOCATION_POLY:
            bc_types = POLYGON_BC_TYPES
        else:
            bc_types = GLOBAL_BC_TYPES
        bc_curves = self._curves.get(from_comp_id, {})
        for bc_type in bc_types:
            bc_curve = bc_curves.get(bc_type)
            if bc_curve is None:
                bc_curve = self._get_bc_curve_from_file(from_comp_id, bc_type, False)
            if bc_curve is not None:
                self._curves.setdefault(to_comp_id, {})[bc_type] = bc_curve.copy(deep=True)

    @staticmethod
    def _get_new_bc_atts(comp_id, gridded):
        """Get a new dataset with default attributes for a BC arc.

        Args:
            comp_id (int): The unique XMS component id of the BC arc. If UNINITIALIZED_COMP_ID, a new one is
                generated.
            gridded (bool): True if a gridded BC, False if point, arc, or global. Adds several parameters that are only
                applicable to gridded BC types.

        Returns:
            (xarray.Dataset): A new default dataset for a BC arc. Can later be concatenated to persistent dataset.
        """
        bc_table = get_default_bc_data(fill=True, gridded=gridded)
        coords = {'comp_id': [comp_id]}
        ds = xr.Dataset(data_vars=bc_table, coords=coords)
        return ds

    @property
    def info(self):
        """Return the info dataset.

        Need to override because we may be in a subgroup
        Returns:
            xarray.Dataset: Dict of parameters
        """
        if self._info is None:
            group_name = f'{self._group_path}/' if self._group_path else ''
            self._info = self.get_dataset(f'{group_name}info', False)
        return self._info

    @property
    def globals(self):
        """Load the globals dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the global dataset in the main file

        """
        if self._globals is None:
            group_name = f'{self._group_path}/' if self._group_path else ''
            self._globals = self.get_dataset(f'{group_name}globals', False)
        return self._globals

    @globals.setter
    def globals(self, dset):
        """Setter for the globals dataset."""
        if dset:
            self._globals = dset

    @property
    def bcs(self):
        """Load the bcs dataset from disk.

        Notes:
            Stores arc, global, and gridded BC types. Point BCs are stored in a separate Dataset because BC coverages
            can have both arcs and points.

        Returns:
            xarray.Dataset: Dataset interface to the bcs dataset in the main file

        """
        if self._bcs is None:
            group_name = f'{self._group_path}/' if self._group_path else ''
            self._bcs = self.get_dataset(f'{group_name}bcs', False)
        return self._bcs

    @bcs.setter
    def bcs(self, dset):
        """Setter for the bcs dataset."""
        if dset:
            self._bcs = dset

    @property
    def points(self):
        """Load the points dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the points dataset in the main file

        """
        if self._points is None:
            group_name = f'{self._group_path}/' if self._group_path else ''
            self._points = self.get_dataset(f'{group_name}points', False)
        return self._points

    @points.setter
    def points(self, dset):
        """Setter for the points dataset."""
        if dset:
            self._points = dset

    @property
    def polygons(self):
        """Load the polygons dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the polygons dataset in the main file

        """
        if self._polygons is None:
            group_name = f'{self._group_path}/' if self._group_path else ''
            self._polygons = self.get_dataset(f'{group_name}polygons', False)
        return self._polygons

    @polygons.setter
    def polygons(self, dset):
        """Setter for the polygons dataset."""
        if dset:
            self._polygons = dset

    def get_bc_curve(self, comp_id, bc_type, default):
        """Get the times series curve for a BC point(s)/arc(s).

        Args:
            comp_id (int): Component id of the BC feature to retrieve curve for
            bc_type (str): The TUFLOWFV BC type card of the feature
            default (bool): If True, create a default empty curve if the curve does not exist for the given component
                id and BC type

        Returns:
            Union[Dataset, None]: The BC curve xarray Dataset or None if not found.
        """
        bc_curve = self._curves.setdefault(comp_id, {}).get(bc_type)
        if bc_curve is None:
            self._curves[comp_id][bc_type] = self._get_bc_curve_from_file(comp_id, bc_type, default)
        return self._curves[comp_id][bc_type]

    def set_bc_curve(self, comp_id, bc_type, dataset):
        """Set the times series curve for a BC point(s)/arc(s).

        Args:
            comp_id (int): Component id of the BC feature to retrieve curve for
            bc_type (str): The TUFLOWFV BC type card of the feature
            dataset (xr.Dataset): The xarray dataset for the BC curve
        """
        if comp_id >= 0 and dataset is not None:
            self._curves.setdefault(comp_id, {})[bc_type] = dataset

    def update_bc(self, comp_id, new_atts, bc_location):
        """Update the BC attributes of a boundary condition.

        Args:
            comp_id (int): Component id of the BC to update
            new_atts (xarray.Dataset): The new attributes for the BC
            bc_location (int): Location of the BC, one of BC_LOCATION_* constants
        """
        check_for_object_strings_dumb(new_atts, BC_STRING_VARIABLES)
        if bc_location == BC_LOCATION_POINT:
            dset = self.points
        elif bc_location == BC_LOCATION_ARC:
            dset = self.bcs
        else:  # bc_location == BC_LOCATION_POLYGON:
            dset = self.polygons
        dset['type'].loc[dict(comp_id=[comp_id])] = new_atts['type']
        dset['subtype'].loc[dict(comp_id=[comp_id])] = new_atts['subtype']
        dset['name'].loc[dict(comp_id=[comp_id])] = new_atts['name']
        dset['friction_slope'].loc[dict(comp_id=[comp_id])] = new_atts['friction_slope']
        dset['define_default'].loc[dict(comp_id=[comp_id])] = new_atts['define_default']
        dset['default1'].loc[dict(comp_id=[comp_id])] = new_atts['default1']
        dset['default2'].loc[dict(comp_id=[comp_id])] = new_atts['default2']
        dset['define_offset'].loc[dict(comp_id=[comp_id])] = new_atts['define_offset']
        dset['offset1'].loc[dict(comp_id=[comp_id])] = new_atts['offset1']
        dset['offset2'].loc[dict(comp_id=[comp_id])] = new_atts['offset2']
        dset['define_scale'].loc[dict(comp_id=[comp_id])] = new_atts['define_scale']
        dset['scale1'].loc[dict(comp_id=[comp_id])] = new_atts['scale1']
        dset['scale2'].loc[dict(comp_id=[comp_id])] = new_atts['scale2']
        dset['define_update_dt'].loc[dict(comp_id=[comp_id])] = new_atts['define_update_dt']
        dset['update_dt'].loc[dict(comp_id=[comp_id])] = new_atts['update_dt']
        dset['time_units'].loc[dict(comp_id=[comp_id])] = new_atts['time_units']
        dset['define_reference_time'].loc[dict(comp_id=[comp_id])] = new_atts['define_reference_time']
        dset['use_isodate'].loc[dict(comp_id=[comp_id])] = new_atts['use_isodate']
        dset['reference_time_hours'].loc[dict(comp_id=[comp_id])] = new_atts['reference_time_hours']
        dset['reference_time_date'].loc[dict(comp_id=[comp_id])] = new_atts['reference_time_date']
        dset['include_mslp'].loc[dict(comp_id=[comp_id])] = new_atts['include_mslp']
        dset['define_vertical_distribution'].loc[dict(comp_id=[comp_id])] = new_atts['define_vertical_distribution']
        dset['vertical_distribution_type'].loc[dict(comp_id=[comp_id])] = new_atts['vertical_distribution_type']
        dset['vertical_distribution_file'].loc[dict(comp_id=[comp_id])] = new_atts['vertical_distribution_file']
        dset['grid_dataset_file'].loc[dict(comp_id=[comp_id])] = new_atts['grid_dataset_file']
        dset['variable1'].loc[dict(comp_id=[comp_id])] = new_atts['variable1']
        dset['variable2'].loc[dict(comp_id=[comp_id])] = new_atts['variable2']
        dset['variable3'].loc[dict(comp_id=[comp_id])] = new_atts['variable3']
        dset['variable4'].loc[dict(comp_id=[comp_id])] = new_atts['variable4']
        if bc_location == BC_LOCATION_GRID:  # Parameters only applicable to gridded BC types
            self._update_gridded_bc(comp_id, new_atts)

    def _update_gridded_bc(self, comp_id, new_atts):
        """Update the BC attributes that are specific to gridded boundary conditions.

        Args:
            comp_id (int): Component id of the BC to update
            new_atts (xarray.Dataset): The new attributes for the BC
        """
        self.bcs['define_time_units'].loc[dict(comp_id=[comp_id])] = new_atts['define_time_units']
        self.bcs['grid_id'].loc[dict(comp_id=[comp_id])] = new_atts['grid_id']
        self.bcs['default3'].loc[dict(comp_id=[comp_id])] = new_atts['default3']
        self.bcs['default4'].loc[dict(comp_id=[comp_id])] = new_atts['default4']
        self.bcs['default5'].loc[dict(comp_id=[comp_id])] = new_atts['default5']
        self.bcs['default6'].loc[dict(comp_id=[comp_id])] = new_atts['default6']
        self.bcs['default7'].loc[dict(comp_id=[comp_id])] = new_atts['default7']
        self.bcs['default8'].loc[dict(comp_id=[comp_id])] = new_atts['default8']
        self.bcs['offset3'].loc[dict(comp_id=[comp_id])] = new_atts['offset3']
        self.bcs['offset4'].loc[dict(comp_id=[comp_id])] = new_atts['offset4']
        self.bcs['offset5'].loc[dict(comp_id=[comp_id])] = new_atts['offset5']
        self.bcs['offset6'].loc[dict(comp_id=[comp_id])] = new_atts['offset6']
        self.bcs['offset7'].loc[dict(comp_id=[comp_id])] = new_atts['offset7']
        self.bcs['offset8'].loc[dict(comp_id=[comp_id])] = new_atts['offset8']
        self.bcs['scale3'].loc[dict(comp_id=[comp_id])] = new_atts['scale3']
        self.bcs['scale4'].loc[dict(comp_id=[comp_id])] = new_atts['scale4']
        self.bcs['scale5'].loc[dict(comp_id=[comp_id])] = new_atts['scale5']
        self.bcs['scale6'].loc[dict(comp_id=[comp_id])] = new_atts['scale6']
        self.bcs['scale7'].loc[dict(comp_id=[comp_id])] = new_atts['scale7']
        self.bcs['scale8'].loc[dict(comp_id=[comp_id])] = new_atts['scale8']
        self.bcs['variable5'].loc[dict(comp_id=[comp_id])] = new_atts['variable5']
        self.bcs['variable6'].loc[dict(comp_id=[comp_id])] = new_atts['variable6']
        self.bcs['variable7'].loc[dict(comp_id=[comp_id])] = new_atts['variable7']
        self.bcs['variable8'].loc[dict(comp_id=[comp_id])] = new_atts['variable8']
        self.bcs['variable9'].loc[dict(comp_id=[comp_id])] = new_atts['variable9']

    def add_bc_atts(self, bc_location, dset=None):
        """Add the BC attribute dataset for an arc.

        Args:
            bc_location (int): Location of the BC, one of BC_LOCATION_* constants
            dset (xarray.Dataset): The attribute dataset to concatenate. If not provided, a new Dataset of default
                attributes will be generated.

        Returns:
            int: The newly generated component id
        """
        try:
            new_comp_id = int(self.info.attrs['next_comp_id'])
            self.info.attrs['next_comp_id'] += 1  # Increment the unique XMS component id.
            if dset is None:  # Generate a new default Dataset
                dset = self._get_new_bc_atts(comp_id=new_comp_id, gridded=bc_location == BC_LOCATION_GRID)
            else:  # Update the component id of an existing Dataset
                # Copy all the curves to the new component id
                old_comp_id = int(dset.coords['comp_id'][0]) if len(dset.coords['comp_id']) > 0 else \
                    UNINITIALIZED_COMP_ID
                dset.coords['comp_id'] = [new_comp_id for _ in dset.coords['comp_id']]
                # No curves on gridded or curtain BCs
                if bc_location != BC_LOCATION_GRID and dset['type'][0].item() not in CURTAIN_BC_TYPES:
                    self._copy_curves(old_comp_id, new_comp_id, bc_location)

            # Global and gridded bcs stored in the bcs Dataset
            if bc_location not in [BC_LOCATION_POINT, BC_LOCATION_POLY]:
                self._bcs = xr.concat([self.bcs, dset], 'comp_id')
            elif bc_location == BC_LOCATION_POINT:
                self._points = xr.concat([self.points, dset], 'comp_id')
            else:
                self._polygons = xr.concat([self.polygons, dset], 'comp_id')
            return new_comp_id
        except Exception:
            return UNINITIALIZED_COMP_ID

    def clear_and_load_curves(self, comp_ids):
        """Load all curves for given component ids and clear unused curves from memory.

        Args:
            comp_ids (Iterable): Component ids of the BC features to load
        """
        all_bc_types = set(ARC_BC_TYPES.keys()).union(
            set(POINT_BC_TYPES.keys())).union(set(GLOBAL_BC_TYPES.keys())).union(set(POLYGON_BC_TYPES.keys()))
        new_curves = {}
        for comp_id in comp_ids:
            loaded_curves = self._curves.get(comp_id, {})
            for bc_type in all_bc_types:
                # Check if curve is in memory or on disk.
                bc_curve = loaded_curves.get(bc_type, self._get_bc_curve_from_file(comp_id, bc_type, False))
                if bc_curve is not None:
                    new_curves.setdefault(comp_id, {})[bc_type] = bc_curve
        self._curves = new_curves

    def load_all_data(self, load_curves):
        """Load all data from disk into memory.

        Args:
            load_curves (bool): True if BC curves should be loaded into memory
        """
        _ = self.info
        _ = self.globals
        _ = self.bcs
        _ = self.points
        _ = self.polygons
        if load_curves:
            comp_ids = self.bcs.comp_id.data.tolist()
            if self._points is not None:
                comp_ids.extend(self.points.comp_id.data.tolist())
            self.clear_and_load_curves(comp_ids)
        self._info.close()
        self._globals.close()
        self._bcs.close()
        if self._points is not None:
            self._points.close()
        if self._points is not None:
            self._points.close()
        if self._polygons is not None:
            self._polygons.close()

    def commit(self):
        """Save current in-memory component parameters to data file."""
        base_path = f'{self._group_path}/' if self._group_path else ''
        mode = 'a'
        if not os.path.exists(self._filename) or not os.path.isfile(self._filename):
            mode = 'w'
        if self._info is not None:
            group_name = f'{base_path}info'
            if mode != 'w':
                self._drop_h5_groups([group_name])
            self.info.to_netcdf(self._filename, group=group_name, mode=mode)

        if self._globals is not None:
            self._globals.close()
            group_name = f'{base_path}globals'
            self._drop_h5_groups([group_name])
            self._globals.to_netcdf(self._filename, group=group_name, mode='a')
        if self._bcs is not None:
            self._bcs.close()
            group_name = f'{base_path}bcs'
            self._drop_h5_groups([group_name])
            check_for_object_strings_dumb(self._bcs, BC_STRING_VARIABLES)
            self._bcs.to_netcdf(self._filename, group=group_name, mode='a')
        if self._points is not None:
            self._points.close()
            group_name = f'{base_path}points'
            self._drop_h5_groups([group_name])
            check_for_object_strings_dumb(self._points, BC_STRING_VARIABLES)
            self._points.to_netcdf(self._filename, group=group_name, mode='a')
        if self._polygons is not None:
            self._polygons.close()
            group_name = f'{base_path}polygons'
            self._drop_h5_groups([group_name])
            check_for_object_strings_dumb(self._polygons, BC_STRING_VARIABLES)
            self._polygons.to_netcdf(self._filename, group=group_name, mode='a')
        for comp_id, bc_curves in self._curves.items():
            for bc_type, bc_curve in bc_curves.items():
                if bc_curve is None:
                    continue
                dset_path = f'{base_path}bc_curves/{comp_id}/{bc_type}'
                bc_curve.close()
                self._drop_h5_groups([dset_path])
                bc_curve.to_netcdf(self._filename, group=dset_path, mode='a')

    def vacuum(self):
        """Rewrite all data to a new file to reclaim disk space.

        All BC datasets that need to be written to the file must be loaded into memory before calling this method.
        """
        self.load_all_data(False)
        io_util.removefile(self._filename)
        self.commit()

    def update_file_paths(self):
        """Called before resaving an existing project.

        All referenced filepaths should be converted to relative from the project directory. Should already be stored
        in the component main file since this is a resave operation.

        Returns:
            (str): Message on failure, empty string on success
        """
        proj_dir = self.info.attrs['proj_dir']
        if not os.path.exists(proj_dir):
            return 'Unable to update selected file paths to relative from the project directory.'

        flush_data = False
        if self.bcs.sizes['comp_id'] > 0:
            flush_data |= self._update_table_files(self.bcs.variables['vertical_distribution_file'], proj_dir, '')
            flush_data |= self._update_table_files(self.bcs.variables['grid_dataset_file'], proj_dir, '')
        if self.points.sizes['comp_id'] > 0:
            flush_data |= self._update_table_files(self.points.variables['vertical_distribution_file'], proj_dir, '')
        if flush_data:
            self.commit()
        return ''

    def copy_external_files(self):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Returns:
            (str): Message on failure, empty string on success
        """
        old_proj_dir = self.info.attrs['proj_dir']
        # Target directory for package save is three directories above the component UUID folder.
        target_dir = os.path.normpath(os.path.join(os.path.dirname(self._filename), '../../..'))
        # Copy the files referenced in tables
        self._copy_table_files(self.bcs.variables['vertical_distribution_file'], old_proj_dir, target_dir)
        self._copy_table_files(self.bcs.variables['grid_dataset_file'], old_proj_dir, target_dir)
        self._copy_table_files(self.points.variables['vertical_distribution_file'], old_proj_dir, target_dir)
        # Wipe the stored project directory. Paths will be absolute in GUI until resave.
        self.info.attrs['proj_dir'] = ''
        return ''

    def update_proj_dir(self):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Returns:
            (str): Message on failure, empty string on success.
        """
        try:
            old_proj_dir = self.info.attrs['proj_dir']  # Empty if first project save
            # Get the new project location. Three directories above the component UUID folder.
            comp_folder = os.path.dirname(self._filename)
            new_proj_dir = os.path.normpath(os.path.join(comp_folder, '../../..'))
            self.info.attrs['proj_dir'] = new_proj_dir
        except Exception:
            return 'There was a problem updating file paths to be relative from the project directory. Any selected ' \
                   'file paths will remain absolute.\n'

        if self.bcs.sizes['comp_id'] > 0:
            self._update_table_files(self.bcs.variables['vertical_distribution_file'], new_proj_dir, old_proj_dir)
            self._update_table_files(self.bcs.variables['grid_dataset_file'], new_proj_dir, old_proj_dir)
        if self.points.sizes['comp_id'] > 0:
            self._update_table_files(self.points.variables['vertical_distribution_file'], new_proj_dir, old_proj_dir)
        self.commit()  # Save the updated project directory and referenced filepaths.
        return ''  # Don't report errors, leave that to model checks.
