"""Data class for TUFLOWFV simulation component."""
# 1. Standard python modules
import logging
import os
import uuid

# 2. Third party modules
import numpy as np
import pandas as pd
import pkg_resources
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as io_util
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.tuflowfv.data import bc_data as bcd
from xms.tuflowfv.data.material_data import MaterialData
from xms.tuflowfv.data.tuflowfv_data import check_for_object_strings_dumb, fix_datetime_format
from xms.tuflowfv.file_io.bc_csv_reader import parse_tuflowfv_time
from xms.tuflowfv.gui import gui_util


SIM_DATA_MAINFILE = 'sim_comp.nc'
SUPPORTED_OUTPUT_TYPES = [
    'DATV',
    'XMDF',
    'NetCDF',
    'Flux',
    'Mass',
    'Points',
    'Transport',
]
GRID_DEF_STR_VARIABLES = ['file', 'name', 'vert_coord_type', 'x_variable', 'y_variable', 'z_variable']

ELEV_TYPE_SET_ZPTS = 0
ELEV_TYPE_GRID_ZPTS = 1
ELEV_TYPE_CELL_CSV = 2
ELEV_TYPE_ZLINE = 3

ELEV_MOD_TYPE_VAR = 'type'
ELEV_SET_ZPTS_VAR = 'set_zpts'
ELEV_GRID_ZPTS_VAR = 'grid_zpts'
ELEV_CSV_FILE_VAR = 'csv'
ELEV_CSV_TYPE_VAR = 'csv_type'
ELEV_ZLINE_UUID_VAR = 'uuid'
ELEV_ZPOINT1_UUID_VAR = 'point1'
ELEV_ZPOINT2_UUID_VAR = 'point2'
ELEV_ZPOINT3_UUID_VAR = 'point3'
ELEV_ZPOINT4_UUID_VAR = 'point4'
ELEV_ZPOINT5_UUID_VAR = 'point5'
ELEV_ZPOINT6_UUID_VAR = 'point6'
ELEV_ZPOINT7_UUID_VAR = 'point7'
ELEV_ZPOINT8_UUID_VAR = 'point8'
ELEV_ZPOINT9_UUID_VAR = 'point9'

CELL_ELEV_CSV_TYPE_CELL = 'cell_ID'
CELL_ELEV_CSV_TYPE_COORD = 'coordinate'


def get_point_variable_names():
    """Get the xarray DataArray names of the zpoint shapefile variables.

    Returns:
        list: See description
    """
    return [
        ELEV_ZPOINT1_UUID_VAR,
        ELEV_ZPOINT2_UUID_VAR,
        ELEV_ZPOINT3_UUID_VAR,
        ELEV_ZPOINT4_UUID_VAR,
        ELEV_ZPOINT5_UUID_VAR,
        ELEV_ZPOINT6_UUID_VAR,
        ELEV_ZPOINT7_UUID_VAR,
        ELEV_ZPOINT8_UUID_VAR,
        ELEV_ZPOINT9_UUID_VAR,
    ]


def format_output_type(raw_type):
    """Given a supported output type, return it in our standard capitalization form.

    Args:
        raw_type (str): The supported output type

    Returns:
        str: See description
    """
    raw_type_lower = raw_type.lower()
    lookup = {output_type.lower(): output_type for output_type in SUPPORTED_OUTPUT_TYPES}
    if raw_type_lower not in lookup:
        raise ValueError(f'{raw_type} is not a supported output format.')
    return lookup[raw_type_lower]


def get_default_output_blocks_values(importing):
    """Get the default dict of output blocks values.

    Args:
        importing (bool): If False, will enable some common outputs by default

    Returns:
        dict: See description
    """
    return {
        'format': 'DATV',
        'define_start': 0,
        'output_start': 0.0,
        'output_start_date': '',
        'define_final': 0,
        'output_final': 0.0,
        'output_final_date': '',
        'define_interval': 0,
        'output_interval': 300.0,
        'suffix': '',
        'define_compression': 0,
        'compression': 1,
        'define_statistics': 0,
        'statistics_type': 'Both',
        'define_statistics_dt': 0,
        'statistics_dt': 0.0,
        'depth_output': 0 if importing else 1,
        'wse_output': 0 if importing else 1,
        'bed_shear_stress_output': 0,
        'surface_shear_stress_output': 0,
        'velocity_output': 0 if importing else 1,
        'velocity_mag_output': 0,
        'vertical_velocity_output': 0,
        'bed_elevation_output': 0,
        'turb_visc_output': 0,
        'turb_diff_output': 0,
        'air_temp_output': 0,
        'evap_output': 0,
        'dzb_output': 0,
        'hazard_z1_output': 0,
        'hazard_zaem1_output': 0,
        'hazard_zqra_output': 0,
        'lw_rad_output': 0,
        'mslp_output': 0,
        'precip_output': 0,
        'rel_hum_output': 0,
        'rhow_output': 0,
        'sal_output': 0,
        'sw_rad_output': 0,
        'temp_output': 0,
        'turbz_output': 0,
        'w10_output': 0,
        'wq_all_output': 0,
        'wq_diag_all_output': 0,
        'wvht_output': 0,
        'wvper_output': 0,
        'wvdir_output': 0,
        'wvstr_output': 0,
        'row_index': -1,  # Not the Qt row index in the table, just an infinite incrementer.
        'cov_uuid': '',
    }


def parse_tuflow_isodate(card, to_qt=None):
    """Parse a TUFLOWFV ISO date card, which isn't really in standard ISO format.

    Args:
        card (str): The ISO date string in TUFLOWFV's format ('dd/MM/yyyy HH:mm:ss')
        to_qt (Optional[bool]): If None returns a Python datetime object. If True, returns a QDateTime. If False
            returns the default string representation for a QDateTime.

    Returns:
        datetime.datetime: The parsed ISO date as a Python datetime object. If parsing the datetime string fails,
            returns the TUFLOWFV default zero time.
    """
    try:
        parsed_time = parse_tuflowfv_time(card)
        if not parsed_time:
            raise RuntimeError(f'Date/time parsing failed for {card}.')
        dt = parsed_time.to_pydatetime()
        if to_qt is True:  # Convert to Qt
            dt = gui_util.datetime_to_qdatetime(dt)
        elif to_qt is False:  # Convert to string
            dt = dt.strftime(ISO_DATETIME_FORMAT)
    except Exception:
        logging.getLogger('xms.tuflowfv').error(f'Could not parse date/time: {card}.')
        dt = gui_util.get_tuflow_zero_time(to_qt)
    return dt


class SimData(XarrayBase):
    """Manages data file for the hidden simulation component."""

    def __init__(self, data_file):
        """Initializes the data class.

        Args:
            data_file (str): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        self._filename = data_file
        self._info = None
        self._general = None
        self._time = None
        self._globals = None
        self._wind_stress = None
        self._geometry = None
        self._z_modifications = None
        self._initial_conditions = None
        self._output = None
        self._output_points_coverages = None
        self._global_set_mat = None
        self._global_bcs = None  # BcData
        self._gridded_bcs = None  # BcData
        self._grid_definitions = None
        self._linked_simulations = None
        self._advanced_cards = None
        self._get_default_datasets(data_file)
        super().__init__(data_file)

    def _get_default_datasets(self, data_file):
        """Create default datasets if needed.

        Args:
            data_file (str): Name of the data file. If it doesn't exist, it will be created.
        """
        if not os.path.exists(data_file) or not os.path.isfile(data_file):
            info_attrs = {
                'FILE_TYPE': 'TUFLOWFV_SIMULATION',
                'VERSION': pkg_resources.get_distribution('xmstuflowfv').version,
                'domain_uuid': '',
            }
            self._info = xr.Dataset(attrs=info_attrs)
            self._general = xr.Dataset(attrs=self._get_default_general_data())
            self._time = xr.Dataset(attrs=self._get_default_time_data())
            self._globals = xr.Dataset(attrs=self._get_default_globals_data())
            self._wind_stress = xr.Dataset(attrs=self._get_default_wind_data())
            geometry_data, geometry_attrs = self._get_default_geometry_data()
            self._geometry = xr.Dataset(data_vars=geometry_data, attrs=geometry_attrs)
            self._z_modifications = xr.Dataset(data_vars=self._get_default_z_modification_data())
            self._initial_conditions = xr.Dataset(attrs=self._get_default_initial_conditions_data())
            output_data, output_attrs = self._get_default_output_data()
            self._output = xr.Dataset(data_vars=output_data, attrs=output_attrs)
            self._output_points_coverages = xr.Dataset(data_vars=self._get_default_output_points_coverages_data())
            self._global_set_mat = self._get_default_set_mat_dataset()
            self._global_bcs = bcd.BcData(self._filename, bcd.GLOBAL_BC_GROUP)
            self._global_bcs.info.attrs['next_comp_id'] = 1  # Use one-base since it will show up in the GUI
            self._gridded_bcs = bcd.BcData(self._filename, bcd.GRIDDED_BC_GROUP)
            self._gridded_bcs.info.attrs['next_comp_id'] = 1  # Use one-base since it will show up in the GUI
            self._grid_definitions = self._get_default_grid_definition_data()
            self._linked_simulations = xr.Dataset(data_vars=self._get_default_linked_simulations_data())
            self.commit()
        else:
            self._migrate_data()

    def _migrate_data(self):
        """Check for missing data when opening an existing file."""
        if 'include_wave_stress' not in self.globals.attrs:
            # Moved from the BC coverage - only applicable if simulation has a linked BC with a wave boundary
            # condition.
            self.globals.attrs['include_wave_stress'] = 1
            self.globals.attrs['include_stokes_drift'] = 1
        if 'advanced_cards' not in self.general.attrs:
            self.general.attrs['advanced_cards'] = ''

        # Used to use Qt default for converting datetimes to strings. Need to make it locale agnostic.
        self.time.attrs['ref_date'] = fix_datetime_format(self.time.attrs['ref_date'])
        self.time.attrs['start_date'] = fix_datetime_format(self.time.attrs['start_date'])
        self.time.attrs['end_date'] = fix_datetime_format(self.time.attrs['end_date'])

        for i in range(self.output.output_start_date.size):
            self.output.output_start_date[i] = fix_datetime_format(self.output.output_start_date[i].item())
            self.output.output_final_date[i] = fix_datetime_format(self.output.output_final_date[i].item())

    def _get_default_general_data(self):
        """Returns the default attrs for the general Dataset.

        Returns:
            dict: See Description
        """
        general_attrs = {
            'use_salinity': 0,
            'couple_salinity': 1,
            'use_temperature': 0,
            'couple_temperature': 1,
            'use_sediment': 0,
            'couple_sediment': 1,
            'use_heat': 0,
            'define_spatial_order': 0,
            'horizontal_order': 1,
            'vertical_order': 1,
            'define_hardware_solver': 0,
            'hardware_solver': 'CPU',
            'device_id': 0,
            'is_3d': 0,
            'define_display_interval': 0,
            'display_interval': 300.0,
            'projection_warning': 0,
            'tutorial': 'OFF',
            'advanced_cards': '',
        }
        return general_attrs

    def _get_default_time_data(self):
        """Returns the default attrs for the time Dataset.

        Returns:
            dict: See Description
        """
        time_attrs = {
            'use_isodate': 0,
            'ref_date': gui_util.get_tuflow_zero_time(False),
            'ref_date_hours': 0.0,
            'start_hours': 0.0,
            'end_hours': 1.0,
            'start_date': gui_util.get_tuflow_zero_time(False),
            'end_date': gui_util.get_tuflow_zero_time(False),
            'cfl': 1.0,
            'min_increment': 0.01,
            'max_increment': 100.0,
        }
        return time_attrs

    def _get_default_globals_data(self):
        """Returns the default attrs for the globals Dataset.

        Returns:
            dict: See Description
        """
        globals_attrs = {
            'horizontal_mixing': 'None',
            'global_horizontal_viscosity': 0.0,
            'define_horizontal_viscosity_limits': 0,
            'horizontal_viscosity_min': 0.0,
            'horizontal_viscosity_max': 99999.0,
            'define_vertical_mixing': 0,
            'vertical_mixing': 'Constant',
            'global_vertical_viscosity': 0.0,
            'global_vertical_parametric_coefficients1': 0.4,
            'global_vertical_parametric_coefficients2': 0.4,
            'external_vertical_viscosity': '',
            'define_vertical_viscosity_limits': 0,
            'vertical_viscosity_min': 0.0,
            'vertical_viscosity_max': 99999.0,
            'define_turbulence_update': 0,
            'turbulence_update': 900.0,
            'horizontal_scalar_diffusivity_type': 'None',
            'horizontal_scalar_diffusivity': 0.0,
            'horizontal_scalar_diffusivity_coef1': 0.0,
            'horizontal_scalar_diffusivity_coef2': 0.0,
            'define_horizontal_diffusivity_limits': 0,
            'horizontal_scalar_diffusivity_min': 0.0,
            'horizontal_scalar_diffusivity_max': 99999.0,
            'define_vertical_scalar_diffusivity': 0,
            'vertical_scalar_diffusivity': 0.0,
            'vertical_scalar_diffusivity_coef1': 0.0,
            'vertical_scalar_diffusivity_coef2': 0.0,
            'define_vertical_diffusivity_limits': 0,
            'vertical_scalar_diffusivity_min': 0.0,
            'vertical_scalar_diffusivity_max': 99999.0,
            'define_stability_limits': 0,
            'stability_wse': 100.0,
            'stability_velocity': 10.0,
            'define_bc_default_update_dt': 0,
            'bc_default_update_dt': 0.0,
            'include_wave_stress': 1,  # Only applicable if we have a wave boundary
            'include_stokes_drift': 1,  # Only applicable if we have a wave boundary
            'transport_mode': 0,
            'transport_file': '',
        }
        return globals_attrs

    def _get_default_wind_data(self):
        """Returns the default attrs for the wind Dataset.

        Returns:
            dict: See Description
        """
        wind_attrs = {
            'define_wind': 0,
            'method': 1,
            'define_parameters': 0,
            'wa': 0.0,
            'ca': 0.8e-03,
            'wb': 50.0,
            'cb': 4.05e-03,
            'bulk_coefficient': 0.0013,
            'scale_factor': 1.0,
        }
        return wind_attrs

    def _get_default_geometry_data(self):
        """Returns the default data_vars and attrs for the geometry Dataset.

        Returns:
            tuple(dict, dict): Tuple of the data_vars and attrs
        """
        geometry_attrs = {
            'vertical_mesh_type': 'Sigma',
            'layer_faces': 'CSV',
            'layer_faces_file': '',
            'sigma_layers': 4,
            'cell_3d_depth': 1.0,
            'min_bed_layer_thickness': 0.5,
            'define_bed_elevation_limits': 0,
            'min_bed_elevation': 0.0,
            'max_bed_elevation': 0.0,
            'define_cell_depths': 0,
            'dry_cell_depth': 1.0e-6,
            'wet_cell_depth': 1.0e-2,
        }
        geometry_data = {'Z': xr.DataArray(data=np.array([], dtype=np.float64))}
        return geometry_data, geometry_attrs

    def _get_default_z_modification_data(self):
        """Returns the default data_vars for the z modification Dataset.

        Includes Zpts, Zlines, GRID Zpts, and cell elevation files.

        Returns:
            dict: See Description
        """
        return {
            ELEV_MOD_TYPE_VAR: xr.DataArray(data=np.array([], dtype=np.int32)),  # Type of Z modification
            # Set Zpts
            ELEV_SET_ZPTS_VAR: xr.DataArray(data=np.array([], dtype=np.float64)),
            # GRID Zpts
            ELEV_GRID_ZPTS_VAR: xr.DataArray(data=np.array([], dtype=object)),
            # Cell elevation files
            ELEV_CSV_FILE_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_CSV_TYPE_VAR: xr.DataArray(data=np.array([], dtype=object)),
            # Z-lines
            ELEV_ZLINE_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT1_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT2_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT3_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT4_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT5_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT6_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT7_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT8_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
            ELEV_ZPOINT9_UUID_VAR: xr.DataArray(data=np.array([], dtype=object)),
        }

    def _get_default_initial_conditions_data(self):
        """Returns the default attrs for the intial conditions Dataset.

        Returns:
            dict: See Description
        """
        initial_conditions_attrs = {
            'define_initial_water_level': 0,
            'initial_water_level': 0.5,
            'use_restart_file': 0,
            'restart_file': '',
            'use_restart_file_time': 1,
        }
        return initial_conditions_attrs

    def _get_default_output_data(self):
        """Returns the default data_vars and attrs for the output Dataset.

        Returns:
            tuple(dict, dict): Tuple of the data_vars and attrs
        """
        output_data = {
            'format': xr.DataArray(data=np.array([], dtype=object)),
            'define_start': xr.DataArray(data=np.array([], dtype=np.int32)),
            'output_start': xr.DataArray(data=np.array([], dtype=np.float64)),
            'output_start_date': xr.DataArray(data=np.array([], dtype=object)),
            'define_final': xr.DataArray(data=np.array([], dtype=np.int32)),
            'output_final': xr.DataArray(data=np.array([], dtype=np.float64)),
            'output_final_date': xr.DataArray(data=np.array([], dtype=object)),
            'define_interval': xr.DataArray(data=np.array([], dtype=np.int32)),
            'output_interval': xr.DataArray(data=np.array([], dtype=np.float64)),
            'suffix': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'define_compression': xr.DataArray(data=np.array([], dtype=np.int32)),
            'compression': xr.DataArray(data=np.array([], dtype=np.int32)),
            'define_statistics': xr.DataArray(data=np.array([], dtype=np.int32)),
            'statistics_type': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'define_statistics_dt': xr.DataArray(data=np.array([], dtype=np.int32)),
            'statistics_dt': xr.DataArray(data=np.array([], dtype=np.float64)),
            'depth_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'wse_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'bed_shear_stress_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'surface_shear_stress_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'velocity_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'velocity_mag_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'vertical_velocity_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'bed_elevation_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'turb_visc_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'turb_diff_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'air_temp_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'evap_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'dzb_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'hazard_z1_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'hazard_zaem1_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'hazard_zqra_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'lw_rad_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'mslp_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'precip_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'rel_hum_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'rhow_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'sal_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'sw_rad_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'temp_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'turbz_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'w10_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'wq_all_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'wq_diag_all_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'wvht_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'wvper_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'wvdir_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'wvstr_output': xr.DataArray(data=np.array([], dtype=np.int32)),
            'row_index': xr.DataArray(data=np.array([], dtype=np.int32)),
        }
        output_attrs = {
            'log_dir': '',
            'output_dir': '',
            'write_check_files': 0,
            'check_files_dir': '',
            'write_empty_gis_files': 0,
            'empty_gis_files_dir': '',
            'write_restart_file': 0,
            'restart_file_interval': 24.0,
            'overwrite_restart_file': 1,
            'next_row_index': 0    # Not the Qt row index in the table, just an infinite incrementer.
        }
        return output_data, output_attrs

    def _get_default_output_points_coverages_data(self):
        """Returns the default data_vars for the output points coverages Dataset.

        Returns:
            xr.Dataset: See description
        """
        data_vars = {
            'uuid': ('index', np.array([], dtype=object)),
            'row_index': ('index', np.array([], dtype=np.int32)),
        }
        coords = {'index': []}
        return xr.Dataset(data_vars=data_vars, coords=coords)

    def _get_default_set_mat_dataset(self):
        """Returns the default Dataset for the global set mat Dataset.

        Returns:
            xr.Dataset: The set mat Dataset
        """
        data_dict = MaterialData.default_data_dict()
        # Add an extra row so we can hide the unassigned in fixed size table
        for variable, data_values in data_dict.items():
            if variable == 'name':  # Generate a name for the set mat material the user will never pick for another
                data_values.append(str(uuid.uuid4()))
            else:
                data_values.append(data_values[0])
        dset = pd.DataFrame(data_dict).to_xarray()
        dset.attrs['define_set_mat'] = 0
        dset.attrs['define_global_roughness'] = 0
        dset.attrs['global_roughness'] = 'Manning'
        dset.attrs['global_roughness_coefficient'] = 0.05
        return dset

    def _get_default_grid_definition_data(self):
        """Returns the default data_vars and attrs for the grid definition Dataset.

        Returns:
            xr.Dataset: The default grid definition Dataset
        """
        data_vars, coords = self._get_new_grid_definition()
        grid_def_attrs = {'next_grid_id': 1}
        return xr.Dataset(data_vars=data_vars, coords=coords, attrs=grid_def_attrs)

    def _get_default_linked_simulations_data(self):
        """Returns the default data_vars linked simulations Dataset.

        Returns:
            xr.Dataset: The default linked simulations Dataset
        """
        return {'uuid': xr.DataArray(data=np.array([], dtype=object))}

    def _get_new_grid_definition(self, grid_id=None):
        """Get a new Dataset for a single BC grid definition.

        Args:
            grid_id (int): If provided, will create a single row with default values, empty Dataset if None

        Returns:
            tuple(dict, dict): The data_vars variables and coordinates
        """
        if grid_id is not None:
            num_grids = self.grid_definitions.sizes['grid_id']
            default_var = gui_util.DEFAULT_VALUE_VARIABLE
            data_vars = {
                'file': ('grid_id', np.array([''], dtype=object)),
                'name': ('grid_id', np.array([f'Grid_{num_grids + 1}'], dtype=object)),
                'x_variable': ('grid_id', np.array([default_var], dtype=object)),
                'y_variable': ('grid_id', np.array([default_var], dtype=object)),
                'z_variable': ('grid_id', np.array([default_var], dtype=object)),
                'define_vert_coord_type': ('grid_id', np.array([0], dtype=np.int32)),
                'vert_coord_type': ('grid_id', np.array(['Elevation'], dtype=object)),
                'cell_gridmap': ('grid_id', np.array([1], dtype=np.int32)),
                'boundary_gridmap': ('grid_id', np.array([0], dtype=np.int32)),
                'suppress_coverage_warnings': ('grid_id', np.array([0], dtype=np.int32)),
            }
            coords = {'grid_id': [grid_id]}
        else:
            data_vars = {
                'file': ('grid_id', np.array([], dtype=object)),
                'name': ('grid_id', np.array([], dtype=object)),
                'x_variable': ('grid_id', np.array([], dtype=object)),
                'y_variable': ('grid_id', np.array([], dtype=object)),
                'z_variable': ('grid_id', np.array([], dtype=object)),
                'define_vert_coord_type': ('grid_id', np.array([], dtype=np.int32)),
                'vert_coord_type': ('grid_id', np.array([], dtype=object)),
                'cell_gridmap': ('grid_id', np.array([], dtype=np.int32)),
                'boundary_gridmap': ('grid_id', np.array([], dtype=np.int32)),
                'suppress_coverage_warnings': ('grid_id', np.array([], dtype=np.int32)),
            }
            coords = {'grid_id': []}
        return data_vars, coords

    def update_grid_definition(self, new_atts):
        """Update the BC attributes of a boundary condition.

        Args:
            new_atts (xarray.Dataset): The new attributes for the BC
        """
        check_for_object_strings_dumb(new_atts, GRID_DEF_STR_VARIABLES)
        dset = self.grid_definitions
        grid_id = new_atts.grid_id.item()
        dset['file'].loc[dict(grid_id=[grid_id])] = new_atts['file']
        dset['name'].loc[dict(grid_id=[grid_id])] = new_atts['name']
        dset['x_variable'].loc[dict(grid_id=[grid_id])] = new_atts['x_variable']
        dset['y_variable'].loc[dict(grid_id=[grid_id])] = new_atts['y_variable']
        dset['z_variable'].loc[dict(grid_id=[grid_id])] = new_atts['z_variable']
        dset['define_vert_coord_type'].loc[dict(grid_id=[grid_id])] = new_atts['define_vert_coord_type']
        dset['vert_coord_type'].loc[dict(grid_id=[grid_id])] = new_atts['vert_coord_type']
        dset['cell_gridmap'].loc[dict(grid_id=[grid_id])] = new_atts['cell_gridmap']
        dset['boundary_gridmap'].loc[dict(grid_id=[grid_id])] = new_atts['boundary_gridmap']
        dset['suppress_coverage_warnings'].loc[dict(grid_id=[grid_id])] = new_atts['suppress_coverage_warnings']

    def add_grid_definition(self, dset=None):
        """Add the BC attribute dataset for an arc.

        Args:
            dset (xarray.Dataset): The attribute dataset to concatenate. If not provided, a new Dataset of default
                attributes will be generated.

        Returns:
            int: The newly generated component id
        """
        try:
            new_grid_id = self.grid_definitions.attrs['next_grid_id']
            self.grid_definitions.attrs['next_grid_id'] += 1  # Increment the unique XMS component id.
            if dset is None:  # Generate a new default Dataset
                data_vars, coords = self._get_new_grid_definition(new_grid_id)
                dset = xr.Dataset(data_vars=data_vars, coords=coords)
            else:  # Update the component id of an existing Dataset
                dset.coords['comp_id'] = [new_grid_id for _ in dset.coords['grid_id']]
            self._grid_definitions = xr.concat([self.grid_definitions, dset], 'grid_id')
            return new_grid_id
        except Exception:
            return bcd.UNINITIALIZED_COMP_ID

    def add_output_coverage(self, df):
        """Add an output points coverage.

        Args:
            df (DataFrame): pandas dataset of the output points coverages to add

        Returns:
            int: The number of duplicate or undefined rows that were filtered out
        """
        old_rows = self.output_points_coverages.to_dataframe()
        dset = pd.concat([old_rows, df])
        nonunique_rows = dset.shape[0]
        dset = dset.drop_duplicates(subset=['uuid'])
        dset = dset[dset['uuid'] != '']
        unique_rows = dset.shape[0]
        self._output_points_coverages = dset.to_xarray()
        return nonunique_rows - unique_rows

    def remove_output_coverages(self, uuids, indices):
        """Remove output coverages from the dataset.

        Args:
            uuids (list[str]): UUIDs of the coverages to remove
            indices (list[int]): Row indices of the coverages to remove. The internal counter, not the Qt row index.
        """
        # Convert to pandas because it is easier. Probably a way to do this in xarray, but I don't think performance
        # is an issue here
        df = self.output_points_coverages.to_dataframe()
        df = df[~(df['uuid'].isin(uuids)) & ~(df['row_index'].isin(indices))]
        self.output_points_coverages = df.to_xarray()

    def commit(self):
        """Save current in-memory component parameters to data file."""
        # Always load all data into memory and vacuum the file. Data is small enough and prevents filesystem bloat.
        _ = self.info
        _ = self.general
        _ = self.time
        _ = self.globals
        _ = self.wind_stress
        _ = self.geometry
        _ = self.z_modifications
        _ = self.initial_conditions
        _ = self.output
        _ = self.output_points_coverages
        _ = self.global_set_mat
        _ = self.global_bcs
        _ = self.gridded_bcs
        _ = self.grid_definitions
        _ = self.linked_simulations
        self._global_bcs.load_all_data(True)
        self._gridded_bcs.load_all_data(False)
        self._info.close()
        self._general.close()
        self._time.close()
        self._globals.close()
        self._wind_stress.close()
        self._geometry.close()
        self._z_modifications.close()
        self._initial_conditions.close()
        self._output.close()
        self._output_points_coverages.close()
        self._global_set_mat.close()
        self._grid_definitions.close()
        self._linked_simulations.close()
        io_util.removefile(self._filename)
        super().commit()  # Recreates the NetCDF file
        self._general.to_netcdf(self._filename, group='general', mode='a')
        self._time.to_netcdf(self._filename, group='time', mode='a')
        self._globals.to_netcdf(self._filename, group='globals', mode='a')
        self._wind_stress.to_netcdf(self._filename, group='wind_stress', mode='a')
        self._geometry.to_netcdf(self._filename, group='geometry', mode='a')
        string_variables = [
            ELEV_GRID_ZPTS_VAR,
            ELEV_CSV_FILE_VAR,
            ELEV_CSV_TYPE_VAR,
            ELEV_ZLINE_UUID_VAR,
        ]
        string_variables.extend(get_point_variable_names())
        check_for_object_strings_dumb(self._z_modifications, string_variables)
        self._z_modifications.to_netcdf(self._filename, group='z_modifications', mode='a')
        self._initial_conditions.to_netcdf(self._filename, group='initial_conditions', mode='a')
        string_variables = ['format', 'output_start_date', 'output_final_date', 'suffix', 'statistics_type']
        check_for_object_strings_dumb(self._output, string_variables)
        self._output.to_netcdf(self._filename, group='output', mode='a')
        check_for_object_strings_dumb(self._output_points_coverages, ['uuid'])
        self._output_points_coverages.to_netcdf(self._filename, group='output_points_coverages', mode='a')
        self._global_set_mat.to_netcdf(self._filename, group='global_set_mat', mode='a')
        check_for_object_strings_dumb(self._grid_definitions, GRID_DEF_STR_VARIABLES)
        self._grid_definitions.to_netcdf(self._filename, group='grid_definitions', mode='a')
        check_for_object_strings_dumb(self._linked_simulations, ['uuid'])
        self._linked_simulations.to_netcdf(self._filename, group='linked_simulations', mode='a')
        self._global_bcs.commit()
        self._gridded_bcs.commit()

    def update_file_paths(self):
        """Called before resaving an existing project.

        All referenced filepaths should be converted to relative from the project directory. Should already be stored
        in the component main file since this is a resave operation.

        Returns:
            (str): Message on failure, empty string on success
        """
        proj_dir = self.info.attrs['proj_dir']
        if not os.path.exists(proj_dir):
            return 'Unable to update selected file paths to relative from the project directory.'
        self._convert_filepath(self.globals.attrs, 'external_vertical_viscosity', proj_dir, '')
        self._convert_filepath(self.globals.attrs, 'transport_file', proj_dir, '')
        self._convert_filepath(self.geometry.attrs, 'layer_faces_file', proj_dir, '')
        self._convert_filepath(self.initial_conditions.attrs, 'restart_file', proj_dir, '')
        if self.z_modifications.csv.data.size > 0:
            self._update_table_files(self.z_modifications.variables[ELEV_CSV_FILE_VAR], proj_dir, '')
        if self.z_modifications.grid_zpts.data.size > 0:
            self._update_table_files(self.z_modifications.variables[ELEV_GRID_ZPTS_VAR], proj_dir, '')
        if self.grid_definitions.file.data.size > 0:
            self._update_table_files(self.grid_definitions.variables['file'], proj_dir, '')
        self.global_bcs.update_file_paths()
        self.gridded_bcs.update_file_paths()
        self.commit()
        return ''

    def copy_external_files(self):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Returns:
            (str): Message on failure, empty string on success
        """
        old_proj_dir = self.info.attrs['proj_dir']
        # Target directory for package save is three directories above the component UUID folder
        target_dir = os.path.normpath(os.path.join(os.path.dirname(self._filename), '../../..'))
        self._copy_attr_file(self.globals.attrs, 'external_vertical_viscosity', old_proj_dir, target_dir)
        self._copy_attr_file(self.globals.attrs, 'transport_file', old_proj_dir, target_dir)
        self._copy_attr_file(self.geometry.attrs, 'layer_faces_file', old_proj_dir, target_dir)
        self._copy_attr_file(self.initial_conditions.attrs, 'restart_file', old_proj_dir, target_dir)
        self._copy_table_files(self.z_modifications.variables[ELEV_CSV_FILE_VAR], old_proj_dir, target_dir)
        self._copy_table_files(self.z_modifications.variables[ELEV_GRID_ZPTS_VAR], old_proj_dir, target_dir)
        self._copy_table_files(self.grid_definitions.variables['file'], old_proj_dir, target_dir)
        self.global_bcs.copy_external_files()
        self.gridded_bcs.copy_external_files()
        # Wipe the stored project directory. Paths will be absolute in GUI until resave.
        self.info.attrs['proj_dir'] = ''
        return ''

    def update_proj_dir(self):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Returns:
            (str): Message on failure, empty string on success.
        """
        try:
            old_proj_dir = self.info.attrs['proj_dir']  # Empty if first project save
            # Get the new project location. Three directories above the component UUID folder.
            comp_folder = os.path.dirname(self._filename)
            new_proj_dir = os.path.normpath(os.path.join(comp_folder, '../../..'))
            self.info.attrs['proj_dir'] = new_proj_dir
        except Exception:
            return 'There was a problem updating file TUFLOWFV Model Control paths to be relative from the' \
                   ' project directory. Any selected file paths will remain absolute.\n'

        self._convert_filepath(self.globals.attrs, 'external_vertical_viscosity', new_proj_dir, old_proj_dir)
        self._convert_filepath(self.globals.attrs, 'transport_file', new_proj_dir, old_proj_dir)
        self._convert_filepath(self.geometry.attrs, 'layer_faces_file', new_proj_dir, old_proj_dir)
        self._convert_filepath(self.initial_conditions.attrs, 'restart_file', new_proj_dir, old_proj_dir)
        if self.z_modifications.csv.data.size > 0:
            self._update_table_files(self.z_modifications.variables[ELEV_CSV_FILE_VAR], new_proj_dir, old_proj_dir)
        if self.z_modifications.grid_zpts.data.size > 0:
            self._update_table_files(self.z_modifications.variables[ELEV_GRID_ZPTS_VAR], new_proj_dir, old_proj_dir)
        if self.grid_definitions.file.data.size > 0:
            self._update_table_files(self.grid_definitions.variables['file'], new_proj_dir, old_proj_dir)
        self.global_bcs.update_proj_dir()
        self.gridded_bcs.update_proj_dir()
        self.commit()  # Save the updated project directory and referenced filepaths.
        return ''  # Don't report errors, leave that to model checks.

    @property
    def general(self):
        """Load the general parameters Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the general parameters in the main file
        """
        if self._general is None:
            self._general = self.get_dataset('general', False)
            if self._general is None:
                self._general = xr.Dataset(attrs=self._get_default_general_data())
        return self._general

    @property
    def time(self):
        """Load the time parameters Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the time parameters in the main file
        """
        if self._time is None:
            self._time = self.get_dataset('time', False)
            if self._time is None:
                self._time = xr.Dataset(attrs=self._get_default_time_data())
        return self._time

    @property
    def globals(self):
        """Load the model parameters Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the model parameters in the main file
        """
        if self._globals is None:
            self._globals = self.get_dataset('globals', False)
            if self._globals is None:
                self._globals = xr.Dataset(attrs=self._get_default_globals_data())
        return self._globals

    @property
    def wind_stress(self):
        """Load the wind stress parameters Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the wind stress parameters in the main file
        """
        if self._wind_stress is None:
            self._wind_stress = self.get_dataset('wind_stress', False)
            if self._wind_stress is None:
                self._wind_stress = xr.Dataset(attrs=self._get_default_wind_data())
        return self._wind_stress

    @property
    def geometry(self):
        """Load the geometry parameters Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the geometry parameters in the main file
        """
        if self._geometry is None:
            self._geometry = self.get_dataset('geometry', False)
            if self._geometry is None:
                geometry_data, geometry_attrs = self._get_default_geometry_data()
                self._geometry = xr.Dataset(data_vars=geometry_data, attrs=geometry_attrs)
        return self._geometry

    @geometry.setter
    def geometry(self, dset):
        """Setter for the geometry Dataset."""
        if dset:
            self._geometry = dset

    @property
    def z_modifications(self):
        """Load the z modifications Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the z_modifications in the main file
        """
        if self._z_modifications is None:
            self._z_modifications = self.get_dataset('z_modifications', False)
            if self._z_modifications is None:
                self._z_modifications = xr.Dataset(data_vars=self._get_default_z_modification_data())
        return self._z_modifications

    @z_modifications.setter
    def z_modifications(self, dset):
        """Setter for the z modifications Dataset."""
        if dset:
            self._z_modifications = dset

    @property
    def initial_conditions(self):
        """Load the initial conditions parameters Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the initial conditions parameters in the main file
        """
        if self._initial_conditions is None:
            self._initial_conditions = self.get_dataset('initial_conditions', False)
            if self._initial_conditions is None:
                self._initial_conditions = xr.Dataset(attrs=self._get_default_initial_conditions_data())
        return self._initial_conditions

    @property
    def output(self):
        """Load the output parameters Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the output parameters in the main file
        """
        if self._output is None:
            self._output = self.get_dataset('output', False)
            if self._output is None:
                output_data, output_attrs = self._get_default_output_data()
                self._output = xr.Dataset(data_vars=output_data, attrs=output_attrs)
        return self._output

    @output.setter
    def output(self, dset):
        """Setter for the output Dataset."""
        if dset:
            self._output = dset

    @property
    def output_points_coverages(self):
        """Load the output points coverages Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the output points coverages in the main file
        """
        if self._output_points_coverages is None:
            self._output_points_coverages = self.get_dataset('output_points_coverages', False)
            if self._output_points_coverages is None:
                output_data = self._get_default_output_points_coverages_data()
                self._output_points_coverages = xr.Dataset(data_vars=output_data)
        return self._output_points_coverages

    @output_points_coverages.setter
    def output_points_coverages(self, dset):
        """Setter for the output points coverages Dataset."""
        if dset:
            self._output_points_coverages = dset

    @property
    def global_set_mat(self):
        """Load the global set mat parameters Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the global set mat parameters in the main file
        """
        if self._global_set_mat is None:
            self._global_set_mat = self.get_dataset('global_set_mat', False)
            if self._global_set_mat is None:
                self._global_set_mat = self._get_default_set_mat_dataset()
        return self._global_set_mat

    @global_set_mat.setter
    def global_set_mat(self, dset):
        """Setter for the global set mat Dataset."""
        if dset:
            self._global_set_mat = dset

    @property
    def global_bcs(self):
        """Load the global cell inflow Dataset from disk.

        Returns:
            bcd.BcData: The global BCs data
        """
        if self._global_bcs is None:
            self._global_bcs = bcd.BcData(self._filename, bcd.GLOBAL_BC_GROUP)
            if self._global_bcs is None:
                self._global_bcs = bcd.BcData(self._filename, bcd.GLOBAL_BC_GROUP)
                self._global_bcs.info.attrs['next_comp_id'] = 1  # Use one-base since it will show up in the GUI
        return self._global_bcs

    @global_bcs.setter
    def global_bcs(self, dset):
        """Setter for the global BCs data."""
        if dset:
            self._global_bcs = dset

    @property
    def gridded_bcs(self):
        """Load the gridded boundary conditions data from disk.

        Returns:
            bcd.BcData: The gridded BCs data
        """
        if self._gridded_bcs is None:
            self._gridded_bcs = bcd.BcData(self._filename, bcd.GRIDDED_BC_GROUP)
            if self._gridded_bcs is None:
                self._gridded_bcs = bcd.BcData(self._filename, bcd.GRIDDED_BC_GROUP)
                self._gridded_bcs.info.attrs['next_comp_id'] = 1  # Use one-base since it will show up in the GUI
        return self._gridded_bcs

    @gridded_bcs.setter
    def gridded_bcs(self, dset):
        """Setter for the global boundary conditions Dataset."""
        if dset:
            self._gridded_bcs = dset

    @property
    def grid_definitions(self):
        """Load the grid definitions Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the grid definitions in the main file
        """
        if self._grid_definitions is None:
            self._grid_definitions = self.get_dataset('grid_definitions', False)
            if self._grid_definitions is None:
                self._grid_definitions = self._get_default_grid_definition_data()
        return self._grid_definitions

    @grid_definitions.setter
    def grid_definitions(self, dset):
        """Setter for the grid definitions Dataset."""
        if dset:
            self._grid_definitions = dset

    @property
    def linked_simulations(self):
        """Load the linked simulations Dataset from disk.

        Returns:
            xr.Dataset: Dataset interface to the linked simulations in the main file
        """
        if self._linked_simulations is None:
            self._linked_simulations = self.get_dataset('linked_simulations', False)
            if self._linked_simulations is None:
                self._linked_simulations = xr.Dataset(data_vars=self._get_default_linked_simulations_data())
        return self._linked_simulations

    @linked_simulations.setter
    def linked_simulations(self, dset):
        """Setter for the linked simulations Dataset."""
        if dset:
            self._linked_simulations = dset
