"""Module for exporting a TUFLOWFV simulation with feedback."""
# 1. Standard python modules
import logging
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.tuflowfv.data.coverage_collector import CoverageCollector
from xms.tuflowfv.dmi.xms_data import XmsData
from xms.tuflowfv.file_io import io_util
from xms.tuflowfv.file_io.control_writer import ControlWriter
from xms.tuflowfv.file_io.geom_writer import GeomWriter


class ExportSimulationRunner(ProcessFeedbackThread):
    """Class for exporting a TUFLOWFV simulation in a worker thread."""

    def __init__(self, parent):
        """Constructor.

        Args:
            parent (QWidget): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = logging.getLogger('xms.tuflowfv')
        self._exported_sim_uuids = set()
        self.query = Query()

    def _export_simulation(self, sim_uuid, sim_name):
        """Export a simulation and its linked child simulations recursively.

        Args:
            sim_uuid (str): UUID of the simulation to export, pass empty string for the root simulation
            sim_name (str): Name of the simulation to export, pass empty string for the root simulation
        """
        xms_data = XmsData(self.query)
        if sim_uuid:
            if sim_uuid in self._exported_sim_uuids:
                return  # We have already exported this simulation, nothing to do
            xms_data.set_sim(sim_uuid=sim_uuid, sim_name=sim_name)
        else:  # Make sure we don't export this simulation again if referenced by some other child simulation
            self._exported_sim_uuids.add(xms_data.sim_uuid)

        coverages = CoverageCollector(xms_data)
        self._logger.info('Exporting control file...')
        writer = ControlWriter(xms_data, coverages)
        child_sims = writer.write()
        if xms_data.do_ugrid:  # May not have a geometry if a "create GIS empties" run
            self._logger.info('Exporting geometry file...')
            writer = GeomWriter(xms_data, coverages)
            writer.write()

        # Export the child simulations
        for sim_uuid, sim_name in child_sims:
            self._export_simulation(sim_uuid, sim_name)

    def _do_work(self):
        """Creates the snap preview of coverages onto the mesh."""
        try:
            self._export_simulation('', '')
        except Exception as e:
            self._logger.error(f'Error exporting simulation. Process returned: {str(e)}')


def export_simulation_with_feedback():
    """Export a simulation with a feedback dialog.

    Returns:
        Query: The Query used for testing
    """
    # Export all TUFLOWFV executables to the same folder.
    io_util.move_to_shared_folder(os.getcwd())

    ensure_qapplication_exists()
    worker = ExportSimulationRunner(None)
    display_text = {
        'title': 'TUFLOWFV Export Simulation',
        'working_prompt': 'Exporting TUFLOWFV simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
        'success_prompt': 'Successfully exported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when exporting is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms', worker=worker,
                                      parent=None)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():
        if LogEchoQSignalStream.logged_error:
            XmEnv.report_export_error()
    else:
        XmEnv.report_export_aborted()
    return worker.query
