"""Class to read a TUFLOWFV control input file."""
# 1. Standard python modules
import copy
import datetime
import logging
import os
from pathlib import Path
import re
import uuid

# 2. Third party modules
import numpy as np
import pandas as pd
import xarray as xr

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xfs
from xms.data_objects.parameters import Coverage, FilterLocation, Point, Polygon, Projection, Simulation
from xms.guipy.data.target_type import TargetType
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.tuflowfv.components.tuflowfv_component import get_component_data_object
from xms.tuflowfv.data import bc_data as bcd
from xms.tuflowfv.data import sim_data as smd
from xms.tuflowfv.data import structure_data as std
from xms.tuflowfv.file_io import io_util
from xms.tuflowfv.file_io.bc_component_builder import BcComponentBuilder, build_global_bc_dataset
from xms.tuflowfv.file_io.bc_csv_reader import BcCsvReader
from xms.tuflowfv.file_io.culvert_csv_io import CulvertCsvReader
from xms.tuflowfv.file_io.duplicate_data_checker import DuplicateDataChecker
from xms.tuflowfv.file_io.geom_reader import TwoDMReader
from xms.tuflowfv.file_io.holland_wind_reader import HollandWindReader
from xms.tuflowfv.file_io.material_component_builder import MaterialComponentBuilder
from xms.tuflowfv.file_io.output_component_builder import OutputComponentBuilder
from xms.tuflowfv.file_io.shapefile_converter import ShapefileConverter
from xms.tuflowfv.file_io.structure_component_builder import StructureComponentBuilder
from xms.tuflowfv.file_io.structure_manager import StructureManager
from xms.tuflowfv.gui import gui_util


def check_for_variable_bc_values(atts, att_name, values):
    """Parse a variable number of values from a default, offset, or scale BC block command line.

    Args:
        atts (dict): The BC's attributes
        att_name (str): Base name of the attribute whose xarray variable name is built as <base_name>1,...,<base_name>N
        values (list[str]): The lexed values from the line
    """
    for idx in range(bcd.MAX_NUM_BC_FIELDS):
        try:
            variable_name = f'{att_name}{idx + 1}'
            atts[variable_name] = float(values[idx])
        except Exception:
            return


def z_modification_data_vars():
    """Returns a dict of the z_modifications Dataset data_vars that has no rows.

    Returns:
        dict: See description
    """
    return {
        smd.ELEV_MOD_TYPE_VAR: [],
        smd.ELEV_SET_ZPTS_VAR: [],
        smd.ELEV_GRID_ZPTS_VAR: [],
        smd.ELEV_CSV_FILE_VAR: [],
        smd.ELEV_CSV_TYPE_VAR: [],
        smd.ELEV_ZLINE_UUID_VAR: [],
        smd.ELEV_ZPOINT1_UUID_VAR: [],
        smd.ELEV_ZPOINT2_UUID_VAR: [],
        smd.ELEV_ZPOINT3_UUID_VAR: [],
        smd.ELEV_ZPOINT4_UUID_VAR: [],
        smd.ELEV_ZPOINT5_UUID_VAR: [],
        smd.ELEV_ZPOINT6_UUID_VAR: [],
        smd.ELEV_ZPOINT7_UUID_VAR: [],
        smd.ELEV_ZPOINT8_UUID_VAR: [],
        smd.ELEV_ZPOINT9_UUID_VAR: [],
    }


def bc_token_count(bc_line):
    """Find the number of tokens to parse from the first line of a BC block command.

    Args:
        bc_line (str): The first line of the BC block

    Returns:
        tuple(int, str): The number of values to parse from the bc block command or 0 if unrecognized type, and the
            BC type (upper case).
    """
    bc_type = bc_line.split()[0].split(',')[0].strip().upper()
    num_tokens = 0
    if bc_type in bcd.ARC_BC_TYPES or bc_type == 'QH':  # QH is same as HQ so not in the interface, but still read
        # ZG only has type, id. QN has type, id, friction_slope. Rest have type, id, CSV
        num_tokens = 2
        if bc_type == 'ZG':
            num_tokens = 1
        # num_tokens = 1 if bc_type == 'ZG' else num_tokens = 2
    elif bc_type in bcd.POINT_BC_TYPES:
        # type, x, y, CSV or type, name, CSV if a GIS SA points
        num_tokens = 3
    elif bc_type in bcd.POLYGON_BC_TYPES:
        num_tokens = 2
    elif bc_type in ['QG', 'CYC_HOLLAND', 'TRANSPORT']:
        # type, CSV
        num_tokens = 1
    elif bc_type in bcd.GRIDDED_BC_TYPES:
        # type, grid label, NetCDF file
        num_tokens = 2
    return num_tokens, bc_type


class ControlReader:
    """Class to read a TUFLOWFV shell input file."""
    FILE_CARDS = {
        'shp projection',
        'write empty gis files',
        'geometry 2d',
        'read gis nodestring',
        'read gis z line',
        'read gis mat',
        'cell elevation file',
        'log dir',
        'output dir',
        'write check files',
        'output points file',
        'restart',  # examples use this
        'restart file',  # docs say this
        'grid definition file',
        'read file',
        'include',  # equivalent to 'read file'
        'culvert file',
        'width file',
        'blockage file',
        'energy loss file',
        'flux file',
    }

    def __init__(self, filenames):
        """Constructor.

        Args:
            filenames (list[str]): Absolute paths to the control files to import
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._filenames = [os.path.normpath(filename) for filename in filenames]  # All the .fvcs we are going to read
        self._filename = ''  # The current .fvc we are reading
        self._cards = []  # [(<command>, [<value>])]
        self._card_idx_to_line_idx = {}  # Index of card to line number (might not match because of comment/empty lines)
        self._lines = []  # Store these so we can add unrecognized cards to advanced cards
        self._fvc_to_sim_uuid = {}  # {filename: sim_uuid} - For linking/reading recursive .fvc file references
        self._line_num = 0
        self._duplicate_checker = DuplicateDataChecker()
        self._structure_manager = StructureManager()
        self._culvert_csv_io = CulvertCsvReader()
        self.unrecognized_lines = []

        # Intermediate data read from file - reset with each control file
        self._sim_uuid = ''  # UUID of the current simulation
        self._sim_name = ''  # Basename of the current control file
        self._sim_data = None  # Data for the current simulation
        self._sim_comp_uuid = ''  # UUID of the current simulation's component
        self._bc_reference_time = None  # Reference timestamp of the current BC
        self._2dm_file = ''  # .2dm file for the current simulation
        self._wkt = ''  # Projection of all the data in the current sim
        self._vertical_units = 'METERS'
        self._default_mat_id = None  # This overwrites all material assignments from the 2dm (set mat)
        self._global_bcs = []  # [{variable: value}] - Global BC
        self._materials = {}  # {mat_id: {variable: value}}
        self._nodestrings = {}  # {bc_id: {variable: value}}
        self._bc_points = {}  # {bc_id: {variable: value}}
        self._bc_points_locations = {}  # {BC_ID: (x,y)}
        self._bc_polygons = {}  # {bc_id: {variable: value}}
        self._structure_arcs = {}  # {struct_id: {variable: value}}
        self._structure_polys = {}  # {struct_id: {variable: value}}
        self._structure_poly_locations = []  # [data_object]
        self._gis_bc_files = []  # [filename]
        self._gis_mat_files = []  # [filename]
        self._gis_sa_files = []  # [filename]
        self._gis_po_files = []  # [filename]
        self._gis_struct_zone_files = []  # [filename]
        self._output_blocks = []  # [{variable: value}]
        self._output_coverage_uuids = []  # [uuid]
        self._output_coverage_indices = []  # [index]
        self._z_modifications = {}  # data_vars for the current simulation's z_modifications Dataset
        self._wind_boundary_files = []  # [CSV filename]
        self._grid_definitions = {}  # {label: grid_id}
        self._gridded_bcs = {}  # {grid_label: [{variable: value}]}
        self._child_sim_links = []  # [uuid] - linked child simulations
        # For the structures coverage
        self._arc_id_to_feature_id = {}
        self._arc_id_to_feature_name = {}
        self._poly_id_to_feature_id = {}
        self._poly_id_to_feature_name = {}

        # TUFLOWFV requires all geometric data to be in the same projection. If the GIS projection has been specified,
        # we will read the WKT from that. If no GIS projection but spherical=1, we set to the default GEOGRAPHIC
        # projection. In either case we set the vertical units to meters (default in TUFLOWFV) if Imperial units are
        # not specified in the control file. MIF/MID GIS format (default in TUFLOWFV) is not supported. If we have GIS
        # data to import but the format is set or defaulted to MIF, we report an error.
        self._specified_shp_format = False  # Error if have GIS inputs but use TUFLOWFV default of .mif/.mid files
        self._geographic = False  # If spherical=1 and no GIS projection supplied, set to default GEOGRAPHIC
        self._do_projection = None  # data_object Projection of all geometric data

        # Outputs
        self._do_sims = []  # [Simulation]  # parallel with self._do_sim_comps
        self._do_sim_comps = []  # [SimComponent]  # parallel with self._do_sims
        self._do_ugrids = []  # [UGrid]
        self._sim_links = []  # [(sim_uuid, taken_uuid)]
        self._gis_zline_files = []  # [{uuid: (filename, [(uuid, filename)])}] - For each Z line up to 9 point layers
        self._bc_coverages = []  # [[(Coverage, component)]] - inner list per sim
        self._material_coverages = []  # [[(Coverage, component)]] - inner list per sim
        self._output_points_coverages = []  # [[(Coverage, component)]] - inner list per sim
        self._wind_coverages = []  # [WindCoverage]
        self._structure_coverages = []  # [[(Coverage, component)]] - inner list per sim

        # Card reader methods
        self._single_card_readers = [
            self._process_single_general_card,
            self._process_single_time_card,
            self._process_single_global_card,
            self._process_single_viscosity_card,
            self._process_single_diffusivity_card,
            self._process_single_wind_stress_card,
            self._process_single_geometry_card,
            self._process_single_initial_conditions_card,
            self._process_single_global_output_card,
            self._process_single_global_set_mat_card,
            self._process_single_global_bc_card,
        ]

    def _reset(self, filename):
        """Reset member variables before reading a control file.

        Args:
            filename (str): Absolute path to the control file we are about to read
        """
        self._filename = filename
        # Ensure no spaces in the name
        self._sim_name = os.path.splitext(os.path.basename(filename))[0].replace(' ', '_')
        self._sim_uuid = self._fvc_to_sim_uuid.setdefault(filename, str(uuid.uuid4()))
        self._sim_data = None
        self._sim_comp_uuid = str(uuid.uuid4())
        self._2dm_file = ''
        self._wkt = ''
        self._line_num = 0
        self._lines = []
        self._cards = []
        self._card_idx_to_line_idx = {}
        self._default_mat_id = None
        self._materials = {}
        self._nodestrings = {}
        self._global_bcs = []
        self._bc_points = {}
        self._bc_points_locations = {}
        self._bc_polygons = {}
        self._structure_arcs = {}
        self._structure_polys = {}
        self._structure_poly_locations = []  # Store the polygon locations here because they cannot be in the 2dm file.
        self._gis_bc_files = []
        self._gis_mat_files = []
        self._gis_sa_files = []
        self._gis_po_files = []
        self._gis_struct_zone_files = []
        self._wind_boundary_files = []
        self._output_blocks = []
        self._output_coverage_uuids = []
        self._output_coverage_indices = []
        self._z_modifications = z_modification_data_vars()
        self._grid_definitions = {}
        self._gridded_bcs = {}
        self._child_sim_links = []
        self._specified_shp_format = False
        self._geographic = False
        self._do_projection = None
        self._structure_manager.clear()
        self._arc_id_to_feature_id = {}
        self._arc_id_to_feature_name = {}
        self._poly_id_to_feature_id = {}
        self._poly_id_to_feature_name = {}
        self.unrecognized_lines = []
        # Switch the .fvc filename in the CSV reader but don't create a new one. It caches CSV files it has already read
        self._culvert_csv_io.switch_simulation(self._filename)

        # Append another simulation dimension to the outputs
        self._bc_coverages.append([])
        self._material_coverages.append([])
        self._output_points_coverages.append([])
        self._structure_coverages.append([])
        self._gis_zline_files.append({})

    def _get_input_filepath(self, input_file):
        """Get a clean absolute path to an input file referenced in the control file.

        Args:
            input_file (str): Input file path as parsed from the control file

        Returns:
            str: Normalized absolute path to the input file
        """
        abs_path = None
        for possible_root in self._filenames:
            abs_path = xfs.resolve_relative_path(os.path.dirname(possible_root), input_file)
            abs_path = os.path.normpath(abs_path)
            if os.path.exists(abs_path):
                break  # Found the file, so time to stop looking
            else:
                abs_path = None
        if not abs_path:
            abs_path = xfs.resolve_relative_path(os.path.dirname(self._filename), input_file)
            abs_path = os.path.normpath(abs_path)
            self._logger.error(f'Unable to find input filepath: {input_file}\nResolved to: {abs_path}')
        return abs_path

    def _build_projection(self):
        """Build the data_objects projection for all geometric data after reading the control file."""
        # Set the horizontal units in addition to the vertical units in case we did not read a projection file. This is
        # required so SMS can set the projection to "No projection with units". If we read a WKT or are in default
        # GEOGRAPHIC projection, SMS will ignore the horizontal units.
        self._do_projection = Projection(wkt=self._wkt, vertical_units=self._vertical_units,
                                         horizontal_units=self._vertical_units)
        # If no GIS projection provided but spherical=1, set to default GEOGRAPHIC projection
        if not self._wkt and self._geographic:
            self._do_projection.coordinate_system = 'GEOGRAPHIC'
        self._log_missing_projection_warnings()

    def _check_for_data_in_2dm(self):
        """Read the geometry, boundary condition bcs, and material assignments from the .2dm file."""
        if not os.path.isfile(self._2dm_file):
            if self._2dm_file:  # Don't log error if no geometry defined in file. Might be a "create GIS empties" run.
                self._logger.error(f'Unable to find .2dm geometry file: {self._2dm_file}.')
            return

        # Check if we have already read any matching data from the file.
        mat_uuid = ''
        bc_uuid = ''  # None if no bcs in the file
        ugrid_uuid = self._duplicate_checker.have_read_2dm(self._2dm_file)
        if ugrid_uuid:  # Have already read this 2dm, check if we have matching material or BC atts
            bc_uuid = self._duplicate_checker.have_read_2dm_bcs(self._2dm_file, self._nodestrings)
            mat_uuid = self._duplicate_checker.have_read_2dm_materials(self._2dm_file, self._materials)
        # Read any data in the .2dm that might be missing from previously imported simulations.
        ugrid_uuid, bc_uuid, mat_uuid = self._read_2dm_file(ugrid_uuid, bc_uuid, mat_uuid)
        # Link the built items to the simulation
        self._sim_links.append((self._sim_uuid, ugrid_uuid))
        self._sim_links.append((self._sim_uuid, mat_uuid))
        if bc_uuid:
            self._sim_links.append((self._sim_uuid, bc_uuid))
        self._sim_data.info.attrs['domain_uuid'] = ugrid_uuid

    def _read_2dm_file(self, ugrid_uuid, bc_uuid, mat_uuid):
        """Read a .2dm file.

        Args:
            ugrid_uuid (str): UUID of the UGrid if we have previously read this .2dm, empty string if we have not
            bc_uuid (str): UUID of the UGrid if we have previously read this .2dm, empty string if we have not, None if
                we previously read the .2dm but it contains no bcs
            mat_uuid (str): UUID of the Materials coverage if we have previously read this .2dm, empty string if we have
                not

        Returns:
            tuple(str, str, str): UUID of the UGrid, UUID of the BC coverage (if there are bcs), UUID of the
                Materials coverage
        """
        read_mat = not mat_uuid
        read_bc = False if bc_uuid or bc_uuid is None else True
        if not ugrid_uuid or read_mat or read_bc:
            # Need something, so have to read the whole file.
            twodm_reader = TwoDMReader(filename=self._2dm_file, wkt=self._wkt, do_projection=self._do_projection,
                                       create_bc_component=read_bc, create_mat_component=read_mat,
                                       manager=self._structure_manager)
            twodm_reader.read(materials=self._materials, bcs=self._nodestrings)
            if not ugrid_uuid:
                ugrid_uuid = twodm_reader.do_ugrid.uuid
                self._do_ugrids.append(twodm_reader.do_ugrid)
            if read_mat:
                mat_uuid = twodm_reader.material_coverage.uuid
                self._material_coverages[-1].append((twodm_reader.material_coverage, twodm_reader.material_comp))
            if read_bc and twodm_reader.bc_coverage:
                bc_uuid = twodm_reader.bc_coverage.uuid
                self._bc_coverages[-1].append((twodm_reader.bc_coverage, twodm_reader.bc_comp))
            self._duplicate_checker.add_imported_2dm_data(filename=self._2dm_file, ugrid_uuid=ugrid_uuid,
                                                          bc_uuid=bc_uuid, mat_uuid=mat_uuid,
                                                          bc_parameters=self._nodestrings,
                                                          mat_parameters=self._materials)
        return ugrid_uuid, bc_uuid, mat_uuid

    def _add_z_modification(self, mod_type, constant=0.0, grid_zpts_file='', csv_file='',
                            csv_type=smd.CELL_ELEV_CSV_TYPE_CELL, zline_uuid='', z_point_lines=None):
        """Add a 'Set Zpts', 'GRID Zpts', 'Cell Elevation File', or 'ZLine' Z modification.

        Args:
            mod_type (int): The type of Z modification
            constant (Optional[int]): Constant elevation if adding a 'Set Zpts'
            grid_zpts_file (Optional[int]): Absolute path to a raster file if adding a 'GRID Zpts'
            csv_file (Optional[str]): Absolute path to a CSV file if adding a 'Cell Elevation File'
            csv_type (Optional[str]): The format of the CSV file if adding a 'Cell Elevation File'
            zline_uuid (Optional[str]): The UUID of a ZLine shapefile if adding a 'ZLine'
            z_point_lines (Optional[list[str]]): Up to nine UUIDs of ZPoint shapefiles if adding a 'ZLine'
        """
        self._z_modifications[smd.ELEV_MOD_TYPE_VAR].append(mod_type)
        self._z_modifications[smd.ELEV_SET_ZPTS_VAR].append(constant)
        self._z_modifications[smd.ELEV_GRID_ZPTS_VAR].append(grid_zpts_file)
        self._z_modifications[smd.ELEV_CSV_FILE_VAR].append(csv_file)
        self._z_modifications[smd.ELEV_CSV_TYPE_VAR].append(csv_type)
        self._z_modifications[smd.ELEV_ZLINE_UUID_VAR].append(zline_uuid)
        z_points = z_point_lines if z_point_lines else []
        num_z_points = len(z_points)
        self._z_modifications[smd.ELEV_ZPOINT1_UUID_VAR].append(z_points[0] if num_z_points > 0 else '')
        self._z_modifications[smd.ELEV_ZPOINT2_UUID_VAR].append(z_points[1] if num_z_points > 1 else '')
        self._z_modifications[smd.ELEV_ZPOINT3_UUID_VAR].append(z_points[2] if num_z_points > 2 else '')
        self._z_modifications[smd.ELEV_ZPOINT4_UUID_VAR].append(z_points[3] if num_z_points > 3 else '')
        self._z_modifications[smd.ELEV_ZPOINT5_UUID_VAR].append(z_points[4] if num_z_points > 4 else '')
        self._z_modifications[smd.ELEV_ZPOINT6_UUID_VAR].append(z_points[5] if num_z_points > 5 else '')
        self._z_modifications[smd.ELEV_ZPOINT7_UUID_VAR].append(z_points[6] if num_z_points > 6 else '')
        self._z_modifications[smd.ELEV_ZPOINT8_UUID_VAR].append(z_points[7] if num_z_points > 7 else '')
        self._z_modifications[smd.ELEV_ZPOINT9_UUID_VAR].append(z_points[8] if num_z_points > 8 else '')

    def _read_gis_data(self):
        """Read all the GIS shapefiles referenced in the control file."""
        self._convert_gis_materials()
        self._convert_gis_nodestrings()
        self._convert_gis_sa()
        self._convert_structure_zones()

    def _convert_structure_zones(self):
        """Convert all the GIS structure shapefiles referenced in the control file to coverages."""
        for gis_zone in self._gis_struct_zone_files:
            converter = ShapefileConverter(filename=gis_zone, manager=self._structure_manager)
            loc_map = {poly.id: poly for poly in converter.convert_polygons(create_coverage=False)}
            # Reorganize the structure assignment dict from {feature_id: struct_id} to {struct_id: [feature_id]}
            for feature_id, poly_id, in converter.feature_id_to_att_id.items():
                self._poly_id_to_feature_id.setdefault(poly_id, []).append(feature_id)
                self._structure_manager.add_poly_structure_location(
                    converter.feature_id_to_feature_name[feature_id],
                    loc_map[feature_id]
                )
            self._poly_id_to_feature_name.update(converter.feature_id_to_feature_name)

    def _convert_gis_materials(self):
        """Convert all the GIS material shapefiles referenced in the control file to coverages."""
        for gis_mat in self._gis_mat_files:
            mat_uuid = self._duplicate_checker.have_read_gis_materials(gis_mat, self._materials)
            if not mat_uuid:
                converter = ShapefileConverter(filename=gis_mat)
                do_cov = converter.convert_polygons()
                # Reorganize the material assignment dict from {feature_id: mat_id} to {mat_id: [feature_id]}
                poly_assignments = {}
                for feature_id, mat_id, in converter.feature_id_to_att_id.items():
                    poly_assignments.setdefault(mat_id, []).append(feature_id)
                comp_builder = MaterialComponentBuilder(cov_uuid=do_cov.uuid, from_2dm=False,
                                                        poly_assignments=poly_assignments,
                                                        existing_data=self._materials)
                mat_comp = comp_builder.build_material_component()
                self._material_coverages[-1].append((do_cov, mat_comp))
                mat_uuid = do_cov.uuid
                self._duplicate_checker.add_imported_gis_materials(gis_mat, mat_uuid, self._materials)
            self._sim_links.append((self._sim_uuid, mat_uuid))

    def _convert_gis_nodestrings(self):
        """Convert all the GIS BC nodestring shapefiles referenced in the control file to coverages."""
        for gis_bc in self._gis_bc_files:
            cov_uuid = self._duplicate_checker.have_read_gis_bc(gis_bc, self._nodestrings)
            if not cov_uuid:
                converter = ShapefileConverter(filename=gis_bc, manager=self._structure_manager)
                features = [
                    converter.convert_points(),
                    converter.convert_lines(),
                    converter.convert_polygons(),
                ]
                for feature in features:
                    if feature is None:
                        continue
                    cov_uuid = feature.uuid

                    # Reorganize the assignment dict from {feature_id: bc_id} to {bc_id: feature_id}
                    bc_assignments = {bc_id: feature_id for feature_id, bc_id in converter.feature_id_to_att_id.items()}
                    comp_builder = BcComponentBuilder(
                        cov_uuid=feature.uuid,
                        from_2dm=False,
                        bc_atts=self._nodestrings,
                        nodestring_id_to_feature_id=bc_assignments,
                        nodestring_id_to_feature_name=converter.feature_id_to_feature_name
                    )
                    do_comp = comp_builder.build_bc_component()
                    self._bc_coverages[-1].append((feature, do_comp))
                    self._duplicate_checker.add_imported_gis_bc(gis_bc, cov_uuid, self._nodestrings)
            self._sim_links.append((self._sim_uuid, cov_uuid))

    def _convert_gis_sa(self):
        """Convert all the GIS BC source area, point or polygon."""
        for gis_bc in self._gis_sa_files:
            bc_uuid = self._duplicate_checker.have_read_gis_bc(gis_bc, self._bc_points)
            if not bc_uuid:  # No point coverages, look for polygons
                bc_uuid = self._duplicate_checker.have_read_gis_bc(gis_bc, self._bc_polygons)
            if not bc_uuid:  # Have not read this coverage yet
                converter = ShapefileConverter(filename=gis_bc)
                do_cov = converter.convert_points()
                atts = self._bc_points
                if not do_cov:  # Wasn't a point shapefile, try polygons
                    do_cov = converter.convert_polygons()
                    atts = self._bc_polygons
                if do_cov:
                    # Reorganize the assignment dict from {feature_id: bc_id} to {bc_id: feature_id}
                    bc_assignments = {bc_id: feature_id for feature_id, bc_id in converter.feature_id_to_att_id.items()}
                    comp_builder = BcComponentBuilder(
                        cov_uuid=do_cov.uuid,
                        from_2dm=False,
                        bc_atts=atts,
                        nodestring_id_to_feature_id=bc_assignments,
                        nodestring_id_to_feature_name=converter.feature_id_to_feature_name
                    )
                    do_comp = comp_builder.build_bc_component()
                    self._bc_coverages[-1].append((do_cov, do_comp))
                    bc_uuid = do_cov.uuid
                    self._duplicate_checker.add_imported_gis_bc(gis_bc, bc_uuid, atts)
                    self._sim_links.append((self._sim_uuid, bc_uuid))
                else:
                    self._logger.warning(f'Unable to read shapefile: {gis_bc}')

    def _build_z_modifications_dataset(self):
        """Create references to all Z-line shapefiles in the simulation data."""
        if not self._z_modifications:
            return  # nothing to do
        # Convert dict of lists to dict of xarray DataArrays
        data_vars = {
            smd.ELEV_MOD_TYPE_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_MOD_TYPE_VAR], dtype=np.int32)
            ),
            smd.ELEV_SET_ZPTS_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_SET_ZPTS_VAR], dtype=np.float64)
            ),
            smd.ELEV_GRID_ZPTS_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_GRID_ZPTS_VAR], dtype=object)
            ),
            smd.ELEV_CSV_FILE_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_CSV_FILE_VAR], dtype=object)
            ),
            smd.ELEV_CSV_TYPE_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_CSV_TYPE_VAR], dtype=object)
            ),
            smd.ELEV_ZLINE_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZLINE_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT1_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT1_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT2_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT2_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT3_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT3_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT4_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT4_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT5_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT5_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT6_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT6_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT7_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT7_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT8_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT8_UUID_VAR], dtype=object)
            ),
            smd.ELEV_ZPOINT9_UUID_VAR: xr.DataArray(
                data=np.array(self._z_modifications[smd.ELEV_ZPOINT9_UUID_VAR], dtype=object)
            ),
        }
        self._sim_data.z_modifications = xr.Dataset(data_vars=data_vars)

    def _read_wind_boundaries(self):
        """Read the Holland wind boundaries in this simulation (if needed)."""
        reader = HollandWindReader(reftime=self._bc_reference_time, do_projection=self._do_projection)
        for wind_csv in self._wind_boundary_files:
            cov_uuid = self._duplicate_checker.check_for_imported_file(wind_csv)
            if cov_uuid:  # Already created a coverage for this CSV, just link it since no atts outside the CSV
                cov_uuid = cov_uuid[0]
            else:  # Haven't read this particular wind boundary CSV
                wind_dump = reader.read(wind_csv)
                if wind_dump:
                    cov_uuid = wind_dump.m_cov.uuid
                    self._wind_coverages.append(wind_dump)
                    self._duplicate_checker.add_imported_file_no_atts(cov_uuid, wind_csv)
            if cov_uuid:
                self._sim_links.append((self._sim_uuid, cov_uuid))

    def _parse_cards(self, lines):
        """Parse the cards in the .fvc file.

        Args:
            lines (list of str): The lines of the .fvc file
        """
        self._logger.info('Parsing .fvc file cards...')
        for idx, line in enumerate(lines):
            line = line.strip()
            skip_line = not line or line.startswith('!') or line.startswith('#')  # blank lines and comments
            if skip_line:
                continue
            split_line = line.split('==')  # Split the line into the command and the values
            command = split_line[0].strip().lower() if split_line else ''
            if len(split_line) == 1 and command.startswith('end'):  # Add the end card of a block to the stack
                self._card_idx_to_line_idx[len(self._cards)] = idx
                self._cards.append((command, []))
            elif len(split_line) < 2:
                self._logger.warning(f'Unrecognized line in .fvc file at {idx + 1}: {line}')
            else:
                self._card_idx_to_line_idx[len(self._cards)] = idx
                self._parse_command_values(command, split_line)

    def _parse_command_values(self, command, split_line):
        """Parse the values portion of a command line.

        Args:
            command (str): The command card, all lower case
            split_line (list[str]): The line split on the '==' string
        """
        # Split the values portion. Can be comma or space delimited. Can be files with spaces, but filenames
        # can also contain commas.
        value = re.split('[!#]', split_line[1])[0].strip()  # Strip trailing comments from the line
        if command == 'bc':
            # Special case for BC block card, need to parse based on type
            num_tokens, bc_type = bc_token_count(value)
            if not num_tokens:
                self._logger.error(f'Unrecognized BC type: {value}')
            value = value.replace(',', ' ', num_tokens)
            lexed_value = value.split(maxsplit=num_tokens)
            try:
                lexed_value[num_tokens] = lexed_value[num_tokens].replace('"', '')  # Strip quotes
            except IndexError:  # Try to parse as a GIS SA point file
                num_tokens -= 1
                if bc_type == 'QC' and len(lexed_value) >= num_tokens:
                    lexed_value[num_tokens] = lexed_value[num_tokens].replace('"', '')  # Strip quotes
                else:
                    self._logger.error(f'Unrecognized BC type: {value}')
        elif command in self.FILE_CARDS:
            # If multiple files per line, separated with a vertical bar
            lexed_value = [val.replace('"', '') for val in value.split('|')]  # Strip quotes
            if command == 'cell elevation file':  # Special case for cell elevation file. Second value after last comma.
                lexed_value = lexed_value[0].rsplit(',', 1)
        else:
            lexed_value = value.split(',')
        values = [val.strip() for val in lexed_value]
        self._cards.append((command, values))

    def _process_cards(self):
        """Populated simulation data from parsed cards."""
        self._logger.info('Processing parsed cards...')
        while self._line_num < len(self._cards):
            command, values = self._cards[self._line_num]
            if command == 'bc':
                self._initialize_new_bc()
            elif command == 'output':
                self._process_output_block()
            elif command == 'material':
                self._process_material_block()
            elif command.startswith('grid definition file'):
                self._process_grid_definition_block()
            elif command == 'structure':
                self._process_structure_block()
            elif command.startswith('read gis'):
                self._process_gis_command()
            else:
                self._process_single_card()

    def _process_gis_command(self):
        """Handler for a 'Read GIS*' card."""
        command, values = self._cards[self._line_num]
        if command == 'read gis mat':
            self._gis_mat_files.append(self._get_input_filepath(values[0]))
        elif command == 'read gis nodestring':
            self._gis_bc_files.append(self._get_input_filepath(values[0]))
        elif command == 'read gis z line':
            self._parse_gis_z_line_files(values)
        elif command == 'read gis sa':
            self._gis_sa_files.append(self._get_input_filepath(values[0]))
        elif command == 'read gis po':
            self._gis_po_files.append(self._get_input_filepath(values[0]))
        elif command == 'read gis zone':
            self._gis_struct_zone_files.append(self._get_input_filepath(values[0]))
        else:
            self._add_unrecognized_card('')
        # Increment past end of Read GIS card
        self._line_num += 1

    def _parse_gis_z_line_files(self, values):
        """Parse line and point shapefiles from a read GIS z line line.

        Args:
            values (list): The lexed value portion of the line
        """
        # Generate a UUID for the shapefile so we can store a reference to it in the simulation data.
        line_filename = self._get_input_filepath(values[0])
        line_uuid = self._duplicate_checker.get_gis_zline_uuid(line_filename)
        point_layers = []  # Gather all the point shapefiles that go with this Z line (up to 8)
        point_uuids = []
        for idx in range(1, len(values)):
            point_filename = self._get_input_filepath(values[idx])
            point_uuid = self._duplicate_checker.get_gis_zline_uuid(point_filename)
            point_uuids.append(point_uuid)
            point_layers.append((point_uuid, point_filename))
        self._gis_zline_files[-1][line_uuid] = (line_filename, point_layers)
        self._add_z_modification(mod_type=smd.ELEV_TYPE_ZLINE, zline_uuid=line_uuid, z_point_lines=point_uuids)

    def _initialize_new_bc(self):
        """Setup for reading a new BC block."""
        self._bc_reference_time = None
        command, values = self._cards[self._line_num]  # Get the material ID from the first line
        bc_type = values[0].upper()
        x_is_time = bcd.is_x_time(bc_type)
        csv_idx = 2  # Index of the csv file (last of the values, but before comments)
        if bc_type == 'QC':  # This is a point BC
            csv_idx = 3
            try:  # First try to read the old style with x,y locations in the BC header.
                x = float(values[1])
                y = float(values[2])
                bc_id = f'BC_Point_{len(self._bc_points) + 1}'
                self._bc_points_locations[bc_id] = (x, y)
            except ValueError:  # Assume new GIS
                bc_id = values[1]
                csv_idx = 2
            atts = self._bc_points.setdefault(bc_id, {})
            atts['name'] = bc_id
            default_columns = bcd.POINT_BC_TYPES[bc_type]
        elif bc_type in bcd.POLYGON_BC_TYPES:
            bc_id = len(self._bc_polygons) + 1
            atts = self._bc_polygons.setdefault(bc_id, {})
            default_columns = bcd.POLYGON_BC_TYPES[bc_type]
        elif bc_type in bcd.GLOBAL_BC_TYPES:
            atts, _ = bcd.get_default_bc_data_dict(False)
            self._global_bcs.append(atts)
            default_columns = bcd.GLOBAL_BC_TYPES[bc_type]
            csv_idx = 1
        elif bc_type == 'CYC_HOLLAND':  # Wind track boundary, will read into its own coverage
            csv_idx = 1
            default_columns = []
            atts = {}
        elif bc_type == 'TRANSPORT':
            csv_idx = 1
            self._sim_data.globals.attrs['transport_mode'] = 1
            self._sim_data.globals.attrs['transport_file'] = self._get_input_filepath(values[1])
            default_columns = []
            atts = {}  # No more data for transport mode boundary but need to read until end bc card.
        elif bc_type in bcd.GRIDDED_BC_TYPES:  # Gridded BCs, reference NetCDF files instead of reading curve CSV
            default_columns = []
            atts, _ = bcd.get_default_bc_data_dict(True)
            self._gridded_bcs.setdefault(values[1], []).append(atts)
        else:  # This is a nodestring BC
            try:
                bc_id = int(values[1])
            except ValueError:
                bc_id = values[1]
            atts = self._nodestrings.setdefault(bc_id, {})
            atts['name'] = bc_id
            if bc_type == 'QH':
                bc_type = 'HQ'  # Feedback from Mitch says QH is the same as HQ
            elif bc_type == 'ZG':
                csv_idx = 0  # No curve for zero-gradient but set to zero so we don't blow below
            # Get the default CSV column (or NetCDF variable if curtain BC) names
            default_columns = bcd.ARC_BC_TYPES[bc_type]

        atts['type'] = bc_type
        bc_file = values[csv_idx]
        if bc_type == 'QN':  # Special case for 'QN' Boundary no csv file - last value is friction slope
            atts['friction_slope'] = float(bc_file)
        # We read these separately. I don't think there are any attributes in the BC block, we just need to keep reading
        # until the end card.
        elif bc_type == 'CYC_HOLLAND':
            self._wind_boundary_files.append(self._get_input_filepath(bc_file))
        self._process_bc_block(atts, bc_file, default_columns, x_is_time)

    def _process_bc_block(self, atts, bc_file, default_columns, x_is_time):
        """Process the cards in an BC block.

        Args:
            atts (dict): Dict of BC attributes to fill
            bc_file (str): Path to the BC CSV file
            default_columns (list[str]): Default column header names for the BC curve
            x_is_time (bool): True if the first column is time
        """
        command = ''
        while command != 'end bc':
            self._line_num += 1
            command, values = self._cards[self._line_num]
            if command == 'bc header':
                bc_file = self._get_input_filepath(bc_file)  # CSV or NetCDF file
                bc_type = atts['type']
                if bc_type in bcd.GRIDDED_BC_TYPES.union(bcd.CURTAIN_BC_TYPES):
                    atts['grid_dataset_file'] = bc_file
                    for idx in range(len(values)):
                        atts[f'variable{idx + 1}'] = values[idx]  # xarray Variables named 'variable1',...,'variableN'
                else:
                    csv_reader = BcCsvReader(filename=bc_file, default_columns=default_columns, user_columns=values,
                                             x_is_time=x_is_time)
                    atts['curve'] = csv_reader.read().to_xarray()
                    if csv_reader.uses_isodate:
                        # Make sure this gets set if the curve uses isotime. Docs say hour offsets are the default if
                        # not specified, but that is a lie.
                        atts['time_units'] = 'Isotime'
            else:
                self._add_bc_att(command, values, atts)
        # Increment past end of BC card
        self._line_num += 1

    def _process_output_block(self):
        """Process the cards in an output block."""
        command, values = self._cards[self._line_num]
        parameters = smd.get_default_output_blocks_values(True)
        point_csv_file = ''
        output_type = ''
        gis_po = False  # Need to handle these guys a little different
        row_index = len(self._output_blocks)
        while command != 'end output':
            if command == 'output':
                output_type = values[0].lower()
            elif command == 'output parameters':
                self._process_output_parameters(values, parameters)
            elif command == 'output interval':
                parameters['define_interval'] = 1
                parameters['output_interval'] = float(values[0])
            elif command == 'start output':
                parameters['define_start'] = 1
                try:
                    parameters['output_start'] = float(values[0])
                except ValueError:  # Must be specified as an ISODATE, better be consistent with global time format.
                    parameters['use_isodate'] = 1
                    parameters['output_start_date'] = smd.parse_tuflow_isodate(values[0], to_qt=False)
            elif command == 'final output':
                parameters['define_final'] = 1
                try:
                    parameters['output_final'] = float(values[0])
                except ValueError:  # Must be specified as an ISODATE, better be consistent with global time format.
                    parameters['use_isodate'] = 1
                    parameters['output_final_date'] = smd.parse_tuflow_isodate(values[0], to_qt=False)
            elif command == 'suffix':
                parameters['suffix'] = values[0]
            elif command == 'output compression':
                parameters['define_compression'] = 1
                parameters['compression'] = io_util.str2float2int(values[0])
            elif command == 'output statistics':
                parameters['define_statistics'] = 1
                stats_type = values[0].title()
                if len(values) > 1 and values[1].lower() in ['min', 'max']:
                    stats_type = 'Both'
                parameters['statistics_type'] = stats_type
            elif command == 'output statistics dt':
                parameters['define_statistics_dt'] = 1
                parameters['statistics_dt'] = float(values[0])
            elif command == 'output points file':
                point_csv_file = values[0]
            elif command == 'read gis po':
                self._process_gis_command()
                gis_po = True
            else:
                self._add_unrecognized_card('Unsupported output block command')

            cov_uuid = ''
            if point_csv_file:
                cov_uuid = self._build_output_points_coverage(point_csv_file, parameters)
                point_csv_file = ''  # Make sure we don't link the same coverage multiple times
            elif gis_po:
                cov_uuid = self._build_gis_output_points_coverage(parameters)
                gis_po = False  # Make sure we don't link the same coverage multiple times
            if cov_uuid:
                self._output_coverage_uuids.append(cov_uuid)
                self._output_coverage_indices.append(row_index)
            # If we called _process_gis_command(), the line number has already been incremented.
            if command != 'read gis po':
                self._line_num += 1
            command, values = self._cards[self._line_num]

        parameters['row_index'] = row_index
        if output_type in ['datv', 'xmdf', 'netcdf', 'flux', 'mass', 'points', 'transport']:
            parameters['format'] = smd.format_output_type(output_type)
            self._output_blocks.append(parameters)
        else:
            self._add_unrecognized_card(f'Unrecognized output block type: {output_type}')
        # Increment past end of output card
        self._line_num += 1

    def _add_material_att(self, command, values, atts):
        """Switch statement that adds an attribute inside a material block to the material's data.

        Args:
            command (str): The command card (all lower)
            values (list[str]): The lexed values
            atts (dict): The attribute dict for the material
        """
        if command == 'inactive':
            atts['inactive'] = io_util.str2float2int(values[0])
        elif command == 'bottom roughness':
            atts['override_bottom_roughness'] = 1
            atts['bottom_roughness'] = float(values[0])
        elif command == 'horizontal eddy viscosity':
            atts['override_horizontal_eddy_viscosity'] = 1
            atts['horizontal_eddy_viscosity'] = float(values[0])
        elif command == 'horizontal eddy viscosity limits':
            atts['override_horizontal_eddy_viscosity_limits'] = 1
            atts['horizontal_eddy_viscosity_minimum'] = float(values[0])
            atts['horizontal_eddy_viscosity_maximum'] = float(values[1])
        elif command == 'vertical eddy viscosity limits':
            atts['override_vertical_eddy_viscosity_limits'] = 1
            atts['vertical_eddy_viscosity_minimum'] = float(values[0])
            atts['vertical_eddy_viscosity_maximum'] = float(values[1])
        elif command == 'horizontal scalar diffusivity':
            atts['override_horizontal_scalar_diffusivity'] = 1
            atts['horizontal_scalar_diffusivity'] = float(values[0])
        elif command == 'horizontal scalar diffusivity limits':
            atts['override_horizontal_scalar_diffusivity_limits'] = 1
            atts['horizontal_scalar_diffusivity_minimum'] = float(values[0])
            atts['horizontal_scalar_diffusivity_maximum'] = float(values[1])
        elif command == 'vertical scalar diffusivity limits':
            atts['override_vertical_scalar_diffusivity_limits'] = 1
            atts['vertical_scalar_diffusivity_minimum'] = float(values[0])
            atts['vertical_scalar_diffusivity_maximum'] = float(values[1])
        elif command == 'bed elevation limits':
            atts['override_bed_elevation_limits'] = 1
            atts['bed_elevation_minimum'] = float(values[0])
            atts['bed_elevation_maximum'] = float(values[1])
        elif command == 'spatial reconstruction':
            atts['spatial_reconstruction'] = io_util.str2float2int(values[0])
        elif command == 'end material':
            pass  # Just pass here. This card will trigger exit of the calling material block parsing method.
        else:
            self._add_unrecognized_card('Unsupported material block command')

    def _set_flux_function(self, atts, values):
        """Set the flux function variable, warning if the flux function or struct type is unsupported.

        Args:
            atts (dict): Attributes of the structure
            values (values): The parsed flux function line
        """
        # Be case-insensitive with these guys.
        flux_function = values[0].title()
        unknown_type = flux_function not in std.FLUX_FUNCTION_TYPES and flux_function.lower() != 'weir_dz'
        unsupported_struct = std.FLUX_FUNCTION_UNSUPPORTED_STRUCTS.get(flux_function, ())
        unsupported_struct = True if atts['struct_type'] in unsupported_struct else False
        if unknown_type or unsupported_struct:
            line_num = self._card_idx_to_line_idx[self._line_num] + 1
            if unknown_type:
                self._logger.warning(f'Unsupported flux function found: {values[0]}')
            else:  # Unsupported struct_type
                self._logger.warning(
                    f'Line {line_num}: Unsupported structure type "{atts["struct_type"]}" for '
                    f'flux function "{values[0]}"'
                )
            default_flux_function = std.DEFAULT_FLUX_FUNCTION_POLY if atts['struct_type'] == 'linked zones' else \
                std.DEFAULT_FLUX_FUNCTION_ARC
            flux_function = default_flux_function  # Set to the default structure
            atts['elevation_is_dz'] = 1
        atts['flux_function'] = flux_function

    def _add_structure_atts(self, command, values, atts):
        """Switch statement that adds an attribute inside a structure block to the material's data.

        Args:
            command (str): The command card (all lower)
            values (list[str]): The lexed values
            atts (dict): The attribute dict for the structure
        """
        if command == 'flux function':
            self._set_flux_function(atts, values)
        elif command == 'properties':
            atts['weir_z'] = float(values[0])
            # These are all on the properties line
            if len(values) > 1:
                atts['weir_cw'] = float(values[1])
                atts['define_weir'] = 1  # We are overriding the default weir properties
            if len(values) > 2:
                atts['weir_ex'] = float(values[2])
            if len(values) > 3:
                atts['weir_a'] = float(values[3])
            if len(values) > 4:
                atts['weir_b'] = float(values[4])
            if len(values) > 5:
                atts['weir_csfm'] = float(values[5])
        elif command == 'culvert file':
            # Need to parse this differently because there is an ID field on the row.
            reverse_line = values[0][::-1]  # Reverse the line and split on the first comma.
            tokens = reverse_line.split(',', maxsplit=1)
            culvert_id = io_util.str2float2int(tokens[0].strip())
            # Get the filename back in the right order.
            filename = tokens[1][::-1].strip()
            self._culvert_csv_io.read_culvert_csv(filename=filename, culvert_id=culvert_id, atts=atts)
        elif command == 'energy loss function':
            atts['energy_loss_function'] = values[0].title()
        elif command == 'form loss coefficient':
            atts['form_loss_coefficient'] = float(values[0])
        elif command == 'blockage file':
            atts['blockage_file'] = self._get_input_filepath(values[0])
        elif command == 'width file':
            atts['width_file'] = self._get_input_filepath(values[0])
        elif command == 'energy loss file':
            atts['energy_loss_file'] = self._get_input_filepath(values[0])
        elif command == 'zone inlet/outlet orientation':
            atts['zone_inlet_orientation'] = float(values[0])
            atts['zone_outlet_orientation'] = float(values[1])
        elif command == 'flux file':
            atts['flux_file'] = self._get_input_filepath(values[0])
        elif command == 'end structure':
            pass  # Just pass here. This card will trigger exit of the calling structure block parsing method.
        else:
            self._add_unrecognized_card('Unsupported structure block command')

    def _add_bc_att(self, command, values, atts):
        """Switch statement that adds an attribute inside a bc block to the bc's data.

        Args:
            command (str): The command card (all lower)
            values (list[str]): The lexed values
            atts (dict): The attribute dict for the bc
        """
        if command == 'sub-type':
            atts['subtype'] = io_util.str2float2int(values[0])
        elif command == 'bc default':
            atts['define_default'] = 1
            check_for_variable_bc_values(atts, 'default', values)
        elif command == 'bc offset':
            atts['define_offset'] = 1
            check_for_variable_bc_values(atts, 'offset', values)
        elif command == 'bc scale':
            atts['define_scale'] = 1
            check_for_variable_bc_values(atts, 'scale', values)
        elif command == 'bc update dt':
            atts['define_update_dt'] = 1
            atts['update_dt'] = float(values[0])
        elif command == 'bc time units':
            atts['time_units'] = values[0].title()
            if 'define_time_units' in atts:
                # This attribute only exists for gridded BCs because it is always defined by the BC curve editor for
                # other types. We build global and gridded BCs a little different than point/arc in that we intially
                # fill in the atts here with default values, but the arc/point atts only have attributes that have
                # been read from the file and missing values get defaulted later.
                atts['define_time_units'] = 1
        elif command == 'bc reference time':
            atts['define_reference_time'] = 1
            try:
                atts['reference_time_hours'] = float(values[0])
                self._bc_reference_time = gui_util.get_tuflow_zero_time() + datetime.timedelta(
                    hours=atts['reference_time_hours']
                )
            except ValueError:
                atts['use_isodate'] = 1
                # Need to store the string representation in the attrs but need a datetime for arithmetic
                atts['reference_time_date'] = smd.parse_tuflow_isodate(values[0], to_qt=False)
                self._bc_reference_time = smd.parse_tuflow_isodate(values[0])
        elif command == 'includes mslp':
            atts['include_mslp'] = io_util.str2float2int(values[0])
        elif command == 'vertical coordinate type':
            atts['define_vertical_distribution'] = 1
            atts['vertical_distribution_type'] = values[0].title()
        elif command == 'vertical distribution file':
            atts['define_vertical_distribution'] = 1
            atts['vertical_distribution_file'] = self._get_input_filepath(values[0])
        elif command == 'end bc':
            pass  # Just pass here. This card will trigger exit of the calling bc block parsing method.
        else:
            self._add_unrecognized_card('Unsupported BC block command')

    def _add_grid_definition_att(self, command, values, atts):
        """Switch statement that adds an attribute inside a grid definition block to the grid's data.

        Args:
            command (str): The command card (all lower)
            values (list[str]): The lexed values
            atts (xr.Dataset): The attribute Dataset for the grid
        """
        if command == 'grid definition label':
            atts['name'][0] = values[0]
        elif command == 'grid definition variables':
            atts['x_variable'][0] = values[0]
            atts['y_variable'][0] = values[1]
            if len(values) > 2:  # Z-variable not typically defined
                atts['z_variable'][0] = values[2]
        elif command == 'vertical coordinate type':
            atts['define_vert_coord_type'] = 1
            atts['vert_coord_type'][0] = values[0].title()
        elif command == 'cell gridmap':
            atts['cell_gridmap'][0] = io_util.str2float2int(values[0])
        elif command == 'boundary gridmap':
            atts['boundary_gridmap'][0] = io_util.str2float2int(values[0])
        # Typo in the docs so we will accept the misspelling
        elif command in ['supress coverage warnings', 'suppress coverage warnings']:
            atts['suppress_coverage_warnings'][0] = io_util.str2float2int(values[0])
        elif command == 'end grid':
            pass  # Just pass here. This card will trigger exit of the calling bc block parsing method.
        else:
            self._add_unrecognized_card('Unsupported grid definition block command')

    def _process_material_block(self):
        """Process the cards in a material block."""
        command, values = self._cards[self._line_num]  # Get the material ID from the first line
        atts = self._materials.setdefault(io_util.str2float2int(values[0]), {})
        while command != 'end material':  # Read the optional commands until the end of the material block
            self._line_num += 1
            command, values = self._cards[self._line_num]
            self._add_material_att(command, values, atts)
        # Increment past end of material card
        self._line_num += 1

    def _process_grid_definition_block(self):
        """Process the cards in a grid definition block."""
        command, values = self._cards[self._line_num]
        grid_id = self._sim_data.add_grid_definition()
        atts = self._sim_data.grid_definitions.where(self._sim_data.grid_definitions.grid_id == grid_id, drop=True)
        atts['file'][0] = self._get_input_filepath(values[0])
        while command != 'end grid':  # Read the optional commands until the end of the grid definition block
            self._line_num += 1
            command, values = self._cards[self._line_num]
            self._add_grid_definition_att(command, values, atts)
        self._grid_definitions[atts.name.item()] = grid_id
        self._sim_data.update_grid_definition(atts)
        # Increment past end of material card
        self._line_num += 1

    def _process_structure_block(self):
        """Process the cards in a structure block."""
        command, values = self._cards[self._line_num]
        try:  # See if this is a numeric ID
            struct_id = str(int(values[1]))
        except ValueError:  # Must be a string ID
            struct_id = values[1]

        # Docs have a single word, examples have two words. We'll handle both.
        struct_type = values[0].lower()
        if struct_type == 'linkednodestrings':
            struct_type = 'linked nodestrings'
        elif struct_type == 'linkedzones':
            struct_type = 'linked zones'

        # Check for the name of the second arc/polygon in a set.
        struct2_id = None
        if struct_type in ['linked nodestrings', 'linked zones']:
            try:  # See if this is a numeric ID
                struct2_id = str(int(values[2]))
            except ValueError:  # Must be a string ID
                struct2_id = values[2]
            if struct_type == 'linked nodestrings':
                self._structure_manager.add_arc_structure(struct2_id)
            else:
                self._structure_manager.add_poly_structure(struct2_id)

        if struct_type in ['nodestring', 'linked nodestrings']:
            atts = self._structure_arcs.setdefault(struct_id, {})
            self._structure_manager.add_arc_structure(struct_id)
            target_type = TargetType.arc
        else:
            atts = self._structure_polys.setdefault(struct_id, {})
            self._structure_manager.add_poly_structure(struct_id)
            target_type = TargetType.polygon
        atts['name'] = struct_id
        atts['struct_type'] = struct_type  # If this is a linked structure, we need to know later.
        # Create a link to the second arc/polygon if this is a set.
        atts['connection'] = struct2_id if struct2_id is not None else ''
        atts['upstream'] = 1

        while command != 'end structure':
            self._line_num += 1
            command, values = self._cards[self._line_num]
            # Add the attribute for this line in the structure block
            self._add_structure_atts(command, values, atts)

        atts2 = None
        default_flux_function = std.DEFAULT_FLUX_FUNCTION_ARC
        if target_type == TargetType.arc:
            self._structure_manager.add_arc_structure_atts(struct_id, atts)
            if struct2_id:  # Add the second arc in a set if it exists
                atts2 = copy.deepcopy(atts)
                atts2['connection'] = struct_id
                atts2['name'] = struct2_id
                atts2['upstream'] = 0
                self._structure_manager.add_arc_structure_atts(struct2_id, atts2)
        else:  # TargetType.polygon
            self._structure_manager.add_poly_structure_atts(struct_id, atts)
            default_flux_function = std.DEFAULT_FLUX_FUNCTION_POLY
            if struct2_id:  # Add the second polygon in a set if it exists
                atts2 = copy.deepcopy(atts)
                atts2['connection'] = struct_id
                atts2['name'] = struct2_id
                atts2['upstream'] = 0
                self._structure_manager.add_poly_structure_atts(struct2_id, atts2)
        if atts2 is not None:  # If this is a linked structure, we need to know later.
            atts2['struct_type'] = struct_type

        # If Wier_dz turn into a weir.
        # Remember, though, that not all structures must have flux function defined.
        if 'flux_function' in atts and atts['flux_function'].lower() == 'weir_dz':
            # The default flux function  for arcs is 'Weir', 'unassigned' for polygons.
            atts['flux_function'] = default_flux_function
            atts['elevation_is_dz'] = 1
            if atts2 is not None:
                atts2['flux_function'] = default_flux_function
                atts2['elevation_is_dz'] = 1

        # Increment past end of structure card
        self._line_num += 1

    # def _process_control_block(self):
    #     """Process the cards in a control block."""
    #     command, values = self._cards[self._line_num]
    #     while command != 'end control':
    #         self._line_num += 1
    #         command, values = self._cards[self._line_num]
    #     # Increment past end of BC card
    #     self._line_num += 1

    def _process_output_parameters(self, values, parameters):
        """Process the cards in an output parameters line.

        Args:
            values (list[str]): The output parameters line values, split by card
            parameters (dict): The data attrs to update flags for
        """
        unrecognized = []  # Build up list of all unrecognized and report all at once
        for card in values:
            if card.startswith('!') or card.startswith('#'):
                break  # Done once we encounter a comment character on the line
            card = card.lower()
            if card == 'd':
                parameters['depth_output'] = 1
            elif card == 'h':
                parameters['wse_output'] = 1
            elif card == 'taub':
                parameters['bed_shear_stress_output'] = 1
            elif card == 'taus':
                parameters['surface_shear_stress_output'] = 1
            elif card == 'v':
                parameters['velocity_output'] = 1
            elif card == 'vmag':
                parameters['velocity_mag_output'] = 1
            elif card == 'w':
                parameters['vertical_velocity_output'] = 1
            elif card == 'zb':
                parameters['bed_elevation_output'] = 1
            elif card == 'turb_visc':
                parameters['turb_visc_output'] = 1
            elif card == 'turb_diff':
                parameters['turb_diff_output'] = 1
            elif card == 'air_temp':
                parameters['air_temp_output'] = 1
            elif card == 'evap':
                parameters['evap_output'] = 1
            elif card == 'dzb':
                parameters['dzb_output'] = 1
            elif card == 'hazard_z1':
                parameters['hazard_z1_output'] = 1
            elif card == 'hazard_zaem1':
                parameters['hazard_zaem1_output'] = 1
            elif card == 'hazard_zqra':
                parameters['hazard_zqra_output'] = 1
            elif card == 'lw_rad':
                parameters['lw_rad_output'] = 1
            elif card == 'mslp':
                parameters['mslp_output'] = 1
            elif card == 'precip':
                parameters['precip_output'] = 1
            elif card == 'rel_hum':
                parameters['rel_hum_output'] = 1
            elif card == 'rhow':
                parameters['rhow_output'] = 1
            elif card == 'sal':
                parameters['sal_output'] = 1
            elif card == 'sw_rad':
                parameters['sw_rad_output'] = 1
            elif card == 'temp':
                parameters['temp_output'] = 1
            elif card == 'turbz':
                parameters['turbz_output'] = 1
            elif card == 'w10':
                parameters['w10_output'] = 1
            elif card == 'wq_all':
                parameters['wq_all_output'] = 1
            elif card == 'wq_diag_all':
                parameters['wq_diag_all_output'] = 1
            elif card == 'wvht':
                parameters['wvht_output'] = 1
            elif card == 'wvper':
                parameters['wvper_output'] = 1
            elif card == 'wvdir':
                parameters['wvdir_output'] = 1
            elif card == 'wvstr':
                parameters['wvstr_output'] = 1
            elif card:  # Don't whine about trailing comma in list
                unrecognized.append(card)  # Report all unrecognized parameters on line at once
        if unrecognized:
            self._add_unrecognized_card(f'Unsupported output parameters {", ".join(unrecognized)}')

    def _process_single_general_card(self, command, values):
        """Process a global simulation general parameters card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'spatial order':
            self._sim_data.general.attrs['define_spatial_order'] = 1
            self._sim_data.general.attrs['horizontal_order'] = io_util.str2float2int(values[0])
            if len(values) > 1:  # Vertical may or may not be present
                self._sim_data.general.attrs['vertical_order'] = io_util.str2float2int(values[1])
        elif command == 'gis format':
            if values[0].lower() != 'shp':  # We only currently support importing GIS data from shapefiles
                self._logger.error('*.mif/*.mid input specified. SMS only supports importing GIS data from shapefiles.')
            else:
                self._specified_shp_format = True
        elif command == 'tutorial model':
            self._sim_data.general.attrs['tutorial'] = values[0].upper()
        elif command == 'gis projection check':
            self._sim_data.general.attrs['projection_warning'] = 1 if values[0].upper() == 'WARNING' else 0
        elif command == 'shp projection':  # Use specified projection
            prj_file = self._get_input_filepath(values[0])
            with open(prj_file, 'r') as f:
                self._wkt = f.read()
        elif command == 'mif projection':  # This is not allowed
            self._logger.error('*.mif/*.mid input specified. SMS only supports importing GIS data from shapefiles.')
        elif command == 'hardware':
            self._sim_data.general.attrs['define_hardware_solver'] = 1
            self._sim_data.general.attrs['hardware_solver'] = values[0].upper()
        elif command == 'device id':
            self._sim_data.general.attrs['device_id'] = io_util.str2float2int(values[0])
        elif command == 'display dt':
            self._sim_data.general.attrs['define_display_interval'] = 1
            self._sim_data.general.attrs['display_interval'] = io_util.str2float2int(values[0])
        elif command == 'units':
            if values[0].lower() != 'metric':
                self._vertical_units = 'FEET (U.S. SURVEY)'
        elif command == 'spherical':
            # We need to read this because we can at least set projection of imported data to default Geographic
            # projection if there is no GIS projection to read.
            if io_util.str2float2int(values[0]) == 1:
                self._geographic = True
        elif command in ['read file', 'include']:
            fvc_filename = self._get_input_filepath(values[0])
            self._process_read_file_command(fvc_filename)
        else:
            return False
        return True

    def _process_read_file_command(self, fvc_filename):
        """Process a 'read file' command which is a recursive simulation read - make sure no circular references.

        Args:
            fvc_filename (str): Normalized absolute path to the referenced control file
        """
        self._logger.info(f'Reading include file: {fvc_filename}')

        # List members that need to be reset
        reset_atts = ['_filename','_lines', '_cards', '_line_num']

        # Save the current state and reset it to defaults
        old = {}
        for att in reset_atts:
            old[att] = getattr(self, att)
            old_type = type(old[att])
            if old_type is list:
                setattr(self, att, [])
            elif old_type is str:
                setattr(self, att, "")
            elif old_type is int:
                setattr(self, att, 0)
            elif old_type is float:
                setattr(self, att, 0.0)
            else:
                setattr(self, att, None)

        # Set filename to the new file and start a read
        self._filename = str(Path(fvc_filename).absolute())
        self._read_sim(primary=False)

        # Restore old state
        for att in reset_atts:
            setattr(self, att, old[att])

    def _process_single_time_card(self, command, values):
        """Process a global simulation time card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'time format':
            self._sim_data.time.attrs['use_isodate'] = 1 if values[0].lower() == 'isodate' else 0
        elif command == 'reference time':
            try:  # Could be in hours or ISO date format, be careful here
                hour_offset = float(values[0])  # If using hours format, value is a single float
                ref_date = gui_util.get_tuflow_zero_time() + datetime.timedelta(hours=hour_offset)
                self._sim_data.time.attrs['ref_date'] = ref_date.strftime(ISO_DATETIME_FORMAT)
            except ValueError:
                self._sim_data.time.attrs['ref_date'] = smd.parse_tuflow_isodate(values[0], to_qt=False)
        elif command == 'start time':
            try:  # Could be in hours or ISO date format, be careful here
                self._sim_data.time.attrs['start_hours'] = float(values[0])
            except ValueError:
                self._sim_data.time.attrs['start_date'] = smd.parse_tuflow_isodate(values[0], to_qt=False)
        elif command == 'end time':
            try:  # Could be in hours or ISO date format, be careful here
                self._sim_data.time.attrs['end_hours'] = float(values[0])
            except ValueError:
                self._sim_data.time.attrs['end_date'] = smd.parse_tuflow_isodate(values[0], to_qt=False)
        elif command == 'cfl':
            self._sim_data.time.attrs['cfl'] = float(values[0])
        elif command == 'timestep limits':
            self._sim_data.time.attrs['min_increment'] = float(values[0])
            self._sim_data.time.attrs['max_increment'] = float(values[1])
        else:
            return False
        return True

    def _process_single_global_card(self, command, values):
        """Process a global simulation global parameters card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'external turbulence model directory':
            self._sim_data.globals.attrs['external_vertical_viscosity'] = self._get_input_filepath(values[0])
        elif command == 'global vertical eddy viscosity limits':
            self._sim_data.globals.attrs['define_vertical_viscosity_limits'] = 1
            self._sim_data.globals.attrs['vertical_viscosity_min'] = float(values[0])
            self._sim_data.globals.attrs['vertical_viscosity_max'] = float(values[1])
        elif command == 'turbulence update dt':
            self._sim_data.globals.attrs['define_turbulence_update'] = 1
            self._sim_data.globals.attrs['turbulence_update'] = float(values[0])
        elif command == 'stability limits':
            self._sim_data.globals.attrs['define_stability_limits'] = 1
            self._sim_data.globals.attrs['stability_wse'] = float(values[0])
            self._sim_data.globals.attrs['stability_velocity'] = float(values[1])
        else:
            return False
        return True

    def _process_single_viscosity_card(self, command, values):
        """Process a global simulation viscosity parameters card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'momentum mixing model':
            self._sim_data.globals.attrs['horizontal_mixing'] = values[0].title()
        elif command == 'global horizontal eddy viscosity':
            self._sim_data.globals.attrs['global_horizontal_viscosity'] = float(values[0])
        elif command == 'global horizontal eddy viscosity limits':
            self._sim_data.globals.attrs['define_horizontal_viscosity_limits'] = 1
            self._sim_data.globals.attrs['horizontal_viscosity_min'] = float(values[0])
            self._sim_data.globals.attrs['horizontal_viscosity_max'] = float(values[1])
        elif command == 'vertical mixing model':
            self._sim_data.globals.attrs['define_vertical_mixing'] = 1
            model = values[0].title()
            self._sim_data.globals.attrs['vertical_mixing'] = model
            if model == 'External' and not self._sim_data.globals.attrs['external_vertical_viscosity']:
                # If using external vertical mixing model, the turbulence file directory is assumed to be the directory
                # containing the control file unless another location is specified.
                self._sim_data.globals.attrs['external_vertical_viscosity'] = os.path.dirname(self._filename)
        elif command == 'vertical mixing parameters':
            self._sim_data.globals.attrs['global_vertical_parametric_coefficients1'] = float(values[0])
            try:  # May or may not be a second value on the line depending on the vertical mixing model.
                self._sim_data.globals.attrs['global_vertical_parametric_coefficients2'] = float(values[1])
            except Exception:  # If only one value, using constant vertical eddy viscosity
                self._sim_data.globals.attrs['global_vertical_viscosity'] = float(values[0])
        else:
            return False
        return True

    def _process_single_diffusivity_card(self, command, values):
        """Process a global simulation diffusivity parameters card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'scalar mixing model':
            self._sim_data.globals.attrs['horizontal_scalar_diffusivity_type'] = values[0].title()
        elif command == 'global horizontal scalar diffusivity':
            self._sim_data.globals.attrs['horizontal_scalar_diffusivity'] = float(values[0])
            self._sim_data.globals.attrs['horizontal_scalar_diffusivity_coef1'] = float(values[0])
            try:  # May or may not be a second value on this line depending on scalar mixing model
                self._sim_data.globals.attrs['horizontal_scalar_diffusivity_coef2'] = float(values[1])
            except Exception:
                pass
        elif command == 'global horizontal scalar diffusivity limits':
            self._sim_data.globals.attrs['define_horizontal_diffusivity_limits'] = 1
            self._sim_data.globals.attrs['horizontal_scalar_diffusivity_min'] = float(values[0])
            self._sim_data.globals.attrs['horizontal_scalar_diffusivity_max'] = float(values[1])
        elif command == 'global vertical scalar diffusivity':
            self._sim_data.globals.attrs['define_vertical_scalar_diffusivity'] = 1
            self._sim_data.globals.attrs['vertical_scalar_diffusivity'] = float(values[0])
            self._sim_data.globals.attrs['vertical_scalar_diffusivity_coef1'] = float(values[0])
            try:  # May or may not be a second value on this line depending on scalar mixing model
                self._sim_data.globals.attrs['vertical_scalar_diffusivity_coef2'] = float(values[1])
            except Exception:
                pass
        elif command == 'global vertical scalar diffusivity limits':
            self._sim_data.globals.attrs['define_vertical_diffusivity_limits'] = 1
            self._sim_data.globals.attrs['vertical_scalar_diffusivity_min'] = float(values[0])
            self._sim_data.globals.attrs['vertical_scalar_diffusivity_max'] = float(values[1])
        else:
            return False
        return True

    def _process_single_wind_stress_card(self, command, values):
        """Process a global simulation wind stress card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'wind stress model':
            self._sim_data.wind_stress.attrs['method'] = io_util.str2float2int(values[0])
            self._sim_data.wind_stress.attrs['define_wind'] = 1
        elif command == 'wind stress parameters':
            self._sim_data.wind_stress.attrs['define_parameters'] = 1
            # There is always at least one value, but may be more depending on wind stress model.
            first_coefficient = float(values[0])
            self._sim_data.wind_stress.attrs['wa'] = first_coefficient
            self._sim_data.wind_stress.attrs['bulk_coefficient'] = first_coefficient
            self._sim_data.wind_stress.attrs['scale_factor'] = first_coefficient
            try:  # If wind stress model is Wu, there are three more values on this line.
                self._sim_data.wind_stress.attrs['ca'] = float(values[1])
                self._sim_data.wind_stress.attrs['wb'] = float(values[2])
                self._sim_data.wind_stress.attrs['cb'] = float(values[3])
            except Exception:
                pass
        else:
            return False
        return True

    def _process_single_geometry_card(self, command, values):
        """Process a global simulation geometry card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'geometry 2d':
            self._2dm_file = self._get_input_filepath(values[0])
        elif command == 'cell wet/dry depths':
            self._sim_data.geometry.attrs['define_cell_depths'] = 1
            self._sim_data.geometry.attrs['dry_cell_depth'] = float(values[0])
            self._sim_data.geometry.attrs['wet_cell_depth'] = float(values[1])
        elif command == 'cell elevation file':
            csv_type = smd.CELL_ELEV_CSV_TYPE_CELL
            if len(values) > 1 and values[1].lower() != 'cell_id':
                csv_type = smd.CELL_ELEV_CSV_TYPE_COORD  # Optional, cell ids by default
            self._add_z_modification(mod_type=smd.ELEV_TYPE_CELL_CSV, csv_file=self._get_input_filepath(values[0]),
                                     csv_type=csv_type)
        elif command == 'set zpts':
            self._add_z_modification(mod_type=smd.ELEV_TYPE_SET_ZPTS, constant=float(values[0]))
        elif command == 'read grid zpts':
            self._add_z_modification(mod_type=smd.ELEV_TYPE_GRID_ZPTS,
                                     grid_zpts_file=self._get_input_filepath(values[0]))
        else:
            return False
        return True

    def _process_single_initial_conditions_card(self, command, values):
        """Process a global simulation initial conditions card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'initial water level':
            self._sim_data.initial_conditions.attrs['define_initial_water_level'] = 1
            self._sim_data.initial_conditions.attrs['initial_water_level'] = float(values[0])
        elif command in ['restart', 'restart file']:
            self._sim_data.initial_conditions.attrs['use_restart_file'] = 1
            self._sim_data.initial_conditions.attrs['restart_file'] = self._get_input_filepath(values[0])
        elif command == 'use restart file time':
            self._sim_data.initial_conditions.attrs['use_restart_file_time'] = io_util.str2float2int(values[0])
        else:
            return False
        return True

    def _process_single_global_output_card(self, command, values):
        """Process a global simulation output card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'write restart':  # The link to this is broken in the docs. Is this a file?
            self._sim_data.output.attrs['write_restart_file'] = 1
        elif command == 'write restart dt':
            self._sim_data.output.attrs['write_restart_file'] = 1
            self._sim_data.output.attrs['restart_file_interval'] = float(values[0])
        elif command == 'restart overwrite':
            self._sim_data.output.attrs['overwrite_restart_file'] = io_util.str2float2int(values[0])
        elif command in ['log dir', 'logdir']:
            self._sim_data.output.attrs['log_dir'] = values[0]
        elif command == 'output dir':
            self._sim_data.output.attrs['output_dir'] = values[0]
        elif command == 'write check files':
            self._sim_data.output.attrs['write_check_files'] = 1
            self._sim_data.output.attrs['check_files_dir'] = values[0]
        elif command == 'write empty gis files':
            self._sim_data.output.attrs['write_empty_gis_files'] = 1
            self._sim_data.output.attrs['empty_gis_files_dir'] = values[0]
        else:
            return False
        return True

    def _process_single_global_set_mat_card(self, command, values):
        """Process a global simulation output card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'bottom drag model':
            method = values[0].lower()  # Be case insensitive
            self._sim_data.global_set_mat.attrs['global_roughness'] = method if method == 'ks' else 'Manning'
            self._sim_data.global_set_mat.attrs['define_global_roughness'] = 1
        elif command == 'global bottom roughness':
            self._sim_data.global_set_mat.attrs['global_roughness_coefficient'] = float(values[0])
            self._sim_data.global_set_mat.attrs['define_global_roughness'] = 1
        elif command == 'set mat':
            self._sim_data.global_set_mat.attrs['define_set_mat'] = 1
            self._default_mat_id = io_util.str2float2int(values[0])
        else:
            return False
        return True

    def _process_single_global_bc_card(self, command, values):
        """Process a global simulation output card.

        Args:
            command (str): The TUFLOWFV defined card for the line, lowercase
            values (list[str]): The parsed values of the command for the line

        Returns:
            bool: True if the command was handled
        """
        if command == 'bc default update dt':
            self._sim_data.globals.attrs['define_bc_default_update_dt'] = 1
            self._sim_data.globals.attrs['bc_default_update_dt'] = float(values[0])
        elif command == 'include wave stress':
            self._sim_data.globals.attrs['include_wave_stress'] = io_util.str2float2int(values[0])
        elif command == 'include stokes drift':
            self._sim_data.globals.attrs['include_stokes_drift'] = io_util.str2float2int(values[0])
        else:
            return False
        return True

    def _process_single_card(self):
        """Process a card that is on a single line."""
        command, values = self._cards[self._line_num]
        for command_method in self._single_card_readers:
            if command_method(command, values):
                self._line_num += 1
                return
        self._add_unrecognized_card('')
        self._line_num += 1

    def _add_unrecognized_card(self, block_comment):
        """Add an unrecognized card to the advanced cards.

        Args:
            block_comment (str): Specific comment for an unrecognized block command. If not None, will prepend a comment
                character to the advanced card line.
        """
        line_num = self._card_idx_to_line_idx[self._line_num]
        line = f'{self._lines[line_num].rstrip()}  ! Unrecognized card on line {line_num + 1}'
        if block_comment:
            line = f'! {line}: {block_comment}'
        self.unrecognized_lines.append(line)

    def _add_child_sim_links(self):
        """Add all the child simulation links referenced by the current simulation."""
        if not self._child_sim_links:
            return
        child_sims = {'uuid': xr.DataArray(np.array(self._child_sim_links, dtype=object))}
        self._sim_data.linked_simulations = xr.Dataset(data_vars=child_sims)

    def _set_global_bcs(self):
        """If defined, set the global BC attributes on the simulation data."""
        if not self._global_bcs:
            return
        build_global_bc_dataset(sim_data=self._sim_data, global_bcs=self._global_bcs, gridded=False)

    def _set_gridded_bcs(self):
        """If defined, set the gridded BC attributes on the simulation data."""
        if not self._gridded_bcs:
            return
        # Map from grid label in the file to generated grid id. Flatten into a 1D list of BCs
        gridded_bcs = []
        for grid_name, grid_bcs in self._gridded_bcs.items():
            grid_id = self._grid_definitions.get(grid_name)
            if grid_id is None:  # If no grid label, try the default generated one
                grid_id = self._grid_definitions.get(f'Grid_{grid_name}')
            for grid_bc in grid_bcs:
                grid_bc['grid_id'] = grid_id
            gridded_bcs.extend(grid_bcs)
        build_global_bc_dataset(sim_data=self._sim_data, global_bcs=gridded_bcs, gridded=True)

    def _concatenate_output_blocks(self):
        """Concatenate all the read output blocks and create the xarray Dataset."""
        if not self._output_blocks:
            return
        data_vars = {variable: [] for variable in self._output_blocks[0]}
        for output_block in self._output_blocks:  # Build a data cube
            for variable, value in output_block.items():
                data_vars[variable].append(value)
        data_vars = {variable: xr.DataArray(values) for variable, values in data_vars.items()}
        self._sim_data.output = xr.Dataset(data_vars=data_vars, attrs=self._sim_data.output.attrs)
        self._sim_data.output_points_coverages.attrs['next_row_index'] = len(self._output_blocks)

    def _link_output_coverages(self):
        """Link output coverages to the simulation model control."""
        data_vars = {
            'uuid': ('index', np.array(self._output_coverage_uuids, dtype=object)),
            'row_index': ('index', np.array(self._output_coverage_indices, dtype=np.int32)),
        }
        coords = {'index': [i for i in range(len(self._output_coverage_uuids))]}
        self._sim_data.output_points_coverages = xr.Dataset(data_vars=data_vars, coords=coords)

    def _build_output_points_coverage(self, point_csv_file, parameters):
        """Build the output points coverage.

        Args:
            point_csv_file (str): location of the csv file that specifies xy
            parameters (dict): specifies which parameters to report

        Returns:
            str: UUID of the new coverage
        """
        point_csv_file = self._get_input_filepath(point_csv_file)
        if self._duplicate_checker.have_read_output_points(point_csv_file, parameters):
            return parameters['cov_uuid']  # 'cov_uuid' column got set in duplicate data checker

        df = pd.read_csv(point_csv_file, header=0)
        x_coords = df.iloc[:, 0].values.tolist()
        y_coords = df.iloc[:, 1].values.tolist()
        points = [
            Point(x=x_coord, y=y_coord, feature_id=idx + 1) for
            idx, (x_coord, y_coord) in enumerate(zip(x_coords, y_coords))
        ]
        cov_name = os.path.splitext(os.path.basename(point_csv_file))[0]
        cov_uuid = str(uuid.uuid4())
        parameters['cov_uuid'] = cov_uuid  # Reset 'cov_uuid' column value, wrong from duplicate data check
        do_cov = Coverage(name=cov_name, uuid=cov_uuid)
        do_cov.set_points(points)
        do_cov.complete()
        atts = {i: {
            'label': str(i + 1),
            'comment': '',
            'vert_min': 0.0,
            'vert_max': 0.0,
        } for i in range(len(points))}
        point_id_to_feature_id = {i: i + 1 for i in range(len(points))}
        builder = OutputComponentBuilder(cov_uuid=cov_uuid, from_csv=True, atts=atts,
                                         point_id_to_feature_id=point_id_to_feature_id, point_id_to_feature_name={})
        do_comp = builder.build_output_component()
        self._output_points_coverages[-1].append((do_cov, do_comp))
        self._duplicate_checker.add_imported_output_points(point_csv_file, cov_uuid, parameters)
        return cov_uuid

    def _build_gis_output_points_coverage(self, parameters):
        """Build an output points coverage from a GIS PO file.

        Args:
            parameters (dict): The output block parameters

        Returns:
            str: UUID of the new coverage
        """
        if self._duplicate_checker.have_read_output_points(self._gis_po_files[-1], parameters):
            return parameters['cov_uuid']  # 'cov_uuid' column got set in duplicate data checker
        converter = ShapefileConverter(filename=self._gis_po_files[-1])
        do_cov, do_comp = converter.convert_output_points()
        cov_uuid = ''
        if do_cov:
            self._output_points_coverages[-1].append((do_cov, do_comp))
            cov_uuid = do_cov.uuid
            parameters['cov_uuid'] = cov_uuid  # Reset 'cov_uuid' column value, wrong from duplicate data check
            self._duplicate_checker.add_imported_output_points(self._gis_po_files[-1], cov_uuid, parameters)
        return cov_uuid

    def _build_bc_points_coverage(self):
        """Build the boundary condition points coverage."""
        if not self._bc_points_locations:
            return  # No cell BCs in the control file

        cov_uuid = str(uuid.uuid4())
        points = [  # We number these points 1-N but keep them in a dict so we can reuse the component builder code.
            Point(x=x_coord, y=y_coord, feature_id=idx + 1) for
            idx, (x_coord, y_coord) in enumerate(self._bc_points_locations.values())
        ]
        do_cov = Coverage(name=f'{self._sim_name}_-_BC Points', uuid=cov_uuid)
        do_cov.set_points(points)
        do_cov.complete()
        nodestring_id_to_feature_id = {bc_id: idx + 1 for idx, bc_id in enumerate(self._bc_points)}
        nodestring_id_to_feature_name = {bc_id: bc_id for bc_id in self._bc_points}
        comp_builder = BcComponentBuilder(cov_uuid=cov_uuid, from_2dm=True, bc_atts=self._bc_points,
                                          nodestring_id_to_feature_id=nodestring_id_to_feature_id,
                                          nodestring_id_to_feature_name=nodestring_id_to_feature_name)
        do_comp = comp_builder.build_bc_component()
        self._bc_coverages[-1].append((do_cov, do_comp))
        self._sim_links.append((self._sim_uuid, cov_uuid))

    def _build_point_polygon(self, poly):
        """Builds a point Polygon object based on the provided polygon.

        Args:
            poly (Polygon): The polygon to build the point Polygon from.

        Returns:
            Polygon: The point Polygon object.
        """
        p = Polygon(feature_id=poly.id)
        p_pts = poly.get_points(FilterLocation.PT_LOC_ALL)
        p.set_points(points=p_pts, vertice_count=len(p_pts) - 1)
        return p

    def _build_structures_coverage(self):
        """Build the structures coverage."""
        if not self._structure_arcs and not self._structure_polys:
            return  # No structures in the control file

        cov_uuid = str(uuid.uuid4())
        cov_name = f'{self._sim_name}_-_Structures'
        do_cov = Coverage(name=cov_name, uuid=cov_uuid)
        do_cov.arcs = list(self._structure_manager.get_all_arc_locations().values())
        # We have to build the polygon from points instead of arcs for some reason. This is a bug in data_objects.
        polys = [
            self._build_point_polygon(poly) for poly in self._structure_manager.get_all_poly_locations().values()
        ]
        do_cov.polygons = polys
        do_cov.complete()
        comp_builder = StructureComponentBuilder(cov_uuid=cov_uuid, manager=self._structure_manager)
        do_comp = comp_builder.build_structure_component()
        self._structure_coverages[-1].append((do_cov, do_comp))
        self._sim_links.append((self._sim_uuid, cov_uuid))

    def _find_set_mat(self):
        """If a default material has been assigned, find it and add it to the simulation data."""
        if self._default_mat_id is None:
            return  # set mat not defined
        if self._default_mat_id not in self._materials:
            self._logger.error(f'set mat was specified as material {self._default_mat_id}, but no material block was '
                               'found with that id.')
            return
        dim_name = next(iter(self._sim_data.global_set_mat.sizes))
        for variable, value in self._materials[self._default_mat_id].items():
            # There are always 2 and only 2 materials in this Dataset. The unassigned is hidden and just there so we
            # could reuse the GUI code. Overwrite values in the second material (id 1).
            self._sim_data.global_set_mat[variable].loc[{dim_name: 1}] = value

    def _build_xms_data(self):
        """Add all the imported data to the xmsapi Query to send back to SMS."""
        # Add the simulation
        do_sim = Simulation(model='TUFLOWFV', sim_uuid=self._sim_uuid, name=self._sim_name)
        do_sim_comp = get_component_data_object(
            main_file=self._sim_data._filename, comp_uuid=self._sim_comp_uuid, unique_name='SimComponent'
        )
        self._do_sims.append(do_sim)
        self._do_sim_comps.append(do_sim_comp)

        # Set the projection for all the coverages read with this simulation
        for bc_cov, _ in self._bc_coverages[-1]:
            bc_cov.projection = self._do_projection
        for mat_cov, _ in self._material_coverages[-1]:
            mat_cov.projection = self._do_projection
        for output_cov, _ in self._output_points_coverages[-1]:
            output_cov.projection = self._do_projection
        for structure_cov, _ in self._structure_coverages[-1]:
            structure_cov.projection = self._do_projection  # Should only ever be 0 or 1 per simulation

    def _add_build_data(self, query):
        """Adds build data at the end of reading everything.

        Args:
            query (Query): The XMS interprocess communicator
        """
        # Add the simulations
        for do_sim, do_sim_comp in zip(self._do_sims, self._do_sim_comps):
            query.add_simulation(do_sim, components=[do_sim_comp])

        # Add the geometric data read from the .2dm file
        for do_ugrid in self._do_ugrids:
            query.add_ugrid(do_ugrid)

        # Add the Output Points Coverages
        for simulation in self._output_points_coverages:
            for cov, comp in simulation:
                query.add_coverage(cov, model_name='TUFLOWFV', coverage_type='Output Points', components=[comp])

        # Add the point BC coverages and BC coverages read from shapefiles
        for simulation in self._bc_coverages:
            for cov, comp in simulation:
                query.add_coverage(cov, model_name='TUFLOWFV', coverage_type='Boundary Conditions', components=[comp])

        # Add the material coverages read from shapefiles
        for simulation in self._material_coverages:
            for cov, comp in simulation:
                query.add_coverage(cov, model_name='TUFLOWFV', coverage_type='Materials', components=[comp])

        # Add the structures coverages (should be 0 or 1 per simulation)
        for simulation in self._structure_coverages:
            for cov, comp in simulation:
                query.add_coverage(cov, model_name='TUFLOWFV', coverage_type='Structures', components=[comp])

        # Add the Z line shapefiles
        added_shapefiles = set()
        for simulation in self._gis_zline_files:
            for line_uuid, (line_filename, point_layers) in simulation.items():
                if line_uuid not in added_shapefiles:  # Haven't added this one yet
                    query.add_shapefile(line_filename, item_uuid=line_uuid)
                    added_shapefiles.add(line_uuid)
                for point_uuid, point_filename in point_layers:
                    if point_uuid not in added_shapefiles:  # Haven't added this one yet
                        query.add_shapefile(point_filename, item_uuid=point_uuid)
                        added_shapefiles.add(point_uuid)

        # Add the wind boundaries as wind model coverages
        for wind_cov_dump in self._wind_coverages:
            query.add_coverage(wind_cov_dump)

        # Link items
        for taker_uuid, taken_uuid in self._sim_links:
            query.link_item(taker_uuid=taker_uuid, taken_uuid=taken_uuid)

    def _have_gis_data(self):
        """Returns True if we have imported any GIS data for this simulation."""
        return self._gis_bc_files or self._gis_mat_files or self._gis_zline_files[-1] or self._gis_sa_files \
            or self._gis_po_files or self._gis_struct_zone_files

    def _have_geometric_data(self):
        """Returns True if we have imported any geometric data (excluding GIS files) for this simulation."""
        return self._2dm_file or self._bc_points or self._wind_boundary_files or self._output_points_coverages[-1] or \
            self._structure_manager.num_arc_structure_locations() > 0

    def _log_unrecognized_card_warnings(self):
        """Log warnings for the cards we didn't recognize but also didn't treat as fatal."""
        if self.unrecognized_lines:
            bad_cards = "\n".join(self.unrecognized_lines)
            self._logger.warning(f'Unrecognized cards found in file. They will be added to the "Advanced" tab in the '
                                 f'Model Control:\n{bad_cards}')
            self._sim_data.general.attrs['advanced_cards'] = bad_cards

    def _log_missing_projection_warnings(self):
        """Warn the user if no projection card read, maybe valid or maybe not."""
        if not self._wkt:  # No GIS projection was specified in the control file
            have_gis_data = self._have_gis_data()
            have_other_data = self._have_geometric_data()  # 2dm, BC points, output points, or wind boundaries
            if not have_gis_data and not have_other_data:
                return  # Don't worry about projection errors if we have nothing to project.
            # If we have GIS data but the GIS format was specified as MIF or it defaulted to this, it is an error.
            if have_gis_data and not self._specified_shp_format:
                self._logger.error(
                    'GIS input found, but the GIS format has been set or defaulted to "MIF". Importing *.mif/*mid '
                    'files into SMS is not supported.'
                )
            # This is Ok, but the projection will not be complete on the geometric data read from the .2dm.
            elif not self._geographic and have_other_data:
                self._logger.warning(
                    'No projection file was specified in the control file. Meshes and coverages imported into SMS will '
                    f'have a projection of "No projection" with "{self._vertical_units}" units.'
                )

    def _read_file_lines(self, control_file):
        """Read all lines from the control file and parse any included files recursively."""
        self._logger.info(f'Parsing ASCII text from control file: {io_util.logging_filename(self._filename)}')
        with open(control_file, 'r', buffering=io_util.READ_BUFFER_SIZE) as f:
            initial_lines = f.readlines()

            # Read in included file contents
            for line in initial_lines:
                if line.lower().startswith('include =='):
                    include_file = line.split('==')[1]
                    include_file = include_file.replace('\n', '').strip()
                    include_file = self._get_input_filepath(include_file)
                    self._read_file_lines(include_file)
                else:
                    self._lines.append(line)

    def _read_sim(self, primary=True):
        """Read a single .fvc file."""
        try:
            # Load the lines from the file
            self._read_file_lines(self._filename)

            # Parse the loaded lines
            self._parse_cards(self._lines)

            # Create the simulation data
            self._sim_data = smd.SimData(
                os.path.join(io_util.create_component_folder(self._sim_comp_uuid), smd.SIM_DATA_MAINFILE)
            )

            # Process the parse cards
            self._process_cards()
            self._log_unrecognized_card_warnings()
            self._build_projection()  # Do this after reading all cards but before reading the .2dm

            # Add links to child simulations
            self._add_child_sim_links()

            # Set global/gridded BC
            self._set_global_bcs()
            self._set_gridded_bcs()

            # Create the output blocks Dataset
            self._concatenate_output_blocks()
            self._link_output_coverages()

            # Find the default material if specified
            self._find_set_mat()

            # Build the BC coverage of cell points in the .fvc
            self._build_bc_points_coverage()

            # Read the .2dm file (if we haven't gotten everything with a previous simulation)
            self._check_for_data_in_2dm()
            self._build_z_modifications_dataset()

            # Read GIS shapefiles
            self._read_gis_data()

            # Build the structures coverage
            self._build_structures_coverage()

            # Read the Holland wind boundaries
            self._read_wind_boundaries()

            self._logger.info('Committing changes....')
            self._sim_data.commit()
            if primary:
                self._build_xms_data()
            self._logger.info(f'Finished reading {self._filename}!')
        except Exception as e:
            self._logger.error(f'Unexpected error reading control file: {str(e)}.')

    def send(self, query):
        """Send all the imported data back to SMS.

        Args:
            query (Query): The XMS interprocess communicator
        """
        if query:
            self._add_build_data(query)
            query.send()

    def read(self):
        """Read all the TUFLOWFV simulations."""
        if not self._filenames:
            self._logger.error('No control files found to import.')

        for fvc_filename in self._filenames:
            if not os.path.isfile(fvc_filename):
                self._logger.error(f'Unable to find control file: {fvc_filename}')
            else:
                self._reset(fvc_filename)
                self._read_sim()
