"""Creates an Materials and Sediment Materials coverage hidden component."""
# 1. Standard python modules
import os
import uuid

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids
from xms.guipy.data.polygon_texture import PolygonTexture

# 4. Local modules
from xms.tuflowfv.components import material_component as mac
from xms.tuflowfv.components.material_component_display import MaterialComponentDisplay
from xms.tuflowfv.components.tuflowfv_component import get_component_data_object
from xms.tuflowfv.data import material_data as mad
from xms.tuflowfv.file_io.io_util import create_component_folder


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


class MaterialComponentBuilder:
    """Builds hidden component for a TUFLOWFV Materials coverage."""
    def __init__(self, cov_uuid, from_2dm, poly_assignments, existing_data=None):
        """Construct the builder.

        Args:
            cov_uuid (str): UUID of the Material coverage to build component for
            from_2dm (bool): True if geometry read from a .2dm, False if read from a shapefile
            poly_assignments (dict): Material ID to list of polygon IDs with that material assignment
            existing_data (dict): Dict of the material data arrays. If not provided default values will be assigned to
                the material attributes.
        """
        self._cov_uuid = cov_uuid
        self._from_2dm = from_2dm
        self._comp_uuid = str(uuid.uuid4())
        self._comp_dir = create_component_folder(self._comp_uuid)
        self._polys = poly_assignments
        self._next_mat_color = 0  # For randomizing material polygon display options
        self._existing_data = existing_data if existing_data else {}  # {mat_id: {'variable': value}}
        self._names = self._get_default_material_names()

    def _get_default_material_names(self):
        """Get the default material names.

        Returns:
            dict: Key is material id, value is default name
        """
        # If read triggered from a .fvc file we have a material list, so add all of them even if there are no cells
        # assigned the material. Mimics old SMS global materials. If reading the .2dm independently, create a material
        # for each unique assignment in the file
        materials_in_list = self._existing_data if self._existing_data else self._polys
        names = {mat_id: f'Material ({mat_id})' for idx, mat_id in enumerate(materials_in_list) if mat_id > 0}
        names[mad.MaterialData.UNASSIGNED_MAT] = 'Unassigned'  # Make sure unassigned exists and is named 'Unassigned'
        return names

    def _get_variable_array(self, variable, default_value, sorted_mat_ids):
        """Get the list of data values for a material data variable.

        Args:
            variable (str): Name of the MaterialData xarray variable
            sorted_mat_ids:

        Returns:
            np.array: The list of values for the material
        """
        return np.array([
            self._existing_data.get(mat_id, {}).get(variable, default_value) for mat_id in sorted_mat_ids
        ])

    def _get_material_poly_display(self, mat_id):
        """Get a set of randomized polygon fill display options.

        Args:
            mat_id (int): The material to get options for

        Returns:
            str: The randomized display options as a string. Formatted as 'R G B texture', where R, G, B, and texture
                are integers.
        """
        # Check if this material already has display attributes specified.
        if mat_id == mad.MaterialData.UNASSIGNED_MAT:  # Always use the same defaults for unassigned.
            return '0 0 0 1'

        if self._next_mat_color >= len(mad.DEFAULT_MATERIAL_COLORS):
            self._next_mat_color = 0
        # We decided that we want all materials (except unassigned) to have a solid pattern by default.
        clr = mad.DEFAULT_MATERIAL_COLORS[self._next_mat_color]
        clr_str = f'{clr[0]} {clr[1]} {clr[2]} {int(PolygonTexture.null_pattern)}'
        self._next_mat_color += 1
        return clr_str

    def _build_material_data(self):
        """Create the material coverage's hidden component and data.

        Returns:
            xarray.Dataset: The material list dataset
        """
        # Build up the material list.
        sorted_mat_ids = list(sorted(self._names.keys()))  # Make unassigned first
        sorted_mat_names = [self._names[mat_id] for mat_id in sorted_mat_ids]
        style = [self._get_material_poly_display(mat_id) for mat_id in sorted_mat_ids]
        mat_data = {
            'id': xr.DataArray(sorted_mat_ids),
            'color': xr.DataArray(style),
            'name': xr.DataArray(sorted_mat_names),
        }
        default_values = mad.MaterialData.default_data_dict()
        default_values.pop('id')
        default_values.pop('color')
        default_values.pop('name')
        for variable, default in default_values.items():
            mat_data[variable] = xr.DataArray(self._get_variable_array(variable, default[0], sorted_mat_ids))
        return xr.Dataset(data_vars=mat_data), sorted_mat_ids

    def _write_component_id_files(self, comp_dir):
        """Write XMS feature id and component id files so we can initialize map component ids in SMS.

        Args:
            comp_dir (str): Path to the component data directory (folder containing the component main file)
        """
        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        att_ids = []
        comp_ids = []
        for mat_id, poly_ids in self._polys.items():
            if mat_id < 0:  # pragma no cover
                continue  # Don't need to write out unassigned polys since it is the default category.
            non_default_poly_ids = [poly_id for poly_id in poly_ids if poly_id > 0]
            att_ids.extend(non_default_poly_ids)
            comp_ids.extend([mat_id for _ in range(len(non_default_poly_ids))])
        id_file = os.path.join(comp_dir, mac.MAT_INITIAL_ATT_ID_FILE)
        write_display_option_ids(id_file, att_ids)
        id_file = os.path.join(comp_dir, mac.MAT_INITIAL_COMP_ID_FILE)
        write_display_option_ids(id_file, comp_ids)

    def build_material_component(self):
        """Create an SRH-2D Material coverage's hidden component and data_object.

        Returns:
            xms.data_objects.parameters.Component: data_object for the new MaterialComponent to send back to SMS.
        """
        # Create the data_object Component to send back to SMS
        mat_main_file = os.path.join(self._comp_dir, 'mat_comp.nc')
        do_comp = get_component_data_object(main_file=mat_main_file, comp_uuid=self._comp_uuid,
                                            unique_name='MaterialComponent')
        mat_comp = mac.MaterialComponent(mat_main_file)
        mat_data, sorted_mat_ids = self._build_material_data()
        mat_comp.data.materials = mat_data
        mat_comp.data.info.attrs['cov_uuid'] = self._cov_uuid
        mat_comp.data.info.attrs['export_format'] = '2dm' if self._from_2dm else 'Shapefile'
        mat_comp.cov_uuid = self._cov_uuid
        mat_comp.data.commit()

        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        display_helper = MaterialComponentDisplay(mat_comp)
        display_helper.update_display_id_files([], sorted_mat_ids)
        display_helper.update_display_options_file()
        self._write_component_id_files(self._comp_dir)
        return do_comp
