"""Dialog to add gridded BCs."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtWidgets import QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.tuflowfv.gui import gui_util
from xms.tuflowfv.gui.global_bcs_table import GlobalBcsTableWidget


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class AddGriddedBcDialog(XmsDlg):
    """Dialog to add gridded BCs to a grid definition."""

    def __init__(self, bc_data, parent, grid_id, time_formats):
        """Initializes the class, sets up the ui.

        Args:
            bc_data (BcData): The simulation's gridded BCs data to edit
            parent (QDialog): The parent dialog
            grid_id (Union[int, None]): If gridded BC type, id of the grid whose datasets are being edited. Pass None
                if a non-gridded global BC.
            time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
                specifiers for (strftime, qt)
        """
        super().__init__(parent, 'xms.tuflowfv.gui.add_gridded_bc_dialog')
        self.grid_id = grid_id
        self.time_formats = time_formats
        self.gridded_bcs_table = None
        self._setup_ui(bc_data)

    def _setup_ui(self, bc_data):
        """Add the table to the dialog.

        Args:
            bc_data (BcData): The simulation's gridded BCs data to edit
        """
        self.setWindowTitle('Add Gridded BCs')
        # Create the table
        self.gridded_bcs_table = GlobalBcsTableWidget(bc_data=bc_data, parent=self, grid_id=self.grid_id,
                                                      time_formats=self.time_formats)
        # Create the button box
        button_box = gui_util.build_ok_cancel_buttons(self)
        # Add the widgets to the dialog.
        layout = QVBoxLayout(self)
        layout.addWidget(self.gridded_bcs_table)
        layout.addWidget(button_box)
        self.setLayout(layout)
