"""Qt dialog for defining a BC grid."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import Qt

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xfs
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.tuflowfv.gui import assign_bc_consts as const, gui_util
from xms.tuflowfv.gui.bc_grid_definition_dialog_ui import Ui_BcGridDefinitionDialog


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class BcGridDefinitionDialog(XmsDlg):
    """A dialog showing the BC grid definition dialog."""

    def __init__(self, parent, grid_definition, sim_data):
        """Constructor.

        Args:
            parent (QWidget): The parent Qt dialog
            grid_definition (xr.Dataset): The simulation grid definitions Dataset
            sim_data (SimData): The simulation data, needed file selector junk
        """
        super().__init__(parent, 'xms.tuflowfv.gui.bc_grid_definition_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:TUFLOW_FV'
        self.grid_definition = grid_definition
        self.sim_data = sim_data
        self.ui = Ui_BcGridDefinitionDialog()
        self.ui.setupUi(self)
        self._setup_ui()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        # Load the grid file first so we can populate the comboboxes from it
        self._load_grid_file()
        self._add_combobox_options()
        self._connect_signals()
        # Populate rest of the widgets from data
        self._load_data()
        # Initialize state
        self.on_tog_define_vert_coord_type(self.ui.tog_define_vert_coord_type.checkState())

    def _load_grid_file(self):
        """Load grid file into widget - Needs to happen first in UI setup."""
        grid_file = self.grid_definition['file'][0].item()
        if grid_file and self.sim_data.does_file_exist(grid_file):
            self.ui.lbl_selection_file.setText(grid_file)

    def _populate_netcdf_combobxes(self):
        """Populate the NetCDF variable comboboxes if a valid file has been selected."""
        grid_file = xfs.resolve_relative_path(self.sim_data.info.attrs['proj_dir'], self.ui.lbl_selection_file.text())
        comboboxes = [self.ui.cbx_x_variable, self.ui.cbx_y_variable, self.ui.cbx_z_variable]
        for combobox in comboboxes:  # Clear previous options
            combobox.clear()
        enable = gui_util.populate_gridded_bc_netcdf_variables(abs_path=grid_file, comboboxes=comboboxes)
        for combobox in comboboxes:  # Disable if invalid/no NetCDF file selected.
            combobox.setEnabled(enable)

    def _add_combobox_options(self):
        """Add items to combobox widgets common to both arcs and points."""
        gui_util.add_combobox_options(const.BC_CBX_OPTS, self.ui.cbx_vertical_distribution)
        self._populate_netcdf_combobxes()

    def _connect_signals(self):
        """Connect Qt widget signal/slots common to both arcs and points."""
        self.ui.btn_file.clicked.connect(self.on_btn_file)
        self.ui.tog_define_vert_coord_type.stateChanged.connect(self.on_tog_define_vert_coord_type)

    def _load_data(self):
        """Populate widgets from persistent data."""
        # Read options from saved files
        self.ui.edt_name.setText(self.grid_definition['name'][0].item())
        gui_util.set_combobox_from_data(self.ui.cbx_x_variable, self.grid_definition['x_variable'][0].item())
        gui_util.set_combobox_from_data(self.ui.cbx_y_variable, self.grid_definition['y_variable'][0].item())
        gui_util.set_combobox_from_data(self.ui.cbx_z_variable, self.grid_definition['z_variable'][0].item())
        self.ui.tog_define_vert_coord_type.setChecked(
            bool(int(self.grid_definition['define_vert_coord_type'][0].item()))
        )
        gui_util.set_combobox_from_data(
            self.ui.cbx_vertical_distribution, self.grid_definition['vert_coord_type'][0].item()
        )
        self.ui.tog_cell_gridmap.setChecked(bool(int(self.grid_definition['cell_gridmap'][0].item())))
        self.ui.tog_boundary_gridmap.setChecked(bool(int(self.grid_definition['boundary_gridmap'][0].item())))
        self.ui.tog_suppress_coverage_warnings.setChecked(
            bool(int(self.grid_definition['suppress_coverage_warnings'][0].item()))
        )

    def _save_data(self):
        """Store widget values in the persistent dataset on 'OK'."""
        grid_file = self.ui.lbl_selection_file.text()
        if grid_file and grid_file != gui_util.NULL_SELECTION:
            self.grid_definition['file'][0] = grid_file
        self.grid_definition['name'][0] = self.ui.edt_name.text()
        self.grid_definition['x_variable'][0] = self.ui.cbx_x_variable.itemData(self.ui.cbx_x_variable.currentIndex())
        self.grid_definition['y_variable'][0] = self.ui.cbx_y_variable.itemData(self.ui.cbx_y_variable.currentIndex())
        self.grid_definition['z_variable'][0] = self.ui.cbx_z_variable.itemData(self.ui.cbx_z_variable.currentIndex())
        self.grid_definition['define_vert_coord_type'][0] = 1 if self.ui.tog_define_vert_coord_type.isChecked() else 0
        self.grid_definition['vert_coord_type'][0] = self.ui.cbx_vertical_distribution.itemData(
            self.ui.cbx_vertical_distribution.currentIndex()
        )
        self.grid_definition['cell_gridmap'][0] = 1 if self.ui.tog_cell_gridmap.isChecked() else 0
        self.grid_definition['boundary_gridmap'][0] = 1 if self.ui.tog_boundary_gridmap.isChecked() else 0
        self.grid_definition['suppress_coverage_warnings'][0] = \
            1 if self.ui.tog_suppress_coverage_warnings.isChecked() else 0

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_data()
        super().accept()

    def on_btn_file(self):
        """Called when the file selector button is clicked."""
        proj_dir = self.sim_data.info.attrs['proj_dir']
        gui_util.select_file(self, self.ui.lbl_selection_file, 'Select a grid definition file',
                             'Grid definition files (*.nc);;All Files (*.*)', proj_dir, False)
        self._populate_netcdf_combobxes()

    def on_tog_define_vert_coord_type(self, state):
        """Called when define vertical coordinate type toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        self.ui.cbx_vertical_distribution.setEnabled(state == Qt.Checked)
