# -*- coding: utf-8 -*-
import param
import pandas as pd


class SedimentProperties(param.Parameterized):
    cohesive_settling = param.ObjectSelector(
        default=0,
        objects={'0 - Free Settling': 0, '1 - Hwang and Mehta': 1},
        doc='SP CSV STR: Cohesive settling velocity.',
        precedence=1,
    )
    cohesive_settling_a = param.Number(
        default=0.0,
        bounds=(None, None),
        doc='SP CSV ATT: Cohesive settling velocity "a" for Hwang and Mehta.',
        precedence=2,
    )
    cohesive_settling_b = param.Number(
        default=0.0,
        bounds=(None, None),
        doc='SP CSV ATT: Cohesive settling velocity "b" for Hwang and Mehta.',
        precedence=3,
    )
    cohesive_settling_m = param.Number(
        default=0.0,
        bounds=(None, None),
        doc='SP CSV ATT: Cohesive settling velocity "m" for Hwang and Mehta.',
        precedence=4,
    )
    cohesive_settling_n = param.Number(
        default=0.0,
        bounds=(None, None),
        doc='SP CSV ATT: Cohesive settling velocity "n" for Hwang and Mehta.',
        precedence=5,
    )
    wind_wave_stress = param.ObjectSelector(
        default=0,
        objects={'0 - No applied wind-wave stress': 0, '1 - Grant and Madsen': 1, '2 - Teeter': 2},
        doc='SP WWS STR: Wind wave shear.',
        precedence=6,
    )
    suspended_entrainment = param.ObjectSelector(
        default=-1,
        objects={'-1 - No suspended load (bedload only)': -1, '0 - Garcia-Parker': 0, '1 - Wright-Parker': 1,
                 'Van Rijn': 2, 'Yang': 3},
        doc='SP NSE STR: Non-cohesive suspended entrainment.',
        precedence=7,
    )
    bedload_entrainment = param.ObjectSelector(
        default=-1,
        objects={'-1 - No bedload load (suspended load only)': -1, '0 - Van Rijn': 0, '1 - Meyer Peter Mueller': 1,
                 '2 - Meyer Peter Mueller with Wong Parker correction': 2, '3 - Wilcock': 3},
        doc='SP NBE STR: Non-cohesive bedload entrainment.',
        precedence=8,
    )
    critical_shear_sand = param.Number(
        default=0.0,
        bounds=(0, None),
        doc='SP NBE ATT: Critical shear stress for sand.',
        precedence=9,
    )
    critical_shear_clay = param.Number(
        default=0.0,
        bounds=(0, None),
        doc='SP NBE ATT: Critical shear stress for clay.',
        precedence=10,
    )
    hiding_factor = param.ObjectSelector(
        default=0,
        objects={'0 - Karim Holly Yang': 0, '1 - Egiazaroff': 1, '2 - Wu Wang Jia': 2, '3 - Parker and Klingeman': 3},
        doc='SP HID STR: Non-cohesive hiding factor.',
        precedence=11,
    )
    hiding_factor_exponent = param.Number(
        default=0.5,
        bounds=(0, 1),
        doc='SP HID ATT: Hiding factor exponent.',
        precedence=12,
    )
    use_infiltration_factor = param.Integer(
        default=0,
        bounds=(0, 1),
        doc='SP SIF ATT: Sediment infiltration factor.',
        precedence=13,
    )
    infiltration_factor = param.Number(
        default=1.0,
        bounds=(0, None),
        doc='SP SIF ATT: Sediment infiltration factor.',
        precedence=14,
    )
    sediment_diversion = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['CARD', 'S_ID', 'TOP', 'BOTTOM', 'BOTTOM_MAIN']),
        doc='Columns [CARD, S_ID, TOP, BOTTOM, BOTTOM_MAIN], '
            'SDV: Sediment diversion. S_ID: String id, TOP: top elevation of the zone of withdrawal, '
            'BOTTOM: bottom elevation of the zone of withdrawal, BOTTOM_MAIN: bottom elevation of the main channel.',
        precedence=15,
    )
    number_bed_layers = param.Integer(
        default=0,
        bounds=(0, None),
        doc='MP NBL: Number of bed layers.',
        precedence=16,
    )
    bed_layer_thickness_protocol = param.Integer(
        default=0,
        bounds=(0, 1),
        doc='MP NBL: Bed layer thickness assignment protocol.',
        precedence=17,
    )
    number_consolidation_times = param.Integer(
        default=0,
        bounds=(0, None),
        doc='MP NCP: Number of consolidation times.',
        precedence=18,
    )
    global_consolidation = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['MP', 'CARD', 'TIME_ID', 'ELAPSED_TIME', 'POROSITY', 'CRITICAL_SHEAR',
                                               'EROSION_CONSTANT', 'EROSION_EXPONENT']),
        doc='Columns [MP, CARD, TIME_ID, ELAPSED_TIME, POROSITY, CRITICAL_SHEAR, EROSION_CONSTANT,'
            ' EROSION_EXPONENT], '
            'MP CPA: Consolidation time series. TIME_ID: Consolidation time id, '
            'ELAPSED_TIME: elapsed time since sediment deposition in seconds, '
            'POROSITY: porosity at the elapsed time, '
            'CRITICAL_SHEAR: critical shear stress for erosion at the elapsed time,'
            'EROSION_CONSTANT: erosion rate constant at the elapsed time, '
            'EROSION_EXPONENT: erosion rate exponent at the elapsed time.',
        precedence=20,
    )
    material_consolidation = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['MP', 'CARD', 'MATERIAL_ID', 'TIME_ID', 'ELAPSED_TIME', 'POROSITY',
                                               'CRITICAL_SHEAR', 'EROSION_CONSTANT', 'EROSION_EXPONENT']),
        doc='Columns [MP, CARD, MATERIAL_ID, TIME_ID, ELAPSED_TIME, POROSITY, CRITICAL_SHEAR, EROSION_CONSTANT,'
            ' EROSION_EXPONENT], '
            'MP CPM: Consolidation time series for materials. MATERIAL_ID: Material id, '
            'TIME_ID: consolidation time id, '
            'ELAPSED_TIME: elapsed time since sediment deposition in seconds, '
            'POROSITY: porosity at the elapsed time, '
            'CRITICAL_SHEAR: critical shear stress for erosion at the elapsed time,'
            'EROSION_CONSTANT: erosion rate constant at the elapsed time, '
            'EROSION_EXPONENT: erosion rate exponent at the elapsed time.',
        precedence=21,
    )
    global_cohesive_bed = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['MP', 'CARD', 'BED_LAYER_ID', 'POROSITY', 'CRITICAL_SHEAR',
                                               'EROSION_CONSTANT', 'EROSION_EXPONENT']),
        doc='Columns [MP, CARD, BED_LAYER_ID, POROSITY, CRITICAL_SHEAR, EROSION_CONSTANT, EROSION_EXPONENT], '
            'MP CBA: Cohesive bed sediment. BED_LAYER_ID: Bed layer id, '
            'POROSITY: porosity for the bed layer, '
            'CRITICAL_SHEAR: critical shear stress for erosion for the bed layer,'
            'EROSION_CONSTANT: erosion rate constant for the bed layer, '
            'EROSION_EXPONENT: erosion rate exponent for the bed layer.',
        precedence=22,
    )
    material_cohesive_bed = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['MP', 'CARD', 'BED_LAYER_ID', 'MATERIAL_ID', 'POROSITY',
                                               'CRITICAL_SHEAR', 'EROSION_CONSTANT', 'EROSION_EXPONENT']),
        doc='Columns [MP, CARD, BED_LAYER_ID, MATERIAL_ID, POROSITY, CRITICAL_SHEAR, EROSION_CONSTANT,'
            ' EROSION_EXPONENT], '
            'MP CBM: Cohesive bed sediment for materials. BED_LAYER_ID: Bed layer id, MATERIAL_ID: material id, '
            'POROSITY: porosity for the bed layer, '
            'CRITICAL_SHEAR: critical shear stress for erosion for the bed layer,'
            'EROSION_CONSTANT: erosion rate constant for the bed layer, '
            'EROSION_EXPONENT: erosion rate exponent for the bed layer.',
        precedence=23,
    )
    global_bed_layers = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['CARD', 'BED_LAYER_ID', 'THICKNESS']),
        doc='Columns [CARD, BED_LAYER_ID, THICKNESS], '
            'MP SBA: Cohesive bed sediment. BED_LAYER_ID: Bed layer id, '
            'THICKNESS: bed layer thickness or strata elevation horizon.',
        precedence=24,
    )
    material_bed_layers = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['CARD', 'BED_LAYER_ID', 'MATERIAL_ID', 'THICKNESS']),
        doc='Columns [CARD, BED_LAYER_ID, MATERIAL_ID, THICKNESS], '
            'MP SBM: Cohesive bed sediment for materials. BED_LAYER_ID: Bed layer id, MATERIAL_ID: material id, '
            'THICKNESS: bed layer thickness or strata elevation horizon.',
        precedence=25,
    )
    bed_layer_grain_fractions = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['MATERIAL_ID', 'BED_LAYER_ID', 'CONSTITUENT_ID', 'FRACTION',]),
        doc='Columns [MATERIAL_ID, BED_LAYER_ID, CONSTITUENT_ID, FRACTION], '
            'MATERIAL_ID: Material id or 0 if global, '
            'BED_LAYER_ID: bed layer id, '
            'CONSTITUENT_ID: bed layer thickness or strata elevation horizon, '
            'FRACTION: the grain class fraction for the sediment constituent.',
        precedence=26,
    )
    material_displacement_off = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['CARD', 'MATERIAL_ID']),
        doc='Columns [CARD, MATERIAL_ID], '
            'MP NDM: Turn off displacement for materials. MATERIAL_ID: material id.',
        precedence=25,
    )
    material_local_scour = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['CARD', 'MATERIAL_ID']),
        doc='Columns [CARD, MATERIAL_ID], '
            'MP LSM: Local scour for materials. MATERIAL_ID: material id.',
        precedence=27,
    )
    material_diffusion = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['CARD', 'MATERIAL_ID', 'DIFFUSION']),
        doc='Columns [CARD, MATERIAL_ID, DIFFUSION], '
            'MP BLD: Bedload diffusion for materials. MATERIAL_ID: material id, '
            'DIFFUSION: the bedload diffusion constant.',
        precedence=27,
    )

    def __init__(self):
        super(SedimentProperties, self).__init__()

    @param.depends('cohesive_settling', watch=True)
    def _update_cohesive_settling(self):
        self.param.cohesive_settling_a.precedence = -1
        self.param.cohesive_settling_b.precedence = -1
        self.param.cohesive_settling_m.precedence = -1
        self.param.cohesive_settling_n.precedence = -1
        objects = list(self.param.cohesive_settling.get_range())
        if self.cohesive_settling == objects[1]:
            self.param.cohesive_settling_a.precedence = 2
            self.param.cohesive_settling_b.precedence = 3
            self.param.cohesive_settling_m.precedence = 4
            self.param.cohesive_settling_n.precedence = 5

    @param.depends('bedload_entrainment', watch=True)
    def _update_bedload_entrainment(self):
        self.param.critical_shear_sand.precedence = -1
        self.param.critical_shear_clay.precedence = -1
        objects = list(self.param.bedload_entrainment.get_range())
        if self.bedload_entrainment == objects[1]:
            self.param.critical_shear_sand.precedence = 9
            self.param.critical_shear_clay.precedence = 10

    @param.depends('hiding_factor', watch=True)
    def _update_hiding_factor(self):
        self.param.hiding_factor_exponent.precedence = -1
        objects = list(self.param.hiding_factor.get_range())
        if self.hiding_factor == objects[1]:
            self.param.hiding_factor_exponent.precedence = 12
