"""Weir Class."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.variable_group import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable


class CalcOrVarlist(VariableGroup):
    """A base class that defines a class that manages a list of calculators or variables."""

    def __init__(self, class_of_item, app_data=None, model_name=None, project_uuid=None,
                 show_define_btn=True, default_name='item',
                 default_plural_name='items', select_one=True, min_number_of_items=1,
                 max_number_of_items=sys.float_info.max, initial_num_items=1, show_number=True,
                 show_name=True, show_duplicate=True, show_delete=True, *args, **kwargs):
        """Initializes the CalcData or variable list."""
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.calc_support_dual_input = False  # Displays a second column of input (HY-8 Style)
        self.calc_support_warnings = True  # Displays the warnings text box
        self.calc_support_results = True  # Displays the Results table box
        self.calc_support_plot = True  # Displays the plot
        self.calc_save_on_accept = False  # Saves the CalcData to a file

        self.dlg_width = None
        self.dlg_height = None

        self.item_list = []
        self.deleted_list = []

        self.class_of_item = class_of_item
        self.args = args
        self.kwargs = kwargs
        self.default_name = default_name
        self.default_plural_name = default_plural_name
        self.min_number_of_items = min_number_of_items
        self.max_number_of_items = max_number_of_items
        self.select_one = select_one
        self.show_number = show_number  # Size may be managed elsewhere
        self.show_name = show_name  # Name may be set elsewhere
        self.show_duplicate = show_duplicate
        self.show_delete = show_delete

        # This flags are used when the list size is managed elsewhere (e.g. Bridgescour)
        self.btn_pressed = None
        self.btn_item_index = None

        num_items = initial_num_items
        if num_items < self.min_number_of_items:
            num_items = self.min_number_of_items

        self.input['Number of items'] = Variable(f'Number of {default_plural_name}', 'int', num_items,
                                                 limits=[min_number_of_items, max_number_of_items])
        self.input['Selected item'] = Variable(f'Selected {default_name}', 'list', 0, [])

        self.name = default_name

        # Open this wiki page when the help is clicked
        # self.self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Channel_Calculator'

        self.show_define_btn = show_define_btn

        self.unknown = None

        # 2D dict, Allow multiple plots
        self.plots = {}
        # self.plots['profile'] = {}
        # self.plots['profile']['Plot name'] = "Weir Profile"
        # self.plots['profile']['Legend'] = "upper right"

        # reference documents (two otions for use)
        # self.reference['pdf Title'] = document_path
        # self.reference['pdf Title'] = (document_path, page_number)
        self.reference_pdfs = {}

        self.check_list_length()

    def check_list_length(self):
        """Checks the length of the list and adds items if necessary."""
        num_items = self.input['Number of items'].get_val()
        if num_items < self.min_number_of_items:
            num_items = self.min_number_of_items
        while num_items > len(self.item_list):
            self.add_item_to_list(False)

        item_names = []
        for item in self.item_list:
            item_names.append(item.name)
        self.input['Selected item'].value_options = item_names

    def get_input_group(self, unknown=None, sys_complexity=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        # Reset the duplicate and delete pressed flags
        # This flag are used when the list size is managed elsewhere (e.g. Bridgescour)
        self.btn_pressed = None
        self.btn_item_index = None

        input_vars = {}
        # First make sure that our item list is long enough:
        self.check_list_length()

        num_items = self.input['Number of items'].get_val()

        _, null_data = self.get_setting('Null data')

        sys_complexity = self.get_system_complexity_index()

        # Add each item's class
        if self.select_one:
            input_vars['Selected item'] = self.input['Selected item']

        if self.show_number:
            input_vars['Number of items'] = self.input['Number of items']
        index = self.input['Selected item'].get_index(null_data)

        if num_items > 0 and index is not None:
            if self.select_one:
                input_vars.update(self.get_input_group_for_index(index, sys_complexity))
            else:
                for index in range(num_items):
                    result = self.get_input_group_for_index(index, sys_complexity)
                    if result is not None:
                        input_vars.update(result)

        return input_vars

    def get_input_group_for_index(self, index, sys_complexity):
        """Gets the input group for item with the given index.

        Args:
            index (int): index of the item to get input group
            sys_complexity (int): the system complexity
        """
        input_vars = {}
        if index >= len(self.item_list):
            return None
        if self.item_list[index].complexity > sys_complexity:
            return None
        default_name = self.default_name[0].upper() + self.default_name[1:]
        # Name
        if self.show_name:
            input_vars[f'CalcOrVarlist item {index} name'] = Variable(
                f'{default_name} name', 'string', self.item_list[index].name, [])
            if 'Name' in self.item_list[index].input:
                self.item_list[index].name = self.item_list[index].input['Name'].value
                input_vars[f'CalcOrVarlist item {index} name'].value = self.item_list[index].input['Name'].value
                input_vars[f'CalcOrVarlist item {index} name'].uuid = self.item_list[index].input['Name'].uuid
            else:
                input_vars[f'CalcOrVarlist item {index} name'].uuid = f'{self.uuid}, {index}'
        # duplicate
        if self.show_duplicate:
            btn_name = 'Duplicate'
            if not self.show_name:
                btn_name = f'Duplicate {self.item_list[index].name}'
            input_vars[f'CalcOrVarlist item {index} duplicate'] = Variable(
                f'Duplicate {self.default_name}', 'action', lambda index: self.duplicate_item(index), btn_name)
            input_vars[f'CalcOrVarlist item {index} duplicate'].uuid = f'{self.uuid}, {index}'
        # delete
        if self.show_delete:
            if len(self.item_list) > self.min_number_of_items:
                btn_name = 'Delete'
                if not self.show_name:
                    btn_name = f'Delete {self.item_list[index].name}'
                input_vars[f'CalcOrVarlist item {index} delete'] = Variable(
                    f'Delete {self.default_name}', 'action', lambda index: self.delete_item(index), btn_name)
                input_vars[f'CalcOrVarlist item {index} delete'].uuid = f'{self.uuid}, {index}'

        if self.show_define_btn:
            # define
            input_vars[f'CalcOrVarlist item {index} define'] = Variable(
                f'Define {self.default_name} parameters', 'action', self.item_list[index],
                f'Edit {self.item_list[index].name}...')
            input_vars[f'CalcOrVarlist item {index} define'].uuid = f'{self.uuid}, {index}'
        else:
            item_var_list = self.item_list[index].get_input_group(None)
            for var in item_var_list:
                input_vars[f'CalcOrVarlist item {index} {var}'] = item_var_list[var]

        return input_vars

    def get_results_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              results_vars (dictionary of variables): the input variables
        """
        results_vars = self.get_results_group()

        return {'Results': results_vars}

    def add_item_to_list(self, update_number=True):
        """Adds an item to the list.

        Args:
            update_number (bool): whether to update the Number of items
        """
        # new_instance = type(self.class_of_item)(clear_my_own_results=False)
        kwargs = self.kwargs
        if 'kwargs' in self.kwargs:
            kwargs = self.kwargs['kwargs']
        if self.app_data is not None:
            if self.model_name is not None and self.model_name is not None:
                new_instance = self.class_of_item(*self.args, app_data=self.app_data,
                                                  model_name=self.model_name,
                                                  project_uuid=self.project_uuid, **kwargs)
            else:
                new_instance = self.class_of_item(*self.args, app_data=self.app_data, **kwargs)
        else:
            new_instance = self.class_of_item(*self.args, **kwargs)

        new_instance.name = f'{self.default_name} {len(self.item_list) + 1}'
        if 'Name' in new_instance.input:
            new_instance.input['Name'].set_val(new_instance.name)
        self.item_list.append(new_instance)
        if update_number:
            self.input['Number of items'].set_val(self.input['Number of items'].get_val() + 1)

    def get_item_by_name(self, item_name):
        """Gets an item by its name.

        Args:
            item_name (str): name of the item to get

        Returns:
            item: the item
        """
        for item in self.item_list:
            if item.name == item_name:
                return item
        return None

    def set_item_by_name(self, new_item):
        """Gets an item by its name.

        Args:
            new_item (item): the new item

        Returns:
            item: the item
        """
        for index, item in enumerate(self.item_list):
            if item.name == new_item.name:
                self.item_list[index] = copy.deepcopy(new_item)
                return True
        return False

    def set_item_by_index(self, new_item, index):
        """Gets an item by its name.

        Args:
            new_item (item): the new item

        Returns:
            item: the item
        """
        if index < 0 or index >= len(self.item_list):
            return False
        self.item_list[index] = copy.deepcopy(new_item)
        return True

    def unduplicate_item(self, uuid):
        """Unduplicates an item from the list.

        Args:
            uuid (str): the uuid of the item to unduplicate
        """
        self.btn_pressed = 'unduplicate'
        item_index, _ = self.find_item_by_uuid(uuid, False)
        if item_index is not None:
            self.btn_item_index = item_index
            self.delete_item(item_index)

    def duplicate_item(self, item_index):
        """Duplicates an item to the list.

        Args:
            item_index (int): index of the item to duplicate

        Returns:
            str: the uuid of the new item
        """
        self.btn_pressed = 'duplicate'
        if item_index < 0:
            return
        if item_index > len(self.item_list):
            return
        self.btn_item_index = item_index
        new_item = copy.deepcopy(self.item_list[item_index])
        new_item.name += ' (Copy)'
        new_item.uuid = uuid.uuid4()
        index = self.input['Number of items'].get_val()
        self.item_list.insert(index, new_item)
        self.input['Number of items'].set_val(index + 1)
        return new_item.uuid

    def delete_item(self, item_index):
        """Adds an item to the list.

        Args:
            item_index (int): index of the item to delete

        Returns:
            str: the uuid of the deleted item
        """
        self.btn_pressed = 'delete'
        if item_index < 0:
            return
        if item_index > len(self.item_list):
            return
        self.btn_item_index = item_index
        deleted_item = copy.deepcopy(self.item_list[item_index])
        self.deleted_list.append(deleted_item)
        self.item_list.pop(item_index)
        self.input['Number of items'].set_val(self.input['Number of items'].get_val() - 1)
        return deleted_item.uuid

    def undelete_item(self, uuid):
        """Undeletes an item from the list.

        Args:
            uuid (str): the uuid of the item to undelete
        """
        self.btn_pressed = 'undelete'
        undelete_index, undelete_item = self.find_item_by_uuid(uuid, True)

        if undelete_item:
            index = self.input['Number of items'].get_val()
            self.btn_item_index = index
            self.item_list.insert(index, undelete_item)
            self.input['Number of items'].set_val(index + 1)
            self.deleted_list.pop(undelete_index)

    def find_item_by_uuid(self, uuid, use_delete_list=False):
        """Finds an item by its uuid.

        Args:
            uuid (str): the uuid of the item to find
            use_delete_list (bool): whether to use the deleted list

        Returns:
            int: index of the item
            item: the item
        """
        item_list = self.item_list
        if use_delete_list:
            item_list = self.deleted_list

        index = 0
        for item in item_list:
            if item.uuid == uuid:
                return index, item
            index += 1

        return None, None

    def compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.check_list_length()

        return False

    def setup_plot_data(self, axes, index, show_axis_titles):
        """Setup the plot data.

        Args:
            axes (AxesSubplot): The plot axes
            index (int): index of flow to display
            show_axis_titles (bool): whether to show the titles of the axes
        """
        pass
