"""SeriesOptions Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class LineOptions(VariableGroup):
    """Class that defines the data for plotting options."""

    def __init__(self, input_list=None, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Plot Series Options.

        Args:
            input_list (dict): dictionary of input variables
            app_data (AppData): application data (for settings)
            model_name (str): name of the model
            project_uuid (uuid): project uuid
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Plot line Options'
        self.type = 'LineOptions'

        # self.input_list = input_list

        # Input
        self.line_alignments = ['vertical', 'horizontal']
        self.text_v_alignments = ['top', 'center', 'bottom', 'None']  # , 'baseline']
        self.text_h_alignments = ['left', 'center', 'right', 'None']  # , 'baseline']
        self.linetypes = ['solid', 'dashed', 'dotted', 'dash-dot']
        self.patterns = ['solid', 'diagonal', 'reverse diagonal', 'vertical', 'horizontal', 'grid', 'diamond',
                         'small dots', 'medium dots', 'large dots', 'stars']

        self.var = None  # Used for unit conversion; set it to a variable that plots the x or y axis data

        self.input['Label'] = Variable('Label', 'string', self.name)
        self.input['Display'] = Variable('Display', 'bool', True)
        self.input['Multiple labels'] = Variable('Label', 'bool', False, complexity=3)
        self.input['Labels'] = Variable('Labels', 'string_list', self.name, complexity=3)
        self.input['Line intercepts'] = Variable('Line intercepts', 'float_list', complexity=3)
        self.input['Line alignment'] = Variable('Text alignment', 'list', 0, self.line_alignments, complexity=3)
        self.input['Line color'] = Variable('Line color', 'color', 'rgb', (256, 256, 256))
        self.input['Line type'] = Variable('Line type', 'list', 0, self.linetypes)
        # Not currently used
        self.input['Line width'] = Variable('Line width', 'float', 1.0, complexity=4)
        # Not currently used
        self.input['Text color'] = Variable('Line color', 'color', 'rgb', (256, 256, 256), complexity=4)
        self.input['Text v alignment'] = Variable('Text alignment', 'list', 2, self.text_v_alignments)
        self.input['Text h alignment'] = Variable('Text alignment', 'list', 2, self.text_h_alignments)

        self.warnings = []
        self.results = {}
        # self.results['Results'] = Variable('Results', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

    def set_plot_line_options(self, name='', line_intercept=None, line_alignment='vertical', line_color=None,
                              linetype=None, line_width=None, text_color=None, text_alignment=None,
                              multiple_labels=None, labels=None):
        """Set the plot options.

        Args:
            name (str): name of the line
            line_intercept (float): where the line will cross the axis
            line_alignment (string: 'vertical' or 'horizontal'): direction of line
            line_color (color tuple): line color
            linetype (str): line type ['solid', 'dashed', 'dotted', 'dash-dot';]
            line_width (float): line width
            text_color (color tuple): text color of label
            text_alignment (string): alignment of label
            multiple_labels (bool): whether to use multiple labels
            labels (list): list of labels
        """
        if name is not None and name != '':
            self.name = name
            self.input['Label'].set_val(name)
        if line_intercept is not None:
            self.input['Line intercepts'].set_val(line_intercept)
        if line_alignment is not None:
            self.input['Line alignment'].set_val(line_alignment)
        if line_color is not None:
            self.input['Line color'].set_val(line_color)
        if linetype is not None:
            self.input['Line type'].set_val(linetype)
        if line_width is not None:
            self.input['Line width'].set_val(line_width)
        if text_color is not None:
            self.input['Text color'].set_val(text_color)
        if text_alignment is not None:
            if self.input['Line alignment'].get_val() == 'vertical':
                self.input['Text v alignment'].set_val(text_alignment)
            else:
                self.input['Text h alignment'].set_val(text_alignment)
        if multiple_labels is not None:
            self.input['Multiple labels'].set_val(multiple_labels)
        if labels is not None:
            self.input['Labels'].set_val(labels)

    def get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        return True, ''

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}

        # input_vars[self.input['Label'].get_val()] = copy.deepcopy(self.input)

        input_vars = copy.deepcopy(self.input)

        if input_vars['Line alignment'].get_val() == 'vertical':
            input_vars.pop('Text h alignment')
        else:
            input_vars.pop('Text v alignment')

        if not input_vars['Multiple labels'].get_val():
            input_vars.pop('Labels')

        return input_vars

    @staticmethod
    def convert_linetype_text_to_symbol(linetype):
        """Convert the line type text to a symbol.

        Args:
            linetype (str): line type text

        Returns:
            str: line type symbol
        """
        if linetype == 'solid':
            return '-'
        elif linetype == 'dashed':
            return '--'
        elif linetype == 'dotted':
            return ':'
        elif linetype == 'dash-dot':
            return '-.'

    def get_plot_options_dict(self):
        """Get the plot options dictionary."""
        plot_options = {}

        var = self.var
        if var is None:
            plot_options['native_unit'] = ''
            plot_options['selected_us_unit'] = ''
            plot_options['selected_si_unit'] = ''
        else:
            plot_options['native_unit'] = var.native_unit
            plot_options['selected_us_unit'] = var.selected_us_unit
            plot_options['selected_si_unit'] = var.selected_si_unit

        plot_options['Label'] = self.input['Label'].get_val()
        plot_options['Multiple labels'] = self.input['Multiple labels'].get_val()
        plot_options['Labels'] = self.input['Labels'].get_val()
        plot_options['Line intercepts'] = self.input['Line intercepts'].get_val()
        plot_options['Line alignment'] = self.input['Line alignment'].get_val()
        plot_options['Line color'] = self.input['Line color'].get_val()
        plot_options['Line type'] = self.convert_linetype_text_to_symbol(
            self.input['Line type'].get_val())
        plot_options['Line width'] = self.input['Line width'].get_val()
        plot_options['Text color'] = self.input['Text color'].get_val()

        if self.input['Line alignment'].get_val() == 'vertical':
            plot_options['Text alignment'] = self.input['Text v alignment'].get_val()
        else:
            plot_options['Text alignment'] = self.input['Text h alignment'].get_val()

        return plot_options
