"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionDensityCalc:
    """Perform density unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Density
    # ----------------------------------------------------------------------
    lb_ft3 = ['slugs/ft^3', 'slug/ft^3', 'lb/ft^3', 'lb/ft^3', 'lbs/ft^3', 'pound/ft^3', 'pounds/ft^3']
    kg_m3 = ['kg/m^3', 'KG/M^3']
    g_l = ['g/L', 'G/L']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.lb_ft3:
            return True, 'kg/m^3'
        return False, ''

    # ----------------------------------------------------------------------
    # Density
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert density.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.kg_m3 or from_unit in self.g_l:
            return self.convert_from_kg_m3(to_unit, value)
        elif from_unit in self.lb_ft3:
            return self.convert_from_lb_ft3(to_unit, value)

        return False, value

    def convert_from_kg_m3(self, to_unit, value):
        """Convert from the kg/m^3 unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.kg_m3 or to_unit in self.g_l:
            return True, value
        elif to_unit in self.lb_ft3:
            return True, value / 515.3788184

        else:
            return False, value

    def convert_from_lb_ft3(self, to_unit, value):
        """Convert from the lb/ft^3 unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.kg_m3 or to_unit in self.g_l:
            return True, value * 515.3788184
        elif to_unit in self.lb_ft3:
            return True, value

        else:
            return False, value
