"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionFlowCalc:
    """Perform flow unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Flow
    # ----------------------------------------------------------------------
    # US Units
    cfy = ['cfy', 'ft^3/year', 'cubic feet per year', 'cubic feet per year (cfy)', 'cubic feet per year (ft^3/y)']
    cfs = ['cfs', 'ft^3/s', 'cubic feet per second', 'cubic feet per second (cfs)', 'cubic feet per second (ft^3/s)']
    stream = ['stream', 'Stream', 'STREAM']

    gpm = ['gpm', 'gal/min', 'gallons per minute', 'gallons per day (gpm)']
    gpd = ['gpd', 'gal/day', 'gallons per day', 'gallons per day (gpd)']

    # SI Units
    cmy = ['cmy', 'm^3/y', 'cubic meters per year', 'cubic meters per year (cmy)',
           'cubic meters per year (m^3/y)']
    cmh = ['cmh', 'm^3/hr', 'cubic meters per hour', 'cubic meters per hour (cmh)', 'cubic meters per minute (m^3/h)']
    cmm = ['cmm', 'm^3/m', 'cubic meters per minute', 'cubic meters per minute (cmm)',
           'cubic meters per minute (m^3/m)']
    cms = ['cms', 'm^3/s', 'cubic meters per second', 'cubic meters per second (cms)',
           'cubic meter per second (ft^3/s)']
    Mld = ['Mld', 'Ml/day', 'megalitre per day', 'megalitre per day (Mld)', 'megalitre per day (Ml/d)']
    Mls = ['Mls', 'Ml/s', 'megalitre per second', 'megalitre per second (Mls)', 'megalitre per second (Ml/s)']
    lm = ['lm', 'l/min', 'litre per minute', 'litre per minute (l/m)']
    ld = ['ld', 'l/day', 'litre per day', 'litre per day (l/s)']
    ls = ['ls', 'l/s', 'litre per second', 'litre per second (l/s)']

    def __init__(self, us_gallon=True, stream_value=12.0):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """
        self.us_gallon = us_gallon
        self.stream_value = stream_value

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.cfy:
            return True, 'cmy'
        elif from_unit in self.cfs:
            return True, 'cms'
        elif from_unit in self.stream:
            return True, 'stream'
        elif from_unit in self.gpm:
            return True, 'lm'
        elif from_unit in self.gpd:
            return True, 'ld'
        return False, ''

    # ----------------------------------------------------------------------------------------------------------------------
    # Flow
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert flow.

        Args:
            from_unit (string): unit that the value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Units
        if from_unit in self.cfy:
            return self.convert_from_cfy(to_unit, value)
        elif from_unit in self.cfs:
            return self.convert_from_cfs(to_unit, value)
        elif from_unit in self.stream:
            return self.convert_from_stream(to_unit, value)
        elif from_unit in self.gpm:
            return self.convert_from_gpm(to_unit, value)
        elif from_unit in self.gpd:
            return self.convert_from_gpd(to_unit, value)
        # SI Units
        elif from_unit in self.cmy:
            return self.convert_from_cmy(to_unit, value)
        elif from_unit in self.cmh:
            return self.convert_from_cmh(to_unit, value)
        elif from_unit in self.cmm:
            return self.convert_from_cmm(to_unit, value)
        elif from_unit in self.cms:
            return self.convert_from_cms(to_unit, value)
        elif from_unit in self.Mld:
            return self.convert_from_mld(to_unit, value)
        elif from_unit in self.Mls:
            return self.convert_from_mls(to_unit, value)
        elif from_unit in self.lm:
            return self.convert_from_lm(to_unit, value)
        elif from_unit in self.ld:
            return self.convert_from_ld(to_unit, value)
        elif from_unit in self.ls:
            return self.convert_from_ls(to_unit, value)

        return False, value

    def convert_from_cfy(self, to_unit, value):
        """Convert from the cubic feet per year unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value
        elif to_unit in self.cfs or to_unit in self.stream:
            value /= 31536000
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 1.423247e-5
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 0.02049068
        # SI
        elif to_unit in self.cmy:
            return True, value * 0.028317017
        elif to_unit in self.cmh:
            return True, value * 3.232536e-6
        elif to_unit in self.cmm:
            return True, value * 5.3875603e-8
        elif to_unit in self.cms:
            return True, value * 8.97926724e-10
        elif to_unit in self.Mld:
            return True, value * 7.75808689e-8
        elif to_unit in self.Mls:
            return True, value * 8.97926724e-13
        elif to_unit in self.lm:
            return True, value * 5.38756034e-5
        elif to_unit in self.ld:
            return True, value / 12.8984065
        elif to_unit in self.ls:
            return True, value * 8.9792672e-7

        else:
            return False, value

    def convert_from_cfs(self, to_unit, value):
        """Convert from the cubic feet per second unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 31535810
        elif to_unit in self.cfs or to_unit in self.stream:
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 448.8325
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 646190.439
        # SI
        elif to_unit in self.cmy:
            return True, value * 893000.074
        elif to_unit in self.cmh:
            return True, value * 101.940647
        elif to_unit in self.cmm:
            return True, value * 1.699010795
        elif to_unit in self.cms:
            return True, value * 0.02831684
        elif to_unit in self.Mld:
            return True, value * 2.44657554
        elif to_unit in self.Mls:
            return True, value * 2.8316846592E-5
        elif to_unit in self.lm:
            return True, value * 1699.01079
        elif to_unit in self.ld:
            return True, value * 2446575.55
        elif to_unit in self.ls:
            return True, value * 28.316846

        else:
            return False, value

    def convert_from_stream(self, to_unit, value):
        """Convert from the stream unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        value *= self.stream_value
        return self.convert_from_cfs(to_unit, value)

    def convert_from_gpm(self, to_unit, value):
        """Convert from the gallons per minute unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if not self.us_gallon:
            value /= 0.83267384

        if to_unit in self.cfy:
            return True, value * 70262.5
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 0.002228002
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            return True, value
        elif to_unit in self.gpd:
            return True, value * 1440.0
        # SI
        elif to_unit in self.cmy:
            return True, value * 1989.6062
        elif to_unit in self.cmh:
            return True, value * 0.22712
        elif to_unit in self.cmm:
            return True, value * 0.0037854
        elif to_unit in self.cms:
            return True, value * 6.309E-5
        elif to_unit in self.Mld:
            return True, value * 0.005450976
        elif to_unit in self.Mls:
            return True, value * 6.309E-8
        elif to_unit in self.lm:
            return True, value * 3.7854
        elif to_unit in self.ld:
            return True, value * 5451.0
        elif to_unit in self.ls:
            return True, value * 0.06309

        else:
            return False, value

    def convert_from_gpd(self, to_unit, value):
        """Convert from the gallons per day unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if not self.us_gallon:
            value /= 0.83267384

        if to_unit in self.cfy:
            return True, value * 48.7934
        elif to_unit in self.cfs or to_unit in self.stream:
            value /= 646317.0
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            return True, value / 1440.0
        elif to_unit in self.gpd:
            return True, value
        # SI
        elif to_unit in self.cmy:
            return True, value * 1.3819456617003
        elif to_unit in self.cmh:
            return True, value * 0.000157756
        elif to_unit in self.cmm:
            return True, value * 2.6292725679229E-6
        elif to_unit in self.cms:
            return True, value * 4.3821209465381E-8
        elif to_unit in self.Mld:
            return True, value * 3.7861524978089E-6
        elif to_unit in self.Mls:
            return True, value * 4.3821209465381E-11
        elif to_unit in self.lm:
            return True, value * 0.0026292725
        elif to_unit in self.ld:
            return True, value * 3.785
        elif to_unit in self.ls:
            return True, value * 4.3821209465381E-5

        else:
            return False, value

    def convert_from_cmy(self, to_unit, value):
        """Convert from the cubic meters per year unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 35.31445398
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 1.1198207357144E-6
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 0.000502612014
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 0.723617453069
        # SI
        elif to_unit in self.cmy:
            return True, value
        elif to_unit in self.cmh:
            return True, value * 0.00011415525
        elif to_unit in self.cmm:
            return True, value * 1.9025875190259E-6
        elif to_unit in self.cms:
            return True, value / 31536000.0
        elif to_unit in self.Mld:
            return True, value * 2.7397260273973E-6
        elif to_unit in self.Mls:
            return True, value * 3.1709791983765E-11
        elif to_unit in self.lm:
            return True, value * 0.0019025875190
        elif to_unit in self.ld:
            return True, value * 2.73973
        elif to_unit in self.ls:
            return True, value * 3.1709791983765E-5

        else:
            return False, value

    def convert_from_cmh(self, to_unit, value):
        """Convert from the cubic meters per hour unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 309354.616944
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 0.00980962964
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 4.40288124
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 6338.888888
        # SI
        elif to_unit in self.cmy:
            return True, value * 8760.0
        elif to_unit in self.cmh:
            return True, value
        elif to_unit in self.cmm:
            return True, value * 0.01666666666666
        elif to_unit in self.cms:
            return True, value / 3600.0
        elif to_unit in self.Mld:
            return True, value * 0.024
        elif to_unit in self.Mls:
            return True, value * 2.7777777777778E-7
        elif to_unit in self.lm:
            return True, value * 16.666666666667
        elif to_unit in self.ld:
            return True, value * 24000.0
        elif to_unit in self.ls:
            return True, value * 0.2777777777777

        else:
            return False, value

    def convert_from_cmm(self, to_unit, value):
        """Convert from the cubic meters minute unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 18561388.83
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 0.588577778691
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 264.17287472
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 380333.3333333
        # SI
        elif to_unit in self.cmy:
            return True, value * 525600
        elif to_unit in self.cmh:
            return True, value * 60.0
        elif to_unit in self.cmm:
            return True, value
        elif to_unit in self.cms:
            return True, value / 60.0
        elif to_unit in self.Mld:
            return True, value * 1.44
        elif to_unit in self.Mls:
            return True, value * 1.6666666666667E-5
        elif to_unit in self.lm:
            return True, value * 1000
        elif to_unit in self.ld:
            return True, value * 1440000.0
        elif to_unit in self.ls:
            return True, value * 16.666666666667

        else:
            return False, value

    def convert_from_cms(self, to_unit, value):
        """Convert from the cubic meters per second unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 1113676621
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 35.314666721489
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 15850.372483753
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 22820000.0
        # SI
        elif to_unit in self.cmy:
            return True, value * 31536000.0
        elif to_unit in self.cmh:
            return True, value * 3600.0
        elif to_unit in self.cmm:
            return True, value * 60.0
        elif to_unit in self.cms:
            return True, value
        elif to_unit in self.Mld:
            return True, value * 86.4
        elif to_unit in self.Mls:
            return True, value * 0.001
        elif to_unit in self.lm:
            return True, value * 60000
        elif to_unit in self.ld:
            return True, value * 86400000
        elif to_unit in self.ls:
            return True, value * 1000

        else:
            return False, value

    def convert_from_mld(self, to_unit, value):
        """Convert from the Mega liter per day unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 12889853.354327
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 0.40873456853575
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 183.45338522863
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 264120.37037037
        # SI
        elif to_unit in self.cmy:
            return True, value * 365000.0
        elif to_unit in self.cmh:
            return True, value * 41.66666666666
        elif to_unit in self.cmm:
            return True, value * 0.69444444444444
        elif to_unit in self.cms:
            return True, value * 0.011574074074074
        elif to_unit in self.Mld:
            return True, value
        elif to_unit in self.Mls:
            return True, value * 1.1574074074074E-5
        elif to_unit in self.lm:
            return True, value * 694.4444444444
        elif to_unit in self.ld:
            return True, value * 1000000
        elif to_unit in self.ls:
            return True, value * 11.574074074074

        else:
            return False, value

    def convert_from_mls(self, to_unit, value):
        """Convert from the Mega liters per second unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 1113676621000.0
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 35314.666721489
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 00.83267384
            return True, value * 15850372.483753
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 22820000000.0
        # SI
        elif to_unit in self.cmy:
            return True, value * 31536000000.0
        elif to_unit in self.cmh:
            return True, value * 3600000.0
        elif to_unit in self.cmm:
            return True, value * 60000.0
        elif to_unit in self.cms:
            return True, value * 1000.0
        elif to_unit in self.Mld:
            return True, value * 86400.0
        elif to_unit in self.Mls:
            return True, value
        elif to_unit in self.lm:
            return True, value * 60000000.0
        elif to_unit in self.ld:
            return True, value * 8.64e+10
        elif to_unit in self.ls:
            return True, value * 1000000

        else:
            return False, value

    def convert_from_lm(self, to_unit, value):
        """Convert from the liters per minute unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 18561.277016667
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 0.00058857777869148
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 0.26417287472922
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 380.33333333333
        # SI
        elif to_unit in self.cmy:
            return True, value * 525.6
        elif to_unit in self.cmh:
            return True, value * 0.06
        elif to_unit in self.cmm:
            return True, value * 0.001
        elif to_unit in self.cms:
            return True, value * 1.6666666666667E-5
        elif to_unit in self.Mld:
            return True, value * 0.00144
        elif to_unit in self.Mls:
            return True, value * 1.6666666666667E-8
        elif to_unit in self.lm:
            return True, value
        elif to_unit in self.ld:
            return True, value * 1440.0
        elif to_unit in self.ls:
            return True, value * 0.01666666666666

        else:
            return False, value

    def convert_from_ld(self, to_unit, value):
        """Convert from the liters per minute unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 12.8899
        elif to_unit in self.cfs or to_unit in self.stream:
            value /= 2446575.55
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value / 5450.99296896
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value / 3.785
        # SI
        elif to_unit in self.cmy:
            return True, value / 2.7379070069885
        elif to_unit in self.cmh:
            return True, value / 24000.0
        elif to_unit in self.cmm:
            return True, value / 1.44e+6
        elif to_unit in self.cms:
            return True, value * 1.15741e-8
        elif to_unit in self.Mld:
            return True, value * 1e-6
        elif to_unit in self.Mls:
            return True, value * 1.1574074074074e-11
        elif to_unit in self.lm:
            return True, value / 1440.0
        elif to_unit in self.ld:
            return True, value
        elif to_unit in self.ls:
            return True, value / 86400.0

        else:
            return False, value

    def convert_from_ls(self, to_unit, value):
        """Convert from the liter per second unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfy:
            return True, value * 1113676.621
        elif to_unit in self.cfs or to_unit in self.stream:
            value *= 0.035314666721489
            if to_unit in self.stream:
                value /= self.stream_value
            return True, value
        elif to_unit in self.gpm:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 15.850372483753
        elif to_unit in self.gpd:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 22820
        # SI
        elif to_unit in self.cmy:
            return True, value * 31536.0
        elif to_unit in self.cmh:
            return True, value * 3.6
        elif to_unit in self.cmm:
            return True, value * 0.06
        elif to_unit in self.cms:
            return True, value * 0.001
        elif to_unit in self.Mld:
            return True, value * 0.0864
        elif to_unit in self.Mls:
            return True, value * 1.0E-6
        elif to_unit in self.lm:
            return True, value * 60.0
        elif to_unit in self.ld:
            return True, value * 60.0 * 60.0 * 24.0
        elif to_unit in self.ls:
            return True, value

        else:
            return False, value
