"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionVolumeCalc:
    """Perform volume unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Volume
    # ----------------------------------------------------------------------
    # English Units
    cubic_mile = ['mi^3', 'cu mi', 'cubic miles', 'Cubic Miles', 'CUBIC MILES']
    cubic_yard = ['yd^3', 'cu yd', 'cubic yards', 'Cubic Yards', 'CUBIC YARDS']
    cubic_foot = ['ft^3', 'cu ft', 'cubic foot', 'cubic feet', 'Cubic Foot', 'Cubic Feet', 'CUBIC FOOT', 'CUBIC FEET']
    cubic_inch = ['in^3', 'cu in', 'cubic inches', 'Cubic Inches', 'CUBIC INCHES']
    gal = ['gal', 'Gallon', 'Gallon', 'Gallon', 'gallons', 'Gallons', 'GALLONS']
    # SI Units
    cubic_km = ['km^3', 'cu km', 'cubic kilometers', 'Cubic Kilometers', 'CUBIC KILOMETERS']
    cubic_m = ['m^3', 'cu m', 'cubic meters', 'Cubic Meters', 'CUBIC METERS']
    cubic_cm = ['cm^3', 'cu cm', 'cubic centimeters', 'Cubic Centimeters', 'CUBIC CENTIMETERS']
    cubic_mm = ['mm^3', 'cu mm', 'cubic millimeters', 'Cubic Millimeters', 'CUBIC MILLIMETERS']
    liter = ['l', 'liter', 'Liter', 'LITER', 'liters', 'Liters', 'LITERS']
    milliliter = ['ml', 'milliliter', 'Milliliter', 'MILLILITER', 'milliliters', 'Milliliters', 'MILLILITERS']

    def __init__(self, us_gallon=True):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """
        self.us_gallon = us_gallon

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.cubic_mile:
            return True, 'km^3'
        elif from_unit in self.cubic_yard:
            return True, 'm^3'
        elif from_unit in self.cubic_foot:
            return True, 'm^3'
        elif from_unit in self.cubic_inch:
            return True, 'mm^3'
        elif from_unit in self.gal:
            return True, 'l'
        return False, ''

    # ----------------------------------------------------------------------------------------------------------------------
    # Volume
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert volume unit.

        Args:
            from_unit (string): unit that value is currently in
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Units
        if from_unit in self.cubic_mile:
            return self.convert_from_cubic_mile(to_unit, value)
        elif from_unit in self.cubic_yard:
            return self.convert_from_cubic_yard(to_unit, value)
        elif from_unit in self.cubic_foot:
            return self.convert_from_cubic_foot(to_unit, value)
        elif from_unit in self.cubic_inch:
            return self.convert_from_cubic_inch(to_unit, value)
        elif from_unit in self.gal:
            return self.convert_from_gal(to_unit, value)
        # SI Units
        elif from_unit in self.cubic_km:
            return self.convert_from_cubic_km(to_unit, value)
        elif from_unit in self.cubic_m:
            return self.convert_from_cubic_m(to_unit, value)
        elif from_unit in self.cubic_cm:
            return self.convert_from_cubic_cm(to_unit, value)
        elif from_unit in self.cubic_mm or from_unit in self.milliliter:
            return self.convert_from_cubic_mm(to_unit, value)
        elif from_unit in self.liter:
            return self.convert_from_liter(to_unit, value)

        return False, value

    def convert_from_cubic_mile(self, to_unit, value):
        """Convert from the cubic mile unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value
        elif to_unit in self.cubic_yard:
            return True, value * 1760.0 ** 3.0
        elif to_unit in self.cubic_foot:
            return True, value * 5280.0 ** 3.0
        elif to_unit in self.cubic_inch:
            return True, value * 63360.0 ** 3.0
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 1101117130711.3
        # SI
        elif to_unit in self.cubic_km:
            return True, value * 1.609344 ** 3.0
        elif to_unit in self.cubic_m:
            return True, value * 1609.34 ** 3.0
        elif to_unit in self.cubic_cm:
            return True, value * 160934.0 ** 3.0
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 1609340.0 ** 3.0
        elif to_unit in self.liter:
            return True, value * 4168182000000.0

        else:
            return False, value

    def convert_from_cubic_yard(self, to_unit, value):
        """Convert from the cubic yard unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value / 1760.0 ** 3.0
        elif to_unit in self.cubic_yard:
            return True, value
        elif to_unit in self.cubic_foot:
            return True, value * 3.0 ** 3.0
        elif to_unit in self.cubic_inch:
            return True, value * 36.0 ** 3.0
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 201.974029
        # SI
        elif to_unit in self.cubic_km:
            return True, value * 0.0009144 ** 3.0
        elif to_unit in self.cubic_m:
            return True, value * 0.9144 ** 3.0
        elif to_unit in self.cubic_cm:
            return True, value * 91.44 ** 3.0
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 914.4 ** 3.0
        elif to_unit in self.liter:
            return True, value * 9.144 ** 3.0

        else:
            return False, value

    def convert_from_cubic_foot(self, to_unit, value):
        """Convert from the cubic foot unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value * 0.000189394 ** 3.0
        elif to_unit in self.cubic_yard:
            return True, value * 0.33333333 ** 3.0
        elif to_unit in self.cubic_foot:
            return True, value
        elif to_unit in self.cubic_inch:
            return True, value * 12.0 ** 3.0
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 7.48052
        # SI
        elif to_unit in self.cubic_km:
            return True, value * 0.0003048 ** 3.0
        elif to_unit in self.cubic_m:
            return True, value * 0.3048 ** 3.0
        elif to_unit in self.cubic_cm:
            return True, value * 30.48 ** 3.0
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 304.8 ** 3.0
        elif to_unit in self.liter:
            return True, value * 3.048 ** 3.0

        else:
            return False, value

    def convert_from_cubic_inch(self, to_unit, value):
        """Convert from the cubic inch unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value * 1.5783e-5 ** 3.0
        elif to_unit in self.cubic_yard:
            return True, value / 46656.0
        elif to_unit in self.cubic_foot:
            return True, value / 1728.0
        elif to_unit in self.cubic_inch:
            return True, value
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 0.004329003471008008308
        # SI
        elif to_unit in self.cubic_km:
            return True, value * 2.54e-5 ** 3.0
        elif to_unit in self.cubic_m:
            return True, value * 0.0254 ** 3.0
        elif to_unit in self.cubic_cm:
            return True, value * 2.54 ** 3.0
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 25.4 ** 3.0
        elif to_unit in self.liter:
            return True, value * 0.254 ** 3.0

        else:
            return False, value

    def convert_from_gal(self, to_unit, value):
        """Convert from the cubic inch unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if not self.us_gallon:
            value /= 0.83267384

        # US Unit
        if to_unit in self.cubic_mile:
            return True, value / 1101117130711.3
        elif to_unit in self.cubic_yard:
            return True, value / 201.974029
        elif to_unit in self.cubic_foot:
            return True, value / 7.48052
        elif to_unit in self.cubic_inch:
            return True, value / 0.004329003471008008308
        elif to_unit in self.gal:
            return True, value
        # SI
        elif to_unit in self.cubic_km:
            return True, value / 264172000000.0005371
        elif to_unit in self.cubic_m:
            return True, value / 264.1720000000005371
        elif to_unit in self.cubic_cm:
            return True, value * 3785.41
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 3785410.0
        elif to_unit in self.liter:
            return True, value * 3.78541
        else:
            return False, value

    def convert_from_cubic_km(self, to_unit, value):
        """Convert from the cubic km unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value / (1.609344 ** 3.0)
        elif to_unit in self.cubic_yard:
            return True, value * 1093.6132983 ** 3.0
        elif to_unit in self.cubic_foot:
            return True, value * 3280.84 ** 3.0
        elif to_unit in self.cubic_inch:
            return True, value * 39370.1 ** 3.0
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 264172000000.0005371
        # SI
        elif to_unit in self.cubic_km:
            return True, value
        elif to_unit in self.cubic_m:
            return True, value * 1000.0 ** 3.0
        elif to_unit in self.cubic_cm:
            return True, value * 100000.0 ** 3.0
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 1.0e+6 ** 3.0
        elif to_unit in self.liter:
            return True, value * 10000.0 ** 3.0

        else:
            return False, value

    def convert_from_cubic_m(self, to_unit, value):
        """Convert from the cubic m unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value / (1609.344 ** 3.0)
        elif to_unit in self.cubic_yard:
            return True, value * 1.0936133 ** 3.0
        elif to_unit in self.cubic_foot:
            return True, value * 3.280839895 ** 3.0
        elif to_unit in self.cubic_inch:
            return True, value * 39.3700787 ** 3.0
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value * 264.1720000000005371
        # SI
        elif to_unit in self.cubic_km:
            return True, value * 0.001 ** 3.0
        elif to_unit in self.cubic_m:
            return True, value
        elif to_unit in self.cubic_cm:
            return True, value * 100.0 ** 3.0
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 1000.0 ** 3.0
        elif to_unit in self.liter:
            return True, value * 10.0 ** 3.0

        else:
            return False, value

    def convert_from_cubic_cm(self, to_unit, value):
        """Convert from the cubic cm unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value / (160934.4 ** 3.0)
        elif to_unit in self.cubic_yard:
            return True, value / (91.44 ** 3.0)
        elif to_unit in self.cubic_foot:
            return True, value / (30.48 ** 3.0)
        elif to_unit in self.cubic_inch:
            return True, value / (2.54 ** 3.0)
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value / 3785.411784
        # SI
        elif to_unit in self.cubic_km:
            return True, value * 0.00001 ** 3.0
        elif to_unit in self.cubic_m:
            return True, value * 0.01 ** 3.0
        elif to_unit in self.cubic_cm:
            return True, value
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 10.0 ** 3.0
        elif to_unit in self.liter:
            return True, value * 0.1 ** 3.0

        else:
            return False, value

    def convert_from_cubic_mm(self, to_unit, value):
        """Convert from the cubic mm unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value / (1609344.0 ** 3.0)
        elif to_unit in self.cubic_yard:
            return True, value / (914.4 ** 3.0)
        elif to_unit in self.cubic_foot:
            return True, value / (304.8 ** 3.0)
        elif to_unit in self.cubic_inch:
            return True, value / (25.4 ** 3.0)
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value / 3785411.784
        # SI
        elif to_unit in self.cubic_km:
            return True, value * 0.000001 ** 3.0
        elif to_unit in self.cubic_m:
            return True, value * 0.001 ** 3.0
        elif to_unit in self.cubic_cm:
            return True, value * 0.1 ** 3.0
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value
        elif to_unit in self.liter:
            return True, value * 0.01 ** 3.0

        else:
            return False, value

    def convert_from_liter(self, to_unit, value):
        """Convert from the cubic mm unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cubic_mile:
            return True, value / 4168182000000.0
        elif to_unit in self.cubic_yard:
            return True, value / 764.554858
        elif to_unit in self.cubic_foot:
            return True, value / 28.3168
        elif to_unit in self.cubic_inch:
            return True, value / 0.0163871
        elif to_unit in self.gal:
            if not self.us_gallon:
                value *= 0.83267384
            return True, value / 3.78541
        # SI
        elif to_unit in self.cubic_km:
            return True, value * 0.0001 ** 3.0
        elif to_unit in self.cubic_m:
            return True, value * 0.1 ** 3.0
        elif to_unit in self.cubic_cm:
            return True, value * 10.0 ** 3.0
        elif to_unit in self.cubic_mm or to_unit in self.milliliter:
            return True, value * 100.0 ** 3.0
        elif to_unit in self.liter:
            return True, value

        else:
            return False, value
