"""UserArray Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import sys

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable


class UserArray(VariableGroup):
    """Provides a class that will Handle user-selected array like HY-'s discharge array.

    Options: single value; min, design, max; user-supplied; recurrence
    """

    def __init__(self, precision, unit_type, native_unit, us_units, si_units, limits=(0.0, sys.float_info.max),
                 select_name='Select Type', name_append='', app_data=None):
        """Initializes the User Array.

        Args:
            precision (int): Level of precision to report to the user
            unit_type (string): data type of user array
            native_unit (string): the unit that the user array is being worked in and the default
            us_units (list of lists): The US units that are available to the user
            si_units (list of lists): The US units that are available to the user
            limits (tuple): Gives the lower and upper limit of the variable, that will be applied to all variables
            select_name (string): Name to display with select combobox
            name_append (string): string to append at end of min, design, or max (eg, 'flow')
            app_data (AppData): application data (settings)
        """
        super().__init__(app_data=app_data)

        self.name = select_name
        self.type = 'UserArray'

        # Input
        # Make Input a dict
        self.name_append = name_append
        self.input['Select'] = Variable(select_name, 'list', 1, ['single value', 'min, design, max',
                                                                 'min, max, increment', 'User-defined',
                                                                 'Recurrence'])
        self.input['Single value'] = Variable('Single value', 'float', 0.0, [], precision=precision,
                                              unit_type=unit_type, native_unit=native_unit,
                                              us_units=us_units, si_units=si_units)
        string = 'Min'
        if name_append != '':
            string = 'Min ' + name_append
        self.input['Min'] = Variable(string, 'float', 0.0, [], precision=precision,
                                     unit_type=unit_type, native_unit=native_unit,
                                     us_units=us_units, si_units=si_units)
        string = 'Design'
        if name_append != '':
            string = 'Design ' + name_append
        self.input['Design'] = Variable(string, 'float', 0.0, [], precision=precision,
                                        unit_type=unit_type, native_unit=native_unit,
                                        us_units=us_units, si_units=si_units)
        string = 'Max'
        if name_append != '':
            string = 'Max ' + name_append
        self.input['Max'] = Variable(string, 'float', 0.0, [], precision=precision,
                                     unit_type=unit_type, native_unit=native_unit,
                                     us_units=us_units, si_units=si_units)
        string = 'Increment'
        if name_append != '':
            string = 'Increment of ' + name_append
        self.input['Increment'] = Variable(string, 'float', 0.0, [], precision=precision,
                                           unit_type=unit_type, native_unit=native_unit,
                                           us_units=us_units, si_units=si_units)

        self.input['User-defined'] = Variable('User-defined', 'float_list', 0.0, [], precision=precision,
                                              unit_type=unit_type, native_unit=native_unit,
                                              us_units=us_units, si_units=si_units)

        self.input['Recurrence'] = Variable('Recurrence', 'float_list', 0.0, [], precision=precision,
                                            unit_type=unit_type, native_unit=native_unit,
                                            us_units=us_units, si_units=si_units)

        # tuple to give lower and upper limits
        self.limits = limits

        self.results['Results'] = Variable('Results', 'float_list', 0.0, [], precision=precision,
                                           unit_type=unit_type, native_unit=native_unit,
                                           us_units=us_units, si_units=si_units)

    def set_user_list(self, user_list, app_data=None):
        """Sets the user array to user-defined and sets the user-defined list.

        Args:
            user_list (list of values): list to set as the user-defined list
            app_data (AppData): settings to use for the calculation

        Returns:
            results (variable): variable according to the specifications of the user
        """
        self.input['Select'].set_val('User-defined')
        self.input['User-defined'].set_val(user_list, app_data)
        self.compute_data()
        return self.results['Results']

    def set_single_value(self, val, app_data=None):
        """Sets the user array to single value and sets the value.

        Args:
            val (value): value to set as the single value
            app_data (AppData): settings to use for the calculation

        Returns:
            results (variable): variable according to the specifications of the user
        """
        self.input['Select'].set_val('single value')
        self.input['Single value'].set_val(val, app_data)
        self.compute_data()
        return self.results['Results']

    def get_val(self):
        """Computes the resulting variable class and returns it.

        Returns:
            results (variable): variable according to the specifications of the user
        """
        self.compute_data()
        return self.results['Results']

    def get_result(self):
        """Returns the resulting list of values and returns ONLY the list of values.

        Returns:
            (list of values): list of values according to the specifications of the user
        """
        return self.get_val().get_val()

    def add_result(self, value):
        """Appends a result to the list of values.

        Args:
            value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if self.input['Select'].get_val() == 'single value':
            self.input['Single value'].set_val(value)
        elif self.input['Select'].get_val() == 'min, design, max':
            pass  # No way to add result to computed list
        elif self.input['Select'].get_val() == 'User-defined':
            self.input['User-defined'].value_options.append(value)
        elif self.input['Select'].get_val() == 'Recurrence':
            pass  # Can't add to recurrence list; must be by index

        self.compute_data()
        self.results['Results'].add_result(value)

    def set_precision(self, precision):
        """Sets the precision to all input Variables in the class.

        Args:
            precision (int): precision to set to the class
        """
        self.input['Single value'].precision = precision
        self.input['Min'].precision = precision
        self.input['Design'].precision = precision
        self.input['Max'].precision = precision
        self.input['Increment'].precision = precision
        self.input['User-defined'].precision = precision
        self.input['Recurrence'].precision = precision

    def set_unit_types(self, unit_types):
        """Sets the unit type to all input Variables in the class.

        Args:
            unit_types (string): unit_type to set to the class
        """
        self.input['Single value'].unit_type = unit_types
        self.input['Min'].unit_type = unit_types
        self.input['Design'].unit_type = unit_types
        self.input['Max'].unit_type = unit_types
        self.input['Increment'].unit_type = unit_types
        self.input['User-defined'].unit_type = unit_types
        self.input['Recurrence'].unit_type = unit_types

    def set_native_units(self, units):
        """Sets the native units to all input Variables in the class.

        Args:
            units (string): units to set to the class
        """
        self.input['Single value'].native_unit = units
        self.input['Min'].native_unit = units
        self.input['Design'].native_unit = units
        self.input['Max'].native_unit = units
        self.input['Increment'].native_unit = units
        self.input['User-defined'].native_unit = units
        self.input['Recurrence'].native_unit = units

    def set_us_units(self, units):
        """Sets the US units to all input Variables in the class.

        Args:
            units (string): units to set to the class
        """
        self.input['Single value'].available_us_units = units
        self.input['Min'].available_us_units = units
        self.input['Design'].available_us_units = units
        self.input['Max'].available_us_units = units
        self.input['Increment'].available_us_units = units
        self.input['User-defined'].available_us_units = units
        self.input['Recurrence'].available_us_units = units

    def set_si_units(self, units):
        """Sets the SI units to all input Variables in the class.

        Args:
            units (string): units to set to the class
        """
        self.input['Single value'].available_si_units = units
        self.input['Min'].available_si_units = units
        self.input['Design'].available_si_units = units
        self.input['Max'].available_si_units = units
        self.input['Increment'].available_si_units = units
        self.input['User-defined'].available_si_units = units
        self.input['Recurrence'].available_si_units = units

    def set_select_name(self, name):
        """Sets the name to the select variable.

        Args:
            name (string): name
        """
        self.input['Select'].name = name

    def set_single_value_name(self, name):
        """Sets the name to the single value variable.

        Args:
            name (string): name
        """
        self.input['Single value'].name = name

    def set_min_name(self, name):
        """Sets the name to the minimum variable.

        Args:
            name (string): name
        """
        self.input['Min'].name = name

    def set_design_name(self, name):
        """Sets the name to the design variable.

        Args:
            name (string): name
        """
        self.input['Design'].name = name

    def set_max_name(self, name):
        """Sets the name to the maximum variable.

        Args:
            name (string): name
        """
        self.input['Max'].name = name

    def set_increment_name(self, name):
        """Sets the name to the increment variable.

        Args:
            name (string): name
        """
        self.input['Increment'].name = name

    def set_user_defined_name(self, name):
        """Sets the name to the user defined variable.

        Args:
            name (string): name
        """
        self.input['User-defined'].name = name

    def set_recurrence_name(self, name):
        """Sets the name to the recurrence variable.

        Args:
            name (string): name
        """
        self.input['Recurrence'].name = name

    def get_low_and_high_value(self, unknown=None, name=None):
        """Get the low and high value of the class. uses unknown and name to determine if it is the right variable.

        Args:
            unknown (string): variable that is unknown (that we are calculating for)
            name (string): name of current variable

        Return:
            low_val (float): lowest value in the specified set
            high_val (float): highest value in the specified set
        """
        low_val = 0.0
        high_val = 0.0
        if name and unknown and name == unknown or name is None and unknown is None:
            if self.input['Select'].get_val() == 'Single value':
                low_val = self.input['Single value'].limits[0]
                high_val = self.input['Single value'].limits[1]
            elif self.input['Select'].get_val() in ['min, design, max', 'min, max, increment']:
                low_val = self.input['Min'].get_val()
                high_val = self.input['Min'].get_val()
            else:
                val_list = self.results['Results'].get_val()
                low_val = sys.float_info.max
                high_val = -sys.float_info.max
                for value in val_list:
                    if value < low_val:
                        low_val = value
                    if value > high_val:
                        high_val = value

        return low_val, high_val

    def get_input_group(self, unknown=None):
        """Gets the input group to display to the user depending on the selections made.

        Returns:
            input_vars (list of Variables): Variables to be added to the input table
        """
        input_vars = {}
        input_vars['Select'] = self.input['Select']

        if self.input['Select'].get_val() == 'single value':
            input_vars['Single value'] = self.input['Single value']
        elif self.input['Select'].get_val() == 'min, design, max':
            input_vars['Min'] = self.input['Min']
            input_vars['Design'] = self.input['Design']
            input_vars['Max'] = self.input['Max']
        elif self.input['Select'].get_val() == 'min, max, increment':
            input_vars['Min'] = self.input['Min']
            input_vars['Max'] = self.input['Max']
            input_vars['Increment'] = self.input['Increment']
        elif self.input['Select'].get_val() == 'User-defined':
            input_vars['User-defined'] = self.input['User-defined']
        elif self.input['Select'].get_val() == 'Recurrence':
            input_vars['Recurrence'] = self.input['Recurrence']

        for var in input_vars:
            input_vars[var].limits = self.limits
        return input_vars

    def get_can_compute(self):
        """Determines whether there is enough information to complete a computation."""
        return True

    def compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        if self.input['Select'].get_val() == 'single value':
            self.results['Results'].set_val([self.input['Single value'].get_val()])
        elif self.input['Select'].get_val() == 'min, design, max':
            num_discharges = 11
            min_value = self.input['Min'].get_val()
            design_value = self.input['Design'].get_val()
            max_value = self.input['Max'].get_val()
            if 0.0 <= design_value >= max_value:
                return
            change_in_flow = (max_value - min_value) / (num_discharges - 1)
            new_list = []
            for index in range(num_discharges):
                new_list.append(min_value + change_in_flow * index)
            design_index = 0
            distance_from_design = sys.float_info.max
            tol = 0.01
            for index in range(num_discharges):
                cur_dist = abs(design_value - new_list[index])
                if cur_dist <= distance_from_design + tol:
                    distance_from_design = cur_dist
                    design_index = index
            max_and_design_same = False
            if max_value - tol < design_value < max_value + tol:
                max_and_design_same = True
            if design_index == 10 and not max_and_design_same:
                design_index -= 1
            new_list[design_index] = design_value
            self.results['Results'].set_val(new_list)
        elif self.input['Select'].get_val() == 'min, max, increment':
            min_value = self.input['Min'].get_val()
            change_in_flow = self.input['Increment'].get_val()
            max_value = self.input['Max'].get_val()
            if 0.0 < change_in_flow >= max_value:
                return
            cur_value = min_value
            new_list = [min_value]
            while cur_value < max_value:
                cur_value += change_in_flow
                if cur_value > max_value:
                    cur_value = max_value
                new_list.append(cur_value)
            self.results['Results'].set_val(new_list)
        elif self.input['Select'].get_val() == 'User-defined':
            self.results['Results'].set_val(self.input['User-defined'].get_val())
        elif self.input['Select'].get_val() == 'Recurrence':
            self.results['Results'].set_val(self.input['Recurrence'].get_val())

    def check_limits(self, result, warnings):
        """Checks that the value (if applicable) is within limits.

        Args:
            result (bool): Set to False if anything fails a check; otherwise, UNMODIFIED
            warnings (list): list of strings of warnings to provide to the user
        """
        input_vars = self.get_input_group()
        for var in input_vars:
            if not input_vars[var].check_limits(result, warnings, self.name_append):
                result = False
        return result
