"""Variable Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import re
import sys
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc

# self.length = {}
# self.length['all_US'] = ['mi', 'yd', 'ft', 'in']
# self.length['all_SI'] = ['km', 'm', 'cm', 'mm']
# self.length['long_US'] = ['mi', 'yd', 'ft']
# self.length['long_SI'] = ['km', 'm']
# self.length['med_US'] = ['yd', 'ft', 'in']
# self.length['med_SI'] = ['m', 'cm', 'mm']
# self.length['short_US'] = ['ft', 'in', 'mm']
# self.length['short_SI'] = ['cm', 'mm']

# self.flow_all_units_US = ['cfy', 'cfs', 'stream', 'gpm', 'gpd']
# self.flow_all_units_SI = ['cmy', 'cmh', 'cmm', 'cms', 'Mld', 'Mls', 'lm', 'ls']

# self.flow_units_US = ['cfs', 'stream', 'gpm']
# self.flow_units_SI = ['cms', 'lm', 'ls']


class Variable:
    """Provides a class that will store a variable, handle interface, and file IO."""
    string_types = ['string', 'zip', 'phone', 'email', 'url', ]
    supported_color_formats = ['rgb']

    def __init__(self, name: str = '', var_type: str = '', value=0.0, value_options=None,
                 limits=(0.0, sys.float_info.max),
                 precision: int = 2, unit_type=None, native_unit: str = "", us_units=None, si_units=None,
                 note: str = "", complexity: int = 0, help_url: str = None):
        """Initializes the variable values.

        Args:
            name (string): name of the variable to be displayed in the UI
            var_type (string): Data type 'bool', 'int', 'float', 'string', 'Zip', 'Phone', 'email', 'url', 'list',
                'float_list', 'string_list', 'class', 'file', 'image', 'color'
            value (float): Value of the variable.  Type will change as needed
            value_options (list): provides the options for list
            limits (tuple): the limits the variable is allowed to range across
            precision (int): amount of precision to display for the variable
            unit_type (): specifies types of units to provide
            native_unit (string): the native and default unit - units the variable will convert to for internal use
            us_units (list of lists): the US units to provide to the user
            si_units (list of lists): the SI units to provide to the user
            note (string): The note to display to the user.
            complexity (int): Complexity level of the variable
            help_url (string): URL to help the user understand the variable
        """
        if value_options is None:
            value_options = []
        if unit_type is None:
            unit_type = []
        if us_units is None:
            us_units = [[]]
        if si_units is None:
            si_units = [[]]

        self.name = name
        # Type: 'bool', 'int', 'float', 'string', 'list', 'float_list', 'string_list', 'class',
        # 'file', 'image', 'color', 'calc', 'calc_list', 'uuid_dict', 'UserArray', 'table', 'date'
        self.type = var_type
        # bool if type = bool, int if type = int, float if type = bool, int if type = list
        self.value = value
        self.default_value = value  # Record the default value so we can reset if the complexity is changed
        if not isinstance(value, Path) and hasattr(value, 'name'):
            value.name = name
        # If the type is list, this holds the list; if a file, this holds the filters
        # example of file filter: ["Hydraulic Toolbox Files (*.htb)", "All Files (*)"]
        self.value_options = value_options

        self.list_types = ['list', 'float_list', 'string_list', 'calc_list', 'list_of_lists']

        # Properly handle the default value and value_options for double_list
        if self.type in self.list_types and isinstance(self.value_options, list):
            if len(self.value_options) > 0:
                self.default_value = value_options[0]
            else:
                self.default_value = 0.0
        elif self.type in self.list_types and not isinstance(self.value_options, list):
            self.value_options = [value_options]

        # tuple to give lower and upper limits
        self.limits = limits
        # number of significant digits for this variable
        self.precision = precision
        # unit type can hold length, speed, time, percent, etc
        self.unit_type = unit_type
        # This is the unit that we record the data as
        self.native_unit = native_unit
        # This is the unit that we display the data as
        self.selected_us_unit = native_unit
        _, self.selected_si_unit = ConversionCalc(None).get_si_complementary_unit(native_unit)
        # Units to be displayed for the user to select
        self.available_us_units = us_units
        self.available_si_units = si_units
        # A note for describing or assisting the user
        self.note = note
        # The complexity level that unlocks displaying this variable
        self.complexity = complexity
        # URL to help the user understand the variable
        self.help_url = help_url
        # Can the user modify this variable
        self.read_only = False
        # file mode ('folder', 'existing file', 'new file', 'any')
        self.file_mode = 'existing file'
        # Action (used for push button; functor)
        self.action = None
        # If set to true, it will set the row as a header or subheader (depending on recurrence_level)
        self.set_row_as_header = False
        # This will change the color of the row
        # 'approved' = green, 'normal' = (based on row/cell type), 'incomplete' = yellow, 'warning' = orange,
        # 'failure' = red
        self.data_status = 'normal'  # 'approved', 'normal', 'incomplete', 'warning', 'failure'
        # UUID for the variable, used for undo/redo commands
        self.uuid = uuid.uuid4()

        # TODO: Add an option that will remember the last value the user entered for a variable
        # (when the variable is init)
        # TODO: Add an option that will remember the last units the user entered for a variable
        # (when the variable is init)

        # An example of these features would be, specifying an option to use priority raster or highest resolution
        # raster when # merging rasters. A user is likely to have a preference that we would want to persist to the
        # next time they run the tool.
        # Or if a user wants to specify channel depth in inches instead of feet, we would want to remember that
        # preference.
        # I imagine most of the time, we would not want these to persist, but there are cases where it would be useful.

    def get_val(self, app_data=None):
        """Returns the value of the variable.

        Returns:
            ? (?): The return type depends on the data type (specified in self.type)
        """
        selected_unit_system = 'U.S. Customary Units'
        if app_data is not None:
            _, selected_unit_system = app_data.get_setting('Selected unit system', selected_unit_system)
        if self.type in ['bool', 'int', 'string', 'class', 'UserArray', 'table', 'file', 'calc']:
            return self.value
        elif self.type in ['float']:
            _, val = self._prepare_double(self.value, app_data, selected_unit_system)
            return val
        elif self.type in ['float_list']:
            new_list = []
            for val in self.value_options:
                result, new_val = self._prepare_double(val, app_data, selected_unit_system, set_val=False)
                if result:
                    new_list.append(new_val)
            return new_list
        elif self.type == 'list':
            if len(self.value_options):
                if self.value < 0:
                    self.value = 0
                if self.value >= len(self.value_options):
                    self.value = len(self.value_options) - 1
                return self.value_options[self.value]
            return ''
        elif self.type in ['calc_list']:
            return self.value
        elif self.type in ['color']:
            return self.value_options
        elif self.type in self.list_types:  # list is already handled, because we select one item from the list
            return self.value_options
        elif self.type in ['uuid_dict']:
            if self.value in self.value_options:
                return self.value
        return self.value

    def set_val(self, new_value, app_data=None, index=None):
        """Sets the value of the variable.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_data (AppSettingsData): The settings CalcData to use for unit conversion
            index (int): The index of the list to set the value to (if applicable)

        Returns:
            succeeded (bool): True if the value was set successfully; otherwise, False
        """
        selected_unit_system = 'U.S. Customary Units'
        if app_data is not None:
            _, selected_unit_system = app_data.get_setting('Selected unit system', selected_unit_system)
        if self.type in ['bool', 'class', 'UserArray', 'file', 'calc', 'table'] or self.type in Variable.string_types:
            if self.type in ['file'] or self.type in Variable.string_types:
                result, val = self._prepare_string(new_value, app_data, selected_unit_system)
                if result:
                    self.value = val
            else:
                if self.type == 'bool':
                    result, val = self._prepare_bool(new_value, app_data, selected_unit_system)
                    if result:
                        self.value = val
                else:
                    self.value = new_value
        elif self.type in ['int']:
            result, val = self._prepare_int(new_value, app_data, selected_unit_system)
            if result:
                self.value = val
        elif self.type in ['float']:
            result, val = self._prepare_double(new_value, app_data, selected_unit_system)
            if result:
                self.value = val
        elif self.type == 'list':
            if isinstance(new_value, int) or isinstance(new_value, float):
                self.value = int(new_value)
            else:
                if new_value in self.value_options:
                    self.value = self.value_options.index(new_value)
                else:
                    return False
        elif self.type == 'uuid_dict':
            if new_value in self.value_options.values():
                for key, value in self.value_options.items():
                    if value == new_value:
                        self.value = key
            elif new_value in self.value_options.keys():
                self.value = new_value
        elif self.type in self.list_types:
            # set the list, if given a list
            prepare_func = None
            if self.type == 'string_list':
                prepare_func = self._prepare_string
            elif self.type == 'float_list':
                prepare_func = self._prepare_double
            elif self.type == 'int_list':
                prepare_func = self._prepare_int
            try:
                self.value = int(self.value)
            except ValueError:
                self.value = 0
            if isinstance(new_value, list) or isinstance(new_value, tuple):
                new_list = []
                for i in range(len(new_value)):
                    if prepare_func is None:
                        new_list.append(new_value[i])
                        continue
                    result, val = prepare_func(new_value[i], app_data, selected_unit_system)
                    if result:
                        new_list.append(val)
                self.value_options = new_list
                self.value = len(self.value_options)
            elif index is not None:  # Set a single value in the list
                if len(self.value_options) > index:
                    result, val = prepare_func(new_value, app_data, selected_unit_system)
                    if result:
                        self.value_options[index] = val
                elif len(self.value_options) == index:
                    self.value += 1
                    result, val = prepare_func(new_value, app_data, selected_unit_system)
                    if result:
                        self.value_options.append(val)
            elif index is None:
                if prepare_func is None:
                    self.value_options = [new_value]
                    self.value = int(1)
                else:
                    result, val = prepare_func(new_value, app_data, selected_unit_system)
                    if result:
                        self.value_options = [val]
                        self.value = int(1)
        elif self.type in ['image']:
            if isinstance(new_value, np.ndarray):
                self.value = new_value
        elif self.type in ['color']:
            if new_value in Variable.supported_color_formats:
                self.value = new_value
            else:
                if isinstance(new_value, (tuple, list)) and len(new_value) == 3:
                    self.value_options = new_value
        return True

    def _prepare_string(self, new_value, app_settings, selected_unit_system):
        """Prepare the string value for setting the value.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_settings (AppSettingsData): The settings CalcData to use for unit conversion
            selected_unit_system (string): The unit system selected by the user

        Returns:
            succeeded (bool): True if the value was set successfully; otherwise, False
            new_value (float): The new value to set to the variable
        """
        if not isinstance(new_value, str):
            new_value = str(new_value)

        self.data_status = 'normal'
        if self.type == 'zip' and new_value != '':
            # Remove any non-digit characters
            digits = ''.join(filter(str.isdigit, new_value))

            # Check if the number of digits is correct for a ZIP code
            if len(digits) == 5:
                # Short form ZIP code
                new_value = digits
            elif len(digits) == 9:
                # Long form ZIP code
                formatted_zipcode = f"{digits[:5]}-{digits[5:]}"
                new_value = formatted_zipcode
            else:
                # Invalid ZIP code
                self.data_status = 'warning'
        elif self.type == 'phone' and new_value != '':
            # Remove any non-digit characters
            digits = ''.join(filter(str.isdigit, new_value))

            # Check if the number of digits is correct for a phone number
            if len(digits) != 10:
                self.data_status = 'warning'
            else:
                # Format the phone number
                formatted_number = f"({digits[:3]}) {digits[3:6]}-{digits[6:]}"
                new_value = formatted_number
        elif self.type == 'email' and new_value != '':
            # Regular expression for validating an email
            email_regex = r'^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$'

            if re.match(email_regex, new_value):
                # Email is valid
                pass
            else:
                # Email is invalid
                self.data_status = 'warning'
        elif self.type == 'url' and new_value != '':
            # Regular expression for validating a URL
            url_regex = re.compile(
                r'^(?:http|ftp)s?://'  # http:// or https://
                r'(?:(?:[A-Z0-9](?:[A-Z0-9-]{0,61}[A-Z0-9])?\.)+(?:[A-Z]{2,6}\.?|[A-Z0-9-]{2,}\.?)|'  # domain...
                r'localhost|'  # localhost...
                r'\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}|'  # ...or ipv4
                r'\[?[A-F0-9]*:[A-F0-9:]+\]?)'  # ...or ipv6
                r'(?::\d+)?'  # optional port
                r'(?:/?|[/?]\S+)$', re.IGNORECASE)

            if re.match(url_regex, new_value):
                # URL is valid
                pass
            else:
                # URL is invalid
                self.data_status = 'warning'
        return True, new_value

    def _prepare_bool(self, new_value, app_settings, selected_unit_system):
        """Prepare the string value for setting the value.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_settings (AppSettingsData): The settings CalcData to use for unit conversion
            selected_unit_system (string): The unit system selected by the user

        Returns:
            succeeded (bool): True if the value was set successfully; otherwise, False
            new_value (float): The new value to set to the variable
        """
        if isinstance(new_value, str):
            if new_value.lower() in ['true', 't', 'yes', 'y', '1']:
                new_value = True
            elif new_value.lower() in ['false', 'f', 'no', 'n', '0', '-1']:
                new_value = False
            else:
                try:
                    new_value = float(new_value)
                except ValueError:
                    return False, new_value
        if isinstance(new_value, float):
            if new_value > 0:
                new_value = True
            else:
                new_value = False
        if not isinstance(new_value, bool):
            new_value = bool(new_value)
        return True, new_value

    def _prepare_int(self, new_value, app_settings, selected_unit_system):
        """Prepare the float value for setting the value.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_settings (AppSettingsData): The settings CalcData to use for unit conversion
            selected_unit_system (string): The unit system selected by the user

        Returns:
            new_value (float): The new value to set to the variable
        """
        # Clean up type if it was incorrectly set
        if not isinstance(new_value, int):
            try:
                new_value = int(new_value)
            except ValueError:
                try:
                    new_value = float(new_value)
                    new_value = int(new_value)
                except ValueError:
                    return False, new_value
        # Convert the value to the native unit
        # if app_settings is not None:  # Convert if we were given AppSettingsData
        #     selected_unit = self.get_selected_unit(selected_unit_system)
        #     if selected_unit != "" and self.native_unit != "":
        #         unit_converter = ConversionCalc(app_settings)
        #         _, new_value = unit_converter.convert_units(selected_unit, self.native_unit, new_value)
        # Check the limits
        (lower_limit, upper_limit) = self.limits
        if new_value < lower_limit:
            new_value = lower_limit
        if new_value > upper_limit:
            new_value = upper_limit

        return True, new_value

    def _prepare_double(self, new_value, app_settings, selected_unit_system, set_val=True):
        """Prepare the float value for setting the value.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_settings (AppSettingsData): The settings CalcData to use for unit conversion
            selected_unit_system (string): The unit system selected by the user
            set_val (bool): If called from set_val (as opposed to get_val), then True

        Returns:
            succeeded (bool): True if the value was set successfully; otherwise, False
            new_value (float): The new value to set to the variable
        """
        # Clean up type if it was incorrectly set
        if not isinstance(new_value, float):
            try:
                if new_value is None:
                    return False, new_value
                new_value = float(new_value)
            except ValueError:
                return False, new_value
        # Convert the value to the native unit
        if app_settings is not None:  # Convert if we were given AppSettingsData
            selected_unit = self.get_selected_unit(selected_unit_system)
            if selected_unit != "" and self.native_unit != "":
                unit_converter = ConversionCalc(app_settings)
                if set_val:
                    _, new_value = unit_converter.convert_units(selected_unit, self.native_unit, new_value)
                else:
                    _, new_value = unit_converter.convert_units(self.native_unit, selected_unit, new_value)
        # Check the limits
        (lower_limit, upper_limit) = self.limits
        if new_value < lower_limit:
            new_value = float(lower_limit)
        if new_value > upper_limit:
            new_value = float(upper_limit)

        return True, new_value

    def add_result(self, val):
        """Add result to a list.

        Args:
            val (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if val is None:
            return
        if self.type in ['float_list', 'string_list']:
            # If you passed a list, it would extend the existing variable; what if we want a list of lists?
            self.value_options.append(val)
            self.value = len(self.value_options)

    def set_index(self, var):
        """Sets the index of a list (by index or option).

        Args:
            var (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if self.type in ['bool', 'int', 'float', 'float_list', 'string_list', 'calc_list', 'calc', 'class',
                         'UserArray']:
            return
        elif self.type == 'list':
            if isinstance(var, int):
                # Make sure the variable is within limits; set tp closest limit
                if var < 0:
                    var = 0
                if var >= len(self.value_options):
                    var = len(self.value_options) - 1
                self.value = var
            else:
                self.value = self.value_options.index(var)

    def get_index(self, null_value, val=None):
        """Returns the index of a list.

        Args:
            null_value (float): what to return if it is not found
            val (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if val is None or val == '':
            return self.value
        if val not in self.value_options:
            return null_value
        index = null_value
        if self.type in ['float_list', 'list', 'string_list', 'calc_list', 'UserArray']:
            index = self.value_options.index(val)
        return index

    def append(self, val):
        """Append to the list.

        Args:
            val (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if self.type in ['bool', 'int', 'float', 'class', 'UserArray']:
            return
        elif self.type in self.list_types:
            self.value_options.append(val)

    def get_list(self):
        """Returns the list (value options).

        Returns
            value_options (list): list of options available
        """
        return self.value_options

    def set_list(self, new_list):
        """Sets the list of items.

        Args:
            new_list (list): The new list
        """
        self.value_options = new_list

    def get_selected_unit(self, selected_unit_system):
        """Returns the selected units."""
        selected_unit = self.selected_si_unit
        if selected_unit_system in ['U.S. Customary Units', 'Both']:
            selected_unit = self.selected_us_unit
        return selected_unit

    def set_selected_unit(self, selected_unit_system, selected_unit):
        """Returns the selected units."""
        if selected_unit_system in ['', None]:
            return None
        if selected_unit in ['', None]:
            return None
        if selected_unit_system in ['U.S. Customary Units', 'Both']:
            self.selected_us_unit = selected_unit
        else:
            self.selected_si_unit = selected_unit

        return selected_unit

    def _remove_duplicates(self, units_list):
        """Removes duplicates from a list.

        Args:
            units_list (list): The list to remove duplicates from
        """
        new_units_list = []
        for unit in units_list:
            if unit not in new_units_list:
                new_units_list.append(unit)
        return new_units_list

    def get_units_list_and_index(self, selected_unit_system):
        """Returns the units list and index of the selected units.

        Args:
            selected_unit_system (string): The unit system selected by the user
        """
        units_list = []
        index = None
        selected_unit = self.selected_si_unit
        # if len(self.unit_type) == 0:
        #     return units_list, index

        if selected_unit_system in ['U.S. Customary Units', 'Both']:
            units_list.extend(self.available_us_units[0])
            selected_unit = self.selected_us_unit
        if selected_unit_system in ['SI Units (Metric)', 'Both']:
            units_list.extend(self.available_si_units[0])
            units_list = self._remove_duplicates(units_list)

        if len(units_list) <= 0:
            return units_list, None
        elif selected_unit == "":
            index = 0
            if selected_unit_system in ['U.S. Customary Units', 'Both']:
                self.selected_us_unit = units_list[index]
            else:
                self.selected_si_unit = units_list[index]
            return units_list, index

        index = None
        try:
            index = units_list.index(selected_unit)
        except ValueError:
            pass
        return units_list, index

    def check_limits(self, result, warnings, name=None):
        """Checks that the value (if applicable) is within limits.

        Args:
            result (bool): Set to False if anything fails a check; otherwise, UNMODIFIED
            warnings (list): list of strings of warnings to provide to the user
            name (string): Additional name for context to the user in the error message
        """
        if self.type in ['float', 'int']:
            if self.limits[0] > self.value or self.value > self.limits[1]:
                result = False
                txt_name = self.name
                if name is not None:
                    txt_name = name + ': ' + self.name
                warnings.append(f'Please select a value for {txt_name} that is within its limits.')
        if self.type in ['UserArray']:
            if not self.get_val().check_limits(result, warnings):
                result = False
        return result

    def find_item_by_uuid(self, uuid):
        """Finds a variable by its uuid.

        Args:
            uuid (string): uuid of the variable

        Returns:
            result: True if item found
            variable: the variable
        """
        if self.uuid == uuid:
            return True, self

        if self.type in ['class', 'calc', 'uuid_dict', 'UserArray', 'table']:
            item = self.get_val()
            if hasattr(item, 'find_item_by_uuid'):
                result, item = self.get_val().find_item_by_uuid(uuid)
                if result:
                    return result, item

        return False, None
