"""Class that will run the application."""

__copyright__ = "(C) Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
# from typing import Any, Callable, Dict, Optional, Tuple
import copy
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.presenter.main.about_presenter import AboutPresenter
from xms.FhwaVariable.interface_adapters.view_model.main.message_box import MessageBox
from ..interactors.dialog_interactor import DialogInteractor
from ...core_data.app_data.app_data import AppData
# from xms.FhwaVariable.core_data.model.commands.command import CommandManager


class AppInteractor():
    """Class that will run the application."""

    app_name = ''
    view = None
    secondary_windows = {}
    dlg_interactors = {}
    window_interactors = {}
    main_presenter = None

    _initialized = False

    def __init__(self, app_name: str = None, injector=None):
        """Initialize the application class."""
        self.app_data = AppData()
        if not AppInteractor._initialized:
            if app_name is None:
                raise ValueError("The application name was not set.")
            AppInteractor.app_name = app_name
            AppInteractor.view = None

            AppInteractor.main_presenter = None

            # Plug-in functors to be accessible from AppData
            # Keep this list as short as possible
            AppData.message_box_functor = self.show_message_box
            AppData.select_file_functor = self.show_select_file
            AppData.edit_item_functor = self.edit_calculator
            AppData.about_functor = self.show_about
            AppData.exit_functor = self.close_application

            AppInteractor._initialized = True

            self.dialog_dict = {}

            # Leave this as the last thing to do in the constructor
            self.create_application()

    # Application Interactor methods
    def create_application(self) -> None:
        """
        Create the application view from a view model.
        """
        if self.app_data.view is None:
            raise ValueError("The view was not set in the model.")
        AppInteractor.view = self.app_data.view
        # Assign functors to model (keep this list short as possible)
        # self.app_data.model_dict[AppInteractor.app_name].set_interactor_functors(self.show_about,
        #                                                                          self.close_application)

        # Placed inside this function to prevent circular imports
        from xms.FhwaVariable.interface_adapters.presenter.main.app_presenter import AppPresenter
        AppInteractor.main_presenter = AppPresenter()
        # We don't have a clipboard before we create the application
        AppInteractor.main_presenter.create_application(self.app_data.model_dict, AppInteractor.view)

    def update_application(self, wiki_url: str = None) -> None:
        """
        Update the application view from a view model.
        """
        AppInteractor.main_presenter.update_application(self.app_data.model_dict, AppInteractor.view, wiki_url)

    def update_document_windows(self, ) -> None:
        """Update the document windows."""
        for window in self.app_data.doc_windows.values():
            # if window.uuid not in AppInteractor.window_interactors:
            #     AppInteractor.window_interactors[window.uuid] =
            window.update_window()

    def run(self):
        """Run the application."""
        self.update_application()
        AppInteractor.main_presenter.run(AppInteractor.view)

    def show_about(self):
        """Show the about dialog."""
        about_presenter = AboutPresenter(self.app_name, AppInteractor.view)
        about_presenter.show_about_dialog(self.app_data.model_dict)

    def show_context_menu(self, menu_dict, pos, item_uuid):
        """
        Show the context menu.

        Args:
            menu_dict (dict): The menu dictionary.
            pos (object): The position to create the menu.
            item_uuid (str): The uuid of the item.
        """
        AppInteractor.main_presenter.show_context_menu(menu_dict, pos, item_uuid, AppInteractor.view)

    def store_window_size_dict(self, geometry_dict):
        """Store the window size.

        Args:
            geometry_dict (dict): The geometry dictionary.
        """
        if geometry_dict is None:
            return
        self.store_window_size(geometry_dict['x'], geometry_dict['y'], geometry_dict['width'], geometry_dict['height'],
                               geometry_dict['maximized'], geometry_dict['window_name'])

    def store_window_size(self, x, y, width, height, maximized, window_name):
        """Store the dialog size."""
        app_settings = self.app_data.app_settings
        app_name = AppInteractor.app_name

        if self.view is not None:
            screens_info = self.view.get_screens_info()
        screen_setup_str = screens_info['screen_setup_str']

        # if name == 'Settings calc' and model == '' or model is None:
        #     model = 'AppData'
        if maximized:
            app_settings.save_persistant_dialog_data(screen_setup_str, window_name, app_name, 'maximized', True)
            app_settings.save_persistant_dialog_data(screen_setup_str, window_name, app_name, 'x', x)
            app_settings.save_persistant_dialog_data(screen_setup_str, window_name, app_name, 'y', y)
        else:
            app_settings.save_persistant_dialog_data(screen_setup_str, window_name, app_name, 'maximized', False)
            app_settings.save_persistant_dialog_data(screen_setup_str, window_name, app_name, 'x', x)
            app_settings.save_persistant_dialog_data(screen_setup_str, window_name, app_name, 'y', y)
            app_settings.save_persistant_dialog_data(screen_setup_str, window_name, app_name, 'width', width)
            app_settings.save_persistant_dialog_data(screen_setup_str, window_name, app_name, 'height', height)

    # Misc Interactor methods
    def show_message_box(self, message_box):
        """Show a message box.

        Args:
            message_box (object): The message box object.
        """
        return AppInteractor.view.message_box(message_box)

    def show_select_file(self, select_file):
        """Show a select file dialog.

        Args:
            select_file (object): The select file object.
        """
        return AppInteractor.view.select_file(select_file)

    def edit_calculator(self, old_calc, model_name=None, model_base=None, icon=None, parent_uuid=None):
        """Edit the CalcData.

        Args:
            model_base (object): The model base object.
            old_calc (object): The old CalcData object.

        Returns:
            new_calc (object): The new CalcData object.
        """
        if model_base is None and model_name is None:
            return False, None
        class_name = None
        if hasattr(old_calc, 'class_name'):
            class_name = old_calc.class_name
        if class_name in ['Folder', 'Proposed', 'Existing']:
            return None
        if class_name in ['Project']:
            old_calc = old_calc.project_settings
        if icon is None and model_base is not None and class_name in model_base.calcdata_dict:
            icon = model_base.calcdata_dict[class_name].icon
        if model_base is not None and model_name is None:
            model_name = model_base.model_name

        old_calc.app_data = self.app_data  # Just make sure it is up to date

        if not old_calc.check_for_needed_data():
            msg_box = MessageBox()
            msg_box.window_title = "Warnings"
            msg_box.message_type = "warning"
            msg_box.message_text = "\n".join(old_calc.warnings)
            msg_box.buttons = ['OK']
            AppInteractor.view.message_box(msg_box)
            return False, old_calc

        dlg_uuid = uuid.uuid4()
        self.dlg_interactors[dlg_uuid] = DialogInteractor(AppInteractor.view, dlg_uuid, self.edit_calculator)
        result, new_calc = self.dlg_interactors[dlg_uuid].launch_dialog_with_calculator(
            old_calc, dlg_uuid, current_model=model_name, modal=True, icon=icon,
            parent_uuid=parent_uuid)

        return result, new_calc

    def action_triggered(self, tool_uuid, right_click_uuid, current_text=''):
        """
        Send an action to be handled.

        Args:
            tool_uuid (str): The uuid of the tool.
            right_click_uuid (str): The uuid of the right click.
            current_text (str): The current text (of the tool, if changed).
        """
        for model in self.app_data.model_dict.values():
            model.app_data = self.app_data  # Make sure the data is up to date
            result, _ = model.perform_action(tool_uuid, right_click_uuid, current_text)
            if result:
                self.update_application()
                return True
        return False

    def toggle_show_panel(self, panel_name):
        """Toggle the visibility of a panel.

        Args:
            panel_name (str): The name of the panel to toggle.
        """
        for model in self.app_data.model_dict.values():
            result = model.toggle_show_panel(panel_name)
            if result:
                self.update_application()
                return True
        return False

    def handle_gui_tool(self, view_uuid, gui_tool, point_1, point_2):
        """Handle the GUI tool.

        Args:
            view_uuid (str): The uuid of the view.
            gui_tool (str): The GUI tool to handle.
            point_1 (object): The first point.
            point_2 (object): The second point.
        """
        # if view_uuid not in self.window_interactors:
        #     return False
        # self.window_interactors[view_uuid].handle_gui_tool(gui_tool, point_1, point_2)
        if view_uuid not in AppData.doc_windows:
            return
        doc = AppData.doc_windows[view_uuid]
        if doc is None or doc.window_type != 'graphics':
            return

        orientation = doc.camera_orientation
        new_orient = AppData.graphics_view.gui_tool_change(view_uuid, orientation, point_1, point_2, tool=gui_tool)
        if new_orient:
            doc.camera_orientation = new_orient
            self.update_application()

    def set_graphics_view_size(self, view_uuid, width, height, update=True):
        """Set the size of the graphics view.

        Args:
            view_uuid (str): The uuid of the view.
            width (int): The width of the view.
            height (int): The height of the view.
            update (bool): Whether to update the application.
        """
        gui_uuid = uuid.UUID(view_uuid)
        if gui_uuid not in AppData.doc_windows:
            return
        doc = AppData.doc_windows[gui_uuid]
        if doc is None or doc.window_type != 'graphics':
            return
        if doc.resulting_image.image_width == width and doc.resulting_image.image_height == height:
            return
        doc.resulting_image.image_width = width
        doc.resulting_image.image_height = height
        if update:
            self.update_application()

    def get_context_menu(self, item_uuid, pos):
        """
        Show the context menu.

        Args:
            item_uuid (str): The uuid of the item.
            pos (object): The position to create the menu.
        """
        for model in self.app_data.model_dict.values():
            result, menu_dict = model.get_context_menu_dict(item_uuid)
            if result:
                self.show_context_menu(menu_dict, pos, item_uuid)
                return True
        return False

    def data_tree_add_folder(self, class_name):
        """
        Add a folder to the data tree.

        Args:
            class_name (str): The class name of the folder.
        """
        for model in self.app_data.model_dict.values():
            result = model.add_tree_folder(class_name)
            if result:
                self.update_application()
                return True
        return False

    # Data Tree Interactor methods
    def data_tree_handle_drop(self, target_uuid, is_ctrl, is_shift):
        """Handle the drop event in the data tree.

        Args:
            target_uuid (str): The uuid of the target item.
            is_ctrl (bool): Whether the Ctrl key is pressed.
            is_shift (bool): Whether the Shift key is pressed.
        """
        # First determine everything selected across all model dicts
        target_item = None
        target_model = None
        selected_items = []
        models_with_selected_items = []
        for model in self.app_data.model_dict.values():
            new_items = model.get_all_selected_items()
            selected_items.extend(new_items)
            result, item = model.find_item_by_uuid(target_uuid)
            if result:
                target_item = item
                target_model = model
            if len(new_items) > 0:
                models_with_selected_items.append(model)

        # Determine if we have nothing selected and fail if so
        if len(selected_items) == 0:
            return False
        if target_item is None:
            return False

        # Check that the targt accepts drops
        if hasattr(target_item, 'is_drop_enabled'):
            if not target_item.is_drop_enabled:
                return False
        if hasattr(target_item, 'tree_data'):
            if not target_item.tree_data.is_drop_enabled:
                return False

        # When we move items, we also move children. Make sure we don't move them twice.
        filtered_items = self.filter_out_items_included_as_children(selected_items)

        # Filter out items that are not drag enabled
        cleaned_items = []
        for item in filtered_items:
            if hasattr(item, 'is_drag_enabled') and item.is_drag_enabled:
                cleaned_items.append(item)
            elif hasattr(item, 'tree_data') and item.tree_data.is_drag_enabled:
                cleaned_items.append(item)

        # Copy or move elements over
        children = None
        if hasattr(target_item, 'children'):
            children = target_item.children
        if hasattr(target_item, 'tree_data'):
            if hasattr(target_item.tree_data, 'children'):
                children = target_item.tree_data.children
        if children is not None:
            for item in cleaned_items:
                new_item = item
                if is_ctrl:
                    # Copy, then everything needs a new uuid
                    new_item = copy.deepcopy(item)
                    target_model.assign_new_uuid_to_item_and_children_recursive(new_item)
                else:
                    # Move: Remove the item from its old parent (get parent_uuid, then parent, then remove)
                    parent_uuid = None
                    if hasattr(item, 'parent_uuid'):
                        parent_uuid = new_item.parent_uuid
                    if hasattr(new_item, 'tree_data'):
                        parent_uuid = new_item.tree_data.parent_uuid
                    if parent_uuid:
                        parent = None
                        for model in models_with_selected_items:
                            result, parent = model.find_item_by_uuid(parent_uuid)
                            if result:
                                break
                        if parent:
                            p_children = None
                            if hasattr(parent, 'children'):
                                p_children = parent.children
                            if hasattr(parent, 'tree_data'):
                                if hasattr(parent.tree_data, 'children'):
                                    p_children = parent.tree_data.children
                            if p_children is not None:
                                p_children.remove(new_item)
                # Now we assign the new item to the target item
                if hasattr(target_item, 'children'):
                    target_item.children.append(new_item)
                elif hasattr(target_item, 'tree_data'):
                    if hasattr(target_item.tree_data, 'children'):
                        target_item.tree_data.children.append(new_item)

        self.update_application()
        return True

    def filter_out_items_included_as_children(self, selected_items):
        """
        Filter out items that are included as children of other selected items.

        Args:
            selected_items (list): The list of selected items.

        Returns:
            list: The filtered list of parent items.
        """
        # Create a set to store all children of selected items
        all_children = set()
        for item in selected_items:
            self._collect_all_children_recursive(item, all_children)

        # Filter out items that are in the all_children set
        filtered_items = [item for item in selected_items if item not in all_children]

        return filtered_items

    def convert_item_list_to_uuid_list(self, item_list):
        """Convert a list of items to a list of UUIDs.

        Args:
            item_list (list): The list of items to convert.

        Returns:
            list: The list of UUIDs.
        """
        uuid_list = []
        for item in item_list:
            uuid_list.append(item.uuid)
        return uuid_list

    def _collect_all_children_recursive(self, item, children_set):
        """Recursively collect all children of an item.

        Args:
            item (object): The item to collect children from.
            children_set (set): The set to store the collected children.
        """
        children = None
        if hasattr(item, 'children'):
            children = item.children
        if hasattr(item, 'tree_data'):
            if hasattr(item.tree_data, 'children'):
                children = item.tree_data.children
        if children is not None:
            for child in item.children:
                children_set.add(child)
                self._collect_all_children_recursive(child, children_set)

    def data_tree_item_selected(self, selected_items):
        """Handle the selection of items in the data tree.

        Args:
            selected_items (list): The list of selected items.
        """
        run_update = False
        for model in self.app_data.model_dict.values():
            result, found_uuids = model.set_selected_items(selected_items)
            if result and len(found_uuids) >= len(selected_items):
                self.update_application()
                return True
            else:
                run_update = True

        if run_update:  # In case we find some, but not all
            self.update_application()
        return False

    def data_tree_item_expanded(self, item_uuid):
        """Handle the expansion of an item in the data tree.

        Args:
            item_uuid (str): The uuid of the item.
        """
        for model in self.app_data.model_dict.values():
            result, item = model.find_item_by_uuid(item_uuid)
            if result:
                if hasattr(item, 'is_expanded'):
                    item.is_expanded = True
                if hasattr(item, 'tree_data'):
                    item.tree_data.is_expanded = True
                self.update_application()
                return True

        return False

    def data_tree_item_collapsed(self, item_uuid):
        """Handle the collapse of an item in the data tree.

        Args:
            item_uuid (str): The uuid of the item.
        """
        for model in self.app_data.model_dict.values():
            result, item = model.find_item_by_uuid(item_uuid)
            if result:
                if hasattr(item, 'is_expanded'):
                    item.is_expanded = False
                if hasattr(item, 'tree_data'):
                    item.tree_data.is_expanded = False
                self.update_application()
                return True

        return False

    def data_tree_rename_selected(self, item_uuid):
        """Handle the collapse of an item in the data tree.

        Args:
            item_uuid (str): The uuid of the item.
        """
        for model in self.app_data.model_dict.values():
            result, item = model.find_item_by_uuid(item_uuid)
            if result:
                if hasattr(item, 'is_editing'):
                    item.is_editing = True
                if hasattr(item, 'tree_data'):
                    item.tree_data.is_editing = True
                self.update_application()
                return True

        return False

    def data_tree_item_changed(self, item_uuid, name):
        """Handle the change of an item in the data tree.

        Args:
            item_uuid (str): The uuid of the item.
            name (str): The name of the item.
        """
        # TODO: Add to undo/redo
        for model in self.app_data.model_dict.values():
            result = model.set_item_name_by_uuid(name, item_uuid, add_undo=True)
            if result:
                self.update_application()
                return True

        return False

    # def data_tree_item_deleted(self, item_uuid_list, update_app=True):
    #     """Handle the deletion of an item in the data tree.

    #     Args:
    #         item_uuid_list (list): The list of uuids of the items to delete.
    #         update_app (bool): Whether to update the application.
    #     """
    #     # TODO: Add to undo/redo
    #     unfound_list = item_uuid_list
    #     changes = False
    #     for model in self.app_data.model_dict.values():
    #         result, items, unfound_list = model.get_items_with_uuids(unfound_list)
    #         for item in items:
    #             changes = True
    #             parent = None
    #             if hasattr(item, 'parent'):
    #                 parent = item.parent
    #             if hasattr(item, 'tree_data'):
    #                 if hasattr(item.tree_data, 'parent'):
    #                     parent = item.tree_data.parent

    #             if parent:
    #                 if hasattr(item, 'children'):
    #                     parent.children.remove(item)
    #                 if hasattr(item, 'tree_data'):
    #                     if hasattr(item.tree_data, 'children'):
    #                         parent.tree_data.children.remove(item.tree_data)
    #         if result:
    #             break

    #     if changes and update_app:
    #         self.update_application()
    #         return True

    #     return False

    def set_active_window(self, window_uuid):
        """Handle the action to set a window as active.

        Args:
            window_uuid (str): The uuid of the window to set as active.
        """
        if not window_uuid:
            return False
        w_uuid = uuid.UUID(window_uuid)
        if w_uuid not in AppData.doc_windows:
            return False
        if AppData.doc_windows[w_uuid].is_active:
            return False

        # Update the active state of all windows
        for window in AppData.doc_windows.values():
            window.is_active = window.window_uuid == w_uuid

        self.update_application()
        return True

    def window_changed(self, tool_uuid):  # camera_changed
        """Handle the change of the window.

        Args:
            tool_uuid (str): The uuid of the tool.
        """
        raise NotImplementedError("window_changed")

    def edit_calc_by_uuid(self, item_uuid):
        """Edit the selected CalcData.

        Args:
            item_uuid (UUID): The uuid of the selected tree item
        """
        for model_base in self.app_data.model_dict.values():
            _, old_calc = model_base.find_item_by_uuid(item_uuid)
            if old_calc:

                result, new_calc = self.edit_calculator(old_calc, model_base=model_base)

                if result:
                    model_base.set_calc(new_calc, old_calc)
        return None

    # Support Windows
    def set_support_window_show_plot(self, index, show_plot):
        """Get the show plot flag for a support window.

        Args:
            index(int): The index of the support window.
            show_plot(bool): The show plot flag.
        """
        AppData.set_support_window_show_plot(index, show_plot)
        self.update_application()

    def set_support_window_plots_dict(self, index, plots_dict):
        """Set the plot dictionary for a support window.

        Args:
            index(int): The index of the support window.
            plots_dict(dict): The plot dictionary.
        """
        raise NotImplementedError("get_support_window_plots_dict")

    def set_support_window_show_wiki(self, index, show_wiki):
        """Set the show wiki flag for a support window.

        Args:
            index(int): The index of the support window.
            show_wiki(bool): The show wiki flag.
        """
        AppData.set_support_window_show_wiki(index, show_wiki)
        self.update_application()

    def set_support_window_use_only_this_wiki(self, index, use_only_this_wiki):
        """Set the use only this wiki flag for a support window.

        Args:
            index(int): The index of the support window.
            use_only_this_wiki(bool): The use only this wiki flag.
        """
        AppData.set_support_window_use_only_this_wiki(index, use_only_this_wiki)
        self.update_application()

    def set_support_window_wiki_url(self, index, wiki_url):
        """Set the wiki URL for a support window.

        Args:
            index(int): The index of the support window.
            wiki_url(str): The wiki URL.
        """
        raise NotImplementedError("set_support_window_wiki_url")

    def set_support_window_plot_dock_area(self, index, plot_dock_area, update=True):
        """Set the plot dock area for a support window.

        Args:
            index(int): The index of the support window.
            plot_dock_area(str): The plot dock area.
            update(bool): Whether to update the application.
        """
        AppData.set_support_window_plot_dock_area(index, plot_dock_area)
        if update:
            self.update_application()

    def set_support_window_wiki_dock_area(self, index, wiki_dock_area, update=True):
        """Set the wiki dock area for a support window.

        Args:
            index(int): The index of the support window.
            wiki_dock_area(str): The wiki dock area.
            update(bool): Whether to update the application.
        """
        AppData.set_support_window_wiki_dock_area(index, wiki_dock_area)
        if update:
            self.update_application()

    def close_application(self):
        """Close the application."""
        # TODO: Check for unsaved changes, if so, ask if they want to save
        geometry = self.view.get_main_window_geometry()
        self.store_window_size_dict(geometry)

        for i in range(len(AppData.support_windows)):
            self.close_support_window(i, False)

        self.view.close()

    def close_support_window(self, index, save_number_of_windows=True):
        """Close a support window.

        Args:
            index(int): The index of the support window.
            save_number_of_windows(bool): Whether to save the number of windows.
        """
        geometry = self.view.get_support_window_geometry(index)
        self.store_window_size_dict(geometry)
        AppData.close_support_window(index, save_number_of_windows)
        if save_number_of_windows:
            self.update_application()

    # Dialog Interactor methods
    def dialog_changed(self, dlg_uuid, row_uuid, new_val, index, is_unit_change, add_undo, uuid_index,
                       update_dialog=True):
        """Change request from a dialog.

        Args:
            dlg_uuid (UUID): The dialog UUID
            row_uuid (UUID): The row UUID
            new_val (Any): The new value
            index (int): The index
            is_unit_change (bool): Whether the change is a unit change
            add_undo (bool): Whether to add to the undo stack
            uuid_index (int): The index of the UUID in the list
            update_dialog (bool): Whether to update the dialog
        """
        if dlg_uuid not in self.dlg_interactors:
            return False
        self.dlg_interactors[dlg_uuid].set_val_by_uuid(row_uuid, new_val, index=index, is_unit_change=is_unit_change,
                                                       add_undo=add_undo, uuid_index=uuid_index,
                                                       update_dialog=update_dialog)
        return True

    def update_dialog(self, dlg_uuid):
        """Update the dialog.

        Args:
            dlg_uuid (UUID): The dialog UUID
        """
        if dlg_uuid not in self.dlg_interactors:
            return False
        self.dlg_interactors[dlg_uuid].update_dialog()
        return True

    def btn_clicked(self, dlg_uuid, row_uuid, uuid_index, is_unit_col, btn_txt, add_undo, icon):
        """Button clicked event.

        Args:
            dlg_uuid (str): the UUID of the dialog
            row_uuid (str): the UUID of the row
            uuid_index (int): the index of the UUID
            is_unit_col (bool): whether the column is a unit column
            btn_txt (str): the text of the button
            add_undo (bool): whether to add to the undo stack
            icon (str): the icon of the button
        """
        if dlg_uuid not in self.dlg_interactors:
            return False
        return self.dlg_interactors[dlg_uuid].btn_clicked(row_uuid, uuid_index, is_unit_col, btn_txt, add_undo, icon)

    def show_panel(self, dlg_uuid: uuid.UUID, panel_name: str, state: int):
        """Show a panel.

        Args:
            dlg_uuid (UUID): The dialog UUID
            panel_name (str): The name of the panel
            state (int): Whether the panel is checked
        """
        if dlg_uuid not in self.dlg_interactors:
            return False
        self.dlg_interactors[dlg_uuid].show_panel(panel_name, state)
        return True

    def handle_help(self, dlg_uuid, item_uuid):
        """Handle the help request.

        Args:
            dlg_uuid (UUID): The dialog UUID
            item_uuid (UUID): The item UUID
        """
        _, use_only_dock_wiki = self.app_data.model_dict[AppInteractor.app_name].get_setting('Use only dockable wiki')
        if use_only_dock_wiki:
            wiki_url = self.get_help_url(dlg_uuid, item_uuid)
            if wiki_url:
                self.update_application(wiki_url)

        if dlg_uuid not in self.dlg_interactors:
            return False
        self.dlg_interactors[dlg_uuid].handle_help(item_uuid)
        return True

    def get_help_url(self, dlg_uuid, item_uuid):
        """Get the help URL.

        Args:
            dlg_uuid (UUID): The dialog UUID
            item_uuid (UUID): The item UUID
        """
        if dlg_uuid in self.dlg_interactors:
            return self.dlg_interactors[dlg_uuid].get_help_url(item_uuid)
        for model in self.app_data.model_dict.values():
            result, item = model.find_item_by_uuid(item_uuid)
            if result:
                url = item.help_url
                return True, url
        return False, None

    def plot_btn(self, dlg_uuid):
        """Handle the plot button click.

        Args:
            dlg_uuid (str): The dialog UUID
        """
        if dlg_uuid not in self.dlg_interactors:
            return False

        plot_interactor = DialogInteractor(AppInteractor.view, dlg_uuid, self.edit_calculator)
        self.dlg_interactors[dlg_uuid].plot_btn(plot_interactor)

    # I'm keeping the extra arguments, to make it easier to align with store_dlg_size
    def close_plot(self, dlg_uuid, x, y, width, height, maximized, index, change_count=True):
        """Close the plot window, but store the dialog size.

        Args:
            dlg_uuid (UUID): The dialog UUID
            x (int): The x position
            y (int): The y position
            width (int): The width
            height (int): The height
            maximized (bool): Whether the dialog is maximized
            show_wiki (bool): Whether the wiki is shown
            show_plot (bool): Whether the plot is shown
            splitter_h (list): The horizontal splitter size info
            splitter_v (list): The vertical splitter size info
            index (int): The index of the plot
            change_count (bool): Whether to change the count of plots (no, if we are closing the dialog)
        """
        if dlg_uuid not in self.dlg_interactors:
            return False
        self.dlg_interactors[dlg_uuid].close_plot(x, y, width, height, maximized, index, change_count)
        return True

    def store_dlg_size(self, dlg_uuid, x, y, width, height, maximized, show_wiki, show_plot, splitter_h, splitter_v):
        """Store the dialog size.

        Args:
            dlg_uuid (UUID): The dialog UUID
            x (int): The x position
            y (int): The y position
            width (int): The width
            height (int): The height
            maximized (bool): Whether the dialog is maximized
            show_wiki (bool): Whether the wiki is shown
            show_plot (bool): Whether the plot is shown
            splitter_h (list): The horizontal splitter size info
            splitter_v (list): The vertical splitter size info
        """
        if dlg_uuid not in self.dlg_interactors:
            return False
        self.dlg_interactors[dlg_uuid].store_dlg_size(x, y, width, height, maximized, show_wiki, show_plot, splitter_h,
                                                      splitter_v)
        return True

    def open_pdf(self, dlg_uuid, reference_name):
        """Open a PDF.

        Args:
            dlg_uuid (UUID): The dialog UUID
            reference_name (str): The name of the reference
        """
        if dlg_uuid not in self.dlg_interactors:
            return False
        self.dlg_interactors[dlg_uuid].open_pdf(reference_name)
        return True
