"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.calculator_list import CalcOrVarlist
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class ClassExampleCalc(CalcData):
    """A class that defines an example of variable data to test the File IO."""

    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the example calculator."""
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Calc Calc"
        self.type = "CalcCalc"

        self.input['int test'] = Variable('int test', 'int', 0, None)

        self.results['Thalweg elevation'] = Variable(
            'Thalweg elevation (prior to scour)',
            "float_list",
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)


class IoExampleCalc(CalcData):
    """A class that defines an example of variable data to test the File IO."""

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Bridge scour calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "IO Example Calc"
        self.type = "IOExampleCalc"

        self.calc_support_dual_input = False

        # Input
        self.input = {}  # Clear the base class input

        # name='', var_type='', value=0.0, value_options=None, limits=(0.0, sys.float_info.max),
        #          precision=2, unit_type=None, native_unit="", us_units=None, si_units=None, note="",
        #          complexity=0

        self.input['int test'] = Variable('int test', 'int', 0, None)
        self.input['float test'] = Variable('float test', 'float', 0.0, None, native_unit='ft',
                                            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['string test'] = Variable('string test', 'string', '', None)
        self.input['bool test'] = Variable('bool test', 'bool', False, None)
        self.list_options = ['option 1', 'option 2', 'option 3']
        self.input['list test'] = Variable('list test', 'list', 0, self.list_options)
        self.input['float_list test'] = Variable('float_list test', 'float_list', 0, [])
        self.input['string_list test'] = Variable('string_list test', 'string_list', 0, [])
        self.uuid_dict_options = {uuid.uuid4(): 'option 1', uuid.uuid4(): 'option 2', uuid.uuid4(): 'option 3'}
        self.input['uuid_dict test'] = Variable('uuid_dict test', 'uuid_dict', None, self.uuid_dict_options)
        self.input['file test'] = Variable('file test', 'file', '', None)
        self.input['color test'] = Variable('color test', 'color', '', None)
        self.input['image test'] = Variable('image test', 'image', '', None)

        self.input['calc test'] = Variable('calc test', 'calc', ClassExampleCalc(app_data=app_data,
                                                                                 model_name=model_name,
                                                                                 project_uuid=project_uuid), None)

        self.input['calc list test'] = Variable(
            'calc list test', 'calc list', CalcOrVarlist(
                ClassExampleCalc, show_define_btn=True, default_name='example', default_plural_name='examples',
                app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        self.input['dict group test'] = {}
        self.input['dict group test']['int test'] = Variable('int test', 'int', 0, None)
        self.input['dict group test']['float test'] = Variable('float test', 'float', 0.0, None, native_unit='ft',
                                                               us_units=self.us_mid_length,
                                                               si_units=self.si_mid_length)

        # self.input['class test'] = Variable('class test', 'class', '', None)
        # self.input['UserArray test'] = Variable('UserArray test', 'UserArray', '', None)
        # self.input['table test'] = Variable('table test', 'table', '', None)
        # self.input['PersonalDetails test'] = Variable('PersonalDetails test', 'PersonalDetails', '', None)

        self.input_dict = {}

        self.input['Plot options'] = {}
        self.add_plot_options_if_needed()

        self.unknown = None

        # Intermediate

        # Results
        self.clear_my_own_results = clear_my_own_results

        self.results['Thalweg elevation'] = Variable(
            'Thalweg elevation (prior to scour)',
            "float_list",
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.warnings = []

        self.unit_converter = ConversionCalc(app_data)

        # Reference Manuals
        self.reference_pdfs['HEC-18'] = (
            'HEC-18, 5th Edition - Evaluating Scour at Bridges.pdf', 167)

        # plot
        self.plots = {}
        # self.plots['Bridge cross-section'] = {}
        # self.plots['Bridge cross-section']['plot name'] = "Profile"
        # self.plots['Bridge cross-section']['legend'] = 'best'
        # self.plots['Bridge cross-section']['index'] = 1

    def add_plot_options_if_needed(self):
        """Add plot options for scenarios if they do not exist."""
        # for scenario in self.input['Scenarios'].get_val().item_list:
        #     if scenario.name not in self.input['Plot options']:
        #         self.add_plot_option_for_scenario(scenario.name)
