"""GradationLayerData Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.gradations.gradation_layer_calc import GradationLayerCalc
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class GradationLayerData(CalcData):
    """Provides a class that will define the site data of a culvert barrel."""

    soil_types = ['Noncohesive', 'Cohesive',]

    def __init__(self, show_elevations=False, show_soil_type=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Gradation Layer.

        Args:
            show_elevations (bool): whether to show the elevations
            show_soil_type (bool): whether to show the soil type
            app_data (AppData): The application data (settings).
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)
        # Input
        # self.name = f'culvert {index_of_culvert}'

        self.name = 'Soil Layer'
        self.type = 'SoilLayer'

        self.calculator = GradationLayerCalc()

        self.theme = self.get_theme()

        self.show_elevations = show_elevations
        self.show_soil_type = show_soil_type

        self.noncohesive_results = 'None'
        self.gradation_name = None

        # self.gradation_entry_list = ['enter complete gradation']
        # if self.allow_only_selected_gradations:
        #     self.gradation_entry_list.append('enter only required gradations')

        # self.input['gradation entry'] = Variable('Embedment entry', "list", 0,
        #                                          self.gradation_entry_list)

        self.input['Name'] = Variable('Name', "string", 'New soil layer', [])

        self.input['Top elevation'] = Variable('Top elevation', 'float', 0.0,
                                               limits=(-sys.float_info.max, sys.float_info.max),
                                               precision=2, unit_type=['length'],
                                               native_unit='ft', us_units=self.us_mid_length,
                                               si_units=self.si_mid_length)
        self.input['Bottom elevation'] = Variable('Bottom elevation', 'float', 0.0,
                                                  limits=(-sys.float_info.max, sys.float_info.max),
                                                  precision=2, unit_type=['length'], native_unit='ft',
                                                  us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input['Soil type'] = Variable('Soil type', "list", 0, GradationLayerData.soil_types)

        self.input['Critical shear stress (τc)'] = Variable(
            "Critical shear stress (τc)",
            "float",
            0.0,
            [],
            precision=4,
            unit_type=['stress'],
            native_unit='psf',
            us_units=[['psf']],
            si_units=[['pa']],
            note='The level of shear required to erode the soil layer')

        _, gamma_s = self.get_setting('Unit weight of soil (γs)')
        self.input['Unit weight of soil (γs)'] = Variable(
            'Unit weight of soil (γs)',
            'float',
            gamma_s,
            precision=4,
            unit_type=["specific_weight"],
            native_unit="lb/ft^3",
            us_units=[["lb/ft^3"]],
            si_units=[["kN/m^3", "N/m^3"]])

        self.input['Angle of repose (Θ)'] = Variable('Angle of repose (Θ)', 'float', math.radians(44), [],
                                                     limits=(math.radians(30), math.radians(44)),
                                                     precision=2, unit_type='angle',
                                                     native_unit='radians',
                                                     us_units=['degrees', 'radians', 'H:1V'],
                                                     si_units=['degrees', 'radians', 'H:1V'])
        self.input['Angle of repose (Θ)'].selected_us_unit = 'degrees'

        self.input['Gradation'] = Variable('Gradation', 'table', None)
        _, grad = copy.deepcopy(self.get_setting('Default gradation'))
        grad.set_app_data_model_name_and_project_uuid(self.app_data, self.model_name, self.project_uuid)

        grad.set_name('Gradation')
        self.input['Gradation'].set_val(grad)

        soil_color = (102, 51, 0)
        soil_fill_color = (215, 211, 199)
        if self.theme is not None:
            soil_color = self.theme['Plot soil color']
            soil_fill_color = self.theme['Plot soil fill color']

        gradation = {
            'Sediment particle diameter': grad.input['Data input'].get_val().input['Sediment particle diameter'],
            'Percent passing': grad.input['Data input'].get_val().input['Percent passing']
        }
        input_dict = {}
        input_dict['Gradation'] = gradation
        self.input['Plot options'] = {}
        self.input['Plot options']['Gradation'] = Variable(
            'Plot options', 'class', PlotOptions('Gradation layers',
                                                 input_dict=input_dict,
                                                 show_series=True, app_data=app_data, model_name=model_name,
                                                 project_uuid=project_uuid),
            complexity=1)
        self.input['Plot options']['Gradation'].get_val().set_plot_series_options(
            related_index=0, index=0, x_axis='Station', y_axis='Elevation', name='Gradation',
            line_color=soil_color, linetype='solid', line_width=1.0, fill_below_line=False,
            fill_color=soil_fill_color, pattern='sand')

        # Intermediate variables
        # self.compute_prep_functions = []
        self.intermediate_to_copy.extend([
            'd50', 'd84', 'tau', 'ks', 'd_star', 'manning_n', 'noncohesive_results', 'gradation_name', 'soil_type',
            'sediment_dia', 'percent_pass', 'num_layers', 'show_elevations', 'show_soil_type', 'bottom_elev'])
        self.d50 = 0.0
        self.d84 = 0.0
        self.tau = 0.0
        self.ks = 0.0
        self.d_star = 0.0
        self.manning_n = 0.0
        self.bottom_elev = 0.0

        self.noncohesive_results = 'None'
        self.gradation_name = None

        self.soil_type = None
        self.sediment_dia = None
        self.percent_pass = None

        self.num_layers = None

        self.warnings = []

        self.results = {}
        self.results['Gradation'] = {}
        self.results['Gradation']['D15'] = Variable(
            'D15',
            'float',
            0.0, [],
            precision=6,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_short_length,
            si_units=self.si_short_length,
            note='')
        self.results['Gradation']['D15'].selected_us_unit = 'mm'
        self.results['Gradation']['D15'].selected_si_unit = 'mm'

        self.results['Gradation']['D50'] = Variable(
            'D50',
            'float',
            0.0, [],
            precision=6,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_short_length,
            si_units=self.si_short_length,
            note='')
        self.results['Gradation']['D50'].selected_us_unit = 'mm'
        self.results['Gradation']['D50'].selected_si_unit = 'mm'

        self.results['Gradation']['D85'] = Variable(
            'D85',
            'float',
            0.0, [],
            precision=6,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_short_length,
            si_units=self.si_short_length,
            note='')
        self.results['Gradation']['D85'].selected_us_unit = 'mm'
        self.results['Gradation']['D85'].selected_si_unit = 'mm'

        self.results['Gradation']['D95'] = Variable(
            'D95',
            'float',
            0.0, [],
            precision=6,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_short_length,
            si_units=self.si_short_length,
            note='')
        self.results['Gradation']['D95'].selected_us_unit = 'mm'
        self.results['Gradation']['D95'].selected_si_unit = 'mm'

        self.results['Gradation']['Cu'] = Variable(
            'Uniformity coefficient (Cu)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['coefficient'],
            note='')

        self.results['Gradation']['Cc'] = Variable(
            'Coefficient of curvature (Cc)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['coefficient'],
            note='')

        self.results['Gradation']['Graded'] = Variable(
            'Gradation classification',
            "string",
            False, [],
            note='')

        self.results['Gradation']['Classification'] = Variable(
            'Classification',
            'string',
            '',
            [],
            note='You can change the gradation classification system in the settings.')

        self.results['Gradation']['Prior classification'] = Variable(
            'Prior classification',
            'string',
            '',
            [],
            note='You can change the category limits in the settings.')

        self.results['Gradation']['Calculate texture'] = Variable(
            "Calculate texture",
            "bool",
            "",
            [],
            note='')
        self.results['Gradation']['Texture'] = Variable(
            "USDA soil texture",
            "string",
            "",
            [],
            note='')

        self.results['Elevations'] = {}
        self.results['Elevations']['Elevations computed'] = Variable(
            'Elevations computed', 'bool', False, [],
            note='Whether the elevations have been computed for this layer')
        self.results['Elevations']['Top elevation'] = Variable(
            'Top elevation', 'float', 0.0, precision=2, limits=(-sys.float_info.max, sys.float_info.max),
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Elevations']['Bottom elevation'] = Variable(
            'Bottom elevation', 'float', 0.0, precision=2, limits=(-sys.float_info.max, sys.float_info.max),
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Shear decay parameters'] = {}
        self.results['Shear decay parameters']['Sediment gradation coefficient (σ)'] = Variable(
            'Sediment gradation coefficient (σ)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['coefficient'],
            note='D85 / D50')
        self.results['Shear decay parameters']['Dimensionless grain size diameter (D*)'] = Variable(
            "Dimensionless grain size diameter (D*)",
            "float",
            0.0,
            [],
            precision=4,
            unit_type=['coefficient'],
            note="Dimensionless grain size diameter computed using Gou's equation (2002)")
        self.results['Shear decay parameters']['Shields number (ks)'] = Variable(
            "Shields number (ks)",
            "float",
            0.0,
            [],
            precision=4,
            unit_type=['coefficient'],
            note='The level of shear required to erode the soil layer')
        self.results['Shear decay parameters']['Critical shear stress (τc)'] = Variable(
            "Critical shear stress (τc)",
            "float",
            0.0,
            [],
            precision=4,
            unit_type=['stress'],
            native_unit='psf',
            us_units=[['psf']],
            si_units=[['pa']],
            note='The level of shear required to erode the soil layer')
        self.results['Shear decay parameters']['Manning n'] = Variable(
            "Manning n (n)",
            'float',
            0.0, [],
            precision=4,
            unit_type=['coefficient'],
            note="n value computed using Strickler's equation")

        self.plot_names = ['Gradation']
        self.plots = {}
        for name in self.plot_names:
            self.plots[name] = {}
            self.plots[name]['Plot name'] = name
            self.plots[name]['Legend'] = "best"

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
            (str): warning message
        """
        if self.input['Soil type'].get_val() == 'Noncohesive':
            grad = self.input['Gradation'].get_val()
            num_items = grad.input['Table options'].get_val().input['Number of items'].get_val()
            if num_items <= 0:
                self.warnings.append('Gradation data is empty. Please enter gradation data.')
                return False
            sediments = grad.input['Data input'].get_val().input['Sediment particle diameter'].get_val()[:num_items]
            passing = grad.input['Data input'].get_val().input['Percent passing'].get_val()[:num_items]

            sed_filtered_values = [value for value in sediments if value != 0.0]
            sed_unique_values = set(sed_filtered_values)

            pass_filtered_values = [value for value in passing if value != 0.0]
            pass_unique_values = set(pass_filtered_values)

            if len(sed_unique_values) == 0 or len(pass_unique_values) == 0:
                self.warnings.append('Gradation data is empty. Please enter gradation data.')
                return False

        else:  # Cohesive
            result = self.check_float_vars_to_greater_zero(['Critical shear stress (τc)'])
            return result
        return True

    def get_val(self):
        """Computes and returns the results.

        Returns:
            self.results['result'] (variable): result of the computations.
        """
        self.compute_data()
        return self.results

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}
        input_vars['Name'] = self.input['Name']

        if self.show_elevations and self.num_layers is None or self.show_elevations and self.num_layers and \
                self.num_layers > 1:
            input_vars['Top elevation'] = self.input['Top elevation']
            # input_vars['Bottom elevation'] = self.input['Bottom elevation']

        if self.show_soil_type:
            input_vars['Soil type'] = self.input['Soil type']

        if self.input['Soil type'].get_val() == 'Noncohesive':
            input_vars['Gradation'] = self.input['Gradation']
        else:
            input_vars['Critical shear stress (τc)'] = self.input['Critical shear stress (τc)']

        input_vars['Angle of repose (Θ)'] = copy.deepcopy(self.input['Angle of repose (Θ)'])

        return input_vars

    def get_embedment_depth(self):
        """Returns the embedment depth.

        Returns:
            embedment depth (float): embedment depth
        """
        self.compute_data()
        return self.input['Embedment depth'].get_val()

    def get_results_tab_group(self, unknown=None):
        """Get the results tab group.

        Args:
            unknown (variable): unknown variable

        Returns:
            results_vars (list of variables): results group of variables
        """
        results_vars = {}

        if self.soil_type == 'Noncohesive':
            if self.noncohesive_results == 'Full gradation':
                results_vars['D15'] = self.results['Gradation']['D15']
                results_vars['D50'] = self.results['Gradation']['D50']
                results_vars['D85'] = self.results['Gradation']['D85']
                results_vars['D95'] = self.results['Gradation']['D95']
                results_vars['Cu'] = self.results['Gradation']['Cu']
                results_vars['Cc'] = self.results['Gradation']['Cc']
                results_vars['Graded'] = self.results['Gradation']['Graded']
                results_vars['Classification'] = self.results['Gradation']['Classification']

            elif self.noncohesive_results == 'Single D50':
                results_vars['D50'] = self.results['Gradation']['D50']
                results_vars['Classification'] = self.results['Gradation']['Classification']

            elif self.noncohesive_results == 'Single value' and self.gradation_name is not None:
                results_vars[self.gradation_name] = self.results[self.gradation_name]['D50']

        if self.results['Gradation']['Calculate texture'].get_val():
            results_vars['Texture'] = self.results['Gradation']['Texture']

        if self.soil_type == 'Noncohesive':
            results_vars['Shear decay parameters'] = self.results['Shear decay parameters']
        else:
            results_vars['Shear decay parameters'] = {}
            results_vars['Shear decay parameters']['Critical shear stress (τc)'] = self.results[
                'Shear decay parameters']['Critical shear stress (τc)']
            results_vars['Shear decay parameters']['Manning n'] = self.results['Shear decay parameters']['Manning n']

        return {'Results': results_vars}

    def get_diameter_for_passing(self, passing):
        """Returns the Percent passing for a given diameter.

        Args:
            Percent passing (float): Percent passing for the given diameter

        Returns:
            diameter (float): diameter of the sediment particle
        """
        gradation = self.input['Gradation'].get_val()
        num_items = gradation.input['Table options'].get_val().input['Number of items'].get_val()
        percent_passing = gradation.input['Data input'].get_val().input['Percent passing'].get_val()[:num_items]
        sediment_particle_diameter = \
            gradation.input['Data input'].get_val().input['Sediment particle diameter'].get_val()[:num_items]
        if passing in percent_passing:
            index = percent_passing.index(passing)
            return sediment_particle_diameter[index]
        else:
            y_vals = sediment_particle_diameter
            x_vals = percent_passing

            # Check if all values in x_vals or y_vals are zero
            if all(val == 0 for val in x_vals):
                return 0.0
            if all(val == 0 for val in y_vals):
                return 0.0

            paired_lists = list(zip(x_vals, y_vals))
            paired_lists.sort()
            x_vals_sorted, y_vals_sorted = zip(*paired_lists)

            _, null_data = self.get_setting('Null data')
            _, zero_tol = self.get_setting('Zero tolerance')
            interpolator = Interpolation(x_vals_sorted, y_vals_sorted, app_data=self.app_data, null_data=null_data,
                                         zero_tol=zero_tol)
            return float(interpolator.interpolate_y(passing)[0])

    def get_passing_for_diameter(self, diameter):
        """Returns the Percent passing for a given diameter.

        Args:
            diameter (float): diameter of the sediment particle

        Returns:
            Percent passing (float): Percent passing for the given diameter
        """
        gradation = self.input['Gradation'].get_val()
        num_items = gradation.input['Table options'].get_val().input['Number of items'].get_val()
        percent_passing = gradation.input['Data input'].get_val().input['Percent passing'].get_val()[:num_items]
        sediment_particle_diameter = \
            gradation.input['Data input'].get_val().input['Sediment particle diameter'].get_val()[:num_items]
        if diameter in sediment_particle_diameter:
            index = sediment_particle_diameter.index(diameter)
            return percent_passing[index]
        else:
            y_vals = percent_passing
            x_vals = sediment_particle_diameter

            # Check if all values in x_vals or y_vals are zero
            if all(val == 0 for val in x_vals):
                return 0.0
            if all(val == 0 for val in y_vals):
                return 0.0

            paired_lists = list(zip(x_vals, y_vals))
            paired_lists.sort()
            x_vals_sorted, y_vals_sorted = zip(*paired_lists)

            _, null_data = self.get_setting('Null data')
            _, zero_tol = self.get_setting('Zero tolerance')
            interpolator = Interpolation(x_vals_sorted, y_vals_sorted, app_data=self.app_data, null_data=null_data,
                                         zero_tol=zero_tol)
            passing, _ = interpolator.interpolate_y(diameter, extrapolate=True)
            if passing > 1.0:
                passing = 1.0
            return passing

    def get_critical_shear_stress(self):
        """Returns the critical shear stress.

        Returns:
            critical shear stress (float): critical shear stress
        """
        return self.results['Shear decay parameters']['Critical shear stress (τc)'].get_val()

    def get_d50(self):
        """Returns the D50 of the gradation.

        Returns:
            D50 (float): D50 of the gradation
        """
        if self.input['Soil type'].get_val() == 'Noncohesive':
            return self.get_diameter_for_passing(0.5)
        return 0.0

    def get_d84(self):
        """Returns the D84 of the gradation.

        Returns:
            D84 (float): D84 of the gradation
        """
        return self.get_diameter_for_passing(0.84)

    def get_plot_options(self, plot_name):
        """Get the plot options.

        Args:
            plot_name (str): name of the plot

        Returns:
            dict: plot options
        """
        plot_options = self.input['Gradation'].get_val().get_plot_options(plot_name)

        return plot_options

    # def setup_plot_data(self, axes, index, show_axis_titles, plot_name, can_compute=None):
    #     """Setup the plot data.

    #     Args:
    #         axes (AxesSubplot): The plot axes
    #         index (int): index of flow to display
    #         show_axis_titles (bool): whether to show the titles of the axes
    #         plot_name (str): name of the plot
    #         can_compute (bool): whether we can compute
    #     """
    #     self.input['Gradation'].get_val().compute_data()
    #     self.input['Gradation'].get_val().setup_plot_data(axes, index, show_axis_titles, plot_name, can_compute)
