"""CalcData for performing Gradually Varied Flow operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.calculator.table_data import TableData
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.channel_geometry_data import ChannelGeometryData
from xms.HydraulicToolboxCalc.hydraulics.GVF.gvf_calc import GVFCalc
from xms.HydraulicToolboxCalc.hydraulics.GVF.specify_boundary_data import SpecifyBoundaryData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.composite_manning_n_data import CompositeManningNData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_data import ManningNData


class GVFData(CalcData):
    """A class that defines a channel and performs Manning's n computations."""

    def __init__(self, clear_my_own_results=True, geometry=None, app_data=None, model_name=None, project_uuid=None):
        """Initializes the GVF calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            geometry (ChannelGeometryCalc): Sets the geometry of the channel geometry of the input variable
            flows (list of floats): sets the user list of flows for the input variable
            app_data (AppData): application data (settings, etc.)
            model_name (str): name of the model
            project_uuid (str): the project uuid
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Gradually-Varied Flow (GVF) Calculator"
        self.type = "GVFCalc"
        self.class_name = 'Gradually-Varied Flow (GVF) Calculator'

        self.calc_support_dual_input = False

        self.calculator = GVFCalc()

        # Input
        self.input['Flows'] = Variable(
            'Flow(s)', 'UserArray', UserArray(2, ['flow'], 'cfs', us_units=self.us_flow, si_units=self.si_flow,
                                              select_name='specify flow(s) as:', name_append='flow'))

        if geometry is None:
            geometry = ChannelGeometryData(stand_alone_calc=False, app_data=app_data, model_name=model_name,
                                           project_uuid=project_uuid)
        geometry.stand_alone_calc = False
        self.input['Geometry'] = Variable('Geometry', 'class', geometry)

        self.input['Composite n'] = Variable(
            "Manning's n", 'class',
            CompositeManningNData(shape='circle', open_shape=True, embedment_present=False, gradation_present=False,
                                  total_embedment=0.0, stations=[], elevations=[], app_data=app_data,
                                  model_name=model_name, project_uuid=project_uuid))

        self.input['Upstream invert station'] = Variable(
            'Upstream invert station', 'float', 0.0, [], precision=2,
            limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Upstream invert elevation'] = Variable(
            'Upstream invert elevation', 'float', 0.0, [], precision=2,
            limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Upstream water depth'] = Variable(
            'Upstream water depth', 'class',
            SpecifyBoundaryData(boundary_name='upstream', allow_normal_depth=False, app_data=app_data,
                                model_name=model_name, project_uuid=project_uuid))

        self.input['Downstream invert station'] = Variable(
            'Downstream invert station', 'float', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Downstream invert elevation'] = Variable(
            'Downstream invert elevation', 'float', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Downstream water depth'] = Variable(
            'Downstream water depth', 'class',
            SpecifyBoundaryData(boundary_name='downstream', allow_normal_depth=True, app_data=app_data,
                                model_name=model_name, project_uuid=project_uuid))

        self.input['Display channel slope as flat'] = Variable(
            'Display channel slope as flat', 'bool', False)

        self.input['Selected profile'] = Variable(
            'Selected profile', 'int', 0, [],
            note='This selects which flow profile to plot and show in the specified results.')
        self.plot_index_set = False

        self.unknown = None

        # Plot options
        self.input['Plot options'] = {}
        self.add_plot_options()

        # Intermediate
        self.compute_prep_functions.extend([])
        self.compute_finalize_functions.extend([self.assign_profile_results, self.set_plotting_data])
        self.intermediate_to_copy.extend([
            'direction', 'starting_depth', 'starting_velocity', 'up_starting_depth', 'up_starting_velocity',
            'down_starting_depth', 'down_starting_velocity', 'tw_depth', 'increment', 'wse_stations', 'wse_elevations',
            'froude', 'increase_in_depth', 'total_length', 'start_station', 'slope', 'slope_type', 'normal_depths',
            'normal_depth', 'normal_velocity', 'critical_depths', 'critical_depth', 'brink_depth', 'brink_depth_edr',
            'critical_flowarea', 'critical_velocity', 'flows', 'flow', 'min_embedment_depth', 'channel_is_embedded',
            'upstream_pressure_head', 'upstream_depth', 'upstream_flow_area', 'upstream_velocity', 'upstream_energy',
            'downstream_depth', 'downstream_flow_area', 'downstream_velocity', 'downstream_energy',
            'accept_low_results',
            # lists for direct step point data
            'intermediate_results', 'depth', 'flow_area', 'wetted_perimeter', 'top_width', 'hydraulic_radius',
            'manning_n', 'velocity', 'energy', 'energy_loss', 'energy_slope', 'friction_slope_ave', 'delta_x',
            'x', 'y', 'sequent_depth', 'froude', 'boundary_shear_stress', 'max_shear_stress', 'results_upstream',
            'results_downstream', 'plot_x', 'plot_y',
            # If this is a downstream segement, we may need to accept the profile starting upstream to downstream
            'enforce_upstream_depth',
            # Vena Contracta
            'compute_vena_contracta', 'vena_contracta_length', 'vena_contracta_depth', 'vena_contracta_x',
            'vena_contracta_y', 'vena_contracta_lengths', 'vena_contracta_y', 'vena_contracta_xs',
            'vena_contracta_ys', 'vena_contracta_coefficient',
            # Hydraulic Jump
            'ignore_energy_in_full_flow_calcs', 'hyd_jump_station', 'hyd_jump_depth', 'hyd_jump_sequent_depth',
            'hyd_jump_length', 'hyd_jump_type_b_over_break', 'hyd_jump_exists', 'hyd_jump_swept_out', 'froude_jump',
            'hyd_jump_station_absolute', 'hyd_jump_final_station_absolute', 'hyd_jump_elevation',
            'hyd_jump_sequent_elevation', 'join_curves', 'join_station',
            'hyd_jump_stations', 'hyd_jump_depths', 'hyd_jump_sequent_depths', 'hyd_jump_lengths',
            'hyd_jump_type_b_over_breaks', 'hyd_jump_exists_list', 'hyd_jump_swept_outs', 'froude_jumps',
            'froude_jumps', 'hyd_jump_station_absolutes', 'hyd_jump_final_station_absolutes', 'hyd_jump_elevations',
            'hyd_jump_sequent_elevations', 'froude_class', 'froude_classes', 'froude_class_note', 'froude_class_notes',
            'full_hyd_jump_length', 'full_hyd_jump_sequent_depth', 'full_hyd_jump_lengths',
            'full_hyd_jump_sequent_depths',
            # Full flow
            'inlet_control_depth', 'closed_shape', 'full_flow', 'full_flow_area', 'full_flow_hydraulic_radius',
            'full_flow_velocity', 'full_flow_n', 'span', 'rise', 'length_flowing_full', 'flowing_full', 'ke',
            'headloss', 'outlet_control_depth',

            'inlet_invert_elevation', 'outlet_invert_elevation', 'vel_outlet', 'depth_outlet',
            'show_oscillitary_flow_note', 'manning_n_calc',
            # settings
            'null_data', 'zero_tol', 'flow_profile', 'flow_profiles', 'usgs_type', 'usgs_types',
            'use_direct_step'
        ])

        self.use_direct_step = True

        manning_n_data = ManningNData(app_data=app_data, model_name=model_name, project_uuid=project_uuid)
        manning_n_data.prepare_for_compute()
        manning_n_data.prepare_input_dict()
        self.manning_n_calc = manning_n_data.calculator

        self.direction = 'downstream'  # 'upstream' or 'downstream'
        self.starting_depth = 0.0
        self.starting_velocity = 0.0
        self.up_starting_depth = 0.0
        self.up_starting_velocity = 0.0
        self.down_starting_depth = 0.0
        self.down_starting_velocity = 0.0
        self.tw_depth = 0.0
        self.increment = 0.02
        self.wse_stations = []
        self.wse_elevations = []
        self.froude = 0.0
        self.increase_in_depth = False  # this is along computational direction
        self.total_length = 0.0
        self.start_station = 0.0
        self.slope = 0.0
        self.slope_type = ''
        self.normal_depths = []
        self.normal_depth = 0.0
        self.normal_velocity = 0.0
        self.critical_depths = []
        self.critical_depth = 0.0
        self.brink_depth = 0.0
        self.brink_depth_edr = 0.8
        self.critical_flowarea = 0.0
        self.critical_velocity = 0.0
        self.flows = 0.0
        self.flow = 0.0
        self.flow_profile = ''
        self.flow_profiles = []
        self.usgs_type = None
        self.usgs_types = []

        # The next variable is not the full embedment depth!
        # It is the min embedment depth ( at the thalweg of low flow channel)
        # So it will give the invert of the embeded channel
        self.min_embedment_depth = 0.0
        self.channel_is_embedded = False

        self.upstream_pressure_head = 0.0
        self.upstream_depth = 0.0
        self.upstream_flow_area = 0.0
        self.upstream_velocity = 0.0
        self.upstream_energy = 0.0
        self.downstream_depth = 0.0
        self.downstream_flow_area = 0.0
        self.downstream_velocity = 0.0
        self.downstream_energy = 0.0

        self.accept_low_results = False
        # lists for direct step point data
        self.intermediate_results = {
            'Depth': [],
            'flow_area': [],
            'wetted_perimeter': [],
            'top_width': [],
            'hydraulic_radius': [],
            'manning_n': [],
            'Velocity': [],
            'Energy': [],
            'energy_loss': [],
            'energy_slope': [],
            'friction_slope_ave': [],
            'delta_x': [],
            'x': [],
            'y': [],
            'sequent_depth': [],
            'froude': [],
            'boundary_shear_stress': [],
            'max_shear_stress': [],
        }

        self.depth = []
        self.flow_area = []
        self.wetted_perimeter = []
        self.top_width = []
        self.hydraulic_radius = []
        self.manning_n = []
        self.velocity = []
        self.energy = []
        self.energy_loss = []
        self.energy_slope = []
        self.friction_slope_ave = []
        self.delta_x = []
        self.x = []
        self.y = []
        self.sequent_depth = []
        self.froude = []
        self.boundary_shear_stress = []
        self.max_shear_stress = []

        self.results_upstream = {}
        self.results_downstream = {}

        self.plot_x = []
        self.plot_y = []

        self.upstream_depths = []
        self.upstream_flow_areas = []
        self.upstream_velocities = []
        self.upstream_energies = []

        self.downstream_depths = []
        self.downstream_flow_areas = []
        self.downstream_velocities = []
        self.downstream_energies = []

        # If this is a downstream segement, we may need to accept the profile starting upstream to downstream
        self.enforce_upstream_depth = False

        # Vena Contracta
        self.compute_vena_contracta = False
        self.vena_contracta_length = 0.0
        self.vena_contracta_depth = 0.0
        self.vena_contracta_x = []
        self.vena_contracta_y = []

        self.vena_contracta_lengths = []
        self.vena_contracta_depths = []
        self.vena_contracta_xs = []
        self.vena_contracta_ys = []
        self.vena_contracta_coefficient = 0.7

        # Hydraulic Jump
        self.ignore_energy_in_full_flow_calcs = False
        self.join_curves = False
        self.join_station = 0.0
        self.hyd_jump_station = 0.0
        self.hyd_jump_depth = 0.0
        self.hyd_jump_sequent_depth = 0.0
        self.hyd_jump_length = 0.0
        self.hyd_jump_type_b_over_break = False
        self.hyd_jump_exists = False
        self.hyd_jump_swept_out = False
        self.froude_jump = False
        self.hyd_jump_station_absolute = 0.0
        self.hyd_jump_final_station_absolute = 0.0
        self.hyd_jump_elevation = 0.0
        self.hyd_jump_sequent_elevation = 0.0
        self.full_hyd_jump_length = self.hyd_jump_length
        self.full_hyd_jump_sequent_depth = self.hyd_jump_sequent_depth

        self.hyd_jump_stations = []
        self.hyd_jump_depths = []
        self.hyd_jump_sequent_depths = []
        self.hyd_jump_lengths = []
        self.hyd_jump_type_b_over_breaks = []
        self.hyd_jump_exists_list = []
        self.hyd_jump_swept_outs = []
        self.froude_jumps = []
        self.hyd_jump_station_absolutes = []
        self.hyd_jump_final_station_absolutes = []
        self.hyd_jump_elevations = []
        self.hyd_jump_sequent_elevations = []
        self.froude_class = ''
        self.froude_classes = []
        self.froude_class_note = ''
        self.froude_class_notes = []
        self.full_hyd_jump_lengths = []
        self.full_hyd_jump_sequent_depths = []

        # Full flow
        self.inlet_control_depth = 0.0
        self.closed_shape = False
        self.full_flow = 0.0
        self.full_flow_area = 0.0
        self.full_flow_hydraulic_radius = 0.0
        self.full_flow_velocity = 0.0
        self.full_flow_n = 0.0
        self.span = 0.0
        self.rise = 0.0
        self.length_flowing_full = 0.0
        self.flowing_full = False
        self.ke = 0.0
        self.headloss = 0.0
        self.outlet_control_depth = 0.0

        self.inlet_invert_elevation = 0.0
        self.outlet_invert_elevation = 0.0
        self.vel_outlet = 0.0
        self.depth_outlet = 0.0
        self.show_oscillitary_flow_note = False

        self.zero_tol = 1e-6
        self.null_data = -9999.0

        # Results
        self.clear_my_own_results = clear_my_own_results
        self.results = {}

        self.results['Flow profile'] = Variable(
            'Flow profile', 'string_list', 0, [], complexity=1)

        self.results['Normal depth'] = Variable(
            'Normal depth', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Critical depth'] = Variable(
            'Critical depth', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Downstream depth'] = Variable(
            'Downstream depth', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Downstream velocity'] = Variable(
            'Downstream velocity', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft/s', us_units=self.us_velocity, si_units=self.si_velocity)
        self.results['Length flowing full'] = Variable(
            'Length flowing full', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Length flowing free'] = Variable(
            'Length flowing full', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        # TODO: Add total length, slope, slope type, to the output message

        self.results['Water surface profile'] = {}
        self.results['Water surface profile']['Station'] = Variable(
            'Station', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Water surface profile']['WSE'] = Variable(
            'WSE', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Water surface profile']['Distance from inlet'] = Variable(
            'Distance from inlet', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Water surface profile']['Depth'] = Variable(
            'Depth', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Water surface profile']['Flow area'] = Variable(
            'Flow area', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft^2',
            us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])
        self.results['Water surface profile']['Wetted perimeter'] = Variable(
            'Wetted perimeter', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Water surface profile']['Top width'] = Variable(
            'Top width', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Water surface profile']['Hydraulic radius'] = Variable(
            'Hydraulic radius', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Water surface profile']['Manning n'] = Variable(
            'Critical depth', 'float_list', 0.0, [], precision=4, unit_type=['coefficient'], native_unit='coefficient',
            us_units=[['coefficient']], si_units=[['coefficient']])
        self.results['Water surface profile']['Velocity'] = Variable(
            'Velocity', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft/s',
            us_units=[['yd/s', 'ft/s', 'in/s']], si_units=[['m/s', 'mm/s']])
        self.results['Water surface profile']['Energy'] = Variable(
            'Energy', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Water surface profile']['Energy loss'] = Variable(
            'Energy loss', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Water surface profile']['Energy slope'] = Variable(
            'Energy slope', 'float_list', 0.0, limits=[-10.0, 10.0], precision=6, unit_type=['slope'],
            native_unit='ft/ft', us_units=[['ft/ft']], si_units=[['m/m']])
        self.results['Water surface profile']['Boundary shear stress'] = Variable(
            'Boundary shear stress', 'float_list', 0.0, precision=2, unit_type=['slope'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']])
        self.results['Water surface profile']['Max shear stress'] = Variable(
            'Max shear stress', 'float_list', 0.0, precision=2, unit_type=['slope'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']])

        # Profile results
        # structure_color = self.theme['Plot structure color']
        # structure_fill_color = self.theme['Plot structure fill color']

        # soil_color = self.theme['Plot soil color']
        # soil_fill_color = self.theme['Plot soil fill color']

        # cross_section = {
        #     'Station': Variable('Station', 'float_list', 0, [0.0], precision=2,
        #                         limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
        #                         native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length),
        #     'Elevation': Variable('Elevation', 'float_list', 0, [0.0], precision=2,
        #                           limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
        #                           native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        # }

        #   Water Surface Profile Data
        # # Other arguments (if any) can be added to args and kwargs
        # args = ()  # Add positional arguments here if needed
        # kwargs = {'input': copy.deepcopy(self.results['Water surface profile']), 'theme': self.theme,
        #           'name': 'Water surface profile', 'plot_names': [], }
        name = 'Water surface profile data'
        self.results[name] = Variable(name, 'class_list', 0, [])

        self.water_surface_profile_data = TableData(
            self.theme, name=name, plot_names=['Bridge cross-section'],
            input=copy.deepcopy(self.results['Water surface profile']), min_items=0, app_data=app_data,
            model_name=model_name, project_uuid=project_uuid)

        # name, 'class_list', CalcOrVarlist(
        #         WaterSurfaceProfileData, app_data=app_data, model_name=model_name, project_uuid=project_uuid,
        #         show_define_btn=False, default_name='Plot series',
        #         default_plural_name='Plot series', select_one=False, min_number_of_items=0, initial_num_items=1,
        #         show_number=False, show_duplicate=False, show_delete=False, args=args, kwargs=kwargs))

        # Set Colors?
        # self.input[name].get_val().set_plot_series_options(
        #     name, related_index=0, index=0, name=name, x_axis='Station', y_axis='Elevation',
        #     line_color=soil_color, linetype='solid', line_width=1.5, fill_below_line=False,
        #     fill_color=soil_fill_color, pattern='sand')

        self.warnings = []

        # plot
        self.plots['Longitudinal profile'] = {}
        self.plots['Longitudinal profile']['Plot name'] = 'Longitudinal profile'
        self.plots['Longitudinal profile']['Legend'] = 'best'
        self.plots['Longitudinal profile']['index'] = 1

    def add_plot_options(self):
        """Add plot options for a scenario.
        """
        self.station_var = Variable('Station', 'float_list', 0, [0.0], precision=2,
                                    limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                    native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.elevation_var = Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                      limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                      native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input_dict = {}
        self.input_dict['Channel'] = {
            'Station': copy.copy(self.station_var),
            'Elevation': copy.copy(self.elevation_var)
        }

        self.min_y = sys.float_info.max

        # Load theme colors
        self.theme = self.get_theme()
        # channel_bank_color = self.theme['Plot channel bank line color']  # Bank lines, not channel lines

        structure_color = self.theme['Plot structure color']
        structure_fill_color = self.theme['Plot structure fill color']

        # embankment_color = self.theme['Plot embankment color']
        # embankment_fill_color = self.theme['Plot embankment fill color']

        soil_color = self.theme['Plot soil color']
        soil_fill_color = self.theme['Plot soil fill color']

        wsp_color = self.theme['Plot WSP color']
        wsp_fill_color = self.theme['Plot WSP fill color']

        normal_color = self.theme['Plot normal depth color']
        normal_fill_color = self.theme['Plot normal depth fill color']

        critical_color = self.theme['Plot critical depth color']
        # critical_fill_color = self.theme['Plot critical depth fill color']

        hydraulic_jump_color = self.theme['Plot hydraulic jump color']

        vena_contracta_color = self.theme['Plot vena contracta color']

        # Need plot items for thalweg, point properties (toes, abutment? user-defined?) Auto determine?
        # Need plot items for ineffective flow areas, n values?

        self.input['Plot options']['Longitudinal profile'] = Variable(
            'Plot options', 'class', PlotOptions(
                'Longitudinal profile', input_dict=self.input_dict, show_series=True, app_data=self.app_data,
                model_name=self.model_name, project_uuid=self.project_uuid), complexity=1)

        # Lines

        # Series
        self.input['Plot options']['Longitudinal profile'].get_val().set_plot_series_options(
            related_index=0, index=0, x_axis='Station', y_axis='Elevation', name='Channel',
            line_color=structure_color, linetype=None, line_width=4, fill_below_line=None,
            fill_color=structure_fill_color, pattern=None, density=None)
        self.input['Plot options']['Longitudinal profile'].get_val().set_plot_series_options(
            related_index=0, index=1, x_axis='Station', y_axis='Elevation', name='Embedment', line_color=soil_color,
            linetype=None, line_width=2, fill_below_line=None, fill_color=soil_fill_color, pattern='earth',
            density=None)
        self.input['Plot options']['Longitudinal profile'].get_val().set_plot_series_options(
            related_index=0, index=2, x_axis='Station', y_axis='Elevation', name='Water-surface profile',
            line_color=wsp_color, linetype=None, line_width=2, fill_below_line=None,
            fill_color=wsp_fill_color, pattern=None, density=None)
        self.input['Plot options']['Longitudinal profile'].get_val().set_plot_series_options(
            related_index=0, index=3, x_axis='Station', y_axis='Elevation', name='Vena contracta',
            line_color=vena_contracta_color, linetype='dashed', line_width=5, fill_below_line=None, fill_color=None,
            pattern=None, density=None)
        self.input['Plot options']['Longitudinal profile'].get_val().set_plot_series_options(
            related_index=0, index=4, x_axis='Station', y_axis='Elevation', name='Hydraulic jump',
            line_color=hydraulic_jump_color, linetype='dashed', line_width=5, fill_below_line=None, fill_color=None,
            pattern=None, density=None)
        self.input['Plot options']['Longitudinal profile'].get_val().set_plot_series_options(
            related_index=0, index=5, x_axis='Station', y_axis='Elevation', name='Critical depth',
            line_color=critical_color, linetype='dotted', line_width=2, fill_below_line=None, fill_color=None,
            pattern=None, density=None)
        self.input['Plot options']['Longitudinal profile'].get_val().set_plot_series_options(
            related_index=0, index=6, x_axis='Station', y_axis='Elevation', name='Normal depth',
            line_color=normal_color, linetype='dotted', line_width=2.5, fill_below_line=None,
            fill_color=normal_fill_color, pattern=None, density=None)

    def set_plotting_data(self, index=None):
        """Sets up the plotting data for the cross section.

        Args:
            index (int): Not used.
        """
        if index is None:
            index = self.input['Selected profile'].get_val() - 1  # Index for users is 1-based
        self.calculator.set_plotting_data(index)

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        input_vars['Flows'] = self.input['Flows']

        input_vars['Geometry'] = self.input['Geometry']

        input_vars['Composite n'] = self.input['Composite n']

        input_vars['Upstream invert station'] = self.input['Upstream invert station']
        input_vars['Upstream invert elevation'] = self.input['Upstream invert elevation']
        input_vars['Upstream water depth'] = self.input['Upstream water depth']

        input_vars['Downstream invert station'] = self.input['Downstream invert station']
        input_vars['Downstream invert elevation'] = self.input['Downstream invert elevation']
        input_vars['Downstream water depth'] = self.input['Downstream water depth']

        input_vars['Display channel slope as flat'] = self.input['Display channel slope as flat']

        num_calcs = 0
        if 'Flows' in input_vars:
            num_calcs = len(input_vars['Flows'].get_val().get_result())
        elif 'Depths' in input_vars:
            num_calcs = len(input_vars['Depths'].get_val().get_result())
        elif 'WSE' in input_vars:
            num_calcs = len(input_vars['WSE'].get_val().get_result())
        elif 'Width of spread' in input_vars:
            num_calcs = len(input_vars['Width of spread'].get_val().get_result())

        if num_calcs > 1:
            self.input['Selected profile'].limits = (1, num_calcs)
            input_vars['Selected profile'] = self.input['Selected profile']
            if not self.plot_index_set:
                self.input['Selected profile'].set_val(num_calcs)  # Default to last index
                self.plot_index_set = True
        else:
            self.input['Selected profile'].limits = (0, num_calcs)
            self.input['Selected profile'].set_val(0)
            self.plot_index_set = False

        return input_vars

    def get_results_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        result_vars = {}

        if not self.can_compute:
            return result_vars

        # result_vars['Water surface profile'] = {}
        results = self.results['Water surface profile']

        if self.input['Display channel slope as flat'].get_val():
            result_vars['Distance from inlet'] = results['Distance from inlet']
            result_vars['Depth'] = results['Depth']
        else:
            result_vars['Station'] = results['Station']
            result_vars['WSE'] = results['WSE']
        result_vars['Flow area'] = results['Flow area']
        result_vars['Wetted perimeter'] = results['Wetted perimeter']
        result_vars['Top width'] = results['Top width']
        result_vars['Hydraulic radius'] = results['Hydraulic radius']
        result_vars['Velocity'] = results['Velocity']
        result_vars['Energy slope'] = results['Energy slope']
        result_vars['Boundary shear stress'] = results['Boundary shear stress']
        result_vars['Max shear stress'] = results['Max shear stress']

        return result_vars

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
                result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        input_vars = self.get_input_group(unknown)
        if 'Flows' in input_vars:
            num_calcs = len(input_vars['Flows'].get_val().get_result())
        elif 'Depths' in input_vars:
            num_calcs = len(input_vars['Depths'].get_val().get_result())
        elif 'WSE' in input_vars:
            num_calcs = len(input_vars['WSE'].get_val().get_result())
        elif 'Width of spread' in input_vars:
            num_calcs = len(input_vars['Width of spread'].get_val().get_result())

        if num_calcs > 1:
            profile_index = self.input['Selected profile'].get_val() - 1  # Index for users is 1-based
            results_group = self.get_results_group(unknown)
            results_vars['Selected profile results'] = {}
            for item in results_group:
                results_vars['Selected profile results'][item] = copy.copy(results_group[item])
                if len(results_vars['Selected profile results'][item].value_options) > profile_index:
                    results_vars['Selected profile results'][item].set_val(
                        results_vars['Selected profile results'][item].get_val()[profile_index])

            results_vars['Full results'] = self.get_results_group(unknown)
        else:
            results_vars['Results'] = self.get_results_group(unknown)

        return results_vars

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        return self.warnings

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        self.input['Geometry'].get_val().clear_results()

    def assign_profile_results(self):
        """Assigns the results for profile variables."""
        input_vars = self.get_input_group()
        if 'Flows' in input_vars:
            num_calcs = len(input_vars['Flows'].get_val().get_result())
        # elif 'Depths' in input_vars:
        #     num_calcs = len(input_vars['Depths'].get_val().get_result())
        # elif 'WSE' in input_vars:
        #     num_calcs = len(input_vars['WSE'].get_val().get_result())
        # elif 'Width of spread' in input_vars:
        #     num_calcs = len(input_vars['Width of spread'].get_val().get_result())

        if num_calcs > 1:
            # Place results in self.results['Water surface profile data'] Table
            self.results['Water surface profile data'].value = num_calcs
            self.results['Water surface profile data'].value_options = \
                [copy.copy(self.water_surface_profile_data)] * num_calcs
            for item in self.calculator.results:
                if item in self.results['Water surface profile']:
                    for i in range(num_calcs):
                        if len(self.calculator.results[item]) > i:
                            self.results['Water surface profile data'].value_options[i].input[
                                'Data input'].value.input[item].set_val(self.calculator.results[item][i])

        else:  # Single result
            # Place results in self.results['Water surface profile'] dict
            for item in self.calculator.results:
                if item in self.results['Water surface profile'] and len(self.calculator.results[item]) > 0:
                    self.results['Water surface profile'][item].set_val(
                        self.calculator.results[item][0])
