"""Pile Cap Data."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier.Complex.pile_cap_calc import PileCapCalc


class PileCap(VariableGroup):
    """Provides a class that will Handle input for complex pier pile caps."""
    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Specify Pile Caps.

        Args:
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.calculator = PileCapCalc()

        self.input = {}

        self.input['Compute pile cap'] = Variable('Compute pile cap scour component', "bool", True)

        pile_cap_shape_options = ['Square Nose', 'Round Nose', 'Sharp Nose']

        self.input['Pile cap shape (K1)'] = Variable('Pile cap shape (K1)', "list", 0, pile_cap_shape_options)
        self.input['Pile cap width (apc)'] = Variable('Pile cap width (apc)', 'float', 0.0, [], precision=2,
                                                      unit_type=['length'], native_unit='ft',
                                                      us_units=self.us_mid_length, si_units=self.si_mid_length,
                                                      note='horizontal size, parallel to the bridge, not adjusted '
                                                      'for skew')
        self.input['Pile cap length (Lpc)'] = Variable('Pile cap length (Lpc)', 'float', 0.0, [], precision=2,
                                                       unit_type=['length'], native_unit='ft',
                                                       us_units=self.us_mid_length, si_units=self.si_mid_length,
                                                       note='horizontal size, perpendicular to the bridge, not '
                                                       'adjusted for skew')
        self.input['Pile cap thickness (T)'] = Variable('Pile cap thickness (T)', 'float', 0.0, [], precision=2,
                                                        unit_type=['length'], native_unit='ft',
                                                        us_units=self.us_mid_length, si_units=self.si_mid_length,
                                                        note='Vertical size of the pile cap')
        self.input['Initial height of the pile cap above bed (h0)'] = \
            Variable('Initial height of the pile cap above bed (h0)', 'float', 0.0, [], precision=2,
                     limits=[-self.max_value, self.max_value],
                     unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
                     note='Negative or zero when bottom of pile cap (footing) is not exposed')

        # Intermediate
        # self.compute_prep_functions = []  # Functions to call before compute_data
        # Define variables to copy to the calculator
        self.intermediate_to_copy.extend(['is_fdot_complex'])

        self.is_fdot_complex = False

    def get_input_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = {}

        if self.is_fdot_complex and not self.input['Compute pile cap'].get_val():
            input_vars['Compute pile cap'] = self.input['Compute pile cap']
            return input_vars

        input_vars = copy.deepcopy(self.input)

        if not self.is_fdot_complex:
            input_vars.pop('Compute pile cap')

        return input_vars
