"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier.Complex.pier_stem_data import PierStem
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier.Complex.pile_cap_data import PileCap
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier.Complex.pile_group_data import PileGroup
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier_scour_calc import PierScourCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_data import ScourBaseData


class PierScourData(ScourBaseData):
    """A class that defines a pier scour at a bridge contraction."""

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the GVF calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Pier Scour Calculator"
        self.type = "PierScourData"
        self.class_name = 'Pier Scour Calculator'

        self.calculator = PierScourCalc()

        self.calc_support_dual_input = False

        computation_methods = ["'Rule of Thumb' Simple Pier", 'HEC-18 Simple Pier', 'FDOT Simple Pier',
                               'HEC-18 Complex Pier', 'FDOT Complex Pier']

        # FDOT: pier_shape_options = ['Square Nose', 'Round Nose', 'Sharp Nose']
        bed_options = [
            'Clear-Water Scour', 'Plane Bed and Antidune Flow', 'Small Dunes',
            'Medium Dunes', 'Large Dunes'
        ]
        # bed_material_options = ['Sand Size Bed Material', 'Gravel and coarser Bed Material']

        # Input
        self.input['Computation method'] = Variable(
            'Computation method', "list", 1, computation_methods)

        self.input['Angle of attack (K2)'] = Variable(
            'Angle of attack (K2)', 'float', 0.0, [], limits=(0, 90), precision=2, unit_type=['angle'],
            native_unit='°', us_units=['°'], si_units=['°'])

        self.input['Bed condition (K3)'] = Variable(
            'HEC-18 Bed condition (K3)', "list", 0, bed_options, note='Dune height is N/A')

        # self.input['Bed Material (Ks)'] = Variable('Bed Material (Ks)', "list", 0, bed_material_options)

        self.input['Depth upstream of pier (y1)'] = Variable(
            'Depth upstream of pier (y1)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length,
            note='Depth of flow upstream of the influence of the pier, but at approach section. If using the '
            'thalweg as the scour reference, use the depth at the location of the greatest unit discharge. '
            'Otherwise, use point directly upstream of the pier.')

        self.input['Velocity upstream of pier (v1)'] = Variable(
            'Velocity upstream of pier (v1)', 'float', 0.0, [], precision=2, unit_type=['Velocity'], native_unit='ft/s',
            us_units=self.us_velocity, si_units=self.si_velocity,
            note='velocity of flow upstream of the influence of the pier, but at approach section. If using the '
            'thalweg as the scour reference, use the depth at the location of the greatest unit discharge. '
            'Otherwise, use point directly upstream of the pier.')

        # Cohesive
        self.input['Critical velocity (vc)'] = Variable(
            'Critical velocity (vc)', 'float', 0.0, [], precision=2, unit_type=['Velocity'], native_unit='ft/s',
            us_units=self.us_velocity, si_units=self.si_velocity,
            note='Critical velocity for initiation of erosion of the cohesive material in the bed material.')

        self.input['Scour from single event'] = {}
        self.input['Scour from single event']["Manning's n (n)"] = Variable(
            "Manning's n (n)", 'float', 0.0, [], precision=4, unit_type=['coefficient'])

        self.input['Scour from single event']['Initial erosion rate'] = Variable(
            'Initial erosion rate (żi)', 'float', 0.0, [], precision=2, unit_type=['velocity'], native_unit='ft/hr',
            us_units=[['ft/hr']], si_units=[['m/hr']])

        self.input['Scour from single event']['Duration of flow event (t)'] = Variable(
            'Duration of Flow (t)', 'float', 0.0, [], precision=2, unit_type=['time'], native_unit='hr',
            us_units=[['day', 'hr', 'min']], si_units=[['day', 'hr', 'min']])

        self.input['Pier stem parameters'] = Variable('Pier stem parameters', "class", PierStem(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input['Pile cap parameters'] = Variable('Pile cap parameters', "class", PileCap(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input['Pile group parameters'] = Variable('Pile group parameters', "class", PileGroup(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        self.input['Apply wide pier factor (Kw)'] = Variable(
            'Apply wide pier factor (Kw)', "bool", False,
            note='Engineering judgment should be used in applying Kw because it is based on limited'
                 'data from flume experiments. Engineering judgment should take into consideration'
                 'the volume of traffic, the importance of the highway, cost of a failure (potential loss of'
                 'lives and dollars) and the change in cost that would occur if the Kw factor is used.')

        self.input['Safety factor (SF)'] = Variable(
            'Safety factor (SF)', 'float', 1.25, [], precision=2, complexity=3)

        self.input['Alpha value for local shear decay'] = Variable(
            'Alpha value for local shear decay (α)', 'float', 4.37, [], precision=2, complexity=2)

        self.input['Beta value for local shear decay'] = Variable(
            'Beta value for local shear decay (β)', 'float', 1.33, [], precision=2, complexity=2)

        self.input['Use CSU equation'] = Variable('Use CSU equation for max depth', 'bool', False, [])

        self.input['Scour hole geometry'].is_pier_scour_hole = True

        _, gamma_w = self.get_setting('Unit weight of water (γw)')

        self.input['Unit weight of water (γw)'] = Variable(
            'Unit weight of water (γw)', 'float', gamma_w, precision=4, unit_type=["specific_weight"],
            native_unit="lb/ft^3", us_units=[["lb/ft^3"]], si_units=[["kN/m^3", "N/m^3"]])

        self.unknown = None

        # Intermediate
        # self.compute_prep_functions = []  # Functions to call before compute_data
        # Define variables to copy to the calculator
        self.intermediate_to_copy.extend([
            'unit_converter', 'scour_case', 'k3', 'y1', 'v1', 'ys', 'ys_pier', 'ys_pc', 'h1', 'ks', 'kp',
            'contraction_scour_depth', 'max_flow_depth', 'total_scour_depth', 'streambed_elev_prior_to_scour',
            'local_streambed_after_cs_ltd', 'total_scour_elevation', 'wse_x', 'wse_y', 'con_decay_x', 'con_decay_y',
            'con_decay_x_markers', 'con_decay_y_markers', 'is_computing_shear_decay', 'centerline',
            'centerline_streambed', 'channel_location', 'scour_reference_point', 'bh_dict', 'bh_uuid', 'bh',
            'bh_layers', 'layer_index', 'surface_d50', 'scour_depth'])

        self.unit_converter = ConversionCalc(self.app_data)
        self.theme = self.get_theme()

        self.scour_case = 'Case 2'
        self.k3 = 0.0
        self.y1 = 0.0
        self.v1 = 0.0
        self.ys = 0.0
        self.ys_pier = 0.0
        self.ys_pc = 0.0
        self.h1 = 0.0
        self.ks = 0.0
        self.kp = 0.0

        self.contraction_scour_depth = 0.0

        self.max_flow_depth = 0.0
        self.total_scour_depth = 0.0
        self.streambed_elev_prior_to_scour = 0.0
        self.local_streambed_after_cs_ltd = 0.0
        self.total_scour_elevation = 0.0

        self.wse_x = None
        self.wse_y = None

        self.con_decay_x = []
        self.con_decay_y = []
        self.con_decay_x_markers = []
        self.con_decay_y_markers = []

        # Results
        self.clear_my_own_results = clear_my_own_results

        # self.results['Results'] = {}
        self.results['Case1'] = {}
        self.results['Case2'] = {}
        self.results['FDOT Complex'] = {}
        self.results['FDOT Complex']['Pier stem'] = {}
        self.results['FDOT Complex']['Pile cap'] = {}
        self.results['FDOT Complex']['Pile group'] = {}

        # # Approach Gradation
        # self.results['Results']['Contracted D50'] = Variable('Contracted D50 (D50)',
        #                                                    'float',
        #                                                    0.0, [],
        #                                                    precision=2,
        #                                                    unit_type=['length'],
        #                                                    native_unit='mm',
        #                                                    us_units=self.us_short_length,
        #                                                    si_units=self.si_short_length)
        # # Contracted Gradation
        # self.results['Results']['Approach D50'] = Variable(
        #     'Approach D50 (D50)',
        #     'float',
        #     0.0, [],
        #     precision=2,
        #     unit_type=['length'],
        #     native_unit='mm',
        #     us_units=self.us_short_length,
        #     si_units=self.si_short_length)

        # Cohesive
        # self.results['Initial rate of scour'] = Variable(
        #     'Initial rate of scour (żi)',
        #     'float',
        #     0.0, [],
        #     precision=2,
        #     native_unit='ft/hr',
        #     us_units=[['ft/hr']],
        #     si_units=[['m/hr']])
        self.results['Initial shear stress at the pier (τpier)'] = Variable(
            'Initial shear stress at the pier (τpier)', 'float', 0.0, [], precision=2, unit_type=['force'],
            native_unit='lb/ft^2', us_units=[['lb/ft^2']], si_units=[['kN/m^2', 'N/m^2']])

        self.results['Partial scour depth'] = Variable(
            'Scour depth from flow event (y(t))', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Cohesive scour depth (ys)'] = Variable(
            'Cohesive scour depth (ys)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Results']['Pier shape factor (k1)'] = Variable(
            'Pier shape factor (k1)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Results']['Pier angle of attack factor (k2)'] = Variable(
            'Pier angle of attack factor (k2)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Results']['Pier length over width (L/a)'] = Variable(
            'Pier length over width (L/a)', 'float', 0.0, [], precision=2, unit_type=['coefficient'],
            note='If L/a > 12, use 12')

        self.results['Results']['Angle of attack factor (k2)'] = Variable(
            'Angle of attack factor (k2)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Results']['Bed form factor (k3)'] = Variable(
            'Bed form factor (k3)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Results']['Upstream Froude number (Fr1)'] = Variable(
            'Upstream Froude number (Fr1)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Results']['Scour depth (ys)'] = Variable(
            'Scour depth (ys)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Case1']['Scour depth (ys)'] = self.results['Results']['Scour depth (ys)']
        self.results['Case2']['Scour depth (ys)'] = self.results['Results']['Scour depth (ys)']

        self.results['Results']['Scour factor'] = Variable(
            'Scour factor', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Results']['Wide pier factor (kw)'] = Variable(
            'Wide pier factor (kw)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Results']['Critical velocity (vc)'] = Variable(
            'Critical velocity (vc)', 'float', 0.0, [], precision=2, unit_type=['Velocity'], native_unit='ft/s',
            us_units=self.us_velocity, si_units=self.si_velocity)

        self.results['Results']['Pier stem scour component (yspier)'] = Variable(
            'Pier stem scour component (yspier)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Case1']['Pier stem scour component (yspier)'] = \
            self.results['Results']['Pier stem scour component (yspier)']
        self.results['Case2']['Pier stem scour component (yspier)'] = \
            self.results['Results']['Pier stem scour component (yspier)']

        self.results['Results']['Pile cap scour component (yspc)'] = Variable(
            'Pile cap scour component (yspc)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Case1']['Pile cap scour component (yspc)'] = \
            self.results['Results']['Pile cap scour component (yspc)']
        self.results['Case2']['Pile cap scour component (yspc)'] = \
            self.results['Results']['Pile cap scour component (yspc)']

        self.results['Results']['Pile group scour component (yspg)'] = Variable(
            'Pile group scour component (yspg)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Case1']['Pile group scour component (yspg)'] = \
            self.results['Results']['Pile group scour component (yspg)']

        self.results['Results']['Height of the pile cap after stem scour (h2)'] = Variable(
            'Height of the pile cap (footing) bottom above the bed after stem scour (h2)',
            'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Case1']['Height of the pile cap after stem scour (h2)'] = \
            self.results['Results']['Height of the pile cap after stem scour (h2)']
        self.results['Case2']['Height of the pile cap after stem scour (h2)'] = \
            self.results['Results']['Height of the pile cap after stem scour (h2)']

        self.results['Results']['Scour case'] = Variable(
            'Scour case', "string", '', note='Case 1 computes scour on pier stem, pile cap, and exposed pile group;'
            ' Case 2 computes scour on pier stem and pile cap only')

        self.results['Case1']['Pier stem correction factor (khpier)'] = Variable(
            'Pier stem correction factor (khpier)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])
        self.results['Case2']['Pier stem correction factor (khpier)'] = \
            self.results['Case1']['Pier stem correction factor (khpier)']

        # Pile Cap

        self.results['Case1']['Pile cap shape factor (k1)'] = Variable(
            'Pile cap shape factor (k1)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])
        self.results['Case2']['Pile cap shape factor (k1)'] = self.results['Case1']['Pile cap shape factor (k1)']

        self.results['Case1']['Pile cap equivalent width (apc*)'] = Variable(
            'Pile cap equivalent width (apc*)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Case2']['Grain roughness (ks)'] = Variable(
            'Grain roughness (ks)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='mm',
            us_units=self.us_short_length, si_units=self.si_short_length)

        self.results['Case1']['Adjusted depth of flow upstream (y2)'] = Variable(
            'Adjusted depth of flow upstream of the pier, including degradation, '
            'contraction scour and half the pier stem scour (y2)', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Case2']['Adjusted depth of flow upstream (y2)'] = \
            self.results['Case1']['Adjusted depth of flow upstream (y2)']

        self.results['Case1']['Max adjusted depth of flow upstream (max y2)'] = Variable(
            'Max adjusted depth of flow upstream (max y2)', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Case1']['Adjusted velocity of flow upstream (v2)'] = Variable(
            'Average adjusted velocity in vertical of flow approaching the pier (v2)',
            'float', 0.0, [], precision=2, unit_type=['Velocity'], native_unit='ft/s', us_units=self.us_velocity,
            si_units=self.si_velocity)
        self.results['Case2']['Adjusted velocity of flow upstream (v2)'] = \
            self.results['Case1']['Adjusted velocity of flow upstream (v2)']

        self.results['Case1']['Pile cap Froude number (Fr)'] = Variable(
            'Pile cap Froude number (Fr)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Case1']['Wide pile cap factor (kw)'] = Variable(
            'Wide pile cap factor (kw)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])
        self.results['Case2']['Wide pile cap factor (kw)'] = self.results['Case1']['Wide pile cap factor (kw)']

        self.results['Case1']['Pile cap angle of attack factor (k2)'] = Variable(
            'Pile cap angle of attack factor (k2)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        # Pile Group
        self.results['Case1']['Pile nose shape factor (k1)'] = Variable(
            'Pile nose shape factor (k1)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Case1']['Pile group equivalent width (apg*)'] = Variable(
            'Pile group equivalent width (apg*)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Case1']['Adjusted depth for pile group computations (y3)'] = Variable(
            'Adjusted depth for pile group computations (y3)', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Case1']['Max adjusted depth for pile group computations (max y3)'] = Variable(
            'Max adjusted depth for pile group computations (max y3)', 'float', 0.0, [], precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Case1']['Adjusted velocity for pile group computations (v3)'] = Variable(
            'Adjusted velocity for pile group computations (v3)', 'float', 0.0, [], precision=2,
            unit_type=['Velocity'], native_unit='ft/s', us_units=self.us_velocity, si_units=self.si_velocity)

        self.results['Case1']['Pile group height factor (khpg)'] = Variable(
            'Pile group height factor (khpg)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['Case2']['distance from bed to top of footing (yf)'] = Variable(
            'distance from the bed (after degradation, contraction scour, and pier stem scour) to the top of'
            ' the footing (yf)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Case2']['Average velocity below footing (vf)'] = Variable(
            'Average velocity in the flow zone below the top of the footing (vf)',
            'float', 0.0, [], precision=2, unit_type=['Velocity'], native_unit='ft/s', us_units=self.us_velocity,
            si_units=self.si_velocity)

        # FDOT
        self.results['FDOT Complex']['Critical velocity (vc)'] = Variable(
            'Critical velocity (vc)', 'float', 0.0, [], precision=2, unit_type=['Velocity'], native_unit='ft/s',
            us_units=self.us_velocity, si_units=self.si_velocity)

        self.results['FDOT Complex']['Coefficient of pile spacing (ksp)'] = Variable(
            'Coefficient of pile spacing (ksp)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])
        self.results['FDOT Complex']['Coefficient of aligned rows (km)'] = Variable(
            'Coefficient of aligned rows (km)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Shape factor (ks)'] = Variable(
            'Shape factor (ks)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Projection factor (kp)'] = Variable(
            'Projection factor (kp)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['FDOT Complex']['Shape and projection factors (d*)'] = Variable(
            'Shape and projection factors (d*)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['FDOT Complex']['Live-bed peak velocity (v1p)'] = Variable(
            'Live-bed peak velocity (v1p)', 'float', 0.0, [], precision=2, unit_type=['Velocity'], native_unit='ft/s',
            us_units=self.us_velocity, si_units=self.si_velocity)

        self.results['FDOT Complex']['Critical velocity (vc)'] = Variable(
            'Critical velocity (vc)', 'float', 0.0, [], precision=2, unit_type=['Velocity'], native_unit='ft/s',
            us_units=self.us_velocity, si_units=self.si_velocity)

        self.results['FDOT Complex']['FDOT factor 1 (f1)'] = Variable(
            'FDOT factor 1 (f1)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['FDOT factor 2 (f2)'] = Variable(
            'FDOT factor 2 (f2)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['FDOT factor 3 (f3)'] = Variable(
            'FDOT factor 3 (f3)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        # FDOT
        # Pier Stem
        self.results['FDOT Complex']['Pier stem']['Pier projection factor (kpp)'] = Variable(
            'Pier projection factor (kpp)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pier stem']['Pier shape factor (ksp)'] = Variable(
            'Pier shape factor (ksp)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pier stem']['Pier stem vertical position factor (khp)'] = Variable(
            'Pier stem vertical position factor (khp)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pier stem']['Pier length to width factor (klobp)'] = Variable(
            'Pier length to width factor (klobp)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pier stem']['Pile cap effect on pier stem factor (kf)'] = Variable(
            'Pile cap effect on pier stem factor (kf)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pier stem']['Ratio of overhang to projected pier width (f_ratio)'] = Variable(
            'Ratio of overhang to projected pier width (f_ratio)', 'float', 0.0, [], precision=2,
            unit_type=['coefficient'])

        self.results['FDOT Complex']['Pier stem']['Pier effective diameter (dp*)'] = Variable(
            'Pier effective diameter (dp*)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        # Pile Cap
        self.results['FDOT Complex']['Pile cap']['Pile cap projection factor (kppc)'] = Variable(
            'Pile cap projection factor (kppc)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pile cap']['Pile cap shape factor (kspc)'] = Variable(
            'Pile cap shape factor (kspc)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pile cap']['Pile cap vertical position factor (khpc)'] = Variable(
            'Pile cap vertical position factor (khpc)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pile cap']['Pile cap length to width factor (klobpc)'] = Variable(
            'Pile cap length to width factor (klobpc)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pile cap']['Pile cap effective diameter (dpc*)'] = Variable(
            'Pile cap effective diameter (dpc*)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        # Pile Group
        self.results['FDOT Complex']['Pile group']['Pile group shape factor (kpgs)'] = Variable(
            'Pile group shape factor (kpgs)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pile group']['Pile group projection factor (kpgp)'] = Variable(
            'Pile group projection factor (kpgp)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pile group']['Pile group effective projection factor (kpgpe)'] = Variable(
            'Pile group effective projection factor (kpgpe)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pile group']['Pile group vertical position factor (khpg)'] = Variable(
            'Pile group vertical position factor (khpg)', 'float', 0.0, [], precision=2, unit_type=['coefficient'])

        self.results['FDOT Complex']['Pile group']['Pile group effective diameter (dpg*)'] = Variable(
            'Pile group effective diameter (dpg*)', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        # FDOT Complex
        self.results['FDOT Complex']['Safety factor for complex pier effective diameter (sf)'] = Variable(
            'Safety factor for complex pier effective diameter (sf)', 'float', 0.0, [], precision=2,
            unit_type=['coefficient'])

        self.results['FDOT Complex']['Complex pier effective diameter (dcs*)'] = Variable(
            'Complex pier effective diameter (dcs*)', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Shear decay results']['Shear decay alpha'] = Variable(
            'Shear decay alpha coefficient', 'float', 0.0, [], precision=4, unit_type=['coefficient'])
        self.results['Shear decay results']['Shear decay beta'] = Variable(
            'Shear decay beta coefficient', 'float', 0.0, [], precision=4, unit_type=['coefficient'])

        self.results['Shear decay results']['New depth after cs and ltd (yp)'] = Variable(
            'New depth after contraction scour and long-term degradation (yp)', 'float', 0.0, [], precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Shear decay results']['New velocity after cs and ltd (v2)'] = Variable(
            'New velocity after contraction scour and long-term degradation (v2)', 'float', 0.0, [], precision=2,
            unit_type=['length'], native_unit='ft/s', us_units=self.us_velocity, si_units=self.si_velocity)

        # self.results['Results']['Scour depth (ys)'].set_val(self.ys)

        self.warnings = []

        # # Reference Manuals
        # self.reference_pdfs['HEC-18'] = (
        #     'HEC-18, 5th Edition - Evaluating Scour at Bridges.pdf', 167)

        shear_var = Variable('Shear stress', 'float_list', 0, [0.0], precision=2,
                             limits=(0.0, sys.float_info.max), unit_type=['stress'],
                             native_unit='psf', us_units=[['psf']], si_units=[['pa']],)

        elevation_var = Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                 limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                 native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        shear_vs_elevation = {}
        shear_vs_elevation['Shear stress'] = shear_var
        shear_vs_elevation['Elevation'] = elevation_var

        input_dict = {}
        input_dict['Shear decay'] = shear_vs_elevation

        self.input['Plot options'] = {}
        self.input['Plot options']['Shear decay'] = Variable('Shear decay', 'class',
                                                             PlotOptions('Shear decay',
                                                                         input_dict=input_dict,
                                                                         show_series=True, app_data=app_data,
                                                                         model_name=model_name,
                                                                         project_uuid=project_uuid),
                                                             complexity=1)

        name = 'Shear decay'
        shear_decay_color = (128, 0, 128)  # Deep Violet
        # Other options: Deep Violet: (128, 0, 128), Amber/Gold: (255, 170, 0), Orange-Red: (255, 85, 0),
        # Magenta: (220, 20, 140)
        self.input['Plot options'][name].get_val().set_plot_series_options(
            related_index=0, index=0, name=name, x_axis='Shear stress', y_axis='Elevation',
            line_color=shear_decay_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=shear_decay_color)  # need x_var - it is None

        self.input['Plot options'][name].get_val().set_plot_point_options(
            related_index=0, index=0, name=name, point_color=shear_decay_color,)
        #    point_marker=point_marker, marker_size=marker_size)

        grey_rgb = (127, 127, 127)
        self.input['Plot options'][name].get_val().set_plot_line_options(
            index=0, name='Y max', line_alignment='horizontal',
            line_color=grey_rgb, linetype='dotted', line_width=1.5, text_color=grey_rgb,
            labels='Y max')

        self.plot_names = ['Shear decay']
        self.plots = {}
        for name in self.plot_names:
            self.plots[name] = {}
            self.plots[name]['Plot name'] = name
            self.plots[name]['Legend'] = "best"

    def check_for_needed_data(self):
        """Checks for needed data for the calculator to run.

        Else, show popup with warnings and do not run dialog.
        """
        if self.is_computing_shear_decay:
            self.calc_support_plot = True
        else:
            self.calc_support_plot = False

        return True

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        self.update_gradation_lists()

        input_vars['Computation method'] = copy.deepcopy(self.input['Computation method'])

        if self.clear_my_own_results:
            # This means we are a stand alone calculator and we pop the gradation selections
            # input_vars.input['Approach D50'] = self.input['Approach D50']
            input_vars['Contracted D50'] = self.input['Contracted D50']
            if self.gradations:
                # if self.gradations.input['Approach gradation input options'].get_val() == 'Gradations' or \
                #         self.gradations.input['Approach gradations'].get_val().input[
                #             'Approach locations'].get_val() != 'Single gradation for entire cross-section':
                #     input_vars['Selected approach location'] = self.input['Selected approach location']
                if self.gradations.input['Contracted gradation input options'].get_val() == 'Multiple boreholes':
                    input_vars['Selected borehole'] = self.input['Selected borehole']
        else:
            # input_vars['Selected approach location'] = self.input['Selected approach location']
            input_vars['Selected borehole'] = self.input['Selected borehole']

        if self.input['Computation method'].get_val() in ['HEC-18 Complex Pier']:
            self.input['Pier stem parameters'].get_val().is_hec18_complex = True
        else:
            self.input['Pier stem parameters'].get_val().is_hec18_complex = False

        if self.input['Computation method'].get_val() in ['FDOT Simple Pier', 'FDOT Complex Pier']:
            self.input['Pier stem parameters'].get_val().is_fdot = True
        else:
            self.input['Pier stem parameters'].get_val().is_fdot = False

        if self.input['Computation method'].get_val() in ['FDOT Complex Pier']:
            self.input['Pier stem parameters'].get_val().is_fdot_complex = True
            self.input['Pile cap parameters'].get_val().is_fdot_complex = True
            self.input['Pile group parameters'].get_val().is_fdot_complex = True
        else:
            self.input['Pier stem parameters'].get_val().is_fdot_complex = False
            self.input['Pile cap parameters'].get_val().is_fdot_complex = False
            self.input['Pile group parameters'].get_val().is_fdot_complex = False

        input_vars['Angle of attack (K2)'] = self.input['Angle of attack (K2)']

        # cohesive
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            input_vars.pop('Computation method')
            input_vars['Pier stem parameters'] = self.input['Pier stem parameters']
            input_vars['Depth upstream of pier (y1)'] = self.input['Depth upstream of pier (y1)']
            input_vars['Velocity upstream of pier (v1)'] = self.input['Velocity upstream of pier (v1)']
            input_vars['Critical velocity (vc)'] = self.input['Critical velocity (vc)']
            input_vars['Scour from single event'] = self.input['Scour from single event']

        else:
            input_vars['Bed condition (K3)'] = self.input['Bed condition (K3)']
            # self.input['Bed Material (Ks)'] = Variable('Bed Material (Ks)', "list", 0, bed_material_options)
            # Contracted Gradation
            # if self.gradations:
            #     input_vars['Contracted gradation'] = self.input['Contracted gradation']
            #     sel_contracted_uuid = self.input['Contracted gradation'].get_val()
            #     if sel_contracted_uuid is not None and sel_contracted_uuid != "None":
            #         input_vars['Selected borehole'] = self.input['Selected borehole']
            # else:
            #     input_vars['Contracted D50'] = self.input['Contracted D50']
            input_vars['Depth upstream of pier (y1)'] = self.input['Depth upstream of pier (y1)']
            input_vars['Velocity upstream of pier (v1)'] = self.input['Velocity upstream of pier (v1)']

            input_vars['Pier stem parameters'] = copy.deepcopy(self.input['Pier stem parameters'])

            if self.input['Computation method'].get_val() in ['HEC-18 Complex Pier']:
                input_vars['Pile cap parameters'] = copy.deepcopy(self.input['Pile cap parameters'])
                if self.scour_case in ['Case 1', 'Both']:
                    input_vars['Pile group parameters'] = copy.deepcopy(self.input['Pile group parameters'])

            if self.input['Computation method'].get_val() in ['FDOT Complex Pier']:
                input_vars['Pile cap parameters'] = copy.deepcopy(self.input['Pile cap parameters'])
                input_vars['Pile group parameters'] = copy.deepcopy(self.input['Pile group parameters'])

            if self.input['Computation method'].get_val() in ["'Rule of Thumb' Simple Pier"]:
                if 'Contracted D50' in input_vars:
                    input_vars.pop('Contracted D50')
                if 'Selected borehole' in input_vars:
                    input_vars.pop('Selected borehole')
                # if 'Contracted gradation' in input_vars:
                #     input_vars.pop('Contracted gradation')
                if 'Selected contracted location' in input_vars:
                    input_vars.pop('Selected contracted location')
                input_vars.pop('Bed condition (K3)')
                input_vars['Pier stem parameters'].get_val().input.pop('Pier length (Lp)')
                # rule_shapes = ['Round Nose', 'Circular Cylinder', 'Group of Cylinders']
                # input_vars['Pier stem parameters'].get_val().input['Pier shape (K1)'].value_options = rule_shapes

            if self.input['Computation method'].get_val() in ['HEC-18 Simple Pier',]:
                # D50 is not used with current methods
                input_vars['Apply wide pier factor (Kw)'] = self.input['Apply wide pier factor (Kw)']
                if 'Contracted D50' in input_vars:
                    input_vars.pop('Contracted D50')
                # input_vars.pop('Approach D50')
                if self.input['Apply wide pier factor (Kw)'].get_val():
                    # Add it back on, so it shows up at the end
                    input_vars['Contracted D50'] = self.input['Contracted D50']

            if self.input['Computation method'].get_val() in ['FDOT Simple Pier', 'FDOT Complex Pier']:
                # input_vars['Pier shape (K1)'].value_options = ['Square Nose', 'Round Nose', 'Sharp Nose']
                # input_vars['Pier shape (K1)'].name = 'Pier shape (Ks & Kp)'
                input_vars['Angle of attack (K2)'].name = 'Angle of attack'
                if 'Pile group parameters' in input_vars:
                    input_vars['Pile group parameters'].get_val().input.pop('Staggered pile rows')
                input_vars.pop('Bed condition (K3)')

        # if self.clear_my_own_results:
        #     # This means we are a stand alone calculator and we pop the gradation selections
        #     pass
        # else:
        #     # This means we are part of a bridge project and we pop the D50s
        #     input_vars.pop('Contracted D50')
        #     input_vars.pop('Approach D50')

        if not self.is_computing_shear_decay:
            if 'Use CSU equation' in input_vars:
                input_vars.pop('Use CSU equation')
            if 'Alpha value for local shear decay' in input_vars:
                input_vars.pop('Alpha value for local shear decay')
            if 'Beta value for local shear decay' in input_vars:
                input_vars.pop('Beta value for local shear decay')
        else:
            input_vars['Use CSU equation'] = self.input['Use CSU equation']
            input_vars['Alpha value for local shear decay'] = self.input['Alpha value for local shear decay']
            input_vars['Beta value for local shear decay'] = self.input['Beta value for local shear decay']

        input_vars['Plot options'] = {}
        input_vars['Plot options']['Shear decay'] = copy.deepcopy(self.input['Plot options']['Shear decay'])

        self.pop_property_vars(input_vars)

        return input_vars

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        basic_results = {}
        complete_results = {}

        # can_compute = self.get_can_compute()  # Sets the D50s

        if self.gradations:
            if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
                basic_results['Critical shear stress (τc)'] = self.results['Results']['Critical shear stress (τc)']
                complete_results['Critical shear stress (τc)'] = self.results['Results']['Critical shear stress (τc)']
            else:
                # Approach D50 currently not used by any Pier Scour methods
                # basic_results['Approach D50'] = self.results['Results']['Approach D50']
                # complete_results['Approach D50'] = self.results['Results']['Approach D50']
                basic_results['Contracted D50'] = self.results['Results']['Contracted D50']
                complete_results['Contracted D50'] = self.results['Results']['Contracted D50']

        if not self.can_compute:
            results_vars['Basic Results'] = basic_results
            results_vars['Complete Results'] = complete_results
            return results_vars

        # Cohesive
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            basic_results['Cohesive scour depth (ys)'] = self.results['Cohesive scour depth (ys)']
            complete_results['Cohesive scour depth (ys)'] = self.results['Cohesive scour depth (ys)']
            t = self.input['Scour from single event']['Duration of flow event (t)'].get_val()
            y1 = self.input['Depth upstream of pier (y1)'].get_val()
            v1 = self.input['Velocity upstream of pier (v1)'].get_val()
            n = self.input['Scour from single event']["Manning's n (n)"].get_val()
            if self.cohesive_scour_depth > 0.0 and y1 > 0.0 and v1 > 0.0 and n > 0.0:
                basic_results['Initial erosion rate'] = self.results['Initial shear stress at the pier (τpier)']
                complete_results['Initial erosion rate'] = self.results['Initial shear stress at the pier (τpier)']

                z = self.input['Scour from single event']['Initial erosion rate'].get_val()

                if t > 0.0 and z > 0.0:
                    basic_results['Partial scour depth'] = self.results['Partial scour depth']
                    complete_results['Partial scour depth'] = self.results['Partial scour depth']

            results_vars['Basic Results'] = basic_results
            results_vars['Complete Results'] = complete_results
        else:
            if self.input['Computation method'].get_val() in ['FDOT Simple Pier', 'HEC-18 Complex Pier',
                                                              'FDOT Complex Pier']:
                if self.gradations:
                    basic_results['Contracted D50'] = self.results['Results']['Contracted D50']
                    complete_results['Contracted D50'] = self.results['Results']['Contracted D50']

            if self.input['Computation method'].get_val() in ["'Rule of Thumb' Simple Pier"]:
                complete_results['Upstream Froude number (Fr1)'] = self.results['Results'][
                    'Upstream Froude number (Fr1)']

                complete_results['Scour factor'] = self.results['Results'][
                    'Scour factor']

            if self.input['Computation method'].get_val() in ['HEC-18 Simple Pier', 'HEC-18 Complex Pier']:
                complete_results['Pier shape factor (k1)'] = self.results['Results'][
                    'Pier shape factor (k1)']
                complete_results['Pier length over width (L/a)'] = self.results['Results'][
                    'Pier length over width (L/a)']
                complete_results['Angle of attack factor (k2)'] = self.results['Results'][
                    'Angle of attack factor (k2)']
                complete_results['Bed form factor (k3)'] = self.results['Results'][
                    'Bed form factor (k3)']
                complete_results['Upstream Froude number (Fr1)'] = self.results['Results'][
                    'Upstream Froude number (Fr1)']

            if self.input['Computation method'].get_val() in ['HEC-18 Simple Pier']:
                if self.input['Apply wide pier factor (Kw)'].get_val():
                    # basic_results['Critical velocity (vc)'] = self.results['Results']['Critical velocity (vc)']
                    complete_results['Critical velocity (vc)'] = self.results['Results']['Critical velocity (vc)']

                    basic_results['Wide pier factor (kw)'] = self.results['Results']['Wide pier factor (kw)']
                complete_results['Wide pier factor (kw)'] = self.results['Results']['Wide pier factor (kw)']

            if self.input['Computation method'].get_val() in ['HEC-18 Complex Pier']:
                basic_results['Height of the pile cap after stem scour (h2)'] = \
                    self.results['Results']['Height of the pile cap after stem scour (h2)']

                basic_results['Pier stem scour component (yspier)'] = \
                    self.results['Results']['Pier stem scour component (yspier)']
                # complete_results['Pier stem scour component (yspier)'] =
                #   self.results['Results']['Pier stem scour component (yspier)']

                basic_results['Scour case'] = self.results['Results']['Scour case']
                complete_results['Scour case'] = self.results['Results']['Scour case']

                if self.results['Results']['Scour case'].get_val() == 'Case 1':
                    basic_results['Pile cap scour component (yspc)'] = \
                        self.results['Results']['Pile cap scour component (yspc)']

                    basic_results['Pile group scour component (yspg)'] = \
                        self.results['Results']['Pile group scour component (yspg)']

                else:  # Case 2
                    basic_results['Pile cap scour component (yspc)'] = \
                        self.results['Results']['Pile cap scour component (yspc)']
                    # complete_results['Pile cap scour component (yspc)'] = \
                    #     self.results['Results']['Pile cap scour component (yspc)']

                complete_results['Case 1'] = self.results['Case1']
                complete_results['Case 2'] = self.results['Case2']

            if self.input['Computation method'].get_val() in ['FDOT Simple Pier', 'FDOT Complex Pier']:
                basic_results['Shape factor (ks)'] = self.results['FDOT Complex']['Shape factor (ks)']
                complete_results['Shape factor (ks)'] = self.results['FDOT Complex']['Shape factor (ks)']

                basic_results['Projection factor (kp)'] = self.results['FDOT Complex']['Projection factor (kp)']
                complete_results['Projection factor (kp)'] = self.results['FDOT Complex']['Projection factor (kp)']

                # basic_results['Shape and projection factors (d*)'] = \
                # self.results['Results']['Shape and projection factors (d*)']
                complete_results['Shape and projection factors (d*)'] = \
                    self.results['FDOT Complex']['Shape and projection factors (d*)']

                basic_results['Live-bed peak velocity (v1p)'] = \
                    self.results['FDOT Complex']['Live-bed peak velocity (v1p)']
                complete_results['Live-bed peak velocity (v1p)'] = \
                    self.results['FDOT Complex']['Live-bed peak velocity (v1p)']

                basic_results['Critical velocity (vc)'] = self.results['FDOT Complex']['Critical velocity (vc)']
                complete_results['Critical velocity (vc)'] = self.results['FDOT Complex']['Critical velocity (vc)']

                # basic_results['FDOT factor 1 (f1)'] = self.results['FDOT Complex']['FDOT factor 1 (f1)']
                complete_results['FDOT factor 1 (f1)'] = self.results['FDOT Complex']['FDOT factor 1 (f1)']

                # basic_results['FDOT factor 2 (f2)'] = self.results['FDOT Complex']['FDOT factor 2 (f2)']
                complete_results['FDOT factor 2 (f2)'] = self.results['FDOT Complex']['FDOT factor 2 (f2)']

                # basic_results['FDOT factor 3 (f3)'] = self.results['FDOT Complex']['FDOT factor 3 (f3)']
                complete_results['FDOT factor 3 (f3)'] = self.results['FDOT Complex']['FDOT factor 3 (f3)']

            if self.input['Computation method'].get_val() in ['FDOT Complex Pier']:
                basic_results['Pier effective diameter (dp*)'] = \
                    self.results['FDOT Complex']['Pier stem']['Pier effective diameter (dp*)']
                basic_results['Pile cap effective diameter (dpc*)'] = \
                    self.results['FDOT Complex']['Pile cap']['Pile cap effective diameter (dpc*)']
                basic_results['Pile group effective diameter (dpg*)'] = \
                    self.results['FDOT Complex']['Pile group']['Pile group effective diameter (dpg*)']
                basic_results['Safety factor for complex pier effective diameter (sf)'] = \
                    self.results['FDOT Complex']['Safety factor for complex pier effective diameter (sf)']
                basic_results['Complex pier effective diameter (dcs*)'] = \
                    self.results['FDOT Complex']['Complex pier effective diameter (dcs*)']

                if self.input['Pier stem parameters'].get_val().input['Compute pier stem'].get_val():
                    complete_results['Pier stem'] = self.results['FDOT Complex']['Pier stem']
                if self.input['Pile cap parameters'].get_val().input['Compute pile cap'].get_val():
                    complete_results['Pile cap'] = self.results['FDOT Complex']['Pile cap']
                if self.input['Pile group parameters'].get_val().input['Compute pile group'].get_val():
                    complete_results['Pile group'] = self.results['FDOT Complex']['Pile group']

                basic_results['Safety factor for complex pier effective diameter (sf)'] = \
                    self.results['FDOT Complex']['Safety factor for complex pier effective diameter (sf)']
                basic_results['Complex pier effective diameter (dcs*)'] = \
                    self.results['FDOT Complex']['Complex pier effective diameter (dcs*)']

            basic_results['Scour depth (ys)'] = self.results['Results']['Scour depth (ys)']
            complete_results['Scour depth (ys)'] = self.results['Results']['Scour depth (ys)']

            if self.input['Scour hole geometry'].get_val().can_compute:
                basic_results['Bottom width'] = self.input['Scour hole geometry'].get_val().results['Bottom width']
                complete_results['Bottom width'] = self.input['Scour hole geometry'].get_val().results['Bottom width']

                basic_results['Top width'] = self.input['Scour hole geometry'].get_val().results['Top width']
                complete_results['Top width'] = self.input['Scour hole geometry'].get_val().results['Top width']

            if self.is_computing_shear_decay:
                basic_results['Shear decay results'] = copy.deepcopy(self.results['Shear decay results'])
                basic_results['Shear decay results'].pop('Shear decay curve shear')
                basic_results['Shear decay results'].pop('Shear decay curve elevation')
                basic_results['Shear decay results'].pop('Shear decay curve shear markers')
                basic_results['Shear decay results'].pop('Shear decay curve elevation markers')

                complete_results['Shear decay results'] = copy.deepcopy(basic_results['Shear decay results'])

            results_vars['Basic Results'] = basic_results

            if self.input['Computation method'].get_val() in ['HEC-18 Complex Pier']:
                results_vars['Case 1'] = self.results['Case1']
                results_vars['Case 2'] = self.results['Case2']

            if self.input['Computation method'].get_val() in ['FDOT Complex Pier']:
                if self.input['Pier stem parameters'].get_val().input['Compute pier stem'].get_val():
                    results_vars['Pier stem'] = self.results['FDOT Complex']['Pier stem']
                if self.input['Pile cap parameters'].get_val().input['Compute pile cap'].get_val():
                    results_vars['Pile cap'] = self.results['FDOT Complex']['Pile cap']
                if self.input['Pile group parameters'].get_val().input['Compute pile group'].get_val():
                    results_vars['Pile group'] = self.results['FDOT Complex']['Pile group']

            results_vars['Complete Results'] = complete_results

        if self.is_computing_shear_decay:
            results_vars['Shear decay table'] = {}
            results_vars['Shear decay table']['Shear decay curve shear'] = \
                copy.deepcopy(self.results['Shear decay results']['Shear decay curve shear'])
            results_vars['Shear decay table']['Shear decay curve elevation'] = \
                copy.deepcopy(self.results['Shear decay results']['Shear decay curve elevation'])

        return results_vars

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        return self.warnings
