"""Bridge Scour Hole Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_hole_calc import ScourHoleCalc


class ScourHole(VariableGroup):
    """Provides a class that will define the cross-section of a bridge."""

    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Class.

        Args:
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'ScourHole'
        self.type = 'ScourHole'

        self.calculator = ScourHoleCalc()

        # Input
        self.input = {}

        self.input['Scour depth'] = Variable('Scour depth', 'float', 0.0, [], precision=2,
                                             unit_type='length', native_unit='ft',
                                             us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input['Width to depth ratio'] = Variable(
            'Ratio of bottom width to scour hole depth', 'float', 0.0, [], precision=2, unit_type='coefficient',
            note='A ratio of 1.0 means the bottom width is equal to the scour hole depth')

        self.input['Use pier width'] = Variable('Use pier width as bottom width', 'bool', True, [])

        self.input['Angle of repose (Θ)'] = Variable('Angle of repose (Θ)', 'float', math.radians(44), [],
                                                     limits=(math.radians(30), math.radians(44)),
                                                     precision=2, unit_type='angle',
                                                     native_unit='radians',
                                                     us_units=['degrees', 'radians', 'H:1V'],
                                                     si_units=['degrees', 'radians', 'H:1V'])
        self.input['Angle of repose (Θ)'].selected_us_unit = 'degrees'

        # Intermediate
        # Define variables to copy to the calculator
        self.intermediate_to_copy.extend([
            'is_pier_scour_hole', 'pier_width', 'pier_widths', 'streambed_elevation', 'use_specified_b_width',
            'pier_elevations', 'b_width', 'null_data'])
        # self.compute_prep_functions = []  # Functions to call before compute_data
        _, self.null_data = self.get_setting('Null data')
        self.pier_width = 0.0
        self.pier_widths = []
        self.pier_elevations = []
        self.streambed_elevation = self.null_data
        self.is_pier_scour_hole = False
        self.use_specified_b_width = False
        self.b_width = 0.0

        # Results
        self.warnings = []
        self.results = {}
        self.results['Top width'] = Variable('Top width', 'float_list', 0.0, [], precision=2,
                                             unit_type='length', native_unit='ft',
                                             us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Bottom width'] = Variable('Bottom width', 'float_list', 0.0, [], precision=2,
                                                unit_type='length', native_unit='ft',
                                                us_units=self.us_mid_length, si_units=self.si_mid_length)

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}

        input_vars['Angle of repose (Θ)'] = self.input['Angle of repose (Θ)']
        if self.is_pier_scour_hole:
            input_vars['Use pier width'] = self.input['Use pier width']
            if self.input['Use pier width'].get_val():
                input_vars['Use pier width'] = self.input['Use pier width']
        # input_vars['Scour depth'] = self.input['Scour depth']
        input_vars['Width to depth ratio'] = self.input['Width to depth ratio']

        return input_vars

    def clear_results(self):
        """Clears the results dictionary."""
        self.warnings = []

        self.results['Bottom width'].value_options = []
        self.results['Top width'].value_options = []
