"""SiteData Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.culvert.site_calc import SiteCalc


class SiteData(VariableGroup):
    """Provides a class that will define the site data of a culvert barrel."""

    def __init__(self, embedment_depth, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Site Data.

        Args:
            embedment_depth (int or float): The depth of the culvert embedment.
            app_data (AppData): application data (settings, etc
            model_name (str): the name of the model
            project_uuid (str): the project uuid
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Site Data'
        self.type = 'SiteData'

        self.calculator = SiteCalc()

        self.update_inpute_after_compute = True

        assert isinstance(
            embedment_depth,
            (int, float)), "embedment_depth needs to be an int or float."

        max_value = self.max_value
        # Input
        self.embedment_depth = embedment_depth
        culvert_type_list = [
            'straight', 'single broken-back', 'double broken-back',
            'inverted siphon'
        ]
        self.input['Culvert type'] = Variable('Culvert type', "list", 0, culvert_type_list)
        self.input['Entry options'] = Variable(
            'Site data entry options', 'list', 0,
            ['specify streambed elevation', 'specify culvert barrel invert elevations'])
        self.input['Inlet invert station'] = Variable(
            'Inlet invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.input['Inlet invert elevation'] = Variable(
            'Inlet invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.input['Inlet streambed elevation'] = Variable(
            'Inlet streambed elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.input['Break invert station'] = Variable(
            'Break invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.input['Break invert elevation'] = Variable(
            'Break invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.input['Upper break invert station'] = Variable(
            'Upper break invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.input['Upper break invert elevation'] = Variable(
            'Upper break invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.input['Lower break invert station'] = Variable(
            'Lower break invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.input['Lower break invert elevation'] = Variable(
            'Lower break invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.input['Outlet invert station'] = Variable(
            'Outlet invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.input['Outlet invert elevation'] = Variable(
            'Outlet invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.input['Outlet streambed elevation'] = Variable(
            'Outlet streambed elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.input['Identical barrels'] = Variable(
            'number of identical barrels', 'int', 1, 'Including this barrel; must be 1 or greater', limits=(1, 1000))

        # Intermediate
        self.compute_prep_functions.extend([])
        self.compute_finalize_functions.extend([])
        self.intermediate_to_copy.extend([
            'embedment_depth'
        ])

        self.warnings = []
        self.results = {}

        self.results['Inlet invert station'] = Variable(
            'Inlet invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Inlet invert elevation'] = Variable(
            'Inlet invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Inlet streambed elevation'] = Variable(
            'Inlet streambed elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.results['Break invert station'] = Variable(
            'Break invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Break invert elevation'] = Variable(
            'Break invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.results['Upper break invert station'] = Variable(
            'Upper break invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Upper break invert elevation'] = Variable(
            'Upper break invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.results['Lower break invert station'] = Variable(
            'Lower break invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Lower break invert elevation'] = Variable(
            'Lower break invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.results['Outlet invert station'] = Variable(
            'Outlet invert station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Outlet invert elevation'] = Variable(
            'Outlet invert elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Outlet streambed elevation'] = Variable(
            'Outlet streambed elevation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}
        _, zero_tol = self.get_setting('Zero tolerance')

        input_vars['Culvert type'] = self.input['Culvert type']

        if zero_tol < self.embedment_depth:
            input_vars['Entry options'] = self.input['Entry options']

        input_vars['Inlet invert station'] = self.input['Inlet invert station']
        streambed_entry = False
        if zero_tol < self.embedment_depth:
            if self.input['Entry options'].get_val() == 'specify streambed elevation':
                streambed_entry = True
                input_vars['Inlet streambed elevation'] = self.input['Inlet streambed elevation']
        if not streambed_entry:
            input_vars['Inlet invert elevation'] = self.input['Inlet invert elevation']

        if self.input['Culvert type'].get_val() == 'single broken-back':
            input_vars['Break invert station'] = self.input['Break invert station']
            input_vars['Break invert elevation'] = self.input['Break invert elevation']
        elif self.input['Culvert type'].get_val() == 'double broken-back':
            input_vars['Upper break invert station'] = self.input['Upper break invert station']
            input_vars['Upper break invert elevation'] = self.input['Upper break invert elevation']
            input_vars['Lower break invert station'] = self.input['Lower break invert station']
            input_vars['Lower break invert elevation'] = self.input['Lower break invert elevation']

        input_vars['Outlet invert station'] = self.input['Outlet invert station']
        if streambed_entry:
            input_vars['Outlet streambed elevation'] = self.input['Outlet streambed elevation']
        else:
            input_vars['Outlet invert elevation'] = self.input['Outlet invert elevation']

        input_vars['Identical barrels'] = self.input['Identical barrels']

        return input_vars

    def get_slope_of_final_barrel(self):
        """Computes and returns the slope of the final barrel.

        Returns:
            slope (float): the slope of the final barrel.
        """
        if self.calculator.input_dict == {}:
            self.prepare_for_compute()
            self.calculator.input_dict, self.calculator.plot_dict = self.prepare_input_dict()
            self.compute_data()
        return self.calculator.get_slope_of_final_barrel()
