"""StationComponentDialogHelper class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.adcirc.data.adcirc_data import UNINITIALIZED_COMP_ID
from xms.adcirc.gui.assign_station_dlg import AssignStationDlg
from xms.adcirc.gui.station_plots_dialog import StationPlotsDialog


class StationComponentDialogHelper:
    """Helper class for BC component display options."""
    def __init__(self, station_comp):
        """Create a helper class.

        Args:
            station_comp (:obj:`StationComponent`): The component this helper should help

        """
        self.station_comp = station_comp
        self.selected_att_ids = []
        self.selected_comp_ids = []
        self.dlg_message = ''

    def assign_station(self, params, parent):
        """Display the Assign Recording Stations dialog and persist data if accepted.

        Args:
            params (:obj:`dict`): The ActionRequest parameter map
            parent (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        old_comp_id = self._unpack_xms_data(params[0])
        if not self.selected_att_ids:
            return [('INFO', 'No points selected. Select one or more recording stations.')], []

        increment = True
        comp_id = old_comp_id
        if comp_id == UNINITIALIZED_COMP_ID:  # Create some default atts if this is the initial assignement.
            comp_id = self.station_comp.data.add_station_atts()
            increment = False
        station_data = self.station_comp.data.stations.where(
            self.station_comp.data.stations.comp_id == comp_id, drop=True
        )
        if increment:
            comp_id = self.station_comp.data.add_station_atts(station_data)
        dlg = AssignStationDlg(parent, station_data, self.dlg_message)
        if dlg.exec():
            # Update the attribute datasets
            self.station_comp.data.update_station(comp_id, dlg.data)

            # Associate all selected arcs with the new component id.
            for pt_id in self.selected_att_ids:
                self.station_comp.update_component_id(TargetType.point, pt_id, comp_id)

            self.station_comp.data.commit()  # Flush attribute dataset to disk

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.station_comp.main_file), 'temp'), ignore_errors=True)
        return [], []

    def _unpack_xms_data(self, param_map):
        """Unpack the selection info and component id maps sent by XMS.

        Args:
            param_map (:obj:`dict`): The ActionRequest parameter map

        Returns:
            (:obj:`int`): component id of atts to display

        """
        # Get the XMS attribute ids of the selected for_arcs (if any)
        self.selected_att_ids = param_map.get('selection', [])
        if not self.selected_att_ids:
            return UNINITIALIZED_COMP_ID

        # Get the component id map of the selected for_arcs (if any).
        comp_id = UNINITIALIZED_COMP_ID
        if 'id_files' in param_map and param_map['id_files'] and param_map['id_files'][0]:
            files_dict = {'POINT': (param_map['id_files'][0], param_map['id_files'][1])}
            self.station_comp.load_coverage_component_id_map(files_dict)
            comp_id = self._check_number_selected()

        return comp_id

    def _check_number_selected(self):
        """Determine which attributes to display in Assign BC dialog and any warning message that should be added.

        Returns:
            (:obj:`int`): component id of attributes to display

        """
        num_features = len(self.selected_att_ids)
        # 1 point selected, use those atts
        if num_features == 1:
            comp_id = self.station_comp.get_comp_id(TargetType.point, self.selected_att_ids[0])
            self.selected_comp_ids.append(comp_id)
            return comp_id
        else:  # More than one point selected
            # Get the component ids of all the selected arcs
            try:
                self.selected_comp_ids = list(
                    self.station_comp.comp_to_xms[self.station_comp.cov_uuid][TargetType.point].keys()
                )
            except KeyError:
                return UNINITIALIZED_COMP_ID  # No component ids assigned for any of the selected points

            self.dlg_message = 'Multiple recording station points selected. Changes will apply to all selected points.'
            return UNINITIALIZED_COMP_ID

    def station_plots(self, query, params, win_cont):
        """Opens the dialog to view station point solution plots.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
           (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        ids = params[0].get('selection', [])
        if len(ids) != 1:
            msg = 'One (and only one) point must be selected to view solution plots.'
            return [('INFO', msg)], []

        dlg = StationPlotsDialog(win_cont, ids[0], self.station_comp.cov_uuid, query.project_tree)
        dlg.exec()
        return [], []
