"""BcData class."""

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as io_util

# 4. Local modules
from xms.adcirc.__version__ import version
from xms.adcirc.data.adcirc_data import check_for_object_strings_dumb, UNINITIALIZED_COMP_ID

UNASSIGNED_INDEX = 0
OCEAN_INDEX = 1
MAINLAND_INDEX = 2
ISLAND_INDEX = 3
RIVER_INDEX = 4
LEVEE_OUTFLOW_INDEX = 5
LEVEE_INDEX = 6
RADIATION_INDEX = 7
ZERO_NORMAL_INDEX = 8
FLOW_AND_RADIATION_INDEX = 9

ESSENTIAL_INDEX = 0
NATURAL_INDEX = 1

NODE_VERT_ELEVS = 0
PARAMETRIC_CURVE = 2

# When mapping to a simulation, the new component a copy will be made of the source BC data file named this in the
# mapped component's directory.
BC_MAIN_FILE = 'bc_comp.nc'


def default_arc_atts(fill_num=None, coords=None) -> xr.Dataset:
    """Get some default arc attributes. Free method so we can clear the levee dataset in the BC unmapper.

    Args:
        fill_num (int): If provided, will fill the default dataset with this many rows. If provided, must provide
            `coords`.
        coords (dict): If provided, will fill the default dataset with these coords. If provided, must provide
            `fill_num`.

    Returns:
        (xr.Dataset): Empty arc dataset
    """
    if fill_num:
        bc_table = {
            'type': ('comp_id', [0.0] * fill_num),
            'bc_option': ('comp_id', [0.0] * fill_num),
            'tang_slip': ('comp_id', [0.0] * fill_num),
            'galerkin': ('comp_id', [0.0] * fill_num),
            'use_elevs': ('comp_id', [0.0] * fill_num),
            'subcritical_coeff': ('comp_id', [1.0] * fill_num),
            'supercritical_coeff': ('comp_id', [1.0] * fill_num),
            'constant_q': ('comp_id', [50.0] * fill_num),
            'wse_offset': ('comp_id', [0.0] * fill_num),
        }
    else:
        bc_table = {
            'type': ('comp_id', []),
            'bc_option': ('comp_id', []),
            'tang_slip': ('comp_id', []),
            'galerkin': ('comp_id', []),
            'use_elevs': ('comp_id', []),
            'subcritical_coeff': ('comp_id', []),
            'supercritical_coeff': ('comp_id', []),
            'constant_q': ('comp_id', []),
            'wse_offset': ('comp_id', []),
        }
        coords = {'comp_id': []}
    return xr.Dataset(data_vars=bc_table, coords=coords)


def default_levee_atts() -> xr.Dataset:
    """Get some default levee attributes. Free method so we can clear the levee dataset in the BC unmapper.

    Returns:
        (:obj:`xarray.Dataset`): Empty levee dataset
    """
    # One curve per side on each side of the levee.
    levee_table = {
        'Parametric __new_line__ Length': ('comp_id', []),
        'Parametric __new_line__ Length 2': ('comp_id', []),
        'Zcrest (m)': ('comp_id', []),
        'Subcritical __new_line__ Flow Coef': ('comp_id', []),
        'Supercritical __new_line__ Flow Coef': ('comp_id', []),
    }
    return xr.Dataset(data_vars=levee_table, coords={'comp_id': []})


def default_levee_flags(fill_num=None, coords=None) -> xr.Dataset:
    """Get some default levee flags. Free method so we can clear the levee dataset in the BC unmapper.

    Args:
        fill_num (:obj:`int`): If provided, will fill the default dataset with this many rows. If provided, must provide
            `coords`.
        coords (:obj:`dict`): If provided, will fill the default dataset with these coords. If provided, must provide
            `fill_num`.

    Returns:
        (:obj:`xarray.Dataset`): Empty levee flag dataset
    """
    # One curve per side on each side of the levee.
    if fill_num:
        levee_flag_table = {
            'use_second_side': ('comp_id', np.array([0] * fill_num, dtype=np.int32)),
            'locs': ('comp_id', np.array([''] * fill_num, dtype=object)),
        }
    else:
        levee_flag_table = {
            'use_second_side': ('comp_id', np.array([], dtype=np.int32)),
            'locs': ('comp_id', np.array([], dtype=object)),
        }
        coords = {'comp_id': []}
    return xr.Dataset(data_vars=levee_flag_table, coords=coords)


def default_q() -> xr.Dataset:
    """Get some default river attributes. Free method so we can clear the levee dataset in the BC unmapper.

    Returns:
        (:obj:`xarray.Dataset`): Empty levee dataset
    """
    q_table = {
        'Time': ('comp_id', []),
        'Flow': ('comp_id', []),
    }
    return xr.Dataset(data_vars=q_table, coords={'comp_id': []})


def default_pipe_atts() -> xr.Dataset:
    """Get some default pipe attributes. Free method so we can clear the pipe dataset in the BC unmapper.

    Returns:
        (:obj:`xarray.Dataset`): Empty pipe dataset
    """
    pipe_table = {
        'Height': ('comp_id', []),
        'Coefficient': ('comp_id', []),
        'Diameter': ('comp_id', []),
    }
    return xr.Dataset(data_vars=pipe_table, coords={'comp_id': []})


class BcData(XarrayBase):
    """Manages data file for the boundary conditions coverage hidden component."""
    def __init__(self, data_file):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        # Initialize member variables before calling super so they are available for commit() call
        self._filename = data_file
        self._info = None
        self._arcs = None
        self._levees = None
        self._levee_flags = None
        self._pipes = None
        self._flow_cons = None
        self._q = None
        self.deleted_comp_ids = set()  # Set this to be a list of deleted component ids in the coverage before vacuuming
        # Create the default file before calling super because we have our own attributes to write.
        super().__init__(data_file)
        self._get_default_datasets(data_file)

    def update_bc(self, comp_id, new_atts):
        """Update the BC attributes of a boundary condition.

        Args:
            comp_id (:obj:`int`): Component id of the BC to update
            new_atts (:obj:`xarray.Dataset`): The new attributes for the BC
        """
        self.arcs['type'].loc[dict(comp_id=[comp_id])] = new_atts['type'].values
        self.arcs['bc_option'].loc[dict(comp_id=[comp_id])] = new_atts['bc_option'].values
        self.arcs['tang_slip'].loc[dict(comp_id=[comp_id])] = new_atts['tang_slip'].values
        self.arcs['galerkin'].loc[dict(comp_id=[comp_id])] = new_atts['galerkin'].values
        self.arcs['use_elevs'].loc[dict(comp_id=[comp_id])] = new_atts['use_elevs'].values
        self.arcs['subcritical_coeff'].loc[dict(comp_id=[comp_id])] = new_atts['subcritical_coeff'].values
        self.arcs['supercritical_coeff'].loc[dict(comp_id=[comp_id])] = new_atts['supercritical_coeff'].values
        self.arcs['constant_q'].loc[dict(comp_id=[comp_id])] = new_atts['constant_q'].values
        self.arcs['wse_offset'].loc[dict(comp_id=[comp_id])] = new_atts['wse_offset'].values

    def update_pipe(self, comp_id, new_atts):
        """Update the pipe attributes of a point.

        Args:
            comp_id (:obj:`int`): Component id of the pipe point to update
            new_atts (:obj:`xarray.Dataset`): The new attributes for the pipe
        """
        self.pipes['Height'].loc[dict(comp_id=[comp_id])] = new_atts['Height']
        self.pipes['Coefficient'].loc[dict(comp_id=[comp_id])] = new_atts['Coefficient']
        self.pipes['Diameter'].loc[dict(comp_id=[comp_id])] = new_atts['Diameter']

    @property
    def arcs(self):
        """Load the arcs dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the arcs dataset in the main file
        """
        if self._arcs is None:
            self._arcs = self.get_dataset('arcs', False)
        return self._arcs

    @arcs.setter
    def arcs(self, dset):
        """Setter for the arcs dataset."""
        if dset:
            self._arcs = dset

    @property
    def levees(self):
        """Load the levees dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the levees dataset in the main file
        """
        if self._levees is None:
            self._levees = self.get_dataset('levees', False)
        return self._levees

    @levees.setter
    def levees(self, dset):
        """Setter for the levees dataset."""
        if dset:
            self._levees = dset

    @property
    def levee_flags(self):
        """Load the levee flags dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the levee flags dataset in the main file
        """
        if self._levee_flags is None:
            self._levee_flags = self.get_dataset('levee_flags', False)
            if self._levee_flags is None:  # Lazy migration of data before we had levee flags
                self._levee_flags = default_levee_flags()
        return self._levee_flags

    @levee_flags.setter
    def levee_flags(self, dset):
        """Setter for the levee flags dataset."""
        if dset:
            self._levee_flags = dset

    @property
    def q(self):
        """Load the q dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the levees dataset in the main file
        """
        if self._q is None:
            self._q = self.get_dataset('q', False)
            if self._q is None:  # Lazy migration of data before we had levee flags
                self._q = default_q()
        return self._q

    @q.setter
    def q(self, dset):
        """Setter for the q dataset."""
        if dset:
            self._q = dset

    @property
    def pipes(self):
        """Load the pipes dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the pipes dataset in the main file
        """
        if self._pipes is None:
            self._pipes = self.get_dataset('pipes', False)
        return self._pipes

    @pipes.setter
    def pipes(self, dset):
        """Setter for the pipes dataset."""
        if dset:
            self._pipes = dset

    @property
    def flow_cons(self):
        """Load the flow constituents dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the flow constituents dataset in the main file
        """
        if self._flow_cons is None:
            self._flow_cons = self.get_dataset('flow_cons', False)
        return self._flow_cons

    @flow_cons.setter
    def flow_cons(self, dset):
        """Setter for the flow constituents dataset."""
        if dset:
            self._flow_cons = dset

    def add_bc_atts(self, dset=None):
        """Add the BC attribute dataset for an arc.

        Args:
            dset (:obj:`xarray.Dataset`): The attribute dataset to concatenate. If not provided, a new Dataset
                of default attributes will be generated.

        Returns:
            (:obj:`tuple(int)`): The newly generated component id
        """
        try:
            new_comp_id = int(self.info.attrs['next_comp_id'])
            self.info.attrs['next_comp_id'] += 1  # Increment the unique XMS component id.
            if dset is None:  # Generate a new default Dataset
                dset = self._get_new_arc_atts(new_comp_id)
            else:  # Update the component id of an existing Dataset
                dset.coords['comp_id'] = [new_comp_id for _ in dset.coords['comp_id']]

            if self.arcs.comp_id.size == 0:  # No existing arcs to append. Just use the new dataset.
                self._arcs = dset.copy()
            else:  # Both datasets are non-empty. Concatenate them.
                self._arcs = xr.concat([self.arcs, dset], 'comp_id')
            return new_comp_id
        except Exception:
            return UNINITIALIZED_COMP_ID

    def add_pipe_atts(self, dset=None):
        """Add the pipe attribute dataset for a point.

        Args:
            dset (:obj:`xarray.Dataset`): The attribute dataset to concatenate. If not provided, a new Dataset
                of default attributes will be generated.

        Returns:
            (:obj:`tuple(int)`): The newly generated component id
        """
        try:
            new_comp_id = self.info.attrs['next_comp_id'].item()
            self.info.attrs['next_comp_id'] += 1  # Increment the unique XMS component id.
            if dset is None:  # Generate a new default Dataset
                dset = self._get_new_pipe_atts(new_comp_id)
            else:  # Update the component id of an existing Dataset
                dset.coords['comp_id'] = [new_comp_id for _ in dset.coords['comp_id']]
            self._pipes = xr.concat([self.pipes, dset], 'comp_id')
            return new_comp_id
        except Exception:
            return UNINITIALIZED_COMP_ID

    def add_levee_atts(self, levee_atts):
        """Append to the levee dataset.

        Args:
            levee_atts (:obj:`xarray.Dataset`): The levee atts dataset to append
        """
        if len(self.levees['Zcrest (m)']) == 0:
            self._levees = levee_atts.copy()
        else:
            self._levees = xr.concat([self.levees, levee_atts], 'comp_id')

    def add_levee_flags(self, levee_flags):
        """Append to the levee_flags dataset.

        Args:
            levee_flags (:obj:`xarray.Dataset`): The levee flags dataset to append
        """
        if len(self.levee_flags['locs']) == 0:
            self._levee_flags = levee_flags.copy()
        else:
            self._levee_flags = xr.concat([self.levee_flags, levee_flags], 'comp_id')

    def _get_default_info(self):
        """Get the default info attrs.

        Returns:
            dict: The default info attrs
        """
        return {
            'FILE_TYPE': 'ADCIRC_BC',
            'VERSION': version,
            'display_uuid': '',
            'point_display_uuid': '',
            'cov_uuid': '',
            'next_comp_id': 0,
            'proj_dir': '',
            'periodic_tidal': 1,
            'periodic_flow': 1,
            'hot_start_flow': -1,
            'fort.19': '',
            'fort.20': '',
            'flow_ts': 3600,
            'flow_ts_units': 'seconds',
        }

    def _get_default_datasets(self, data_file):
        """Create default datasets if needed.

        Args:
            data_file (:obj:`str`): Name of the data file. If it doesn't exist, it will be created.
        """
        if not os.path.exists(data_file) or not os.path.isfile(data_file):
            info = self._get_default_info()
            self._info = xr.Dataset(attrs=info)
            self._arcs = default_arc_atts()
            self._levees = default_levee_atts()
            self._levee_flags = default_levee_flags()
            self._pipes = default_pipe_atts()
            self._q = default_q()

            flow_con_data = {
                'Name': xr.DataArray(data=np.array([], dtype=object)),
                'Frequency': xr.DataArray(data=np.array([], dtype=np.float64)),
                'Nodal __new_line__ Factor': xr.DataArray(data=np.array([], dtype=np.float64)),
                'Equilibrium __new_line__ Argument': xr.DataArray(data=np.array([], dtype=np.float64)),
                'Amplitude': xr.DataArray(data=np.array([], dtype=object)),
                'Phase': xr.DataArray(data=np.array([], dtype=object)),
            }
            self._flow_cons = xr.Dataset(data_vars=flow_con_data)
            self.commit()
        else:
            self._check_for_simple_migration()

    def _check_for_simple_migration(self):
        """Check for simple migrations in the data."""
        commit = False
        if 'Parametric __new_line__ Length 2' not in self.levees:
            # This is a file from before we stored both levee curves.
            # Add the missing levee attribute
            arr = np.array([0.0] * self.levees.sizes['comp_id'], dtype=np.float64)
            self.levees['Parametric __new_line__ Length 2'] = ('comp_id', arr)
            # Reorder the columns, so they show up consistently in the GUI.
            self.levees = self.levees[[
                'Parametric __new_line__ Length', 'Parametric __new_line__ Length 2', 'Zcrest (m)',
                'Subcritical __new_line__ Flow Coef', 'Supercritical __new_line__ Flow Coef'
            ]]
            commit = True

        if 'use_elevs' not in self.arcs.data_vars:
            arr = np.array([0.0] * self.arcs.sizes['comp_id'], dtype=np.float64)
            self.arcs['use_elevs'] = ('comp_id', arr)

        if 'subcritical_coeff' not in self.arcs.data_vars:
            sub_arr = np.array([1.0] * self.arcs.sizes['comp_id'], dtype=np.float64)
            self.arcs['subcritical_coeff'] = ('comp_id', sub_arr)
            super_arr = np.array([1.0] * self.arcs.sizes['comp_id'], dtype=np.float64)
            self.arcs['supercritical_coeff'] = ('comp_id', super_arr)

        if 'constant_q' not in self.arcs.data_vars:
            arr = np.array([50.0] * self.arcs.sizes['comp_id'], dtype=np.float64)
            self.arcs['constant_q'] = ('comp_id', arr)

        if 'wse_offset' not in self.arcs.data_vars:
            arr = np.array([0.0] * self.arcs.sizes['comp_id'], dtype=np.float64)
            self.arcs['wse_offset'] = ('comp_id', arr)

        # Make sure we have a levee flag for each unique levee pair.
        levee_pairs = np.unique(self.levees.comp_id)
        if self.levee_flags is None or levee_pairs.size != self.levee_flags.sizes['comp_id']:
            self.levee_flags = default_levee_flags(fill_num=levee_pairs.size, coords={'comp_id': levee_pairs})
            commit = True

        # Make sure we have all the info attrs
        attrs = self._get_default_info()
        for key, value in attrs.items():
            if key not in self.info.attrs:
                # If the attr does not exist, give it a default value.
                self.info.attrs[key] = value
                commit = True

        if commit:
            self.commit()

    @staticmethod
    def _get_new_arc_atts(comp_id):
        """Get a new dataset with default attributes for a BC arc.

        Args:
            comp_id (:obj:`int`): The unique XMS component id of the BC arc. If UNINITIALIZED_COMP_ID, a new one is
                generated.

        Returns:
            (:obj:`xarray.Dataset`): A new default dataset for a BC arc. Can later be concatenated to
            persistent dataset.
        """
        bc_table = {
            'type': ('comp_id', [0]),
            'bc_option': ('comp_id', [1]),
            'tang_slip': ('comp_id', [0]),
            'galerkin': ('comp_id', [0]),
            'use_elevs': ('comp_id', [1]),
            'subcritical_coeff': ('comp_id', [1.0]),
            'supercritical_coeff': ('comp_id', [1.0]),
            'constant_q': ('comp_id', [50.0]),
            'wse_offset': ('comp_id', [0.0]),
        }
        coords = {'comp_id': [comp_id]}
        ds = xr.Dataset(data_vars=bc_table, coords=coords)
        return ds

    @staticmethod
    def _get_new_pipe_atts(comp_id):
        """Get a new dataset with default attributes for a pipe point.

        Args:
            comp_id (:obj:`int`): The unique XMS component id of the pipe point. If UNINITIALIZED_COMP_ID, a new one is
                generated.

        Returns:
            (:obj:`xarray.Dataset`): A new default dataset for a pipe point. Can later be concatenated to
            persistent dataset.
        """
        pipe_table = {
            'Height': ('comp_id', [0.0]),
            'Coefficient': ('comp_id', [0.0]),
            'Diameter': ('comp_id', [0.0]),
        }
        coords = {'comp_id': [comp_id]}
        ds = xr.Dataset(data_vars=pipe_table, coords=coords)
        return ds

    def get_ib_types(self):
        """Get a mapping of comp id to IB type for exporting fort.14."""
        # Build the initial map of component id to internal type index.
        id_to_type = {
            comp_id.item(): self.arcs.type.loc[comp_id.item()].item()
            for comp_id in self.arcs.coords['comp_id']
        }
        # Covert from GUI idx to ADCIRC IBTYPE
        for comp_id, type_idx in id_to_type.items():
            id_to_type[comp_id] = self._idx_to_simple_ibtype(type_idx)
            # Make adjustments to IBTYPES based on simple IBTYPE value
            if type_idx == ZERO_NORMAL_INDEX:  # Check for Galerkin on zero-normal boundaries
                if self.arcs.galerkin.loc[comp_id].item():
                    id_to_type[comp_id] += 1
            elif type_idx in [MAINLAND_INDEX, ISLAND_INDEX, RIVER_INDEX, LEVEE_INDEX, LEVEE_OUTFLOW_INDEX]:
                # Check for essential vs. natural boundaries.
                if self.arcs.bc_option.loc[comp_id].item() == NATURAL_INDEX:
                    id_to_type[comp_id] += 20  # Add 20 to IBTYPE if natural
                elif type_idx != LEVEE_INDEX:  # No tangential slip option for levee pairs
                    # If essential boundary, check if tangential slip has been enabled
                    if self.arcs.tang_slip.loc[comp_id].item() == 0:
                        id_to_type[comp_id] += 10  # Add 10 to IBTYPE if essential without tangential slip
        return id_to_type

    @staticmethod
    def _idx_to_simple_ibtype(idx):
        """Convert boundary condition type index used in the GUI and data to a simple IBTYPE."""
        ibtype = UNINITIALIZED_COMP_ID
        if idx == MAINLAND_INDEX:
            ibtype = 0  # Unassigned should be "generic"?
        elif idx == ISLAND_INDEX:
            ibtype = 1
        elif idx == RIVER_INDEX:
            ibtype = 2
        elif idx == LEVEE_OUTFLOW_INDEX:
            ibtype = 3
        elif idx == LEVEE_INDEX:
            ibtype = 4  # If has pipes, = 5
        elif idx == RADIATION_INDEX:
            ibtype = 30  # Radiation with flux (=32) not supported
        elif idx == ZERO_NORMAL_INDEX:
            ibtype = 40  # = 41 if Galerkin enabled
        elif idx == FLOW_AND_RADIATION_INDEX:
            ibtype = 52
        return ibtype

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._arcs is not None:
            self._arcs.close()
            self._drop_h5_groups(['arcs'])
            self._arcs.to_netcdf(self._filename, group='arcs', mode='a')
        if self._levees is not None:
            self._levees.close()
            self._drop_h5_groups(['levees'])
            self._levees.to_netcdf(self._filename, group='levees', mode='a')
        if self._levee_flags is not None:
            check_for_object_strings_dumb(self._levee_flags, ['locs'])
            self._levee_flags.close()
            self._drop_h5_groups(['levee_flags'])
            self._levee_flags.to_netcdf(self._filename, group='levee_flags', mode='a')
        if self._pipes is not None:
            self._pipes.close()
            self._drop_h5_groups(['pipes'])
            self._pipes.to_netcdf(self._filename, group='pipes', mode='a')
        if self._flow_cons is not None:
            check_for_object_strings_dumb(self._flow_cons, ['Name', 'Amplitude', 'Phase'])
            self._flow_cons.close()
            self._drop_h5_groups(['flow_cons'])
            self._flow_cons.to_netcdf(self._filename, group='flow_cons', mode='a')
        if self._q is not None:
            self._q.close()
            self._drop_h5_groups(['q'])
            self._q.to_netcdf(self._filename, group='q', mode='a')

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space.

        All BC datasets that need to be written to the file must be loaded into memory before calling this method.
        """
        if self._info is None:
            self._info = self.get_dataset('info', False)
        if self._arcs is None:
            self._arcs = self.get_dataset('arcs', False)
        if self._levees is None:
            self._levees = self.get_dataset('levees', False)
        if self._levee_flags is None:
            self._levee_flags = self.get_dataset('levee_flags', False)
        if self._pipes is None:
            self._pipes = self.get_dataset('pipes', False)
        if self._flow_cons is None:
            self._flow_cons = self.get_dataset('flow_cons', False)
        if self._q is None:
            self._q = self.get_dataset('q', False)
        io_util.removefile(self._filename)  # Delete the existing NetCDF file
        self.commit()  # Rewrite all datasets

    def concat_bc_arcs(self, bc_arc_data):
        """Adds the BC arc attributes from bc_arc_data to this instance of BcData.

        Args:
            bc_arc_data (:obj:`BcData`): another BcData instance

        Returns:
            (:obj:`dict`): The old ids of the bc_arc_data as key and the new ids as the data
        """
        next_comp_id = self.info.attrs['next_comp_id']
        # Reassign component id coordinates.
        new_bc_arcs = bc_arc_data.arcs
        num_concat_arcs = new_bc_arcs.sizes['comp_id']
        old_to_new_ids = {}
        if num_concat_arcs:
            old_comp_ids = new_bc_arcs.coords['comp_id'].data.astype('i4').tolist()
            new_bc_arcs.coords['comp_id'] = [next_comp_id + idx for idx in range(num_concat_arcs)]
            self.info.attrs['next_comp_id'] = next_comp_id + num_concat_arcs
            self._arcs = xr.concat([self.arcs, new_bc_arcs], 'comp_id')
            old_to_new_ids = {
                old_comp_id: new_comp_id
                for old_comp_id, new_comp_id in
                zip(old_comp_ids, new_bc_arcs.coords['comp_id'].data.astype('i4').tolist())
            }

            # Concatenate the levee dataset.
            new_levees = bc_arc_data.levees
            comp_id_list = new_levees.coords['comp_id'].data.astype('i4').tolist()
            num_concat_levees = new_levees.sizes['comp_id']
            if num_concat_levees:
                new_levees.coords['comp_id'] = [old_to_new_ids[old_comp_id] for old_comp_id in comp_id_list]
                self._levees = xr.concat([self.levees, new_levees], 'comp_id')

            # Concatenate the levee flag dataset.
            num_concat_levees = new_levees.sizes['comp_id']
            if num_concat_levees:
                xr.Dataset()
                self._levee_flags = xr.concat([self.levee_flags, new_levees], 'comp_id')

        return old_to_new_ids

    def concat_bc_points(self, bc_data):
        """Adds the BC pipe point attributes from bc_data to this instance of BcData.

        Args:
            bc_data (:obj:`BcData`): another BcData instance

        Returns:
            (:obj:`dict`): The old ids of the bc_data as key and the new ids as the data
        """
        next_comp_id = self.info.attrs['next_comp_id']
        # Reassign component id coordinates.
        new_pipe_points = bc_data.pipes
        num_concat_points = new_pipe_points.sizes['comp_id']
        if num_concat_points:
            old_comp_ids = new_pipe_points.coords['comp_id'].data.astype('i4').tolist()
            new_pipe_points.coords['comp_id'] = [next_comp_id + idx for idx in range(num_concat_points)]
            self.info.attrs['next_comp_id'] = next_comp_id + num_concat_points
            self._pipes = xr.concat([self.pipes, new_pipe_points], 'comp_id')
            return {
                old_comp_id: new_comp_id
                for old_comp_id, new_comp_id in
                zip(old_comp_ids, new_pipe_points.coords['comp_id'].data.astype('i4').tolist())
            }
        else:
            return {}

    def migrate_relative_paths(self):
        """
        Convert relative paths to absolute ones if necessary.

        We used to try and make paths relative, but it was hard and there was no clear benefit, so we just keep them
        absolute now.
        """
        proj_dir = self.info.attrs.pop('proj_dir', None)
        if not proj_dir:
            return

        _make_abs(proj_dir, self.info.attrs, 'fort.19')
        _make_abs(proj_dir, self.info.attrs, 'fort.20')
        self.commit()

    def copy_external_files(self):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Returns:
            (:obj:`str`): Message on failure, empty string on success
        """
        old_proj_dir = self.info.attrs['proj_dir']
        # Target folder is the <project name> folder next to the SMS project file.
        target_dir = os.path.normpath(os.path.join(os.path.dirname(self._filename), '../../..'))

        # Copy the single file references
        self._copy_attr_file(self.info.attrs, 'fort.19', old_proj_dir, target_dir)
        self._copy_attr_file(self.info.attrs, 'fort.20', old_proj_dir, target_dir)

        # Wipe the stored project directory. Paths will be absolute in GUI until resave.
        self.info.attrs['proj_dir'] = ''
        return ''

    def levee_pair_data(self):
        """Return the levee dataset and the name of the crest elevation dataset.

        Returns:
            (:obj:`tuple(xarray.Dataset, numpy.array)`): The levee dataset and the levee comp_ids
        """
        # Filter out levee outflow from table
        levee_comp_ids = self.arcs.where(self.arcs.type == LEVEE_INDEX, drop=True).comp_id.data
        levee_pairs = self.levees.where(self.levees.comp_id.isin(levee_comp_ids), drop=True)
        return levee_pairs, levee_comp_ids


def _make_abs(base, attrs, key):
    if not attrs.get(key, None):
        return

    attrs[key] = os.path.join(base, attrs[key])
