"""MappedBcDataManager class."""

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.components.display.display_options_io import read_display_options_from_json, write_display_options_to_json
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList

# 4. Local modules
from xms.adcirc.data import mapped_bc_data as mbcd


class MappedBcDataManager:
    """Helper class for MappedBcData."""
    def __init__(self, bc_comp):
        """Creates a helper class.

        Args:
            bc_comp (:obj:`MappedBcComponent`): The BC component whose data this helper should help

        """
        self.bc_comp = bc_comp

    def update_display_options_file(self, new_main_file, new_path):
        """Generate new UUIDs for the component and display lists.

        Will commit data file in this method.

        Args:
            new_main_file (:obj:`str`): Path to the new component's main file
            new_path (:obj:`str`): The new component's directory.

        Returns:
            (:obj:`ActionRequest`): ActionRequest to initialize display of duplicated component.

        """
        new_data = mbcd.MappedBcData(new_main_file)
        # If duplicating, clear the coverage UUID. Will query XMS for the new coverage's UUID on the create event.
        new_data.info.attrs['cov_uuid'] = ''
        # Set component UUID in arc display options file.
        new_comp_uuid = os.path.basename(new_path)
        basename = os.path.basename(self.bc_comp.disp_opts_files[0])
        fname = os.path.join(new_path, basename)
        json_dict = read_display_options_from_json(fname)
        json_dict['uuid'] = str(uuid.uuid4())  # Generate a new arc display list UUID.
        json_dict['comp_uuid'] = new_comp_uuid
        categories = CategoryDisplayOptionList()
        categories.from_dict(json_dict)
        write_display_options_to_json(fname, categories)
        new_data.info.attrs['display_uuid'] = json_dict['uuid']  # Store arc display list UUID in component data.
        new_data.commit()
        # Return an ActionRequest to initialize duplicate's display lists.
        action = ActionRequest(
            main_file=new_main_file,
            modality='NO_DIALOG',
            class_name=self.bc_comp.class_name,
            module_name=self.bc_comp.module_name,
            method_name='get_initial_display_options',
            comp_uuid=new_comp_uuid
        )
        return action
