"""Class to retrieve all the data needed to export a fort.15 (standalone or full simulation export)."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.data_objects.parameters import FilterLocation

# 4. Local modules
from xms.adcirc.components.station_component import StationComponent
from xms.adcirc.data.mapped_bc_data import MappedBcData
from xms.adcirc.data.mapped_flow_data import MappedFlowData
from xms.adcirc.data.mapped_tidal_data import MappedTidalData
from xms.adcirc.data.sim_data import SimData
from xms.adcirc.dmi.fort13_data_getter import Fort13DataGetter
from xms.adcirc.dmi.fort22_data_getter import Fort22DataGetter
from xms.adcirc.mapping.mapping_util import grid_projection_mismatches_display


class Fort15DataGetter:
    """Class to retrieve all the data needed to export a fort.15 (standalone or full simulation export)."""
    def __init__(self, query, xms_data):
        """Constructor.

        Args:
            query (:obj:`Query`): Object for requesting data from XMS. Should be None if a full simulation export.
            xms_data (:obj:`dict`): The dict of XMS data to fill
        """
        self._query = query
        self._xms_data = xms_data
        self._sim_export = True
        self._sim_item = None

    def _retrieve_sim_data(self):
        """Get the simulation data."""
        if not self._query:  # Simulation level export. Initial Context at the simulation level
            self._query = Query()
            # Get the simulation tree item
            sim_uuid = self._query.current_item_uuid()
            self._sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, sim_uuid)
            # Get the simulation's hidden component data.
            sim_comp = self._query.item_with_uuid(sim_uuid, model_name='ADCIRC', unique_name='Sim_Component')
            self._xms_data['sim_data'] = SimData(sim_comp.main_file)
        elif 'sim_data' in self._xms_data:  # used by CSTORM for export
            self._sim_item = self._xms_data['sim_item']
            self._sim_export = self._xms_data['sim_export']
        else:  # Component menu command export. Initial Context at the simulation hidden component level.
            # Get the simulation's hidden component data.
            sim_comp = self._query.current_item()
            self._xms_data['sim_data'] = SimData(sim_comp.main_file)
            # Go up to the parent simulation and get its tree item.
            sim_uuid = self._query.parent_item_uuid()
            self._sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, sim_uuid)
            self._sim_export = False  # Only export the fort.15

    def _retrieve_domain(self):
        """Get the domain's name, CoGrid file, and projection."""
        mesh_item = tree_util.descendants_of_type(
            self._sim_item, xms_types=['TI_MESH2D_PTR'], allow_pointers=True, recurse=False, only_first=True
        )
        if not mesh_item:
            raise RuntimeError('Error getting mesh geometry when exporting.')
        self._xms_data['domain_name'] = mesh_item.name  # Should be one and only one
        self._xms_data['num_nodes'] = mesh_item.num_points
        do_ugrid = self._query.item_with_uuid(mesh_item.uuid)
        if not do_ugrid:
            raise RuntimeError('Error getting mesh geometry when exporting.')
        self._xms_data['cogrid_file'] = do_ugrid.cogrid_file
        native_proj = do_ugrid.native_projection
        self._xms_data['projection'] = do_ugrid.projection  # The current display projection
        # Ensure that the current display projection matches the ADCIRC mesh.
        projection_error = grid_projection_mismatches_display(native_proj, self._xms_data['projection'])
        if projection_error:
            raise RuntimeError(projection_error)

    def _retrieve_mapped_items(self):
        """Get the mapped BC component and optionally mapped flow and tidal components."""
        # Get the mapped BC coverage - needed for both the fort.14 and fort.15 export scripts.
        bc_comps = tree_util.descendants_of_type(self._sim_item, unique_name='Mapped_Bc_Component', recurse=False)
        if not bc_comps:
            raise RuntimeError('Unable to export ADCIRC simulation because no applied boundary conditions were found.')
        self._xms_data['bc_datas'] = [MappedBcData(bc_comp.main_file) for bc_comp in bc_comps]
        # Find the main mapped BC coverage if there is one
        self._xms_data['main_bc'] = None
        from xms.adcirc.mapping.bc_mapping_manager import is_main_coverage  # Avoid circular dependency
        for mapped_bc_data in self._xms_data['bc_datas']:
            if is_main_coverage(mapped_bc_data):
                self._xms_data['main_bc'] = mapped_bc_data
                break

        # Get the mapped tidal constituents
        tidal_comps = tree_util.descendants_of_type(self._sim_item, unique_name='Mapped_Tidal_Component', recurse=False)
        self._xms_data['tidal_data'] = [MappedTidalData(tidal_comp.main_file) for tidal_comp in tidal_comps]

        # Get the mapped flow components
        flow_comp = tree_util.descendants_of_type(
            self._sim_item, unique_name='Mapped_Flow_Component', recurse=False, only_first=True
        )
        if flow_comp:  # Have a mapped flow component linked to the simulation
            self._xms_data['flow_data'] = MappedFlowData(flow_comp.main_file)
        else:
            self._xms_data['flow_data'] = None

    def _retrieve_optional_sim_takes(self):
        """Get the linked wind grid and recording stations coverage, if they exist.

        Notes:
            We get the wind grid here because if we are using a grid NWS type, we need it to write the fort.15.
        """
        # Look for a wind grid. Have to get it for the fort.15 because dimensions will be written on
        # the WTIMINC line.
        wind_grid_item = tree_util.descendants_of_type(
            tree_root=self._sim_item, xms_types=['TI_CGRID2D_PTR'], allow_pointers=True, only_first=True
        )
        if wind_grid_item:
            do_ugrid = self._query.item_with_uuid(wind_grid_item.uuid)
            self._xms_data['wind_grid'] = do_ugrid.cogrid_file
        else:
            self._xms_data['wind_grid'] = None

        # Look for a recording station coverage.
        station_item = tree_util.descendants_of_type(
            tree_root=self._sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Recording Stations',
            model_name='ADCIRC',
            only_first=True
        )
        if station_item:
            # Get the station point locations.
            station_cov = self._query.item_with_uuid(station_item.uuid)
            station_pts = station_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
            self._xms_data['station_pts'] = {station_pt.id: station_pt for station_pt in station_pts}
            # Get the coverage's hidden component main file
            station_comp = self._query.item_with_uuid(
                station_item.uuid, model_name='ADCIRC', unique_name='Station_Component'
            )
            self._xms_data['station_comp'] = StationComponent(station_comp.main_file)
            # Load the component id mapping files
            self._query.load_component_ids(self._xms_data['station_comp'], points=True)
        else:
            self._xms_data['station_comp'] = None
            self._xms_data['station_cov'] = None

    def retrieve_data(self):
        """Retrieve all the data needed from XMS to export the fort.15."""
        self._retrieve_sim_data()
        self._retrieve_domain()
        self._retrieve_mapped_items()
        self._retrieve_optional_sim_takes()
        if self._sim_export:  # If exporting the entire simulation, get data for the fort.13 and fort.22 as well
            getter = Fort13DataGetter(self._query, self._xms_data)
            getter.retrieve_data()
            getter = Fort22DataGetter(self._query, self._xms_data)
            getter.retrieve_data()
        if XmEnv.xms_environ_running_tests() == 'TRUE':
            self._query.send()  # This is just to get the QueryPlayback files flushed. We shouldn't be adding anything.
