"""Reader for ADCIRC fort.13 nodal attribute files.

https://adcirc.org/home/documentation/users-manual-v52/input-file-descriptions/nodal-attributes-file-fort-13/

Nodal properties which are constant in time but spatially variable. This file is only read when NWP > 0 in the
Model Parameter and Periodic Boundary Condition File.
The basic file structure is shown below. Each line of input data is represented by a line containing the input variable
 name(s) in bold face type. Blank lines are only to enhance readability. Loops indicate multiple lines of input.
 Definitions of each variable are provided via hot links.

AGRID
NumOfNodes
NAttr
for i = 1 to NAttr
    AttrName(i)
    Units(i)
    ValuesPerNode(i)
    DefaultAttrVal(i,k), k = 1, ValuesPerNode(i)
end i loop
for i = 1 to NAttr
    AttrName(i)
    NumNodesNotDefaultVal(i)
    for j = 1 to NumNodesNotDefaultVal(i)
        n, (AttrVal(n,k), k = 1, ValuesPerNode(i))
    end j loop
end i loop
"""

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
import pandas as pd

# 3. Aquaveo modules

# 4. Local modules
from xms.adcirc.dmi.fort13_data_getter import read_fort13_data_json
from xms.adcirc.feedback.xmlog import XmLog


def format_vals(vals):
    """Format nodal attribute dataset values for a line.

    Args:
        vals (:obj:`list`): List of the dataset float values for a node

    Returns:
        (:obj:`str`): The formatted dataset values
    """
    return ' '.join(f'{val:.6f}' for val in vals)


class NodalProperty:
    """Helper class to manage a single nodal attribute."""
    def __init__(self, name, dsets, unit, template):
        """Construct the helper.

        Args:
            name (:obj:`str`): Name of the nodal attribute
            dsets (:obj:`list`): List of either data_objects.parameters.Dataset objects or a single int in the case of
                GeoidOffset
            unit (:obj:`str`): Units of the nodal attribute dataset. I think ADCIRC ignores this. Hope so because it is
                hardcoded on export and ignored on import.
            template (bool): If True will export as a CSTORM template
        """
        self._name = name
        self._unit = unit
        self._template = template
        if self._name == 'sea_surface_height_above_geoid':
            self._values = dsets  # Special case for GeoidOffset - single constant value
            # self._default_value = [dsets[0][0]]  # 39 sec if just use 0.0 as default
        else:  # Load the dataset values into memory. Should be steady-state but will use first timestep if transient.
            self._values = [dset.values[0] for dset in dsets]
            # self._default_value = [0.0 for _ in range(len(dsets))]
        self._default_value = None
        self._get_default_value()

    def _get_default_value(self):
        """Find the most common nodal value.

        If a nodal attribute has multiple datasets, the mode is the most common combination of all dataset values.
        """
        XmLog().instance.info(f'Computing default value for {self._name}...')
        df = pd.DataFrame(np.array(self._values).transpose())
        cols = df.columns.values.tolist()
        series = df.groupby(cols).size().sort_values(ascending=False).index[0]
        if isinstance(series, tuple):  # Convert tuple to a list if more than one dataset for the attribute
            self._default_value = [default.item() for default in series]
        else:  # If only one dataset for attribute, put convert numpy scalar to float and put in a list
            self._default_value = [series.item()]
        XmLog().instance.info(f'Default value for {self._name}: {self._default_value}')

    def write_summary(self, fs):
        """Write the summary section for a nodal attribute.

        Args:
            fs: Open filestream to write to.

        For each enabled nodal attribute:
            att_name
            att_units
            att_num_vals_per_node
            att_default_vals
        """
        fs.write(f"{str(self._name):<40}\n "
                 f"{str(self._unit):<40}\n "
                 f"{str(len(self._values)):<40}\n ")
        if self._template and self._name == 'sea_surface_height_above_geoid':
            line = '%GEOIDOFFSET%'  # Special case for GeoidOffset when writing a CSTORM template.
        else:
            line = ' '.join(f'{self._default_value[k]:.6f}' for k in range(len(self._default_value)))
        fs.write(f'{line}\n')

    def write_att_values(self, fs):
        """Write the dataset values of a nodal attribute.

        Args:
            fs: Open filestream to write to.
        """
        XmLog().instance.info(f'Writing non-default nodal values for {self._name}...')
        data = []  # Values are in columns, extract them by row.
        for i in range(len(self._values[0])):  # Loop through nodes of the mesh (unless GeoidOffset, then single value)
            row = [self._values[k][i] for k in range(len(self._values))]  # Build list of attribute values for this node
            if row != self._default_value:  # Only export the nodal values if different than the default.
                data.append((i + 1, row))  # (Node ID, [nodal values for att])
        fs.write(f'{str(self._name):<40}\n '
                 f'{str(len(data)):<40}\n')  # The number of non-default values.
        # Node_id  att_val1 ...att_valN
        lines = [f' {node[0]} {format_vals(node[1])}' for node in data]
        fs.write('\n'.join(lines))
        if self._name != 'sea_surface_height_above_geoid':
            fs.write('\n')  # Special case for GeoidOffset, constant value


class Fort13Writer:
    """Exporter for the ADCIRC fort.13 file (nodal attribute datasets)."""
    def __init__(self, filename, xms_data):
        """Construct the exporter.

        Args:
            filename (:obj:`str`): Export location filename of the fort.13
            xms_data (:obj:`dict`): All data from XMS required to export the fort.13.
            ::
                {
                    'sim_data': SimData,
                    'domain_name': str,
                    'num_nodes': int,
                    'att_names': [str],
                    'att_dsets': [[]],  # Parallel with att_names
                    'att_units': [str],  # Parallel with att_dsets and att_names
                }
        """
        self._filename = filename
        self._nodal_properties = []
        self._xms_data = xms_data
        self._xms_data.setdefault('template', False)
        self._initialize_nodal_properties()

    def _initialize_nodal_properties(self):
        """Create the nodal property objects for the attributes we need to write."""
        template = self._xms_data['template']
        for name, dsets, unit in zip(
            self._xms_data['att_names'], self._xms_data['att_dsets'], self._xms_data['att_units']
        ):
            nodal_prop = NodalProperty(name, dsets, unit, template)
            self._nodal_properties.append(nodal_prop)

    def write(self):
        """Write the fort.13 file."""
        num_nodal_atts = len(self._nodal_properties)
        if not num_nodal_atts:
            XmLog().instance.error('Cannot export fort.13: No nodal attributes specified in the model control.')
            return  # No nodal attributes to write

        with open(self._filename, 'w', buffering=100000) as fs:
            fs.write(f"{self._xms_data['domain_name']:<40}\n")
            fs.write(f"{self._xms_data['num_nodes']:<40}\n")
            fs.write(f"{num_nodal_atts:<40}\n")
            XmLog().instance.info('Writing nodal attribute summaries...')
            for nodal_prop in self._nodal_properties:  # Write summary for each nodal attribute first
                nodal_prop.write_summary(fs)
            for nodal_prop in self._nodal_properties:  # Write nodal values for each nodal attribute second
                nodal_prop.write_att_values(fs)
        XmLog().instance.info(f'Successfully exported ADCIRC nodal attributes to {self._filename}.')


def export_nodal_atts_for_sim(filename):
    """Entry point for fort.13 exporter when exporting the entire ADCIRC simulation for a run.

    Args:
        filename (:obj:`str`): Path to the fort.13 to write
    """
    # Write a fort.13 in the current working directory (the simulation's model run export location).
    writer = Fort13Writer(filename, read_fort13_data_json())
    writer.write()
