"""Utility methods used by the GUI modules."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import os

# 2. Third party modules
from PySide2.QtCore import QDate, QDateTime, QTime

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as io_util
from xms.guipy import settings
from xms.guipy.dialogs.file_selector_dialogs import get_open_filename

# 4. Local modules

NULL_SELECTION = '(none selected)'


def get_file_selector_start_dir(label_text, proj_dir):
    """Get the directory to open file browser in.

    Args:
        label_text (:obj:`str`): The GUI label text associated with a file selector
        proj_dir (:obj:`str`): Directory of the saved project if it exists. If the project has been saved, any files
            selected at the time were converted to relative paths from the project directory.

    Returns:
        (:obj:`str`): Directory to open file browser in. If previously selected file, use that folder.
        Otherwise, use the last directory that was stored in the registry.
    """
    start_dir = proj_dir  # Default to the project directory if there is one.
    if label_text != NULL_SELECTION:  # Start in the directory of the last selected file, if there is one.
        if not os.path.isabs(label_text):  # Stored relative to the project directory.
            label_text = io_util.resolve_relative_path(proj_dir, label_text)
        start_dir = os.path.dirname(label_text)
    if not start_dir:  # If no project location and no file previously selected, look in registry
        start_dir = settings.get_file_browser_directory()
    return start_dir


def select_file(parent, label, caption, file_filter, proj_dir):
    """Display a file selector dialog.

    Args:
        parent (:obj:`QWidget`): The parent dialog
        label (:obj:`QLabel`): The label widget associated with the file selector.
        caption (:obj:`str`): The dialog caption
        file_filter (:obj:`str`): File extension filter
        proj_dir (:obj:`str`): Directory of the saved project if it exists. If the project has been saved, any files
            selected at the time were converted to relative paths from the project directory.
    """
    start_dir = get_file_selector_start_dir(label.text(), proj_dir)
    filename = get_open_filename(parent, caption, file_filter, start_dir)
    if filename and os.path.isfile(filename):
        label.setText(filename)


def datetime_to_qdatetime(dt_literal):
    """Convert a Python datetime object to a QDateTime.

    Args:
        dt_literal (:obj:`datetime.datetime`): Datetime to convert

    Returns:
        (:obj:`QDateTime`): See description
    """
    return QDateTime(
        QDate(dt_literal.year, dt_literal.month, dt_literal.day),
        QTime(dt_literal.hour, dt_literal.minute, dt_literal.second)
    )


def qdatetime_to_datetime(qdatetime):
    """Convert a QDateTime to a Python datetime object.

    Args:
        qdatetime (:obj:`QDateTime`): QDateTime to convert

    Returns:
        (:obj:`datetime.datetime`): See description
    """
    return datetime.datetime(
        year=qdatetime.date().year(),
        month=qdatetime.date().month(),
        day=qdatetime.date().day(),
        hour=qdatetime.time().hour(),
        minute=qdatetime.time().minute(),
        second=qdatetime.time().second()
    )


def change_df_column_names(df, direction):
    """Replaces special strings in the column names of the data.

    Args:
        df (:obj:`pandas.DataFrame`): the data frame
        direction (:obj:`bool`): If True, newline characters removed. If False, __new_line__ replaced with newline
            characters
    """
    idx = (0, 1) if direction else (1, 0)
    replace_strings = [('\n', '__new_line__'), ('/', '__slash__')]
    cols = df.columns
    new_cols = []
    for c in range(len(cols)):
        new_str = cols[c]
        for rep in replace_strings:
            new_str = new_str.replace(rep[idx[0]], rep[idx[1]])
        new_cols.append(new_str)
    df.columns = new_cols


def fix_df_column_names_for_gui(df):
    """Fix special header tokens for the GUI.

    We want multiline headers for some data frames for the gui but we don't want newlines or
    the like in the files. So this method replaces special characters according to the dict

    Args:
        df (:obj:`pandas.DataFrame`): the data frame
    """
    change_df_column_names(df, False)


def fix_df_column_names_for_io(df):
    """Fix special header tokens for the I/O.

    We want multiline headers for some data frames for the gui but we don't want newlines or
    the like in the files. So this method replaces special characters according to the dict

    Args:
        df (:obj:`pandas.DataFrame`): the data frame
    """
    change_df_column_names(df, True)
