"""Qt table view for entering levee attributes on a Boundary Conditions coverage arc."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QAbstractItemView

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adcirc.gui import levee_table_model as ltm
from xms.adcirc.gui.adcirc_table import AdcircTableWidget


class LeveeTableWidget(AdcircTableWidget):
    """The levee table widget."""
    def __init__(self, data_frame, levee_arcs, parent=None):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (:obj:`pandas.DataFrame`): The model data.
            levee_arcs (list): List of the levee arcs
        """
        super().__init__(
            parent, None, ltm.TBL_COL_LENGTH, {
                'Parametric \n Length': 0.0,
                'Parametric \n Length 2': 0.0,
                'Zcrest (m)': 0.0,
                'Subcritical\n Flow Coef': 1.0,
                'Supercritical \n Flow Coef': 1.0,
            }
        )
        self.two_arcs = len(levee_arcs) == 2
        self.model = ltm.LeveeTableModel(data_frame, levee_arcs, self)
        self.model.set_default_values(self.default_values)
        self.dbl_delegate = None
        self.pos_dbl_delegate = None
        self.length_delegate = None
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        dbl_validator = QxDoubleValidator(parent=self)
        self.dbl_delegate = EditFieldValidator(dbl_validator, self)
        pos_dbl_validator = QxDoubleValidator(bottom=0.0, parent=self)

        self.pos_dbl_delegate = EditFieldValidator(pos_dbl_validator, self)
        delegates = {
            ltm.TBL_COL_LENGTH: self.pos_dbl_delegate,
            ltm.TBL_COL_LENGTH2: self.pos_dbl_delegate,
            ltm.TBL_COL_ZCREST: self.dbl_delegate,
            ltm.TBL_COL_SUB_COEF: self.pos_dbl_delegate,
            ltm.TBL_COL_SUPER_COEF: self.pos_dbl_delegate,
        }

        super()._setup_ui(delegates, False, False)
        self.table_view.setEditTriggers(
            QAbstractItemView.DoubleClicked | QAbstractItemView.EditKeyPressed | QAbstractItemView.AnyKeyPressed
        )
        self.table_view.verticalHeader().setVisible(False)

        # Hide internal datasets
        self.table_view.setColumnHidden(ltm.TBL_COL_LENGTH2, not self.two_arcs)
        self.table_view.setColumnHidden(ltm.TBL_COL_SIDE_FLAG, True)
        self.table_view.setColumnHidden(ltm.TBL_COL_SIDE_LOCS, True)
        self.table_view.setColumnHidden(ltm.TBL_COL_SUB_COEF, True)
        self.table_view.setColumnHidden(ltm.TBL_COL_SUPER_COEF, True)

    def set_is_levee_outflow(self, is_outflow):
        """Set flag indicating whether the BC type is an outflow levee.

        If the BC type is levee outflow, the subcritical coefficient column will be hidden.

        Args:
            is_outflow (:obj:`bool`): True if the BC type is a levee outflow
        """
        # self.table_view.setColumnHidden(ltm.TBL_COL_SUB_COEF, is_outflow)
