"""Qt data model for the mapped BC attribute table view."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from PySide2.QtCore import QAbstractTableModel, Qt
from PySide2.QtGui import QColor

# 3. Aquaveo modules

# 4. Local modules

NUM_COLS = 6
COL_NODES = 0
COL_LEVEE = 1
COL_BC_TYPE = 2
COL_BC_OPTION = 3
COL_TANGENTIAL = 4
COL_GALERKIN = 5
BC_VIEW_COLS = [
    'Node IDs',
    'View BC',
    'Type',
    'BC Option',
    'Tangential\nSlip',
    'Galerkin',
]

BC_TYPES = [
    'Unassigned',
    'Ocean',
    'Mainland',
    'Island',
    'River',
    'Levee outflow',
    'Levee',
    'Radiation',
    'Zero normal',
    'Flow and radiation',
]

BC_OPTIONS = [
    'Essential',
    'Natural',
]


class MappedBcTableModel(QAbstractTableModel):
    """Class derived from QAbstractTableModel to handle mapped BC data."""
    def __init__(self, data, mapped_data, parent=None):
        """Initializes the class.

        Args:
            data (:obj:`BcData`): The BC data.
            mapped_data (:obj:`MappedBcData`): The mapped BC data.
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent)
        self.bc_data = data
        self.mapped_data = mapped_data
        self.read_only_columns = {
            COL_BC_TYPE,
        }
        self.read_only_cells = set()  # Cells that will be read only tuples (row, col)
        self.checkbox_columns = {COL_TANGENTIAL, COL_GALERKIN}  # Columns that will be displayed using a checkbox
        self.combobox_columns = {
            COL_BC_TYPE: BC_TYPES,
            COL_BC_OPTION: BC_OPTIONS,
        }  # Dict of column -> list of strings
        self.defaults = None
        self.show_nan_as_blank = False  # See set_show_nan_as_blank. Nan numbers are displayed as ''

    def rowCount(self, index=None):  # noqa: N802
        """Returns the number of rows the model holds.

        Args:
            index (:obj:`QModelIndex`): The index.

        Returns:
            (:obj:`int`): Number of rows the model holds.
        """
        return self.mapped_data.nodestrings.sizes['dim_0']

    def columnCount(self, index=None):  # noqa: N802
        """Returns the number of columns the model holds.

        Args:
            index (:obj:`QModelIndex`): The index.

        Returns:
            (:obj:`int`): Number of columns the model holds.
        """
        return NUM_COLS

    def data(self, index, role=Qt.DisplayRole):
        """Depending on the index and role given, return data, or None.

        Args:
            index (:obj:`QModelIndex`): The index.
            role (:obj:`int`): The role.

        Returns:
            The data at index, or None.
        """
        if not index.isValid():
            return None

        col = index.column()
        row = index.row()
        if role == Qt.DisplayRole or role == Qt.EditRole:
            # Don't display anything in checkbox columns other than the checkboxes
            if col in self.checkbox_columns:
                return ''

            if col in self.combobox_columns:  # Check for integer combobox option indices
                comp_id = int(self.mapped_data.nodestrings['comp_id'][row].data.item())
                if col == COL_BC_TYPE:
                    value = int(self.bc_data.arcs['type'].loc[comp_id].data.item())
                else:  # COL_BC_OPTION
                    value = int(self.bc_data.arcs['bc_option'].loc[comp_id].data.item())
                if np.issubdtype(type(value), np.integer):
                    s = self._match_index_to_combo_box_value(col, value)
                    if s:
                        return s
            return ''
        elif role == Qt.BackgroundColorRole:
            if col in self.read_only_columns:
                return QColor(240, 240, 240)
        elif role == Qt.CheckStateRole:
            if col in self.checkbox_columns:
                comp_id = int(self.mapped_data.nodestrings['comp_id'][row].data.item())
                if col == COL_TANGENTIAL:
                    i = int(self.bc_data.arcs['tang_slip'].loc[comp_id].data.item())
                else:  # COL_GALERKIN
                    i = int(self.bc_data.arcs['galerkin'].loc[comp_id].data.item())
                return Qt.Checked if i else Qt.Unchecked

        return None

    def setData(self, index, value, role=Qt.EditRole):  # noqa: N802
        """Adjust the data (set it to <value>) depending on index and role.

        Args:
            index (:obj:`QModelIndex`): The index.
            value : The value.
            role (:obj:`int`): The role.

        Returns:
            (:obj:`bool`): True if successful; otherwise False.
        """
        if not index.isValid():
            return False

        if index.column() in self.read_only_columns:
            return False

        col = index.column()
        row = index.row()
        t = (row, col)
        if t in self.read_only_cells:
            return False

        if role == Qt.EditRole or role == Qt.CheckStateRole:
            if col in self.checkbox_columns:
                value = 1 if value else 0  # Assume checkbox columns are integers 0 and 1
                comp_id = int(self.mapped_data.nodestrings['comp_id'][row].data.item())
                if col == COL_TANGENTIAL:
                    self.bc_data.arcs['tang_slip'].loc[comp_id] = value
                else:  # COL_GALERKIN
                    self.bc_data.arcs['galerkin'].loc[comp_id] = value
                self.dataChanged.emit(index, index)
                return True
            elif col in self.combobox_columns and isinstance(value, str):
                comp_id = int(self.mapped_data.nodestrings['comp_id'][row].data.item())
                value = self._match_value_to_combo_box_index(col, value)
                if col == COL_BC_TYPE:
                    self.bc_data.arcs['type'].loc[comp_id] = value
                else:  # COL_BC_OPTION
                    self.bc_data.arcs['bc_option'].loc[comp_id] = value
                self.dataChanged.emit(index, index)
                return True
        return False

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (:obj:`int`): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (:obj:`int`): The role.

        Returns:
            The data.
        """
        if role != Qt.DisplayRole:
            return None

        if orientation == Qt.Horizontal:
            try:
                return BC_VIEW_COLS[section]
            except IndexError:
                return None
        elif orientation == Qt.Vertical:
            try:
                return section + 1
            except IndexError:
                return None

    def flags(self, index):
        """Returns the item flags for the given index.

        Args:
            index (:obj:`QModelIndex`): The index.

        Returns:
            (:obj:`int`): The flags.
        """
        if not index.isValid():
            return Qt.ItemIsEnabled

        # Make it non-editable if needed
        flags = super().flags(index)
        index_column = index.column()
        if index_column in self.read_only_columns:
            flags = flags & (~Qt.ItemIsEditable)
        else:
            flags = flags | Qt.ItemIsEditable

        # Turn on the checkbox option if needed
        if index_column in self.checkbox_columns:
            flags |= Qt.ItemIsUserCheckable
        else:
            flags &= (~Qt.ItemIsUserCheckable)

        return flags

    def set_read_only_columns(self, read_only_columns):
        """Sets which columns are supposed to be read-only.

        Args:
            read_only_columns (:obj:`set{int}`): The read only columns.

        """
        self.read_only_columns = read_only_columns

    def set_checkbox_columns(self, checkbox_columns):
        """Sets which columns are supposed to be displayed as checkboxes.

        Args:
            checkbox_columns (:obj:`set{int}`): The checkbox columns.

        """
        self.checkbox_columns = checkbox_columns

    def set_combobox_column(self, column, items):
        """Tells the model that the column is a combo box delegate with the given items.

        On paste, we check the incoming data against the allowable items.

        Args:
            column (:obj:`int`): The column.
            items (:obj:`list[str]`): The combo box strings.

        """
        self.combobox_columns[column] = items

    def set_default_values(self, defaults):
        """Sets the column default values.

        Args:
            defaults(:obj:`dict{str -> value}`): Column names -> default values.

        """
        self.defaults = defaults

    def get_column_info(self):
        """Returns a tuple with column names and default values.

        Returns:
            (:obj:`tuple`): tuple containing:

                column_names (:obj:`list`): Column names.

                default (:obj:`dict{str -> value}`): Column names -> default values.
        """
        defaults = {}
        if self.defaults:
            defaults = self.defaults

        return [], defaults

    def _match_value_to_combo_box_item(self, value, column):
        """Makes sure the value matches one of the combobox strings.

        Args:
            value (:obj:`str`): The value.
            column (:obj:`int`): The column

        Returns:
            The value.

        """
        combobox_strings = self.combobox_columns[column]
        found = False
        for string in combobox_strings:
            if value.upper() == string.upper():
                value = string
                found = True
                break

        # Set it to the first string if there is not a match
        if not found and len(combobox_strings) > 0:
            value = combobox_strings[0]

        return value

    def _match_index_to_combo_box_value(self, column, index):
        """Returns the string at the combobox option index.

        Args:
            column (:obj:`int`): The column
            index (:obj:`int`): The combobox option index

        Returns:
            (:obj:`str`): Combobox option string at index

        """
        if column in self.combobox_columns:
            combobox_strings = self.combobox_columns[column]
            if len(combobox_strings) > index:
                return combobox_strings[index]

        return ''

    def _match_value_to_combo_box_index(self, column, value):
        """Returns the string at the combobox option index.

        Args:
            column (:obj:`int`): The column
            value (:obj:`str`): The combobox option index

        Returns:
            (:obj:`int`): Index of the combobox option if found, -1 otherwise

        """
        if column in self.combobox_columns:
            combobox_strings = self.combobox_columns[column]
            try:
                return combobox_strings.index(value)
            except ValueError:
                pass
        return -1

    def set_show_nan_as_blank(self, show_nan_as_blank):
        """Sets show_nan_as_blank property. If True, nan numbers are displayed as empty strings ('').

        Args:
            show_nan_as_blank (:obj:`bool`): True to show nan numbers as empty strings.

        """
        self.show_nan_as_blank = show_nan_as_blank
