"""A dialog for assigning flap gate values."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (QDialog, QDialogButtonBox)

# 3. Aquaveo modules
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.gui.flap_dialog_ui import Ui_FlapDialog
import xms.adh.gui.resources.adhqrc  # noqa F401


class FlapGateDialog(QDialog):
    """A dialog for assigning materials to polygons."""
    def __init__(self, win_cont, icon, title):
        """Allows the user to edit arc boundary conditions.

        Args:
            win_cont (QWidget): Parent window
            icon (QIcon): Window icon
            title (str): Window title
            data (MaterialsIO): Materials object
        """
        super().__init__(win_cont)
        self.ui = Ui_FlapDialog()
        self.ui.setupUi(self)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Material_Properties'

        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)
        self.setWindowIcon(icon)
        self.setWindowTitle(title)
        self._setup()

    def _setup(self):
        """Sets up various widget properties and actions."""
        # QDialogButtonBox with Ok and Cancel buttons
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.ui.length.setValidator(QxDoubleValidator())
        self.ui.coefficient_A.setValidator(QxDoubleValidator())
        self.ui.coefficient_B.setValidator(QxDoubleValidator())
        self.ui.coefficient_C.setValidator(QxDoubleValidator())
        self.ui.coefficient_D.setValidator(QxDoubleValidator())
        self.ui.coefficient_E.setValidator(QxDoubleValidator())
        self.ui.coefficient_F.setValidator(QxDoubleValidator())
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())

    def set_assignment(self, upstream_arc_id, downstream_arc_id, upstream_point_id, downstream_point_id):
        """Sets the text for the upstream and downstream roles. Also shows a warning if the structure is invalid.

        Args:
            upstream_arc_id (:obj:`tuple` of str and bool): Upstream arc id string and if the ids are valid.
            downstream_arc_id (:obj:`tuple` of str and bool): Downstream arc id string and if the ids are valid.
            upstream_point_id (:obj:`tuple` of str and bool): Upstream point id string and if the ids are valid.
            downstream_point_id (:obj:`tuple` of str and bool): Downstream point id string and if the ids are valid.
        """
        _ID = 0  # noqa N806
        _VALID = 1  # noqa N806
        option_1 = f'Arc {upstream_arc_id[_ID]} as upstream, arc {downstream_arc_id[_ID]} as downstream'
        option_2 = f'Arc {downstream_arc_id[_ID]} as upstream, arc {upstream_arc_id[_ID]} as downstream'
        self.ui.arc_upstream_downstream.addItems([option_1, option_2])
        option_1 = f'Point {upstream_point_id[_ID]} as upstream, point {downstream_point_id[_ID]} as downstream'
        option_2 = f'Point {downstream_point_id[_ID]} as upstream, point {upstream_point_id[_ID]} as downstream'
        self.ui.point_upstream_downstream.addItems([option_1, option_2])
        valid = upstream_arc_id[_VALID] and downstream_arc_id[_VALID] and \
            upstream_point_id[_VALID] and downstream_point_id[_VALID]
        if not valid:
            self.ui.warning_label.show()
            self.ui.warning_label.setStyleSheet("QLabel { color : red; }")
            self.ui.buttonBox.removeButton(self.ui.buttonBox.button(QDialogButtonBox.Ok))
        else:
            self.ui.warning_label.hide()

    def set_name(self, name):
        """Sets the name of the structure.

        Args:
            name (str): Name of the structure.
        """
        self.ui.flap_gate_name.setText(name)

    def set_options(self, coeff_a, coeff_b, coeff_c, coeff_d, coeff_e, coeff_f, length):
        """Sets the coefficient options a through f.

        Args:
            coeff_a (float): The a coefficient.
            coeff_b (float): The b coefficient.
            coeff_c (float): The c coefficient.
            coeff_d (float): The d coefficient.
            coeff_e (float): The e coefficient.
            coeff_f (float): The f coefficient.
            length (float): The length of the flap gate.
        """
        self.ui.coefficient_A.setText(str(coeff_a))
        self.ui.coefficient_B.setText(str(coeff_b))
        self.ui.coefficient_C.setText(str(coeff_c))
        self.ui.coefficient_D.setText(str(coeff_d))
        self.ui.coefficient_E.setText(str(coeff_e))
        self.ui.coefficient_F.setText(str(coeff_f))
        self.ui.length.setText(str(length))

    def get_arc_assignment_changed(self):
        """Get whether the arc assignment changed.

        Returns:
            True if the assignment changed.
        """
        is_first_index = self.ui.arc_upstream_downstream.currentIndex() == 0
        return not is_first_index

    def get_point_assignment_changed(self):
        """Get whether the point assignment changed.

        Returns:
            True if the assignment changed.
        """
        is_first_index = self.ui.point_upstream_downstream.currentIndex() == 0
        return not is_first_index

    def get_name(self):
        """Gets the name of the flap gate.

        Returns:
            A string name of the flap gate.
        """
        return self.ui.flap_gate_name.text()

    def get_options(self):
        """Gets the coefficients a through f and the length of the flap gate.

        Returns:
            A tuple of floats, starting with coefficients a-f and ending with the flap gate length.
        """
        return float(self.ui.coefficient_A.text()), \
            float(self.ui.coefficient_B.text()), \
            float(self.ui.coefficient_C.text()), \
            float(self.ui.coefficient_D.text()), \
            float(self.ui.coefficient_E.text()), \
            float(self.ui.coefficient_F.text()), \
            float(self.ui.length.text())

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Called when the Ok button is clicked."""
        return QDialog.accept(self)

    def reject(self):
        """Called when the Cancel button is clicked."""
        return QDialog.reject(self)
