"""A widget for assigning transport constituents for materials."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas
from PySide2.QtCore import QModelIndex, Qt

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.gui.widgets.transport_constituent_assignment_widget import TransportConstituentAssignmentWidget


class MaterialAssignmentModel(RenameModel):
    """A model to rename header titles, hide unwanted options."""
    def __init__(self, column_names, parent=None):
        """Initializes the filter model.

        Args:
            column_names (list): The column names.
            parent (Something derived from :obj:`QObject`): The parent object.

        """
        self._ID_SOURCE_COLUMN = 0
        self._NAME_COLUMN = 0
        self._DIFFUSION_SOURCE_COLUMN = 3
        self.show_diffusion = True
        super().__init__(column_names, parent)

    def filterAcceptsColumn(self, source_column: int, source_parent: QModelIndex) -> bool:  # noqa: N802
        """Filters out the 'ID' column.

        Args:
            source_column (int): The column index in the source model.
            source_parent (QModelIndex): The parent index of the view.

        Returns:
            True if the model should keep the column, false if the column should be hidden.
        """
        if source_column == self._DIFFUSION_SOURCE_COLUMN:
            return self.show_diffusion
        return source_column != self._ID_SOURCE_COLUMN

    def flags(self, index: QModelIndex):
        """Decides flags for things like enabled state.

        Args:
            index (QModelIndex): The index in the

        Returns:
            Item flags for the model index.
        """
        local_flags = super(MaterialAssignmentModel, self).flags(index)
        if index.column() == self._NAME_COLUMN:
            local_flags = local_flags & ~Qt.ItemIsEditable
        return local_flags

    def set_visible_viscosity_method(self, viscosity_method):
        """Changes the visible columns based on the eddy viscosity method.

        Args:
            viscosity_method (str): The current eddy viscosity method.
        """
        self.show_diffusion = viscosity_method.find('EVS') != -1
        self.invalidateFilter()


class TransportConstituentMatAssignmentWidget(TransportConstituentAssignmentWidget):
    """A dialog for assigning transport constituents to strings."""
    SOURCE_ID_COLUMN = 0  # Needs to be the source column index.
    SOURCE_NAME_COLUMN = 1
    SOURCE_REFINEMENT_COLUMN = 2
    SOURCE_DIFFUSION_COLUMN = 3

    def __init__(self, parent):
        """Allows the user to edit which transport constituents are used and how.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent, None, True)
        self.filter_model = MaterialAssignmentModel(['Name', 'Refinement\nTolerance', 'Diffusion\nCoefficient'], self)
        self._REFINEMENT_COLUMN = 1  # Needs to be the filtered column index.
        self._DIFFUSION_COLUMN = 2  # Needs to be the filtered column index.

        # Set up the delegate
        self.dbl_validator = QxDoubleValidator()
        self.dbl_validator.setBottom(0.0)
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        self.ui.constituents_table.setItemDelegateForColumn(self._REFINEMENT_COLUMN, self.edit_delegate)
        self.ui.constituents_table.setItemDelegateForColumn(self._DIFFUSION_COLUMN, self.edit_delegate)

        # Hide the group box, but leave the contents visible
        self.ui.transport_constituents_group.setCheckable(False)
        self.ui.transport_constituents_group.setTitle('')
        self.ui.transport_constituents_group.setFlat(True)
        self.ui.transport_constituents_group.setStyleSheet('QGroupBox#transport_constituents_group {border:none}')
        self.ui.transport_constituents_group.setContentsMargins(0, 0, 0, 0)
        self.ui.transport_constituents_group.layout().setContentsMargins(0, 0, 0, 0)
        self.ui.transport_constituents_group.layout().setSpacing(0)

        # Hide the ability to choose the transport constituents component.
        self.ui.current_constituents_label.setVisible(False)
        self.ui.constituents_label.setVisible(False)
        self.ui.constituents_button.setVisible(False)
        self.ui.change_warning_label.setVisible(False)
        self.ui.current_constituents_label.setContentsMargins(0, 0, 0, 0)
        self.ui.constituents_label.setContentsMargins(0, 0, 0, 0)
        self.ui.constituents_button.setContentsMargins(0, 0, 0, 0)
        self.ui.change_warning_label.setContentsMargins(0, 0, 0, 0)

    def set_transport(self, constituents_uuid, constituents, assignments=None, transport_name=''):
        """Sets the transport constituents and current assignments.

        Args:
            constituents_uuid (str): The uuid of the transport constituents component.
            constituents (TransportConstituentsIO): The transport constituents data.
            assignments (pandas.DataFrame): The current assignments to edit.
            transport_name (str): The name of the transport constituents component as it shows in the project explorer.
        """
        super().set_transport(constituents_uuid, constituents, assignments, transport_name)

        assignment_list = []

        # Add assignments for constituents.
        if constituents:
            if constituents.param_control.salinity:
                self._add_constituent(1, 'Salinity', assignments, assignment_list)
            if constituents.param_control.temperature:
                self._add_constituent(2, 'Temperature', assignments, assignment_list)
            if constituents.param_control.vorticity:
                self._add_constituent(3, 'Vorticity', assignments, assignment_list)
            comp_ids = constituents.user_constituents['ID'].data.tolist()
            names = constituents.user_constituents['NAME'].data.tolist()
            for comp_id, name in zip(comp_ids, names):
                self._add_constituent(comp_id, name, assignments, assignment_list)
        self.assignments = pandas.DataFrame(
            assignment_list, columns=['CONSTITUENT_ID', 'NAME', 'REFINEMENT', 'DIFFUSION']
        )
        self.model = QxPandasTableModel(self.assignments, self)
        self.filter_model.setSourceModel(self.model)
        self.ui.constituents_table.setModel(self.filter_model)
        self.ui.constituents_table.update()

    def set_sediment_transport(self, constituents_uuid, constituents, assignments=None):
        """Sets the transport constituents and current assignments.

        Args:
            constituents_uuid (str): The uuid of the transport constituents component.
            constituents (TransportConstituentsIO): The transport constituents data.
            assignments (pandas.DataFrame): The current assignments to edit.
        """
        super().set_transport(constituents_uuid, constituents, assignments)

        assignment_list = []

        for _, row in constituents.param_control.sand.iterrows():
            self._add_constituent(row["ID"], 'Sand', assignments, assignment_list)

        for _, row in constituents.param_control.clay.iterrows():
            self._add_constituent(row["ID"], 'Clay', assignments, assignment_list)

        self.assignments = pandas.DataFrame(
            assignment_list, columns=['CONSTITUENT_ID', 'NAME', 'REFINEMENT', 'DIFFUSION']
        )
        self.model = QxPandasTableModel(self.assignments, self)
        self.filter_model.setSourceModel(self.model)
        self.ui.constituents_table.setModel(self.filter_model)
        self.ui.constituents_table.update()

    @staticmethod
    def _add_constituent(con_id, con_name, assignments, assignment_list):
        """Adds the constituent from the component if possible, otherwise it adds a default.

        Args:
            con_id (int): The constituent id to look for.
            con_name (str): The constituent name.
            assignments (pandas.DataFrame): The current constituent assignments.
            assignment_list (list): The full list of constituent assignments that the current constituent
                                    will be added to.
        """
        con_assignment = None
        if assignments is not None and not assignments.empty:
            con_assignment = assignments.loc[assignments['CONSTITUENT_ID'] == con_id].values.tolist()
        if con_assignment:
            assignment_list.append(con_assignment[0][1:])
        else:
            assignment_list.append([con_id, con_name, 1.0, 0.0])
