"""Query implementation used when recording XMS communication for Python tests."""
# 1. Standard python modules
from collections.abc import Iterable
from distutils.dir_util import copy_tree
import json
import logging
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as io_util
import xms.data_objects.parameters as pydop
from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules
from xms.api._xmsapi.dmi import DataDumpIOBase
from xms.api.dmi._QueryImpl import QueryImpl
from xms.api.dmi import XmsEnvironment as xmenv
from xms.api.tree.tree_node import projection_to_json


class QueryRecordImpl(QueryImpl):
    """Query implementation used when recording XMS communication for Python tests."""
    def __init__(self, recording_folder, **kwargs):
        """Construct the wrapper.

        Args:
            instance (xms.api._xmsapi.dmi.Query, optional): The C++ object to wrap. Mutually exclusive with 'agent'
                kwarg.
            agent (xms.api.XmsAgent, optional): The XMS interprocess communication manager, mutually exclusive with
                'instance' kwarg.
            timeout (int): The interprocess communication timeout interval in milliseconds. Defaults to 5 minutes,
                which might be a little excessive. Used to have lots of script timeouts when we serialized more
                data through the pipe.
            retries (int): Number of times to retry after a failed message request. Defaults to one. I have never
                set it to anything else, and doing sometimes causes duplicate build data being created in XMS
                from import scripts.
            progress_script (bool): True if created in a progress script.
            migrate_script (bool): True if created in a migrate script.
        """
        super().__init__(**kwargs)  # Setup the real Query so we can communicate with XMS while recording
        self._recording_folder = recording_folder
        self._requested_data = {
            'component_id_files': {},  # {comp_uuid: {cov_uuid: {target_type: (att_id_file, comp_id_file)}}}
            'components': {},  # Key is component UUID
            'coverage_attributes': {},  # {cov_uuid: {str_target: [att_file]}}
            'coverages': {},  # Key is coverage UUID
            'current_item': None,
            'current_item_uuid': None,
            'datasets': {},  # Key is dataset UUID
            'display_projection': None,
            'generic_coverages': {},  # Key is coverage UUID
            'global_time': None,
            'global_time_settings': None,
            'named_executable_path': {},  # Key is XML executable name
            'parent_item': None,
            'parent_item_uuid': None,
            'project_tree': None,
            'rasters': {},  # Key is raster UUID
            'read_file': None,
            'selected_component_id_files': {},  # {comp_uuid: {cov_uuid: {int_target: (att_id_file, comp_id_file)}}}
            'simulations': {},  # Key is simulation UUID
            'ugrid_selections': {},  # {ugrid_uuid: {str_target: [selected_ids]}}
            'ugrids': {},  # Key is UGrid UUID
        }
        self._sent_data = None
        self._reset_sent_data()
        self._current_item = False  # True if data retrieved from XMS should be stored as value of current_item()
        self._parent_item = False  # True if data retrieved from XMS should be stored as value of parent_item()

        self._already_flushed = False  # True as soon as we write the baseline files
        self._logger = None

        self._setup_logger()

        # Always get a dump of the current item. If we are recording a hidden component ActionRequest, the script
        # probably never explicitly requests itself, but need the data for setting up the test.
        self.current_item()

    def __del__(self):
        """Write the baseline files if they don't exist yet (script never called Query.send())."""
        self._write_output_files()

    def _reset_sent_data(self):
        """Clear out the maps of sent data."""
        self._sent_data = {
            'components': 0,
            'coverages': 0,
            'coverage_components': 0,
            'datasets': 0,
            'delete_items': 0,
            'generic_coverages': 0,
            'link_items': 0,
            'model_check_errors': 0,
            'rasters': 0,
            # 'shapefiles': 0,  Lazily add this so as not to break backwards compatibility
            'rename_items': 0,
            'simulations': 0,
            'simulation_components': 0,
            'ugrid_elevation_edits': 0,
            'ugrids': 0,
            'unlink_items': 0,
        }

    def _setup_logger(self):
        """Capture logging output and pipe to a file."""
        self._logger = logging.getLogger('xms')
        self._logger.setLevel(logging.DEBUG)
        log_file = os.path.join(self._recording_folder, xmenv.LOGGING_BASE_FILE)
        io_util.removefile(log_file)  # Clear the old logging file
        fh = logging.FileHandler(log_file)
        fh.setLevel(logging.DEBUG)
        formatter = logging.Formatter('%(name)s - %(levelname)s - %(message)s')
        fh.setFormatter(formatter)
        self._logger.addHandler(fh)

    def _write_output_files(self):
        """Write the recording file for playback and the sent data baseline."""
        if self._already_flushed:
            return  # Already wrote the baselines
        self._already_flushed = True
        for handler in self._logger.handlers:  # Disable piping log output to file.
            handler.close()
            self._logger.removeHandler(handler)

        # Write the playback file.
        with open(os.path.join(self._recording_folder, xmenv.PLAYBACK_RECORD_FILE), 'w') as f:
            f.write(json.dumps(self._requested_data))
        # Write the sent data baseline.
        with open(os.path.join(self._recording_folder, xmenv.SENT_DATA_BASE_FILE), 'w') as f:
            f.write(json.dumps(self._sent_data))

    def _copy_component_tree_data(self, tree_node_dict):
        """Ensure that all component tree items have their data copied to the recording folder.

        Args:
            tree_node_dict (dict): JSON dict for the root of the project explorer tree
        """
        # Store the path to the component main file as relative from the recording folder.
        tree_node_dict['main_file'] = self._copy_component_data(tree_node_dict['main_file'])
        for child_item in tree_node_dict['children']:
            self._copy_component_tree_data(child_item)

    def _copy_component_data(self, main_file):
        """Copies a components data to the recording folder if it hasn't been already.

        Args:
            main_file (str): Path to the component main file

        Returns:
            str: The new main file path, relative to the recording folder
        """
        if not main_file or not os.path.isfile(main_file):
            return ''

        # Walk up from the main file to the 'Components' directory to ensure we get the root of the component's data.
        root_folder = os.path.dirname(main_file)
        comp_folder = root_folder
        relative_path = os.path.basename(main_file)  # Build up main file path relative to the recording folder
        while not root_folder.endswith('Components'):
            relative_path = os.path.join(os.path.basename(comp_folder), relative_path)
            comp_folder = root_folder
            root_folder = os.path.dirname(root_folder)
        relative_path = os.path.join('Components', relative_path)

        # Copy component data folder to a 'Components' subfolder of the recording directory if we haven't already.
        recording_comp_folder = os.path.join(self._recording_folder, 'Components')
        os.makedirs(recording_comp_folder, exist_ok=True)
        new_path = os.path.join(recording_comp_folder, os.path.basename(comp_folder))
        if not os.path.isdir(new_path):
            copy_tree(comp_folder, new_path)
        return relative_path

    def _store_simulation(self, simulation):
        """Store a simulation item when it is requested.

        Args:
            simulation (pydop.Simulation): The simulation to store
        """
        if simulation.uuid not in self._requested_data['simulations']:
            self._requested_data['simulations'][simulation.uuid] = {
                'name': simulation.name,
                'sim_uuid': simulation.uuid,
                'model': simulation.model,
            }
        if self._current_item:  # Store component as the value of current_item()
            self._requested_data['current_item'] = self._requested_data['simulations'][simulation.uuid]
        elif self._parent_item:  # Store component as the value of parent_item()
            self._requested_data['parent_item'] = self._requested_data['simulations'][simulation.uuid]

    def _store_coverage(self, coverage):
        """Store a coverage dump when it is requested.

        Args:
            coverage (pydop.Coverage): The coverage to store
        """
        if coverage.uuid not in self._requested_data['coverages']:
            dump_file, group_path = coverage._instance.GetDumpFile()
            coverage_folder = os.path.join(self._recording_folder, 'Coverages')
            os.makedirs(coverage_folder, exist_ok=True)
            new_dump_file = os.path.join(coverage_folder, os.path.basename(dump_file))  # Should be a unique filename
            io_util.copyfile(dump_file, new_dump_file)
            self._requested_data['coverages'][coverage.uuid] = {
                'name': coverage.name,
                'uuid': coverage.uuid,
                # Store relative path to dump file
                'filename': os.path.join('Coverages', os.path.basename(new_dump_file)),
                'group_path': group_path,
                'projection': projection_to_json(coverage.projection),  # The display projection
                'native_projection': projection_to_json(coverage.native_projection),  # The native projection
            }
        if self._current_item:  # Store component as the value of current_item()
            self._requested_data['current_item'] = self._requested_data['coverages'][coverage.uuid]
        elif self._parent_item:  # Store component as the value of parent_item()
            self._requested_data['parent_item'] = self._requested_data['coverages'][coverage.uuid]

    def _store_generic_coverage(self, coverage):
        """Store a generic coverage dump when it is requested.

        Args:
            coverage (DataDumpIOBase): The generic coverage to store
        """
        # Need to copy file before loading.
        filename = coverage.m_fileName
        coverage_folder = os.path.join(self._recording_folder, 'GenericCoverages')
        os.makedirs(coverage_folder, exist_ok=True)
        new_dump_file = os.path.join(coverage_folder, os.path.basename(filename))  # Should be a unique filename
        if not os.path.isfile(new_dump_file):
            io_util.copyfile(filename, new_dump_file)
            self._requested_data['generic_coverages'][coverage.m_cov.uuid] = {
                'name': coverage.m_cov.name,
                'uuid': coverage.m_cov.uuid,
                # Store relative path to dump file
                'filename': os.path.join('GenericCoverages', os.path.basename(new_dump_file)),
                'projection': projection_to_json(coverage.m_cov.projection),  # The display projection
                'native_projection': projection_to_json(coverage.m_cov.native_projection),  # Native projection
                'dump_type': coverage.GetDumpType(),
            }

    def _store_ugrid(self, ugrid):
        """Store a UGrid dump when it is requested.

        Args:
            ugrid (pydop.UGrid): The UGrid to store
        """
        if ugrid.uuid not in self._requested_data['ugrids']:
            ugrid_folder = os.path.join(self._recording_folder, 'UGrids')
            os.makedirs(ugrid_folder, exist_ok=True)
            new_dump_file = os.path.join(ugrid_folder, os.path.basename(ugrid.cogrid_file))
            io_util.copyfile(ugrid.cogrid_file, new_dump_file)  # Should be a unique filename
            self._requested_data['ugrids'][ugrid.uuid] = {
                'name': ugrid.name,
                'uuid': ugrid.uuid,
                # Store relative path to dump file
                'cogrid_file': os.path.join('UGrids', os.path.basename(new_dump_file)),
                'projection': projection_to_json(ugrid.projection),  # The display projection
                'native_projection': projection_to_json(ugrid.native_projection),  # The native projection
            }

    def _store_dataset(self, dataset_reader):
        """Store a dataset dump when it is requested.

        Args:
            dataset_reader (DatasetReader): The dataset to store
        """
        if dataset_reader.uuid not in self._requested_data['datasets']:
            dataset_folder = os.path.join(self._recording_folder, 'Datasets')
            os.makedirs(dataset_folder, exist_ok=True)
            new_dump_file = os.path.join(dataset_folder, os.path.basename(dataset_reader.h5_filename))
            io_util.copyfile(dataset_reader.h5_filename, new_dump_file)  # Should be a unique filename
            self._requested_data['datasets'][dataset_reader.uuid] = {
                # Store relative path to dump file
                'h5_filename': os.path.join('Datasets', os.path.basename(dataset_reader.h5_filename)),
                'group_path': dataset_reader.group_path,
            }

    def _store_gis_data(self, filename, gis_uuid):
        """Store a raster or shapefile filename when it is requested.

        Args:
            filename (str): Path to the GIS file
            gis_uuid (str): UUID of the GIS item
        """
        # When we initially wrote this we only exported rasters. Shapefiles got tacked on later but work exactly the
        # same way as far as the API is concerned. Store shapefiles in "rasters" key of dict so as not to break existing
        # stuff.
        if gis_uuid not in self._requested_data['rasters']:
            folder_name = 'Shapefiles' if os.path.splitext(filename)[1].lower() == '.shp' else 'Rasters'
            gis_folder = os.path.join(self._recording_folder, folder_name)
            os.makedirs(gis_folder, exist_ok=True)
            basename = os.path.basename(filename)
            new_gis_file = os.path.join(gis_folder, basename)
            io_util.copyfile(filename, new_gis_file)  # Should be a unique filename
            self._requested_data['rasters'][gis_uuid] = {
                'filename': os.path.join(folder_name, basename),  # Store relative path to raster file
            }
            if folder_name == 'Shapefiles':
                self._copy_shapefile_files(os.path.dirname(filename), gis_folder, os.path.splitext(basename)[0])

    def _copy_shapefile_files(self, source_dir, dest_dir, basename):
        """Copy a shapefile's .dbf, .shx, and .prj files to the same location as the shapefile was copied.

        Args:
            source_dir (str): Absolute path to the source shapefile's directory
            dest_dir (str): Absolute path to the copied shapefile's directory
            basename (str): Basename of the shapefile without extension
        """
        # Look for associated .dbf file
        source_dbf = os.path.join(source_dir, f'{basename}.dbf')
        if os.path.isfile(source_dbf):
            io_util.copyfile(source_dbf, os.path.join(dest_dir, os.path.basename(source_dbf)))
        # Look for associated .shx file
        source_shx = os.path.join(source_dir, f'{basename}.shx')
        if os.path.isfile(source_shx):
            io_util.copyfile(source_shx, os.path.join(dest_dir, os.path.basename(source_shx)))
        # Look for associated .prj file
        source_prj = os.path.join(source_dir, f'{basename}.prj')
        if os.path.isfile(source_prj):
            io_util.copyfile(source_prj, os.path.join(dest_dir, os.path.basename(source_prj)))

    def _store_component(self, component, item_uuid):
        """Store a component dump and its data when it is requested.

        Args:
            component (pydop.Component): The component to store
            item_uuid (str): UUID of the component, or possibly of the owner item if a hidden component
        """
        if item_uuid not in self._requested_data['components']:
            # May have already been copied with project tree dump
            relative_path = self._copy_component_data(component.main_file)
            unique_name, model_name = component.get_unique_name_and_model_name()
            class_name, module_name = component.get_class_and_module()
            self._requested_data['components'][item_uuid] = {
                'name': component.name,
                'comp_uuid': component.uuid,
                'main_file': relative_path,
                'model_name': model_name,
                'unique_name': unique_name,
                'class_name': class_name,
                'module_name': module_name,
                'locked': component.locked,
            }
        if self._current_item:  # Store component as the value of current_item()
            self._requested_data['current_item'] = self._requested_data['components'][item_uuid]
        elif self._parent_item:  # Store component as the value of parent_item()
            self._requested_data['parent_item'] = self._requested_data['components'][item_uuid]

    def _store_xms_data(self, xms_item, item_uuid):
        """Store a data_object requested by a script.

        Copies any files and creates a JSON dict with enough data to reconstruct the object.

        Args:
            xms_item: The data_object/str/generic coverage dump to serialize
            item_uuid (str): UUID of the item
        """
        if not xms_item:
            return
        elif type(xms_item) == str:  # If requested item was a raster or shapefile, return value is a str
            self._store_gis_data(xms_item, item_uuid)
        elif isinstance(xms_item, DataDumpIOBase):
            self._store_generic_coverage(xms_item)
        elif isinstance(xms_item, pydop.Component):
            self._store_component(xms_item, item_uuid)  # May be the UUID of a hidden component's owner
        elif isinstance(xms_item, pydop.Simulation):
            self._store_simulation(xms_item)
        elif isinstance(xms_item, pydop.Coverage):
            self._store_coverage(xms_item)
        elif isinstance(xms_item, DatasetReader):
            self._store_dataset(xms_item)
        elif isinstance(xms_item, pydop.UGrid):
            self._store_ugrid(xms_item)

    # Data commonly retrieved at the root simulation/component level.
    @property
    def project_tree(self):
        """Returns the XMS project explorer tree."""
        pe_tree = super().project_tree
        if not self._requested_data['project_tree']:
            json_dict = dict(pe_tree)  # Serialize to a json dict.
            self._copy_component_tree_data(json_dict)  # Copy all component tree item data to recording folder
            self._requested_data['project_tree'] = json_dict  # Store after component main file paths have been updated.
        return pe_tree

    @property
    def global_time(self):
        """Returns the current XMS global time."""
        zero_time = super().global_time
        self._requested_data['global_time'] = str(zero_time)  # '%Y-%m-%d %H:%M:%S.%f' by default
        return zero_time

    @property
    def global_time_settings(self):
        """Returns the current XMS global time settings as a JSON string.

        See xms.guipy.time_format.XmsTimeFormatter for more details.
        """
        time_formatting = super().global_time_settings
        self._requested_data['global_time_settings'] = time_formatting
        return time_formatting

    @property
    def display_projection(self):
        """Returns the current XMS display projection."""
        projection = super().display_projection
        self._requested_data['display_projection'] = projection_to_json(projection)
        return projection

    @property
    def cp(self):
        """Returns a dict of GUI module name text to copy protection enable/disable flag."""
        protect_map = super().cp
        self._requested_data['copy_protection'] = protect_map
        return protect_map

    @property
    def read_file(self):
        """Returns the filesystem path to the file being imported by the process.

        Note only applicable if process is an import script. In the case of a recording, we will copy the file
        known to the Query to the recording folder. Any other files needed to read the known read file must be
        manually copied to the recording folder before playback.
        """
        filename = super().read_file
        io_util.copyfile(filename, os.path.join(self._recording_folder, os.path.basename(filename)))
        self._requested_data['read_file'] = os.path.basename(filename)
        return filename

    # Methods for retrieving data from XMS.
    def item_with_uuid(self, item_uuid, generic_coverage=False, model_name=None, unique_name=None):
        """Retrieves an item from XMS given its UUID.

        Note:
            To get a hidden component item_uuid should be the UUID of the hidden component's owner. model_name is the
            hidden component's XML-defined model name. unique_name is the hidden component's XML-defined unique name.

        Args:
            item_uuid (str): UUID of the item
            generic_coverage (bool): True if the item is one of the dumpable generic coverages
            model_name (str): The XML model name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify unique_name if model_name provided.
            unique_name (str): The XML unique name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify model_name if unique_name provided.

        Returns:
            Pure python interface to the requested item.
        """
        xms_item = super().item_with_uuid(
            item_uuid, generic_coverage=generic_coverage, model_name=model_name, unique_name=unique_name
        )
        self._store_xms_data(xms_item, item_uuid)
        return xms_item

    def current_item(self):
        """Returns a KEYWORD_NONE Get request for the current position in the Query Context.

        Note assuming Context position is at something with a UUID. Don't try to use if you are down at the Coverage
        Arc level or something like that.

        Returns:
            The KEYWORD_NONE result for the current Context position or None on failure.
        """
        self._current_item = True
        xms_item = super().current_item()
        item_uuid = xms_item.uuid if xms_item and hasattr(xms_item, 'uuid') else ''
        self._store_xms_data(xms_item, item_uuid)
        self._current_item = False
        return xms_item

    def current_item_uuid(self):
        """Returns the UUID of the item at the current Context position, if it has one.

        Returns:
            str: The current item's UUID or empty string if it doesn't have one.
        """
        my_uuid = super().current_item_uuid()
        self._requested_data['current_item_uuid'] = my_uuid
        return my_uuid

    def parent_item(self):
        """Returns a KEYWORD_NONE Get request for the parent of the current position in the Query Context.

        Note assuming Context position's parent is something with a UUID. Don't try to use if you are down at the
        Coverage Arc level or something like that.

        Returns:
            The KEYWORD_NONE result for the current context position's parent or None on failure.
        """
        self._parent_item = True
        xms_item = super().parent_item()
        item_uuid = xms_item.uuid if xms_item and hasattr(xms_item, 'uuid') else ''
        self._store_xms_data(xms_item, item_uuid)
        self._parent_item = False
        return xms_item

    def parent_item_uuid(self):
        """Returns the UUID of the current Context position's parent, if it has one.

        Returns:
            str: The UUID of the current item's parent or empty string if it doesn't have one.
        """
        parent_uuid = super().parent_item_uuid()
        self._requested_data['parent_item_uuid'] = parent_uuid
        return parent_uuid

    def named_executable_path(self, exe_name):
        """Get the path to a model executable that is defined in the XML.

        Args:
            exe_name (str): The XML-defined executable name

        Returns:
            str: Path to the named executable or empty string if not found
        """
        exe_path = super().named_executable_path(exe_name)
        # Not really sure what to store here that is useful. Path from python folder down? Better not have 'python' in
        # the path before the python root directory.
        exe_path_clean = exe_path.lower()
        python_pos = exe_path_clean.rfind('python')
        if python_pos >= 0:
            self._requested_data['named_executable_path'][exe_name] = exe_path_clean[python_pos:]
        return exe_path

    def load_component_ids(self, component, points, arcs, arc_groups, polygons, only_selected, delete_files):
        """Query XMS for a dump of all the current component ids of the specified entity type.

        Args:
            component (xms.components.bases.coverage_component_base.CoverageComponentBase): The coverage component to
                load id files for
            points (bool): True if you want point component id maps
            arcs (bool): True if you want arc component id maps
            arc_groups (bool): True if you want arc group component id maps
            polygons (bool): True if you want polygon component id maps
            only_selected (bool): If True, only retrieves mappings for the currently selected features
            delete_files (bool): If True, id files will be deleted before return

        Returns:
            dict: The file dict.

            key = entity_type ('POINT', 'ARC', 'POLYGON'),

            value = (att_id_file, comp_id_file)

            Note that if delete_files=True, these files will not exist upon return. If you need to make a copy of
            the component id files, set delete_files=False and manually clean up the files after copying.
        """
        # Don't delete the files until we copy them.
        files_dict = super().load_component_ids(
            component,
            points=points,
            arcs=arcs,
            arc_groups=arc_groups,
            polygons=polygons,
            only_selected=only_selected,
            delete_files=False
        )
        if only_selected:
            comp_dict = self._requested_data['selected_component_id_files'].setdefault(component.uuid, {})
        else:
            comp_dict = self._requested_data['component_id_files'].setdefault(component.uuid, {})
        cov_dict = comp_dict.setdefault(component.cov_uuid, {})

        comp_id_folder = os.path.join(self._recording_folder, 'ComponentIds')
        os.makedirs(comp_id_folder, exist_ok=True)
        for target_type, filenames in files_dict.items():
            if os.path.isfile(filenames[0]):  # No file to copy if requested an entity with no component ids.
                new_att_file = os.path.join(comp_id_folder, os.path.basename(filenames[0]))
                io_util.copyfile(filenames[0], new_att_file)
                new_comp_file = os.path.join(comp_id_folder, os.path.basename(filenames[1]))
                io_util.copyfile(filenames[1], new_comp_file)
                cov_dict[target_type] = (  # Store relative paths to the files
                    os.path.join('ComponentIds', os.path.basename(new_att_file)),
                    os.path.join('ComponentIds', os.path.basename(new_comp_file)),
                )
            else:  # But still leave an entry in the returned map to keep behavior consistent on playback.
                cov_dict[target_type] = ('', '')
            if delete_files:
                io_util.removefile(filenames[0])  # XMS id file
                io_util.removefile(filenames[1])  # Component id file

        return files_dict

    def coverage_attributes(self, cov_uuid, points=False, arcs=False, polygons=False, arc_groups=False):
        """Retrieve coverage attribute table from XMS.

        Args:
            cov_uuid (str): UUID of the coverage
            points (bool): True to retrieve feature point attributes
            arcs (bool): True to retrieve feature arc attributes
            polygons (bool): True to retrieve feature polygon attributes
            arc_groups (bool): True to retrieve feature arc group attributes

        Returns:
            dict: The requested attribute tables. Keyed by 'points', 'arcs', and 'polygons'.
        """
        att_tables = super().coverage_attributes(
            cov_uuid, points=points, arcs=arcs, polygons=polygons, arc_groups=arc_groups
        )

        att_folder = os.path.join(self._recording_folder, 'CoverageAttributes')
        os.makedirs(att_folder, exist_ok=True)
        cov_dict = self._requested_data['coverage_attributes'].setdefault(cov_uuid, {})
        for target_type, filename in att_tables.items():
            new_filename = ''
            if os.path.isfile(filename):
                new_filename = os.path.join(att_folder, os.path.basename(filename))
                io_util.copyfile(filename, new_filename)  # Copy the .csv file
                if os.path.isfile(f'{filename}.def'):
                    io_util.copyfile(f'{filename}.def', f'{new_filename}.def')  # Copy the .def file
                if os.path.isfile(f'{filename}.xy'):
                    io_util.copyfile(f'{filename}.xy', f'{new_filename}.xy')  # Copy the .xy file
                new_filename = os.path.join('CoverageAttributes', os.path.basename(new_filename))
            cov_dict[target_type] = new_filename

        return att_tables

    # Methods for adding data to be built in XMS.
    def add_simulation(self, do_sim, components=None, component_keywords=None):
        """Add a simulation to XMS.

        Args:
            do_sim (pydop.Simulation): The simulation to add
            components (list of pydop.Component): The simulation's hidden components. Component objects must have the
                model name and unique name attributes set.
            component_keywords (list of dict): Parallel with components if provided. Dicts of additional component
                keywords to send with the components, such as 'actions' or 'display_options'. See kwarg docs of
                add_component() for more details.
                ::
                    [{  # Possible component data keywords
                        'actions': [],
                        'messages': [],
                        'display_options': [],
                        'component_coverage_ids': [],
                        'shapefile_atts': {},
                    }]
        """
        super().add_simulation(do_sim, components=components, component_keywords=component_keywords)
        self._sent_data['simulations'] += 1
        if components:
            self._sent_data['simulation_components'] += len(components)

    def add_coverage(
        self,
        do_coverage,
        model_name=None,
        coverage_type=None,
        components=None,
        component_keywords=None,
        folder_path=None
    ):
        """Add a coverage to XMS.

        Args:
            do_coverage (Union[pydop.Coverage, xms.api._xmsapi.dmi.DataDumpIOBase]): The coverage to add. If type is an
                xmscoverage generic coverage dump, all other arguments are ignored.
            model_name (str): XML-defined name of the coverage's model. If specified, must provide coverage_type.
            coverage_type (str): XML-defined coverage type. If specified, must provide model_name.
            components (list of pydop.Component): The simulation's hidden components. Component objects must have the
                model name and unique name attributes set.
            component_keywords (list of dict): Parallel with components if provided. Dicts of additional component
                keywords to send with the components, such as 'actions' or 'display_options'. See kwarg docs of
                add_component() for more details.
                ::
                    [{  # Possible component data keywords
                        'actions': [],
                        'messages': [],
                        'display_options': [],
                        'component_coverage_ids': [],
                        'shapefile_atts': {},
                    }]
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        super().add_coverage(
            do_coverage,
            model_name=model_name,
            coverage_type=coverage_type,
            components=components,
            component_keywords=component_keywords,
            folder_path=folder_path
        )
        if isinstance(do_coverage, pydop.Coverage):
            self._sent_data['coverages'] += 1
            if components:
                self._sent_data['coverage_components'] += len(components)
        else:
            self._sent_data['generic_coverages'] += 1

    def add_ugrid(self, do_ugrid):
        """Add a UGrid to be built in XMS.

        Args:
            do_ugrid (pydop.UGrid): The UGrid to add
        """
        super().add_ugrid(do_ugrid)
        self._sent_data['ugrids'] += 1

    def add_dataset(self, do_dataset, folder_path=None):
        """Add a Dataset to be built in XMS.

        Args:
            do_dataset (Union[pydop.Dataset, DatasetReader, xms.datasets.dataset_writer.DatasetWriter]): The dataset to
                add
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        super().add_dataset(do_dataset, folder_path=folder_path)
        self._sent_data['datasets'] += 1

    def add_raster(self, filename, item_uuid):
        """Add a GIS raster file to be loaded in XMS.

        Args:
            filename (str): Full path to the raster file
            item_uuid (Optional[str]): UUID to assign the raster in XMS
        """
        super().add_raster(filename, item_uuid)
        self._sent_data['rasters'] += 1

    def add_shapefile(self, filename, item_uuid):
        """Add a GIS raster file to be loaded in XMS.

        Args:
            filename (str): Full path to the raster file
            item_uuid (Optional[str]): UUID to assign the shapefile in XMS
        """
        super().add_shapefile(filename, item_uuid)
        # Lazily add this to the output since we added the feature for sending shapefiles after we implemented Query
        # recording and I don't want to have to update every existing recording test.
        if 'shapefiles' not in self._sent_data:
            self._sent_data['shapefiles'] = 0
        self._sent_data['shapefiles'] += 1

    def add_component(
        self,
        do_component,
        actions=None,
        messages=None,
        display_options=None,
        coverage_ids=None,
        shapefile_atts=None,
        folder_path=None
    ):
        """Add a non-hidden Component to be built in XMS.

        Args:
            do_component (pydop.Component): The component to add
            actions (list of pydop.ActionRequest): ActionRequests to send back with the component. See
                xms.components.bases.component_base.ComponentBase for more details.
            messages (list of tuple of str): Messages to send back with component. Should be formatted as follows:
                [('MSG_LEVEL', 'message text')...]  See xms.components.bases.component_base.ComponentBase for more
                details.
            display_options (list): Display options to send back with the component. Data should be in format returned
                by xms.components.bases.component_base.ComponentBase.get_display_options().
            coverage_ids (list): Coverage component id assignments to send back with the component. Data should be in
                format returned by
                xms.components.bases.coverage_component_base.CoverageComponentBase.get_component_coverage_ids().
            shapefile_atts (dict): Shapefile attributes to send back with the component. Data should be in
                format returned by xms.components.bases.component_base.ComponentBase.get_shapefile_att_files().
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        super().add_component(
            do_component,
            actions=actions,
            messages=messages,
            display_options=display_options,
            coverage_ids=coverage_ids,
            shapefile_atts=shapefile_atts,
            folder_path=folder_path
        )
        self._sent_data['components'] += 1

    def add_model_check_errors(self, errors):
        """Adds model check errors to be sent back to XMS.

        Errors will all be set at the Context level we had at construction. It is assumed that nothing else will be
        added during the script.

        Args:
            errors (Union[pydop.ModelCheckError, Sequence[pydop.ModelCheckError]): The model check errors to add. Can
                pass an iterable or a single error
        """
        super().add_model_check_errors(errors)
        if not isinstance(errors, Iterable):  # Allow argument to be a single error
            self._sent_data['model_check_errors'] += 1
        else:
            self._sent_data['model_check_errors'] += len(errors)

    def link_item(self, taker_uuid, taken_uuid):
        """Link one item under another.

        Args:
            taker_uuid (str): UUID of the item to link the other item under. This should be the UUID of the parent tree
                item, even if the item is taken by a hidden component.
            taken_uuid (str): UUID of the item to link under the other item
        """
        super().link_item(taker_uuid, taken_uuid)
        self._sent_data['link_items'] += 1

    def unlink_item(self, taker_uuid, taken_uuid):
        """Unlink one item from under another.

        Args:
            taker_uuid (str): UUID of the item to unlink the other item from. This should be the UUID of the parent tree
                item, even if the item is taken by a hidden component.
            taken_uuid (str): UUID of the item to unlink from the other item
        """
        super().unlink_item(taker_uuid, taken_uuid)
        self._sent_data['unlink_items'] += 1

    def delete_item(self, item_uuid):
        """Delete an item in XMS.

        Args:
            item_uuid (str): UUID of the item to delete
        """
        super().delete_item(item_uuid)
        self._sent_data['delete_items'] += 1

    def rename_item(self, item_uuid, new_name):
        """Rename a tree item in XMS.

        Args:
            item_uuid (str): UUID of the item to rename
            new_name (str): New name to assign the item. Not guaranteed to be the name XMS assigns the item. Will be
                made unique if another item with the same name already exists.
        """
        super().rename_item(item_uuid, new_name)
        self._sent_data['rename_items'] += 1

    def edit_ugrid_top_bottom_elevations(self, json_file):
        """Edit the top and bottom elevations of a UGrid in XMS.

        Args:
            json_file (str): Path to the JSON file containing the new elevations. See xms.components.dataset_metadata
                for more information of the expected JSON structure.
        """
        super().edit_ugrid_top_bottom_elevations(json_file)
        self._sent_data['ugrid_elevation_edits'] += 1

    def get_ugrid_selection(self, ugrid_uuid):
        """Get the currently selected UGrid cells and points.

        Returns:
            dict: Possible keys are 'POINT' and 'CELL'. Values are lists of UGrid point/cell ids. If key does not
            exist, no UGrid entities of that type are selected.
        """
        selection = super().get_ugrid_selection(ugrid_uuid)
        self._requested_data['ugrid_selections'][ugrid_uuid] = selection
        return selection

    # Send any built data to XMS.
    def send(self, component_event=False):
        """Send data queued from add() and set() calls to XMS.

        Args:
            component_event (bool): True if send is being sent from a component event script. They do all sorts of their
                own special messing with the build context, so don't try to manage it.

        Returns:
            xms.api.dmi.SetDataResult: Result object reply from the send request
        """
        self._write_output_files()
        super().send()

    def clear(self):
        """Call this to clear any data added to the Context to prevent it from being sent to XMS."""
        self._reset_sent_data()
        super().clear()
