"""DMI Python definition for the 3D Bridge coverage component."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem
from xms.components.bases.component_base import ComponentBase
from xms.data_objects.parameters import FilterLocation

# 4. Local modules
from xms.bridge.bridge import Bridge
from xms.bridge.bridge_io import BridgeIo
from xms.bridge.gui.bridge_dialog import BridgeDialog


class BridgeComponent(ComponentBase):
    """A Dynamic Model Interface (DMI) component for a MODFLOW 6 simulation."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (:obj:`str`): The main file associated with this component.
        """
        super().__init__(main_file)
        self._ensure_main_file_exists()

    def _ensure_main_file_exists(self):
        """Create a default main file if none exists."""
        if not os.path.isfile(self.main_file):
            bridge = Bridge()
            io = BridgeIo()
            io.write(bridge, self.main_file)

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE - happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE - happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE - happens when re-saving this project.

                SAVE_AS - happens when saving a project in a new location. This happens the first time we save a
                project.

                UNLOCK - happens when the component is about to be changed and it does not have a matching uuid folder
                in the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`):
                tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path if
                necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        return os.path.join(new_path, os.path.basename(self.main_file)), [], []

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (:obj:`list[str]`): A list of the main files of the selected components of this type.

        Returns:
            menu_items (:obj:`list[xms.api.dmi.MenuItem]`):
                A list of menus and menu items to be shown. Note that this list can have objects of type
                xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be added to the list to indicate a
                separator.
        """
        if len(main_file_list) > 1 or not main_file_list:
            return []  # Multi-select or nothing selected

        menu_list = [None]  # None == spacer
        # Add the Open menu command that will display the well dialog when clicked.
        bridge_prop_action = ActionRequest(
            modality='MODAL',
            module_name='xms.bridge.bridge_component',
            class_name='BridgeComponent',
            method_name='edit_bridge_dialog',
            main_file=main_file_list[0][0]
        )
        bridge_prop_item = MenuItem(text='Edit Bridge...', action=bridge_prop_action)
        menu_list.append(bridge_prop_item)
        menu_list.append(None)  # Adds a separator
        return menu_list

    def edit_bridge_dialog(self, query, params, win_cont):
        """Opens the bridge dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`list[str]`): Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages(:obj:`list[tuple(str)]`): A list of tuples with the first element of the tuple
                being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.

                action_requests(:obj:`list[xms.api.dmi.ActionRequest]`): A list of actions for XMS to perform.

        """
        # instantiate the bridge
        bridge_io = BridgeIo()
        bridge = bridge_io.read(self.main_file)

        # get the coverage arc ids
        coverage = query.parent_item()
        arcs = []
        wkt = ''
        if coverage:
            arcs = coverage.arcs
            wkt = coverage.projection.well_known_text

        # Set bridge arc IDs to first two if no match exists (or 0 if not enough arcs)
        if arcs and len(arcs) > 1:
            arc_ids = {arcs[0].id, arcs[1].id}
            if bridge.arc_id_upstream not in arc_ids or bridge.arc_id_downstream not in arc_ids:
                bridge.arc_id_upstream = arcs[0].id
                bridge.arc_id_downstream = arcs[1].id
        else:
            bridge.arc_id_upstream = 0
            bridge.arc_id_downstream = 0

        arc_upstream = self.get_arc(bridge.arc_id_upstream, arcs)
        arc_downstream = self.get_arc(bridge.arc_id_downstream, arcs)

        # Get the arc points
        arc_pts_up_list = self.get_arc_points(arc_upstream)
        arc_pts_down_list = self.get_arc_points(arc_downstream)

        # Run the dialog
        input = {
            'bridge': bridge,
            'arc_up': arc_pts_up_list,
            'arc_down': arc_pts_down_list,
            'wkt': wkt,
            'main_file': self.main_file,
            'cover': coverage
        }
        dlg = BridgeDialog(input, win_cont)
        dlg.setModal(True)
        dlg_result = dlg.exec()
        if dlg_result == QDialog.Accepted:
            bridge_io.write(bridge, self.main_file)
            if dlg.output_ugrid:
                query.add_ugrid(dlg.output_ugrid)

        return [], []

    def get_arc(self, bridge_arc_id, arcs):
        """Returns the arc with bridge_arc_id. If not found, returns arc with index = arc_id_to_use_as_substitute.

        Args:
            bridge_arc_id (:obj:`int`): ID of arc from bridge main file.
            arcs (:obj:`list[arcs]`): List of arcs.

        Returns:
            arc (:obj:`arc`): The arc. See description.
        """
        # Find the arc
        for arc in arcs:
            if arc.id == bridge_arc_id:
                return arc
        return None

    def get_arc_points(self, arc):
        """Extract a list of points from a coverage arc in order from start node to end node.

        Args:
            arc (:obj:`xms.data_objects.parameters.Arc`): The arc to get the points of

        Returns:
            (:obj:`list`): See description
        """
        return [(pt.x, pt.y) for pt in arc.get_points(FilterLocation.PT_LOC_ALL)] if arc else None
