"""CulvertCalculator class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math
import os

# 2. Third party modules
import pandas as pd
from shapely.geometry import box, LineString, Point, Polygon

# 3. Aquaveo modules
from xms.data_objects.parameters import Arc, Coverage, FilterLocation, Point as doPoint
from xms.extractor import UGrid2dDataExtractor
from xms.grid.ugrid.ugrid_utils import read_ugrid_from_ascii_file
from xms.tool.algorithms.mesh_2d.bridge_footprint import ArcType

# 4. Local modules
from xms.bridge import structure_util as su
from xms.bridge.grid_builder import GridBuilder


def _data_extractor_from_ugrid(ugrid):
    """Create a UGrid2dDataExtractor from a UGrid."""
    de = UGrid2dDataExtractor(ugrid=ugrid)
    ug_locs = ugrid.locations
    ug_z = [p[2] for p in ug_locs]
    act = [1] * len(ug_z)
    de.set_grid_point_scalars(ug_z, act, 'points')
    return de


def _top_surface_ugrid(builder_data):
    """Get a UGrid of the top surface 2d mesh.

    Args:
        builder_data (:obj:`dict`): dict with data for the grid builder
    """
    grid_builder = GridBuilder()
    grid_builder.build_top_and_bottom_from_dict(builder_data)
    tmp_dir = os.path.dirname(builder_data['main_file'])
    top_mesh_file = os.path.join(tmp_dir, 'top2d.xmugrid')
    ug = read_ugrid_from_ascii_file(top_mesh_file)
    return ug


def _culvert_base_elev_dataframes(culvert_data):
    """Create dataframes for the base elevation of 3d culvert.

    Args:
        culvert_data (:obj:`dict`): dict of culvert variables
    Returns:
        (tuple(Dataframe, Dataframe)): down stream and up stream dataframes
    """
    base_elev = culvert_data['base_elev']
    top_dn = culvert_data['profiles'][2]
    dist_vals = top_dn['Distance'].tolist()
    dn_profile = [(d, base_elev) for d in dist_vals]
    down_stream = pd.DataFrame(dn_profile, columns=['Distance', 'Elevation'])

    top = culvert_data['profiles'][3]
    dist_vals = top['Distance'].tolist()
    up_profile = [(d, base_elev) for d in dist_vals]
    up_stream = pd.DataFrame(up_profile, columns=['Distance', 'Elevation'])
    return down_stream, up_stream


def _extend_line_to_bounds(ls, bounds):
    """Extend a linestring to the bounds.

    Args:
        ls (:obj:`LineString`): line string
        bounds (:obj:`list`): xmin, ymin, xmax, ymax

    Returns:
        (:obj:`LineString`): line string extended to the bounds
    """
    minx, miny, maxx, maxy = bounds
    a, b = ls.boundary.geoms
    if a.x == b.x:  # vertical line
        extended_line = LineString([(a.x, miny), (a.x, maxy)])
    elif a.y == b.y:  # horizontal line
        extended_line = LineString([(minx, a.y), (maxx, a.y)])
    else:
        # linear equation: y = k*x + m
        k = (b.y - a.y) / (b.x - a.x)
        m = a.y - k * a.x
        y0 = k * minx + m
        y1 = k * maxx + m
        x0 = (miny - m) / k
        x1 = (maxy - m) / k
        points_on_boundary_lines = [Point(minx, y0), Point(maxx, y1), Point(x0, miny), Point(x1, maxy)]
        bb = box(minx, miny, maxx, maxy)
        points_sorted_by_distance = sorted(points_on_boundary_lines, key=bb.distance)
        extended_line = LineString(points_sorted_by_distance[:2])
    return extended_line


def _calc_culvert_wall_piers(offsets, culvert_ls_extended, center_line, bridge_width):
    """Calculate the wall pier linestrings for the culvert.

    Args:
        offsets (:obj:`list`): offsets for the wall piers
        culvert_ls_extended (:obj:`LineString`): extended culvert line
        center_line (:obj:`LineString`): center line of the bridge
        bridge_width (:obj:`float`): width of the bridge
    """
    width = bridge_width
    cl_buf = center_line.buffer(0.5 * bridge_width)
    wall_piers = []
    for offset in offsets:
        ls = culvert_ls_extended.parallel_offset(offset)
        if not center_line.intersects(ls):
            raise RuntimeError('Error culvert barrel walls extend beyond embankment centerline. Aborting.')
        wp = ls.intersection(cl_buf)
        while wp.length > center_line.length:
            width = 0.9 * width
            cl_buf = center_line.buffer(0.5 * width)
            wp = ls.intersection(cl_buf)
        wall_piers.append(ls.intersection(cl_buf))
    return wall_piers


def _calc_barrel_polygon(offsets, culvert_ls_extended, center_line, bridge_width):
    """Create the barrel polygon from the wall piers.

    Args:
        offsets (:obj:`list`): offsets for the wall piers
        culvert_ls_extended (:obj:`LineString`): extended culvert line
        center_line (:obj:`LineString`): center line of the bridge
        bridge_width (:obj:`float`): width of the bridge
    """
    cl_buf = center_line.buffer(3 * bridge_width)  # create a large offset to handle skew
    cl_extended = _extend_line_to_bounds(center_line, cl_buf.bounds)
    coords = [cl_extended.coords[0]] + list(center_line.coords) + [cl_extended.coords[1]]
    cl_extended = LineString(coords)

    cl_buf = cl_extended.buffer(0.5 * bridge_width)
    wp_lines = []
    for offset in offsets:
        ls = culvert_ls_extended.parallel_offset(offset)
        wp_lines.append(ls.intersection(cl_buf))
    right_list = list(wp_lines[0].coords)
    left_list = list(wp_lines[-1].coords)
    return right_list + left_list[::-1]


def _trim_arc_lengths(arc_ls_list, center_line):
    """Trim the arc lengths to be no longer than the center line length.

    Args:
        arc_ls_list (:obj:`list`): list of linestrings
        center_line (:obj:`LineString`): center line of the bridge

    Returns:
        (:obj:`list`): list of trimmed linestrings
    """
    cl_length = center_line.length
    factor = 0.4
    buf = center_line.buffer(factor * center_line.length)
    out_ls = []
    for ls in arc_ls_list:
        out_ls.append(ls)
        if not ls.intersects(center_line):
            continue
        while ls.length > cl_length:
            ls = ls.intersection(buf)
            if ls.length > cl_length:
                factor *= 0.9
                buf = center_line.buffer(factor * center_line.length)
        out_ls[-1] = ls
    return out_ls


class _CulvertData:
    """Class for culvert data variables."""
    def __init__(self, culvert_data):
        """Constructor.

        Args:
            culvert_data (:obj:`dict`): dict of culvert variables
        """
        self.type = culvert_data.get('culvert_type', 'Circular')
        self.bridge_width = culvert_data.get('bridge_width', 0.0)
        self.diameter = culvert_data.get('culvert_diameter', 0.0)
        self.num_seg_barrel = culvert_data.get('culvert_num_seg_barrel', 1)
        self.cov_file = culvert_data.get('coverage_file', '')
        self.has_abutments = culvert_data.get('has_abutments', False)
        self.embed_depth = culvert_data.get('culvert_embed_depth', 0.0)
        self.rise = culvert_data.get('culvert_rise', None)
        self.span = culvert_data.get('culvert_span', 0.0)
        self.ave_top = culvert_data.get('ave_top', 0.0)
        self.ave_bot = culvert_data.get('ave_bot', 0.0)
        self.num_barrels = culvert_data.get('num_barrels', 1)
        self.wall_width = culvert_data.get('culvert_wall_width', 0.0)
        self.culvert_width = self.diameter if self.type == 'Circular' else self.span
        if self.wall_width <= 0.0:
            self.wall_width = self.culvert_width / self.num_seg_barrel
        self.arc_types = culvert_data.get('culvert_arc_properties', {})

    def error_check_culvert_data(self):
        """Check culvert data for errors."""
        if self.bridge_width <= 0.0:
            raise RuntimeError
        self._error_check_circular_culvert_diameter()
        self._error_check_box_culvert_rise_span()
        self._check_culvert_arc_types()

    def _error_check_circular_culvert_diameter(self):
        """Check culvert diameter for errors."""
        if self.type != 'Circular':
            return

        if self.diameter <= 0.0:
            raise RuntimeError('Culvert diameter must be greater than 0.0.')
        if self.embed_depth >= self.diameter / 2.0:
            raise RuntimeError('Culvert embedment depth must be less than the culvert radius.')
        culvert_height = self.diameter
        if self.ave_bot + culvert_height >= self.ave_top:
            val = self.ave_top - self.ave_bot
            raise RuntimeError(
                f'Culvert diameter is too large. The diameter must be less than {val} '
                'to be lower than the top of the embankment height.'
            )

    def _error_check_box_culvert_rise_span(self):
        """Error check the box culvert values."""
        if self.type != 'Box':
            return
        if self.rise <= 0.0:
            raise RuntimeError('Culvert rise must be greater than 0.0.')
        if self.span <= 0.0:
            raise RuntimeError('Culvert span must be greater than 0.0.')
        if self.embed_depth >= self.rise:
            raise RuntimeError('Culvert embedment depth must be less than the culvert rise.')
        culvert_height = self.rise
        if self.ave_bot + culvert_height >= self.ave_top:
            val = self.ave_top - self.ave_bot
            raise RuntimeError(
                f'Culvert rise is too large. The rise must be less than {val} '
                'to be lower than the top of the embankment height.'
            )

    def _check_culvert_arc_types(self):
        """Verify that the culvert arc types are valid."""
        if len(self.arc_types) < 1:
            return
        arc_types = list(self.arc_types.values())
        if arc_types.count('Culvert') != 1:
            raise RuntimeError('There must be exactly one arc of type "Culvert". Aborting.')
        if arc_types.count('Embankment') != 1:
            raise RuntimeError('There must be exactly one arc of type "Embankment". Aborting.')
        if arc_types.count('Abutment') == 1 or arc_types.count('Abutment') > 2:
            raise RuntimeError('There must be two arcs of type "Abutment" if using abutments. Aborting.')


class CulvertCalculator:
    """Class for doing various culvert calculations."""
    def __init__(self, struct_comp, coverage):
        """Constructor.

        Args:
            struct_comp (:obj:`StructureComponent`): the structure component
            coverage (:obj:`data_objects - Coverage`): xms coverage
        """
        self._comp = struct_comp
        self._coverage = coverage
        self._culvert_data = None
        cl_arc, cl_pts = su.bridge_centerline_from_coverage(coverage, struct_comp)
        self._cl_arc = cl_arc
        self._cl_pts = cl_pts
        self._cl_ls = LineString(cl_pts)
        self._culvert_arc = None
        self._culvert_pts = None
        self._culvert_ls = None
        self._culvert_barrel_poly = None
        self.has_abutments = None
        self._culvert_width = None
        self._culvert_right_ls = None
        self._culvert_left_ls = None
        self._new_cl_pts = None
        self._culvert_ls_extended = None
        self.wall_width = None
        self.bridge_width = None
        self.barrels_poly = []
        self._wall_piers = []
        self._barrel_cl = []
        self.err_msg = ''
        self.arc_data = {}

    def has_culvert_arc(self):
        """Tells if the class has a culvert arc.

        Returns:
            (:obj:`bool`): True if it has a culvert arc
        """
        if self._culvert_arc is None:
            return False
        return True

    def _update_bridge_width_for_skew(self):
        """Update the bridge width to account for skew angle."""
        bridge_width = self._culvert_data.bridge_width
        embank_ls = self._cl_ls
        v1_x = embank_ls.coords[1][0] - embank_ls.coords[0][0]
        v1_y = embank_ls.coords[1][1] - embank_ls.coords[0][1]
        culvert_ls = self._culvert_ls
        v2_x = culvert_ls.coords[1][0] - culvert_ls.coords[0][0]
        v2_y = culvert_ls.coords[1][1] - culvert_ls.coords[0][1]
        dot = v1_x * v2_x + v1_y * v2_y
        magnitude_v1 = math.sqrt(v1_x**2 + v1_y**2)
        magnitude_v2 = math.sqrt(v2_x**2 + v2_y**2)
        cosine_theta = dot / (magnitude_v1 * magnitude_v2)
        cosine_theta = max(-1.0, min(1.0, cosine_theta))
        angle_radians = math.acos(cosine_theta)
        self._culvert_data.bridge_width = math.sin(angle_radians) * bridge_width
        self.bridge_width = self._culvert_data.bridge_width

    def _offset_culvert_line(self):
        """Offset the culvert line to the locations where wall piers will be located."""
        bridge_width = self._culvert_data.bridge_width
        self.wall_width = self._culvert_data.wall_width
        # multiple barrels
        total_width = self._culvert_data.culvert_width + self._culvert_data.wall_width
        offsets = []
        offset_factor = self._culvert_data.num_barrels / 2.
        for i in range(self._culvert_data.num_barrels + 1):
            offsets.append((offset_factor - i) * total_width)

        barrel_center_offsets = []
        for i in range(self._culvert_data.num_barrels):
            barrel_center_offsets.append((offset_factor - i - 0.5) * total_width)

        cl_buf = self._cl_ls.buffer(3 * bridge_width)  # create a large offset to handle skew
        culvert_ls_extended = _extend_line_to_bounds(self._culvert_ls, cl_buf.bounds)
        self._culvert_ls_extended = culvert_ls_extended
        self._wall_piers = _calc_culvert_wall_piers(offsets, culvert_ls_extended, self._cl_ls, bridge_width)
        self._calc_barrel_polygon(offsets, culvert_ls_extended)
        # calculate the lines that represent the barrel centers
        self._barrel_cl = []
        for offset in barrel_center_offsets:
            self._barrel_cl.append(culvert_ls_extended.parallel_offset(offset))

    def _calc_barrel_polygon(self, offsets, culvert_ls_extended):
        """Create the barrel polygon from the wall piers."""
        bridge_width = self._culvert_data.bridge_width
        self.barrels_poly = _calc_barrel_polygon(offsets, culvert_ls_extended, self._cl_ls, bridge_width)
        self._culvert_barrel_poly = Polygon(self.barrels_poly).buffer(self.wall_width * 0.1)

    def _calc_new_centerline(self):
        """Calculate the coordinates of a new centerline used to generate the culvert mesh."""
        cl_ls = self._cl_ls
        cl_svals = [cl_ls.project(Point(p[0], p[1])) / cl_ls.length for p in self._cl_pts]
        # calculate the parametric range of the wall piers on the embankment centerline
        wall_pier_s = (
            cl_ls.project(self._wall_piers[0].intersection(cl_ls),
                          normalized=True), cl_ls.project(self._wall_piers[-1].intersection(cl_ls), normalized=True)
        )
        wall_pier_s = sorted(wall_pier_s)
        if math.isnan(wall_pier_s[0]) or math.isnan(wall_pier_s[1]):
            raise RuntimeError('Error culvert barrel walls extend beyond embankment centerline. Aborting.')
        # remove all vertices on the embankment centerline between the wall piers
        new_cl_svals = [s for s in cl_svals if s < wall_pier_s[0]]
        # insert a cluster of vertices at the center of each culvert barrel,
        # the footprint mesh tool will redistribute these points inside the barrel between the wraps on the wall piers
        n_pts_cluster = self._culvert_data.num_seg_barrel - 3
        if n_pts_cluster > 0:
            # delta s for the cluster of points
            delta_s_pts = (n_pts_cluster - 1) * 0.001
            ds = (wall_pier_s[1] - wall_pier_s[0])
            delta_s = ds / self._culvert_data.num_barrels
            # center of the first barrel
            s_center = wall_pier_s[0] + delta_s / 2.
            for _ in range(self._culvert_data.num_barrels):
                # first location in the cluster
                s_pt = s_center - (delta_s_pts / 2.)
                for _ in range(n_pts_cluster):
                    new_cl_svals.append(s_pt)
                    s_pt += 0.001
                s_center += delta_s
        new_cl_svals.extend([s for s in cl_svals if s > wall_pier_s[1]])
        new_cl_pts = [cl_ls.interpolate(s, normalized=True) for s in new_cl_svals]
        self._new_cl_pts = [(pp.coords[0][0], pp.coords[0][1]) for pp in new_cl_pts]

    def _create_culvert_coverage(self):
        """Make a new coverage to generate the culvert mesh."""
        cov_arcs = self._coverage.arcs
        cl_pts = self._new_cl_pts
        start_node = doPoint(x=cl_pts[0][0], y=cl_pts[0][1], z=0.0, feature_id=1)
        end_node = doPoint(x=cl_pts[-1][0], y=cl_pts[-1][1], z=0.0, feature_id=2)
        verts = []
        next_node_id = 3
        for v in cl_pts[1:-1]:
            p = doPoint(x=v[0], y=v[1], z=0.0, feature_id=next_node_id)
            next_node_id += 1
            verts.append(p)
        new_arcs = [Arc(feature_id=1, start_node=start_node, end_node=end_node, vertices=verts)]
        skip_arcs = [self._culvert_arc.id, self._cl_arc.id]
        if self._culvert_data.arc_types:
            arc_ids = [arc.id for arc in cov_arcs]
            for arc_id in arc_ids:
                if self._culvert_data.arc_types[arc_id] == 'Unassigned':
                    skip_arcs.append(arc_id)

        cl_ls = LineString(cl_pts)
        next_arc_id = 2
        other_arcs = []
        for arc in cov_arcs:
            if arc.id in skip_arcs:
                continue
            # These are abutment arcs. They must be shorter than the embankment centerline.
            p0 = (arc.start_node.x, arc.start_node.y)
            p1 = (arc.end_node.x, arc.end_node.y)
            ls = LineString([p0, p1])
            other_arcs.append(ls)

        other_arcs = other_arcs + self._wall_piers
        other_arcs = _trim_arc_lengths(other_arcs, cl_ls)

        for ls in other_arcs:
            arc_pts = list(ls.coords)
            sn = arc_pts[0]
            start_node = doPoint(x=sn[0], y=sn[1], z=0.0, feature_id=next_node_id)
            next_node_id += 1
            en = arc_pts[1]
            end_node = doPoint(x=en[0], y=en[1], z=0.0, feature_id=next_node_id)
            next_node_id += 1
            new_arc = Arc(feature_id=next_arc_id, start_node=start_node, end_node=end_node)
            new_arcs.append(new_arc)
            next_arc_id += 1

        cov = Coverage()
        cov.arcs = new_arcs
        cov.write_h5(self._culvert_data.cov_file)

    def _calc_arc_data_for_display(self):
        """Fill a list with the arc id and arc type to be drawn in the preview window."""
        self.arc_data = self._get_arcs_from_coverage()
        arc_types = self._culvert_data.arc_types
        for arc in self.arc_data:
            if arc['id'] == self._cl_arc.id:
                arc['arc_type'] = ArcType.BRIDGE
            elif arc['id'] == self._culvert_arc.id:
                arc['arc_type'] = ArcType.PIER
            else:
                if arc_types:
                    if arc_types[arc['id']] == 'Abutment':
                        arc['arc_type'] = ArcType.ABUTMENT
                    else:
                        arc['arc_type'] = 'Unassigned'
                else:
                    arc['arc_type'] = ArcType.ABUTMENT

    def _get_arcs_from_coverage(self) -> dict:
        """Get the arcs from a coverage.

        Returns:
            (dict): A dictionary of 'id', 'arc_pts', 'start_node', 'end_node', 'line_string' (shapely LineString),
            and 'length'.
        """
        arcs = []
        coverage_arcs = self._coverage.arcs
        for arc in coverage_arcs:
            pts = [(p.x, p.y, p.z) for p in arc.get_points(FilterLocation.PT_LOC_ALL)]
            arc_dict = {'id': arc.id, 'arc_pts': pts, 'start_node': arc.start_node.id, 'end_node': arc.end_node.id}
            arcs.append(arc_dict)
        return arcs

    def _culvert_arc_to_linestring(self):
        """Build linestring from the culvert arc."""
        sn = self._culvert_arc.start_node
        en = self._culvert_arc.end_node
        pts = [(sn.x, sn.y), (en.x, en.y)]
        self._culvert_pts = pts
        self._culvert_ls = LineString(pts)

    def _set_arcs_from_arc_types(self):
        """Set the centerline and culvert arc based on user specified arc types."""
        for arc in self._coverage.arcs:
            if self._culvert_data.arc_types[arc.id] == 'Culvert':
                self._culvert_arc = arc
                self._culvert_arc_to_linestring()
            elif self._culvert_data.arc_types[arc.id] == 'Embankment':
                self._cl_arc = arc
                self._cl_pts = [(pt.x, pt.y) for pt in arc.get_points(FilterLocation.PT_LOC_ALL)]
                self._cl_ls = LineString(self._cl_pts)
            elif self._culvert_data.arc_types[arc.id] == 'Abutment':
                self.has_abutments = True

    def _validate_coverage_for_culvert_mesh(self):
        """Makes sure that the coverage is valid for a culvert mesh."""
        self.err_msg = ''
        self._culvert_arc = None
        if self._cl_arc is None:
            raise RuntimeError('No embankment arc defined in coverage.')

        if self._culvert_data and self._culvert_data.arc_types:
            self._set_arcs_from_arc_types()
            return

        cov_arcs = {}
        arcs = self._coverage.arcs
        for arc in arcs:
            if arc.id == self._cl_arc.id:
                continue
            sn = arc.start_node
            en = arc.end_node
            pts = [(sn.x, sn.y), (en.x, en.y)]
            arc_ls = LineString(pts)
            if self._cl_ls.intersects(arc_ls):
                output = self._cl_ls.intersection(arc_ls)
                if type(output) is Point:
                    my_dist = self._cl_ls.project(output)
                    cov_arcs[my_dist] = arc
                else:
                    msg = f'Arc id: {arc.id} does not intersect the embankment centerline at a point. Aborting.'
                    raise RuntimeError(msg)
            else:
                raise RuntimeError(f'Arc id: {arc.id} does not intersect the embankment centerline. Aborting.')

        if self.has_abutments:
            if len(cov_arcs) != 3:
                msg = (
                    f'3 arcs must intersect the embankment centerline for a culvert structure with '
                    f'abutments. Number of arcs that intersect the centerline: {len(cov_arcs)}.'
                )
                raise RuntimeError(msg)
        elif len(cov_arcs) != 1:
            msg = (
                f'Only 1 arc must intersect the embankment centerline for culvert structure without'
                f' abutments. Number of arcs that intersect the centerline: {len(cov_arcs)}.'
            )
            raise RuntimeError(msg)

        sorted_dist = sorted(cov_arcs)
        self._culvert_arc = cov_arcs[sorted_dist[0]] if len(cov_arcs) == 1 else cov_arcs[sorted_dist[1]]
        self._culvert_arc_to_linestring()

        # make sure that the centerline is long enough for the culvert dimensions
        min_length = self._culvert_data.culvert_width + 5 * self._culvert_data.wall_width
        if self._cl_ls.length < min_length:
            msg = f'The embankment centerline is too short for the culvert dimensions. Minimum length: {min_length}.'
            raise RuntimeError(msg)

    def new_coverage_for_culvert_mesh(self, culvert_data):
        """Creates a new coverage used to generate a culvert mesh.

        The input coverage has an embankment centerline arc, culvert arc, and optionally abutment arcs. The new
        coverage will have the embankment centerline arc, pier arcs that define the outer walls of the culvert,
        and optionally abutment arcs.

        Args:
            culvert_data (:obj:`dict`): dictionary with culvert data for creating the new coverage
        """
        self.err_msg = ''
        self._culvert_arc = None
        self._culvert_data = _CulvertData(culvert_data)
        self.has_abutments = self._culvert_data.has_abutments
        try:
            self._culvert_data.error_check_culvert_data()
            # make sure the coverage is valid for making a culvert mesh
            self._validate_coverage_for_culvert_mesh()
            # calculate the "bridge width" this is affected by the skew of the culvert arc to the embankment centerline
            self._update_bridge_width_for_skew()
            # offset the culvert line based on culvert width and spacing of elements across culvert barrel
            self._offset_culvert_line()
            # create new points to define the center line so that we get the correct elements
            # in the culvert barrel
            self._calc_new_centerline()
            # create the arcs in the coverage
            self._create_culvert_coverage()
            self._calc_arc_data_for_display()
        except RuntimeError as e:
            self.err_msg = str(e)

    def update_bridge_elev(self, culvert_data):
        """Sets the bridge elevations from the culvert data.

        Args:
            culvert_data (:obj:`dict`): needed data to update the bridge mesh elevations
        """
        if self._culvert_arc is None:
            return
        culvert_data['culvert_ls_extended'] = self._culvert_ls_extended
        culvert_data['barrel_cl'] = self._barrel_cl
        # this creates upstream and downstream profiles for the top half and bottom half of the culvert
        culvert_data['profiles'] = self.profiles_from_culvert_data(culvert_data)
        self._interp_culvert_top(culvert_data)
        self._interp_culvert_barrel(culvert_data)

    def _interp_culvert_barrel(self, culvert_data):
        """Sets the bridge elevations from the top profile of the culvert data.

        Args:
            culvert_data (:obj:`dict`): needed data to update the bridge mesh elevations
        """
        down_stream, up_stream = _culvert_base_elev_dataframes(culvert_data)
        builder_data = {
            'up_stream': up_stream,
            'down_stream': down_stream,
            'top_dn': culvert_data['profiles'][2],
            'top': culvert_data['profiles'][3],
            'up_arc': culvert_data['up_arc'],
            'down_arc': culvert_data['down_arc'],
            'wkt': culvert_data['wkt'],
            'main_file': culvert_data['tmp_main_file'],
            'match_parametric_values': False,
        }
        self._interp_grid_elevations(culvert_data, builder_data, self._culvert_barrel_poly)

    def _interp_grid_elevations(self, culvert_data, builder_data, polygon):
        """Interpolates elevations to the 2d mesh based on surfaces created for the 3d structure.

        Args:
            culvert_data (:obj:`dict`): dict with culvert parameters
            builder_data (:obj:`dict`): dict with data for the grid builder
            polygon (:obj:`Polygon`): polygon for the culvert barrel
        """
        ug = _top_surface_ugrid(builder_data)
        if ug is not None:
            locs = culvert_data['locs']
            elev = culvert_data['elev']
            de = _data_extractor_from_ugrid(ug)
            de.extract_locations = locs
            ug_elev = de.extract_data()
            if polygon is None:
                for i, z in enumerate(ug_elev):
                    if not math.isnan(z):
                        elev[i] = z
            else:
                ug_elev = de.extract_data()
                for i, z in enumerate(ug_elev):
                    if polygon.contains(Point((locs[i][0], locs[i][1]))):
                        elev[i] = z

    def _interp_culvert_top(self, culvert_data):
        """Sets the bridge elevations from the top profile of the culvert data.

        Args:
            culvert_data (:obj:`dict`): needed data to update the bridge mesh elevations
        """
        self._top_profile_dataframes_matching_culvert_dataframes(culvert_data)
        builder_data = {
            'up_stream': culvert_data['profiles'][0],
            'down_stream': culvert_data['profiles'][1],
            'top': self.top_up_df,
            'top_dn': self.top_dn_df,
            'up_arc': culvert_data['up_arc'],
            'down_arc': culvert_data['down_arc'],
            'wkt': culvert_data['wkt'],
            'main_file': culvert_data['tmp_main_file'],
        }
        # get the mesh of the top of the 3d structure, interpolate elevations
        self._interp_grid_elevations(culvert_data, builder_data, None)

    def _top_profile_dataframes_matching_culvert_dataframes(self, culvert_data):
        """Create dataframes for the top profile of the culvert with the same parametric values as the culvert barrel.

        Args:
            culvert_data (:obj:`dict`): dict with culvert parameters
        """
        top_df = culvert_data['top_profile']
        top_vals = list(top_df.itertuples(index=False, name=None))
        top_ls = LineString(top_vals)
        up_df = culvert_data['profiles'][1]
        up_dist = up_df['Distance'].tolist()
        top_up_vals = []
        for d in up_dist:
            o = top_ls.interpolate(d, normalized=True)
            top_up_vals.append((d, o.coords[0][1]))
        self.top_up_df = pd.DataFrame(top_up_vals, columns=['Distance', 'Elevation'])
        dn_df = culvert_data['profiles'][0]
        dn_dist = dn_df['Distance'].tolist()
        top_dn_vals = []
        for d in dn_dist:
            o = top_ls.interpolate(d, normalized=True)
            top_dn_vals.append((d, o.coords[0][1]))
        self.top_dn_df = pd.DataFrame(top_dn_vals, columns=['Distance', 'Elevation'])

    def profiles_from_culvert_data(self, culvert_data):
        """Get the bridge centerline arc from a coverage.

        Args:
            culvert_data (:obj:`dict`): dict with culvert parameters

        Returns:
            (:obj:`tuple(pandas.DataFrame, pandas.DataFrame)`): upstream and downstream profiles
        """
        rval = None
        if self._culvert_arc is None:
            return rval

        culvert_data['up_ls'] = LineString(culvert_data['up_arc'])
        culvert_data['dn_ls'] = LineString(culvert_data['down_arc'])

        up_df, dn_df = self._top_half_of_culvert(culvert_data)
        bot_up_df, bot_dn_df = self._bottom_half_of_culvert(culvert_data)
        rval = up_df, dn_df, bot_up_df, bot_dn_df

        return rval

    def _culvert_elev_offsets(self, diameter, center_z, culvert_ls, bridge_face_ls, is_box, rise):
        """Compute the x,y offsets from circle center given the diameter for 6 points on a half circle.

        Args:
            diameter (:obj:`float`): circle diameter
            center_z (:obj:`float`): elevation of circle center
            cuvlert_ls (:obj:`LineString`): Culvert line
            bridge_face_ls (:obj:`LineString`): the bridge face line where these locations are being calculated
            is_box (:obj:`bool`): True if the culvert is a box culvert
            rise (:obj:`float`): Box culvert rise

        Returns:
            (:obj:`list[x,y]`): list of offset coords from circle center
        """
        r = diameter / 2.0
        sqrt_3_over_2 = math.sqrt(3.0) / 2.0
        sqrt_2_over_2 = math.sqrt(2.0) / 2.0
        # locations left to right
        unit_circle = [
            (-1.0, 0.0), (-sqrt_3_over_2, 0.5), (-sqrt_2_over_2, sqrt_2_over_2), (-0.5, sqrt_3_over_2), (0.0, 1.0),
            (0.5, sqrt_3_over_2), (sqrt_2_over_2, sqrt_2_over_2), (sqrt_3_over_2, 0.5), (1.0, 0.0)
        ]
        if is_box:
            if rise == 0.0:
                unit_circle = [(-1.0, 0.0), (-0.99, 0.0), (0.99, 0.0), (1.0, 0.0)]
            else:
                unit_circle = [(-1.0, 0.0), (-0.99, rise / r), (0.99, rise / r), (1.0, 0.0)]
        locs = []
        for loc in unit_circle:
            offset = abs(r) * loc[0]
            offset_ls = culvert_ls.parallel_offset(distance=offset, side='left', resolution=1)
            sh_pt = offset_ls.intersection(bridge_face_ls)
            s_val = bridge_face_ls.project(sh_pt) / bridge_face_ls.length
            locs.append((s_val, center_z + r * loc[1]))
        return locs

    def _top_half_of_culvert(self, culvert_data):
        """Compute the top half of the culvert.

        Args:
            culvert_data (:obj:`dict`): dict with culvert parameters

        Returns:
            (:obj:`tuple(pandas.DataFrame, pandas.DataFrame)`): upstream and downstream profiles
        """
        diameter = culvert_data['culvert_diameter']
        center_diameter = diameter
        is_box = False
        rise = 0.0
        up_z = culvert_data['culvert_up_invert']
        if culvert_data['culvert_type'] == 'Box':
            is_box = True
            rise = culvert_data['culvert_rise']
            center_diameter = 0.0
            up_z = up_z + culvert_data['culvert_embed_depth']

        up_culvert_center_z = up_z + (center_diameter / 2)
        up_profile = [(0.0, up_culvert_center_z)]
        for culv_ls in culvert_data['barrel_cl']:
            circle_offsets = self._culvert_elev_offsets(
                diameter, up_culvert_center_z, culv_ls, culvert_data['up_ls'], is_box, rise
            )
            up_profile.extend(circle_offsets)
        up_profile.extend([(1.0, up_culvert_center_z)])
        up_profile.sort()
        up_df = pd.DataFrame(up_profile, columns=['Distance', 'Elevation'])

        dn_z = culvert_data['culvert_dn_invert']
        dn_culvert_center_z = dn_z + (center_diameter / 2)
        dn_profile = [(0.0, dn_culvert_center_z)]
        for culv_ls in culvert_data['barrel_cl']:
            circle_offsets = self._culvert_elev_offsets(
                diameter, dn_culvert_center_z, culv_ls, culvert_data['dn_ls'], is_box, rise
            )
            dn_profile.extend(circle_offsets)
        dn_profile.extend([(1.0, dn_culvert_center_z)])
        dn_profile.sort()
        dn_df = pd.DataFrame(dn_profile, columns=['Distance', 'Elevation'])
        return up_df, dn_df

    def _insert_embedment_locations(self, culvert_data, offsets, z, culvert_ls, bridge_face_ls):
        """Insert th embedment locations for the bottom half of the culvert.

        Args:
            culvert_data (:obj:`dict`): dict with culvert parameters
            offsets (:obj:`list`): current list of circle offsets
            z (:obj:`float`): invert elevation
            cuvlert_ls (:obj:`LineString`): Culvert line
            bridge_face_ls (:obj:`LineString`): the bridge face line where these locations are being calculated

        Returns:
            (:obj:`list`): new list of circle offsets
        """
        embed = culvert_data['culvert_embed_depth']
        if embed == 0.0:
            return offsets

        diameter = culvert_data['culvert_diameter']
        embed = culvert_data['culvert_embed_depth']
        radius = diameter / 2.0
        elev = z + embed
        # see https://en.wikipedia.org/wiki/Circular_segment#Chord_length_and_height
        # 2 * sqrt(2 * R * h - h^2)
        cord = 2 * math.sqrt((2 * radius * embed) - embed**2)
        offset = 0.5 * cord
        offset_ls = culvert_ls.parallel_offset(distance=-offset, side='left', resolution=1)
        sh_pt = offset_ls.intersection(bridge_face_ls)
        s0 = bridge_face_ls.project(sh_pt) / bridge_face_ls.length
        offset_ls = culvert_ls.parallel_offset(distance=offset, side='left', resolution=1)
        sh_pt = offset_ls.intersection(bridge_face_ls)
        s1 = bridge_face_ls.project(sh_pt) / bridge_face_ls.length
        s0, s1 = min(s0, s1), max(s0, s1)
        idx0 = idx1 = -1
        for i in range(1, len(offsets)):
            if offsets[i - 1][0] < s0 < offsets[i][0]:
                idx0 = i
            if offsets[i - 1][0] < s1 < offsets[i][0]:
                idx1 = i
        return offsets[:idx0] + [(s0, elev), (s1, elev)] + offsets[idx1:]

    def _bottom_half_of_culvert(self, culvert_data):
        """Compute the bottom half of the culvert.

        Args:
            culvert_data (:obj:`dict`): dict with culvert parameters

        Returns:
            (:obj:`tuple(pandas.DataFrame, pandas.DataFrame)`): upstream and downstream profiles
        """
        diameter = culvert_data['culvert_diameter']
        center_diameter = diameter
        is_box = False
        rise = 0.0
        up_z = culvert_data['culvert_up_invert']
        if culvert_data['culvert_type'] == 'Box':
            is_box = True
            # rise = culvert_data['culvert_rise']
            center_diameter = 0.0
            up_z = up_z + culvert_data['culvert_embed_depth']

        up_culvert_center_z = up_z + (center_diameter / 2)
        up_profile = [(0.0, up_culvert_center_z)]
        for culv_ls in culvert_data['barrel_cl']:
            # make diameter negative to get bottom half of circle
            circle_offsets = self._culvert_elev_offsets(
                -diameter, up_culvert_center_z, culv_ls, culvert_data['up_ls'], is_box, -rise
            )
            circle_offsets.sort()
            # insert the embedment locations
            if not is_box:
                circle_offsets = self._insert_embedment_locations(
                    culvert_data, circle_offsets, up_z, culv_ls, culvert_data['up_ls']
                )
            up_profile.extend(circle_offsets)
        up_profile.extend([(1.0, up_culvert_center_z)])
        up_profile.sort()
        up_df = pd.DataFrame(up_profile, columns=['Distance', 'Elevation'])

        dn_z = culvert_data['culvert_dn_invert']
        dn_culvert_center_z = dn_z + (center_diameter / 2)
        dn_profile = [(0.0, dn_culvert_center_z)]
        for culv_ls in culvert_data['barrel_cl']:
            # make diameter negative to get bottom half of circle
            circle_offsets = self._culvert_elev_offsets(
                -diameter, dn_culvert_center_z, culv_ls, culvert_data['dn_ls'], is_box, -rise
            )
            circle_offsets.sort()
            # insert the embedment locations
            if not is_box:
                circle_offsets = self._insert_embedment_locations(
                    culvert_data, circle_offsets, dn_z, culv_ls, culvert_data['dn_ls']
                )
            dn_profile.extend(circle_offsets)
        dn_profile.extend([(1.0, dn_culvert_center_z)])
        dn_profile.sort()
        dn_df = pd.DataFrame(dn_profile, columns=['Distance', 'Elevation'])

        return up_df, dn_df

    def culvert_arc_data(self, has_abutments):
        """Gets the culvert arc data from the coverage.

        Args:
            has_abutments (:obj:`bool`): flag to tell if there are abutments
        Returns:
            (:obj:`dict`): contains culvert data
        """
        self.err_msg = ''
        self._culvert_arc = None
        self.has_abutments = has_abutments
        try:
            self._validate_coverage_for_culvert_mesh()
            if not self.err_msg and self._culvert_arc is not None:
                return {
                    'cl_arc': self._cl_arc,
                    'cl_pts': self._cl_pts,
                    'cl_ls': self._cl_ls,
                    'culvert_arc': self._culvert_arc,
                    'culvert_pts': self._culvert_pts,
                }
        except RuntimeError as e:
            self.err_msg = str(e)
        return None
