"""Utilities used by some writers."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['cells_are_contiguous', 'neighbors_of']

# 1. Standard Python modules
from itertools import pairwise

# 2. Third party modules

# 3. Aquaveo modules
from xms.grid.ugrid import UGrid

# 4. Local modules


def cells_are_contiguous(ugrid: UGrid, cells: list[int]) -> bool:
    """
    Check whether a list of cells are contiguous within the grid.

    Args:
        cells: Cells to check.

    Returns:
        Whether the cells are all contiguous.
    """
    for first, second in pairwise(cells):
        adjacent_to_first = neighbors_of(ugrid, first)
        if second not in adjacent_to_first:
            return False
    return True


def neighbors_of(ugrid: UGrid, cell: int) -> set[int]:
    """
    Get the neighbors of a cell.

    Neighbors are those cells which share an edge with the given cell.

    Args:
        cell: The cell to get the neighbors of.

    Returns:
        All the cells that are neighbors of the input cell.
    """
    tuples = {ugrid.get_cell_edge_adjacent_cells(cell, edge) for edge in range(ugrid.get_cell_edge_count(cell))}
    neighbors = {tup[0] for tup in tuples if tup}
    return neighbors
