"""A reader for CMS-Wave simulation files."""

# 1. Standard Python modules
import datetime
import logging
import os
import uuid

# 2. Third party modules
import pandas

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.coverage.grid.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.data_objects.parameters import Component, Coverage, Point, Projection, Simulation, UGrid
from xms.guipy.time_format import datetime_to_string, string_to_datetime

# 4. Local modules
from xms.cmswave.data import cmswave_consts as const
from xms.cmswave.data import simulation_data
from xms.cmswave.data.simulation_data import SimulationData
from xms.cmswave.data.structures_data import StructuresData
from xms.cmswave.file_io.dataset_reader import DatasetReaderCMSWAVE
from xms.cmswave.file_io.eng_reader import EngReader
from xms.cmswave.file_io.sim_file_reader import logging_filename, SimFileReader
from xms.cmswave.file_io.structures_component_builder import StructuresComponentBuilder
from xms.cmswave.file_io.structures_reader import StructuresReader


class SimulationReader:
    """A class for reading CMS-Wave simulation files."""
    def __init__(self):
        """Constructor."""
        self._logger = logging.getLogger('xms.cmswave')
        self._reftime = datetime.datetime(year=1950, month=1, day=1)
        self.query = None
        self.filename = ''
        self.sim_name = ''
        self.using_darcy = False
        self.grid_z = None
        self.grid_proj = Projection()
        self.geom_uuid = ""
        self.grid = None
        self.input_files = {}  # {dset_name: [filename, format]}
        self.case_times = []
        self.idd_cards = {}
        self.snap_idds = []
        self.observation_pts = []
        self.nest_pts = []
        self.const_mag = []
        self.const_dir = []
        self.const_surge = []
        self.comp_dir = ''
        self.sim_comp_dir = ''
        self.sim_data = None
        self.sim_uuid = ''
        self.do_sim = None
        self.do_sim_comp = None
        self.setup_query()
        self.obs_cells = []
        self.nest_cells = []
        self.spec_xy = None
        self.spec_xy2 = None  # Location of the side 3 spectral point
        self._case_time_data = None
        self._sim_filename = ''
        self._struct_cov = None
        self._struct_cov_do_comp = None

    def setup_query(self):
        """Sets up timeout, retries, simulation model name, and toot index."""
        self.query = Query()
        self.filename = self.query.read_file
        self.sim_name = os.path.splitext(os.path.basename(self.filename))[0]
        self.sim_uuid = str(uuid.uuid4())
        self.comp_dir = os.path.join(self.query.xms_temp_directory, 'Components')
        # Create the simulation component
        comp = Component()
        comp_uuid = str(uuid.uuid4())
        self.sim_comp_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(self.sim_comp_dir, exist_ok=True)
        sim_mainfile = os.path.join(self.sim_comp_dir, 'simulation_comp.nc')
        self.sim_data = SimulationData(sim_mainfile)  # This will create a default model control
        comp.uuid = comp_uuid
        comp.name = 'sim_comp'
        comp.main_file = sim_mainfile
        comp.set_unique_name_and_model_name('Sim_Component', 'CMS-Wave')
        comp.locked = False
        self.do_sim_comp = comp
        # Create the data_objects simulation
        self.do_sim = Simulation(name=self.sim_name, model='CMS-Wave', sim_uuid=self.sim_uuid)

    def get_case_times(self):  # for idd_spec_types without snap_idds
        """
        Gets case times for 'none' key.

        Returns:
            (:obj:`list[datetime.datetime]`): List of times.
        """
        ts_times = []
        for case_time in self.case_times:
            ts_times.append(case_time)
        return ts_times

    def get_idd_to_case_time_map(self):  # for idd_spec_types with snap_idds
        """
        Gets the case times and corresponding snap idd mapped together.

        Returns:
            (:obj:`dict{datetime.datetime}`): snap idd : case time
        """
        idd_to_time = {}
        for idx, snap_idd in enumerate(self.snap_idds):
            idd_to_time[snap_idd] = self.case_times[idx]
        return idd_to_time

    def read_opts_file(self, filename):
        """
        Reads a CMS-Wave opts (.std) file.

        Arguments:
            filename(:obj:`str`):  The filename of the file to read.
        """
        self._logger.info(f'Reading std file {logging_filename(filename)}')
        iprp = -2
        ibnd = -1
        iwnd = -1
        format_version = 0
        kout = 0
        inest = 0
        with open(filename, 'r') as in_file:
            firstline = True
            self._set_wind_only(0)  # Set the default value to 0 if the WV_WIND_ONLY card is not present
            for line in in_file:
                line = line.strip().replace("\'", '')
                if not line or line[0] == '#' or line[0] == '&' or line[0] == '/':
                    continue
                line_data = [x.strip() for x in line.split(' ') if x != '']
                if firstline:
                    if line_data[0].upper() == 'CMS_WAVE_STD':
                        # Get the card-based version number
                        format_version = int(line_data[1])
                    else:
                        # No card version header line
                        iprp = int(line_data[0])  # source type
                        self._set_icur(line_data[1])
                        self._set_ibreak(int(line_data[2]))
                        self._set_radstress(int(line_data[3]))
                        kout = int(line_data[4])
                        ibnd = int(line_data[5])
                        self._set_iwet(int(line_data[6]))
                        self._set_ibf(int(line_data[7]))
                        self._set_iark(int(line_data[8]))
                        self._set_iarkr(int(line_data[9]))
                        self._set_akap(float(line_data[10]))
                        self._set_bf(line_data[11])
                        self._set_ark(line_data[12])
                        self._set_arkr(line_data[13])
                        self._set_iwvbk(int(line_data[14]))

                        # The remaining items are optional and a default should be loaded if missing
                        nitems = len(line_data)
                        val = int(line_data[15]) if nitems >= 16 else 0
                        self._set_nonln(val)

                        val = int(line_data[16]) if nitems >= 17 else 0
                        self._set_igrav(val)

                        val = int(line_data[17]) if nitems >= 18 else 0
                        self._set_irunup(val)

                        val = int(line_data[18]) if nitems >= 19 else 0
                        self._set_imud(val)

                        iwnd = int(line_data[19]) if nitems >= 20 else 0
                        self._set_iwnd(iwnd)

                        val = int(line_data[20]) if nitems >= 21 else 0
                        self._set_isolv(val)

                        val = int(line_data[21]) if nitems >= 22 else 0
                        self._set_ixmdf(val)

                        val = int(line_data[22]) if nitems >= 23 else 0
                        self._set_iproc(str(val))  # weird

                        val = int(line_data[23]) if nitems >= 24 else 0
                        self._set_iview(val)

                        val = int(line_data[24]) if nitems >= 25 else 0
                        self._set_iroll(val)

                    firstline = False
                else:
                    if format_version == 1:
                        # Newer style:  SMS 11.2 to 13.1
                        card = ''
                        if kout > 0:
                            # Special case for kout -- values should be:  I J
                            self.obs_cells.append((int(line_data[0]), int(line_data[1])))
                            kout -= 1
                            line_data = []  # Clear out line data so there's no card
                        if inest > 0:
                            # Special case for inest -- values should be:  I J
                            self.nest_cells.append((int(line_data[0]), int(line_data[1])))
                            inest -= 1
                            line_data = []  # Clear out line data so there's no card
                        # Get the card, then store the data needed
                        for data in line_data:
                            if data[0] == '!':
                                card = data[1:].lower()
                        if card == 'iprp':
                            iprp = int(line_data[0])
                        elif card == 'icur':
                            self._set_icur(line_data[0])
                        elif card == 'ibreak':
                            self._set_ibreak(int(line_data[0]))
                        elif card == 'irs':
                            self._set_radstress(int(line_data[0]))
                        elif card == 'kout':
                            kout = int(line_data[0])
                        elif card == 'ibnd':
                            ibnd = int(line_data[0])
                        elif card == 'iwet':
                            self._set_iwet(int(line_data[0]))
                        elif card == 'ibf':
                            self._set_ibf(int(line_data[0]))
                        elif card == 'iark':
                            self._set_iark(int(line_data[0]))
                        elif card == 'iarkr':
                            self._set_iarkr(int(line_data[0]))
                        elif card == 'iwvbk':
                            self._set_iwvbk(int(line_data[0]))
                        elif card == 'nonln':
                            self._set_nonln(int(line_data[0]))
                        elif card == 'igrav':
                            self._set_igrav(int(line_data[0]))
                        elif card == 'irunup':
                            self._set_irunup(int(line_data[0]))
                        elif card == 'imud':
                            self._set_imud(int(line_data[0]))
                        elif card == 'iwnd':
                            iwnd = self._set_iwnd(int(line_data[0]))
                        elif card == 'akap':
                            self._set_akap(float(line_data[0]))
                        elif card == 'bf':
                            self._set_bf(line_data[0])
                        elif card == 'ark':
                            self._set_ark(line_data[0])
                        elif card == 'arkr':
                            self._set_arkr(line_data[0])
                        elif card == 'isolv':
                            self._set_isolv(int(line_data[0]))
                        elif card == 'ixmdf':
                            self._set_ixmdf(int(line_data[0]))
                        elif card == 'iproc':
                            self._set_iproc(line_data[0])
                        elif card == 'iview':
                            self._set_iview(int(line_data[0]))
                        elif card == 'inest':
                            inest = int(line_data[0])
                    elif format_version == 2:
                        # New style:  SMS 13.2 and later
                        card = line_data[0]
                        if card == 'WV_PROPAGATION_TYPE':
                            val = line_data[1]
                            if val.upper() == 'WIND_AND_SPECTRA':
                                iprp = 0
                            elif val.upper() == 'SPECTRA_ONLY':
                                iprp = 1
                            elif val.upper() == 'FAST-MODE':
                                iprp = -1
                            elif val.upper() == 'FAST-MODE_SPECTRA_ONLY':
                                iprp = -2
                        elif card == 'WV_PLANE_DEFINITION':
                            val = line_data[1]
                            choices = ['HALF-PLANE', 'FULL-PLANE', 'FULL-PLANE_WITH_REVERSE_SPECTRA']
                            self._set_iview(choices.index(val.upper()))
                        elif card == 'WV_MATRIX_SOLVER':
                            val = line_data[1]
                            choices = ['GAUSS-SEIDEL', 'ADI']
                            self._set_isolv(choices.index(val.upper()))
                        elif card == 'WV_CURRENT_TYPE':
                            val = line_data[1]
                            self.sim_data.info.attrs['current_interaction'] = const.TEXT_NONE if val.upper() == 'OFF' \
                                else const.TEXT_USE_DATASET
                        elif card == 'WV_BOUNDARY_NESTING':
                            val = line_data[1]
                            choices = ['OFF', 'AVERAGE_SPECTRA', 'INVERSE_DISTANCE']
                            ibnd = choices.index(val.upper())
                        elif card == 'WV_BOTTOM_FRICTION':
                            val = line_data[1]
                            choices = [
                                'OFF', 'CONSTANT_DARCY_WEISBACH', 'VARIABLE_DARCY_WEISBACH', 'CONSTANT_MANNINGS',
                                'VARIABLE_MANNINGS'
                            ]
                            self._set_ibf(choices.index(val.upper()))
                        elif card == 'WV_FWD_REFLECTION':
                            val = line_data[1]
                            choices = ['OFF', 'CONSTANT', 'VARIABLE']
                            self._set_iark(choices.index(val.upper()))
                        elif card == 'WV_BWD_REFLECTION':
                            val = line_data[1]
                            choices = ['OFF', 'CONSTANT', 'VARIABLE']
                            self._set_iarkr(choices.index(val.upper()))
                        elif card == 'WV_WETTING_DRYING':
                            val = line_data[1]
                            val = 1 if val == 'OFF' else -1 if val == 'ON_WITH_SEA-SWELL_FILES' else 0
                            self._set_iwet(val)
                        elif card == 'WV_DIFFRACTION_INTENSITY':
                            self._set_akap(float(line_data[1]))
                        elif card == 'WV_NUM_THREADS':
                            self._set_iproc(line_data[1])
                        elif card == 'WV_NESTING_CELLS':
                            num_cells = int(line_data[1])
                            for cell in range(num_cells):
                                self.nest_cells.append((int(line_data[2 + cell * 2]), int(line_data[3 + cell * 2])))
                        elif card == 'WV_OBSERVATION_CELLS':
                            num_cells = int(line_data[1])
                            for cell in range(num_cells):
                                self.obs_cells.append((int(line_data[2 + cell * 2]), int(line_data[3 + cell * 2])))
                        elif card == 'WV_BOTTOM_FRICTION_COEFF':
                            self._set_bf(line_data[1])
                        elif card == 'WV_FWD_REFLECTION_COEFF':
                            self._set_ark(line_data[1])
                        elif card == 'WV_BWD_REFLECTION_COEFF':
                            self._set_arkr(line_data[1])
                        elif card == 'WV_BREAKING_FORMULA':
                            val = line_data[1]
                            choices = [
                                'EXT_GODA', 'EXT_MICHE', 'BATTJES_JANSSEN_78', 'CHAWLA_KIRBY', 'BATTJES_JANSSEN_07',
                                'MICHE_ORIGINAL', 'LIFTING_BREAKING'
                            ]
                            self._set_iwvbk(choices.index(val.upper()))
                        elif card == 'WV_SET_GAMMA_BJ78':
                            self._set_gamma_bj78(float(line_data[1]))
                        elif card == 'WV_ENABLE_INFRAGRAVITY':
                            val = line_data[1]
                            choices = ['OFF', 'ON']
                            self._set_igrav(choices.index(val.upper()))
                        elif card == 'WV_ENABLE_MUDDY_BED':
                            val = line_data[1]
                            choices = ['ON', 'OFF']
                            self._set_imud(choices.index(val.upper()))
                        elif card == 'WV_ENABLE_NONLINEAR_WAVES':
                            val = line_data[1]
                            choices = ['OFF', 'ON']
                            self._set_nonln(choices.index(val.upper()))
                        elif card == 'WV_ROLLER_EFFECT':
                            val = line_data[1]
                            choices = ['OFF', '25_PERCENT', '50_PERCENT', '75_PERCENT', '100_PERCENT']
                            self._set_iroll(choices.index(val.upper()))
                        elif card == 'WV_ENABLE_RUNUP':
                            val = line_data[1]
                            choices = ['OFF', 'REL_TO_ABS_DATUM', 'REL_TO_UPDATED_MWL']
                            self._set_irunup(choices.index(val.upper()))
                        elif card == 'WV_ENABLE_WIND':
                            val = line_data[1]
                            choices = ['ON', 'OFF', 'ON-LIMIT_WAVE_INFLATION']
                            iwnd = self._set_iwnd(choices.index(val.upper()))
                        elif card == 'WV_BREAKING_OUTPUT':
                            val = line_data[1]
                            choices = ['OFF', 'BREAKING_INDICES', 'DISSIPATION_VALUES']
                            self._set_ibreak(choices.index(val.upper()))
                        elif card == 'WV_RAD_STRESS_OUTPUT':
                            val = line_data[1]
                            choices = ['OFF', 'RAD_STRESS_FILE', 'STRESS_SETUP_FILES']
                            self._set_radstress(choices.index(val.upper()))
                        elif card == 'WV_OUTPUT_XMDF':
                            val = line_data[1]
                            choices = ['OFF', 'ON']
                            self._set_ixmdf(choices.index(val.upper()))
                        elif card == 'WV_LIMIT_OBSERVATION_OUTPUT':
                            val = line_data[1]
                            choices = ['OFF', 'ON']
                            self.sim_data.info.attrs['limit_observation_output'] = choices.index(val.upper())
                        elif card == 'WV_WIND_ONLY':  # added 13.4
                            val = line_data[1]
                            choices = ['OFF', 'ON']
                            self._set_wind_only(choices.index(val.upper()))
                    elif format_version == 0:
                        # Old style:  SMS 11.1 and earlier
                        if kout > 0:
                            # Special case for kout -- values should be:  I J
                            self.obs_cells.append((int(line_data[0]), int(line_data[1])))
                            kout -= 1
                        elif inest > 0:
                            # Special case for inest -- values should be:  I J
                            self.nest_cells.append((int(line_data[0]), int(line_data[1])))
                            inest -= 1
                        elif len(line_data) == 1:
                            inest = int(line_data[0])

        if format_version == 0 and (iprp == -2 or ibnd == -1 or iwnd == -1):
            raise IOError(f'Error reading grid file:  {filename}')
        self.sim_data.info.attrs['fastmode'] = 1 if iprp < 0 else 0  # Negative value means fast mode enabled
        self.sim_data.info.attrs['source_terms'] = 'Propagation only' if iprp in [1, -2] else \
            'Source terms and propagation'
        if ibnd != 0:
            self.sim_data.info.attrs['interpolation'] = 'Average Spectra' if ibnd == 1 else 'IDW'
        self.sim_data.info.attrs['wind'] = const.TEXT_USE_DATASET if iwnd == 1 else self.sim_data.info.attrs['wind']

    def _set_icur(self, data):
        """
        Sets the current data from the value passed in.

        Arguments:
            data(:obj:`str`):  The data read in from the grid file to process.
        """
        val = int(data)
        val = const.TEXT_NONE if val == 0 else const.TEXT_USE_DATASET
        self.sim_data.info.attrs['current_interaction'] = val

    def _set_ibreak(self, val):
        """
        Sets the breaking data from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        val = const.TEXT_NONE if val == 0 else const.TEXT_DISSIPATION_INDICES if val == 1 \
            else const.TEXT_DISSIPATION_VALUES
        self.sim_data.info.attrs['breaking_type'] = val

    def _set_radstress(self, val):
        """
        Sets the radiation stress data from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        self.sim_data.info.attrs['rad_stress'] = val

    def _set_iwet(self, val):
        """
        Sets the allow wet dry data from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        self.sim_data.info.attrs['wet_dry'] = 0 if val == 1 else 1
        self.sim_data.info.attrs['sea_swell'] = 1 if val == -1 else 0

    def _set_ibf(self, val):
        """
        Sets the bottom friction data from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        if val == 0:
            self.sim_data.info.attrs['friction'] = const.TEXT_NONE
        elif val == 1:
            self.sim_data.info.attrs['friction'] = const.TEXT_DARCY_CONSTANT
        elif val == 2:
            self.sim_data.info.attrs['friction'] = const.TEXT_DARCY_DATASET
        elif val == 3:
            self.sim_data.info.attrs['friction'] = const.TEXT_MANNINGS_CONSTANT
        elif val == 4:
            self.sim_data.info.attrs['friction'] = const.TEXT_MANNINGS_DATASET

    def _set_iark(self, val):
        """
        Sets the forward reflection option from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        if val == 0:
            self.sim_data.info.attrs['forward_reflection'] = const.TEXT_NONE
        elif val == 1:
            self.sim_data.info.attrs['forward_reflection'] = const.TEXT_REFLECTION_CONSTANT
        elif val == 2:
            self.sim_data.info.attrs['forward_reflection'] = const.TEXT_USE_DATASET

    def _set_iarkr(self, val):
        """
        Sets the backward reflection option from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        if val == 0:
            self.sim_data.info.attrs['backward_reflection'] = const.TEXT_NONE
        elif val == 1:
            self.sim_data.info.attrs['backward_reflection'] = const.TEXT_REFLECTION_CONSTANT
        elif val == 2:
            self.sim_data.info.attrs['backward_reflection'] = const.TEXT_USE_DATASET

    def _set_akap(self, val):
        """
        Sets the diffraction option data from the value passed in.

        Arguments:
            val(:obj:`float`):  The data read in from the grid file to process.
        """
        if 0.0 <= val <= 4.0:
            self.sim_data.info.attrs['diffraction_option'] = 1
        else:
            self.sim_data.info.attrs['diffraction_option'] = 0
        self.sim_data.info.attrs['diffraction_intensity'] = val

    def _set_bf(self, data):
        """
        Sets the bed friction constant data from the value passed in.

        Arguments:
            data(:obj:`str`):  The data read in from the grid file to process.
        """
        val = float(data)
        if self.sim_data.info.attrs['friction'] == const.TEXT_DARCY_CONSTANT:
            self.sim_data.info.attrs['darcy'] = val
        elif self.sim_data.info.attrs['friction'] == const.TEXT_MANNINGS_CONSTANT:
            self.sim_data.info.attrs['manning'] = val

    def _set_ark(self, data):
        """
        Sets the forward reflection const data from the value passed in.

        Arguments:
            data(:obj:`str`):  The data read in from the grid file to process.
        """
        val = float(data)
        if 0.0 <= val <= 1.0:
            self.sim_data.info.attrs['forward_reflection_const'] = val

    def _set_arkr(self, data):
        """
        Sets the backward reflection const data from the value passed in.

        Arguments:
            data(:obj:`str`):  The data read in from the grid file to process.
        """
        val = float(data)
        if 0.0 <= val <= 1.0:
            self.sim_data.info.attrs['backward_reflection_const'] = val

    def _set_iwvbk(self, val):
        """
        Sets the wave breaking formula from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        if val == 0:
            self.sim_data.info.attrs['wave_breaking_formula'] = 'Extended Goda'
        elif val == 1:
            self.sim_data.info.attrs['wave_breaking_formula'] = 'Extended Miche'
        elif val == 2:
            self.sim_data.info.attrs['wave_breaking_formula'] = 'Battjes and Janssen 1978'
        elif val == 3:
            self.sim_data.info.attrs['wave_breaking_formula'] = 'Chawla and Kirby'
        elif val == 4:
            self.sim_data.info.attrs['wave_breaking_formula'] = 'Battjes and Janssen 2007'
        elif val == 5:
            self.sim_data.info.attrs['wave_breaking_formula'] = 'Miche original'
        elif val == 6:
            self.sim_data.info.attrs['wave_breaking_formula'] = 'Lifting breaking'

    def _set_gamma_bj78(self, val):
        """
        Sets the wave breaking gamma value for Battjes & Janssen 78.

        Arguments:
            val(:obj:`float`):  The data read in from the grid file to process.
        """
        if 0.4 <= val <= 0.8:
            self.sim_data.info.attrs['gamma_bj78'] = val

    def _set_nonln(self, val):
        """
        Sets the nonlinear wave effect from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        self.sim_data.info.attrs['nonlinear_wave'] = val

    def _set_iroll(self, val):
        """
        Sets the roller flag from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        self.sim_data.info.attrs['roller'] = val

    def _set_igrav(self, val):
        """
        Sets the infragravity option from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        self.sim_data.info.attrs['infragravity'] = val

    def _set_irunup(self, val):
        """
        Sets the runup option from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        self.sim_data.info.attrs['runup'] = val

    def _set_imud(self, val):
        """
        Sets the muddy bed option from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        if val == 0:
            self.sim_data.info.attrs['muddy_bed'] = const.TEXT_USE_DATASET
        elif val == 1:
            self.sim_data.info.attrs['muddy_bed'] = const.TEXT_NONE

    def _set_iwnd(self, val):
        """
        Sets the muddy bed option from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.

        Returns:
            (:obj:`int`):  The iwnd value
        """
        if val == 0:
            self.sim_data.info.attrs['wind'] = const.TEXT_CONSTANT
        elif val == 1:
            self.sim_data.info.attrs['wind'] = const.TEXT_USE_DATASET
        elif val == 2:
            self.sim_data.info.attrs['limit_wave_inflation'] = 1
        return val

    def _set_isolv(self, val):
        """
        Sets the matrix solver from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        if val == 0:
            self.sim_data.info.attrs['matrix_solver'] = 'Gauss-Seidel'
        elif val == 1:
            self.sim_data.info.attrs['matrix_solver'] = 'ADI'

    def _set_iproc(self, data):
        """
        Sets the solver number of threads from the value passed in.

        Arguments:
            data(:obj:`str`):  The data read in from the grid file to process.
        """
        self.sim_data.info.attrs['num_threads'] = int(data)

    def _set_iview(self, val):
        """
        Sets the plane type / view from the value passed in.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        if val == 0:
            self.sim_data.info.attrs['plane'] = const.CBX_TEXT_HALF_PLANE
        elif val == 1:
            self.sim_data.info.attrs['plane'] = const.CBX_TEXT_FULL_PLANE
        elif val == 2:
            self.sim_data.info.attrs['plane'] = const.CBX_TEXT_FULL_PLANE_REVERSE

    def _set_ixmdf(self, val):
        """
        Sets the XMDF option.

        Arguments:
            val(:obj:`int`):  The data read in from the grid file to process.
        """
        pass

    def _set_wind_only(self, val):
        """
        Sets the combo box for 'Wind Only' or 'Spectra (+ Wind)'.
        """
        if val == 0:
            self.sim_data.info.attrs['boundary_source'] = 'Spectra (+ Wind)'
        elif val == 1:
            self.sim_data.info.attrs['boundary_source'] = 'Wind Only'

    def build_grid(self, name):
        """
        Builds a Rectilinear Grid from grid data members.

        Args:
            name (:obj:`str`): Name of grid.
        """
        cogrid_file = os.path.join(self.sim_comp_dir, 'cmswave_domain.xmc')
        self.grid.write_to_file(cogrid_file, True)
        ugrid = UGrid(cogrid_file, name=name)
        ugrid.projection = self.grid_proj
        self.geom_uuid = str(uuid.uuid4())
        ugrid.uuid = self.geom_uuid
        self.query.add_ugrid(ugrid)
        self.query.link_item(self.sim_uuid, self.geom_uuid)

        # Get nesting and observation points now that we have the grid, if there
        xm_ugrid = self.grid.ugrid
        pt_id = 1
        for cell in self.nest_cells:
            cell_idx = self.grid.get_cell_index_from_ij(cell[0], cell[1])
            location = xm_ugrid.get_cell_centroid(cell_idx=cell_idx)
            if location and location[0] is True:
                point = Point(location[1][0], location[1][1], location[1][2])
                point.id = pt_id
                pt_id += 1
                self.nest_pts.append(point)
        pt_id = 1
        for cell in self.obs_cells:
            cell_idx = self.grid.get_cell_index_from_ij(cell[0], cell[1])
            location = xm_ugrid.get_cell_centroid(cell_idx=cell_idx)
            if location and location[0] is True:
                point = Point(location[1][0], location[1][1], location[1][2])
                point.id = pt_id
                pt_id += 1
                self.observation_pts.append(point)

        # Get the spectral point location... midway between corners of grid
        num_i = len(self.grid.locations_x) - 1
        num_j = len(self.grid.locations_y) - 1
        idx_start = self.grid.get_cell_index_from_ij(1, 1)
        idx_end = self.grid.get_cell_index_from_ij(1, num_j)
        spec_0 = xm_ugrid.get_cell_locations(idx_start)[0]
        spec_3 = xm_ugrid.get_cell_locations(idx_end)[-1]
        self.spec_xy = [(spec_0[0] + spec_3[0]) / 2.0, (spec_0[1] + spec_3[1]) / 2.0]
        # Get the location for the side 3 spectral point
        idx_start = self.grid.get_cell_index_from_ij(num_i, 1)
        idx_end = self.grid.get_cell_index_from_ij(num_i, num_j)
        spec_0 = xm_ugrid.get_cell_locations(idx_start)[1]
        spec_3 = xm_ugrid.get_cell_locations(idx_end)[2]
        self.spec_xy2 = [(spec_0[0] + spec_3[0]) / 2.0, (spec_0[1] + spec_3[1]) / 2.0]

    def build_coverages(self):
        """
        Builds coverages from points and grid members.

        Monitor, nesting points, output, and subset coverages are all built
        and loaded.
        """
        # build the monitor coverage
        if self.observation_pts:
            observation_cov = Coverage()
            observation_cov.name = "CMS-Wave Observation Cells"
            observation_cov.set_points(self.observation_pts)
            observation_cov.projection = self.grid_proj
            observation_cov.uuid = str(uuid.uuid4())
            self.sim_data.info.attrs['observation'] = 1
            self.sim_data.info.attrs['observation_uuid'] = observation_cov.uuid
            observation_cov.complete()
            self.query.add_coverage(observation_cov)

        # build the nesting points coverage
        if self.nest_pts:
            nest_cov = Coverage()
            nest_cov.name = "CMS-Wave Nesting Points"
            nest_cov.set_points(self.nest_pts)
            nest_cov.projection = self.grid_proj
            nest_cov.uuid = str(uuid.uuid4())
            self.sim_data.info.attrs['nesting'] = 1
            self.sim_data.info.attrs['nesting_uuid'] = nest_cov.uuid
            nest_cov.complete()
            self.query.add_coverage(nest_cov)

        if self._struct_cov:
            self.query.add_coverage(
                self._struct_cov,
                model_name='CMS-Wave',
                coverage_type='Structures',
                components=[self._struct_cov_do_comp]
            )
            self.query.link_item(self.sim_uuid, self._struct_cov.uuid)

    def build_spatial_datasets(self):
        """Builds the spatial datasets and adds it to query."""
        # read the input datasets
        ts_times = self.get_case_times()
        dsets = []
        for k, v in self.input_files.items():
            if os.path.isfile(v):
                if k in ['rads', 'wind', 'curr', 'break', 'mud', 'fric', 'fref', 'bref', 'eta']:
                    reftime = string_to_datetime(self.sim_data.info.attrs['reftime'])
                    reader = DatasetReaderCMSWAVE(
                        k, v, ts_times, self.geom_uuid, angle=self.grid.angle, reftime=reftime
                    )
                    # reader = DatasetReader(k, v[0], ts_times, self.geom_uuid)
                    dsets.append(reader.read())
            elif k == 'wind':
                # If wind was enabled, but the dataset file does not exist, set the option back to constant
                self.sim_data.info.attrs['surge'] = const.TEXT_CONSTANT
            elif k == 'mud':
                # If muddy bed was enabled, but the dataset file does not exist, set the option back off
                self.sim_data.info.attrs['muddy_bed'] = const.TEXT_NONE
        for dset in dsets:
            if dset:
                name = dset.name
                is_z = False
                # push an empty node in for the widget
                if name in ['Current', 'curr']:
                    self.sim_data.info.attrs['current_uuid'] = dset.uuid
                elif name in ['Friction', 'fric']:
                    if self.using_darcy:
                        self.sim_data.info.attrs['darcy_uuid'] = dset.uuid
                    else:
                        self.sim_data.info.attrs['manning_uuid'] = dset.uuid
                elif name in ['Surge', 'eta']:
                    # There is no flag in the std file for transient surge so set the model control value here
                    self.sim_data.info.attrs['surge'] = const.TEXT_USE_DATASET
                    self.sim_data.info.attrs['surge_uuid'] = dset.uuid
                elif name in ['Wind', 'wind']:
                    self.sim_data.info.attrs['wind'] = const.TEXT_USE_DATASET
                    self.sim_data.info.attrs['wind_uuid'] = dset.uuid
                elif name in ['Forward Reflection', 'fref']:
                    self.sim_data.info.attrs['forward_reflection_uuid'] = dset.uuid
                elif name in ['Backward Reflection', 'bref']:
                    self.sim_data.info.attrs['backward_reflection_uuid'] = dset.uuid
                elif name in ['Muddy Bed', 'mud']:
                    self.sim_data.info.attrs['muddy_bed_uuid'] = dset.uuid
                else:
                    continue
                # add the dataset itself as a node in the context
                if not is_z:
                    self.query.add_dataset(dset)

    def read_spectral_cov(self, side_three):
        """Builds an energy reader, then reads the data and loads it into widgets.

        Args:
            side_three (:obj:`bool`): True if spectral coverage for side 3, False for side 1
        """
        location = self.spec_xy2 if side_three else self.spec_xy  # Default location if none in the file
        filename = self.input_files['spec2'] if side_three else self.input_files['spec']
        angle = self.grid.angle
        if side_three:
            if angle < 180.0:
                angle += 180.0
            else:
                angle -= 180.0
        spec_reader = EngReader(
            filename, None, origin_x=self.grid.origin[0], origin_y=self.grid.origin[1], azimuth=angle, spec_xy=location
        )
        spec_cov, num_freqs, min_freqs, delta_freqs, date_format = spec_reader.read()
        if spec_cov:
            self.query.add_coverage(spec_cov)
            self._reftime, self._case_time_data = spec_reader.get_case_time_data()
            if side_three:
                self.sim_data.info.attrs['spectral2_uuid'] = spec_cov.m_cov.uuid
                self.sim_data.info.attrs['side3'] = 'Specified spectrum'
            else:
                self.sim_data.info.attrs['spectral_uuid'] = spec_cov.m_cov.uuid
                self.sim_data.info.attrs['side1'] = 'Specified spectrum'
                self.sim_data.info.attrs['num_frequencies'] = num_freqs
                self.sim_data.info.attrs['min_frequency'] = min_freqs
                self.sim_data.info.attrs['delta_frequency'] = delta_freqs
                self.sim_data.info.attrs['date_format'] = date_format

    def read_structures_cov(self):
        """Builds a structures coverage."""
        struct_file = self.input_files.get("struct", "")
        if not os.path.isfile(struct_file):
            return
        struct_data = StructuresData(self._sim_filename)
        struct_reader = StructuresReader(self.grid, struct_data)
        struct_reader.read(struct_file)

        # don't create a coverage if there are no structures
        if struct_reader.structure_count < 1:
            return

        num_cells = self.grid.ugrid.cell_count
        cell_structures = [0 for _ in range(num_cells)]
        for structure, cells in struct_reader.structure_cells.items():
            for cell in cells:
                cell_structures[cell] = structure

        cov_builder = GridCellToPolygonCoverageBuilder(
            self.grid, cell_structures, self.grid_proj, 'Structures', null_value=0
        )
        self._struct_cov = cov_builder.create_polygons_and_build_coverage()
        new_cov_uuid = self._struct_cov.uuid

        comp_builder = StructuresComponentBuilder(self.comp_dir, struct_data, cov_builder.dataset_polygon_ids)
        self._struct_cov_do_comp = comp_builder.build_structures_component(new_cov_uuid)

    def read(self):
        """Reads all the simulation, coverage, and grid data and sends it to SMS."""
        sim_file_reader = SimFileReader(self.filename)
        if not sim_file_reader.read():
            return
        self.grid_proj = sim_file_reader.grid_proj
        self.input_files = sim_file_reader.input_files
        self.grid = sim_file_reader.grid

        if 'opts' in self.input_files:
            self.read_opts_file(self.input_files['opts'])
        self.build_grid(self.sim_name)  # build the domain cartesian grid
        self.build_spatial_datasets()

        self.read_structures_cov()
        self.build_coverages()  # build the coverages

        if os.path.isfile(self.input_files.get('spec', '')):  # build the spectral coverage if there is one
            # self.sim_data.info.attrs['boundary_source'] = 'Spectra (+ Wind)'  This now gets set on sim read.
            self.read_spectral_cov(False)
            if os.path.isfile(self.input_files.get('spec2', '')):
                self.read_spectral_cov(True)

            # build the case data table
            if self._case_time_data:
                if len(self.const_mag) < len(self._case_time_data):
                    for data in self._case_time_data:
                        self.const_mag.append(data[1])
                if len(self.const_dir) < len(self._case_time_data):
                    for data in self._case_time_data:
                        self.const_dir.append(data[2])
                if len(self._case_time_data[0]) >= 5:
                    if len(self.const_surge) < len(self._case_time_data):
                        for data in self._case_time_data:
                            self.const_surge.append(data[4])
                        # Only set to constant if we didn't read spatially varying surge dataset.
                        all_zero = all([float(surge) == 0.0 for surge in self.const_surge])
                        if self.sim_data.info.attrs['surge'] != const.TEXT_USE_DATASET:
                            if all_zero:
                                self.sim_data.info.attrs['surge'] = const.TEXT_NONE
                            else:
                                self.sim_data.info.attrs['surge'] = const.TEXT_CONSTANT
                one_hour = datetime.timedelta(hours=1)
                local_case_times = [
                    (case_time_data[0] - self._reftime) / one_hour for case_time_data in self._case_time_data
                ]
                case_time_data = simulation_data.case_data_table(
                    local_case_times, self.const_dir, self.const_mag, self.const_surge
                )
                self.sim_data.case_times = pandas.DataFrame(case_time_data).to_xarray()
                self.sim_data.info.attrs['reftime'] = datetime_to_string(self._reftime)
                self.sim_data.info.attrs['reftime_units'] = 'hours'
        else:
            self.sim_data.info.attrs['boundary_source'] = const.TEXT_NONE

        self.sim_data.commit()
        self.query.add_simulation(self.do_sim, [self.do_sim_comp])
