"""Determines how to run an CMS-Wave simulation and read the solution."""

# 1. Standard Python modules
import logging
import os
import re

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.guipy.time_format import string_to_datetime

# 4. Local modules
from xms.cmswave.data.simulation_data import SimulationData
from xms.cmswave.file_io.dataset_reader import DatasetReaderCMSWAVE
from xms.cmswave.file_io.eng_reader import EngReader
from xms.cmswave.file_io.sim_file_reader import SimFileReader


class SolutionReader:
    """Reader class for CMS-Wave solutions."""
    def __init__(self):
        """Constructor."""
        self.query = None
        self.geom_uuid = 'CCCCCCCC-CCCC-CCCC-CCCC-CCCCCCCCCCCC'
        self.sim_name = 'Sim'
        self.filename = ''
        self.arg_list = []
        self.read_data = False
        self.rect_grid = None
        self.half_plane = False
        self.ref_time = None
        self.output_coverages = []  # this is just for testing
        self.eng_reader = None
        self.win_cont = None
        self._logger = logging.getLogger('xms.cmswave')

    def setup_query(self):
        """Get data needed to read solutions from SMS."""
        self.query = Query()
        sim_item = self.query.current_item()
        self.sim_name = sim_item.name
        if not self.sim_name:
            return  # not called from a simulation
        comp_item = self.query.item_with_uuid(sim_item.uuid, unique_name='Sim_Component', model_name='CMS-Wave')
        sim_data = SimulationData(comp_item.main_file)
        self.ref_time = string_to_datetime(str(sim_data.info.attrs['reftime']))
        self.half_plane = sim_data.info.attrs['plane'] == 'Half plane'

    def read_grid_definition(self):
        """
        Read the grid definition from the .sim input file.

        Don't want to read the whole file, quit once we have what we need.
        """
        # Look for a .sim file in the same directory with the same base name. If it exists, grab the grid definition
        # from it.
        sim_in = os.path.join(os.path.dirname(self.filename), f'{self.sim_name}.sim')
        if os.path.isfile(sim_in):
            sim_file_reader = SimFileReader(sim_in)
            sim_file_reader.read()
            self.rect_grid = sim_file_reader.grid

    def add_spectral_coverages(self):
        """Reads specral coverage from file_name, and adds it to a spectral dictionary that is sent to query."""
        filename_lower = self.filename.lower()

        is_obse = False
        if filename_lower.endswith('.obs'):
            cover_name = f'{self.sim_name} - OBSE'
            is_obse = True
        elif filename_lower.endswith('.nst'):
            cover_name = f'{self.sim_name} - NEST'
        elif filename_lower.endswith('.eng'):
            cover_name = f'{self.sim_name} - ENG'
        else:
            return  # This is not a spectral coverage output file. Keep looking.

        # If we are reading a solution spectral coverage, we need to read the CMS-Wave grid definition. CMS-Wave output
        # spectral files specify node locations by i-j coordinates on the input CMS-Wave grid. We cannot read this
        # solution if we don't have a matching .sim file in the same directory.

        # read the output spectral coverage
        reader = EngReader(
            self.filename, {},
            rect_grid=self.rect_grid if is_obse else None,
            reftime=self.ref_time,
            azimuth=self.rect_grid.angle if self.rect_grid else None
        )
        reader.win_cont = self.win_cont
        reader.cov_name = cover_name
        self.eng_reader = reader
        out_spec_cov, dum1, dum2, dum3, dum4 = reader.read()
        if out_spec_cov:
            out_spec_cov.m_cov.name = cover_name
            self.query.add_coverage(out_spec_cov)
            self.read_data = True
            self.output_coverages.append(out_spec_cov)

    def add_grid_datasets(self):
        """Adds the dataset in file to query. Could be a wave, break, radiation stress, or ends with .tp.out."""
        ts_times = []
        dset_reader = None

        grid_angle = 0.0
        if self.rect_grid:
            grid_angle = self.rect_grid.angle

        if self.filename.lower().endswith('.wav'):
            self._logger.info(f'Reading wave output file {os.path.basename(self.filename)}')
            dset_reader = DatasetReaderCMSWAVE(
                'WAVE', self.filename, ts_times, self.geom_uuid, grid_angle, reftime=self.ref_time
            )
            dsets = dset_reader.read_wave_file()
            if dsets:
                for dset in dsets:
                    if dset:
                        self.query.add_dataset(dset)
                        self.read_data = True
            return  # We are done if this is the wave dataset. Special case.
        elif self.filename.lower().endswith('.brk'):
            # read the breaking output dataset
            self._logger.info(f'Reading breaking output file {os.path.basename(self.filename)}')
            dset_reader = DatasetReaderCMSWAVE(
                'Wave Breaking', self.filename, ts_times, self.geom_uuid, grid_angle, reftime=self.ref_time
            )
        elif self.filename.lower().endswith('.rad'):
            # read the radiation stress output dataset (Vector)
            self._logger.info(f'Reading rad stress output file {os.path.basename(self.filename)}')
            dset_reader = DatasetReaderCMSWAVE(
                'Radiation Stresses', self.filename, ts_times, self.geom_uuid, grid_angle, reftime=self.ref_time
            )
        if dset_reader:
            dset = dset_reader.read()
            if dset:
                self.query.add_dataset(dset)
                self.read_data = True

    def read(self):
        """Reads spectral coverage and grid definition and sends them to query."""
        if not os.path.isfile(self.filename):
            return  # Not a real file

        # Parse out the project name from the read file.
        basename = os.path.basename(self.filename)
        m = re.search(r'(.+)\..+\.out', basename)
        if m:
            self.sim_name = m.group(1)
        else:
            self.sim_name = os.path.splitext(basename)[0]

        # CMS-Wave uses i-j locations of the input grid to specify points in output spectra. This is opposed to CMS-Wave
        # input spectral files, which specify point locations with x,y coordinates. To read solution spectra, we need
        # the grid definition. Also need the grid angle for reading grid solution datasets. This is only true for OBSE
        # files.
        self.read_grid_definition()

        # read and add the spectral coverages first
        self.add_spectral_coverages()

        # add the solution dataset
        self.add_grid_datasets()
