"""This is a dialog for specifying CMS-Wave model control values."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import webbrowser

# 2. Third party modules
import numpy as np
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QHeaderView, QMessageBox, QWidget
import xarray as xr

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import FilterLocation, julian_to_datetime
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.resources.resources_util import get_tree_icon_from_xms_typename
from xms.guipy.time_format import datetime_to_qdatetime, datetime_to_string, string_to_datetime
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator
from xms.guipy.widgets import widget_builder as wbd
from xms.guipy.widgets.twod_rect_grid_preview_plot import TwoDRectGridPlot

# 4. Local modules
from xms.cmswave.data import cmswave_consts as const
from xms.cmswave.data import simulation_data
from xms.cmswave.dmi.xms_data import XmsData
from xms.cmswave.gui.model_control_dialog_ui import Ui_ModelControlDialog
from xms.cmswave.gui.ref_time_dialog import RefTimeDialog
from xms.cmswave.gui.widgets.case_data_table import CaseDataTable


def is_spectral_if_dset(item):
    """Check if a tree item is a spectral coverage but only if it is a coverage.

    Args:
        item (:obj:`TreeNode`): The item to check

    Returns:
        (:obj:`bool`): True if the tree item is a spectral coverage not a coverage.
    """
    if item.item_typename == 'TI_COVER':
        if item.coverage_type == 'SPECTRAL':
            return True
        return False
    return True


class ModelControlDialog(XmsDlg):
    """A dialog for viewing model control data for a simulation."""
    def __init__(self, data, pe_tree, query, time_format, grid_data, parent=None):
        """
        Initializes the class, sets up the ui.

        Args:
            data (:obj:`SimulationData`): The simulation data for the dialog.
            pe_tree (:obj:`TreeNode`): The project explorer tree from XMS.
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS.
            time_format (:obj:`str`): The absolute datetime format to use from the SMS preferences.
                Should use Qt specifiers.
            grid_data (:obj:`dict`): Domain CGrid data:
                ::
                    {
                        'cogrid': CoGrid,
                        'grid_name': str,
                    }
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent, 'cmswave.gui.model_control_dialog')
        self._data = data
        self._previous_friction = self._data.info.attrs['friction']
        self._time_format = time_format
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:CMS-Wave_Model_Control'
        self.ui = Ui_ModelControlDialog()
        self.ui.setupUi(self)
        self._scalar_dataset = tree_util.copy_tree(pe_tree)
        self._vector_dataset = tree_util.copy_tree(pe_tree)
        self._output_stations_coverage = tree_util.copy_tree(pe_tree)
        self._monitor_coverage = tree_util.copy_tree(pe_tree)
        self._nesting_coverage = tree_util.copy_tree(pe_tree)
        self._trimmed_spectral_tree = tree_util.copy_tree(pe_tree)
        if self._trimmed_spectral_tree:
            tree_util.filter_project_explorer(self._trimmed_spectral_tree, is_spectral_if_dset)
        reftime = string_to_datetime(self._data.info.attrs['reftime'])
        self._case_data_table = CaseDataTable(reftime, self._data.case_times.to_dataframe(), self)
        self._case_data_table.table_view.setColumnWidth(0, 150)
        self._query = query
        self._spec_cov = None
        self._spec_cov2 = None
        self._grid_data = grid_data

        self.ui.button_box.helpRequested.connect(self.help_requested)

        self.double_validator = QxDoubleValidator(parent=self)
        self.int_validator = QxIntValidator(parent=self)

        self._setup_dialog()

    def _setup_dialog(self):
        """Called when the dialog is loaded to set up the initial values in the dialog."""
        # Set up the tabs on the dialog, initializing values.
        self._setup_parameters_tab()
        self._setup_boundary_tab()
        self._setup_output_tab()
        self._setup_options_tab()

        # Set up the connections for the widgets on the tabs.
        self._parameters_tab_connections()
        self._boundary_tab_connections()
        self._options_tab_connections()

    @property
    def _simulation_grid(self):
        """Makes a copy of the project tree filtered to simulation's grid."""
        xms_data = XmsData(self._query)
        do_ugrid = xms_data.do_ugrid
        if do_ugrid is None:
            return None
        pe_tree = self._query.copy_project_tree()
        tree = tree_util.trim_project_explorer(pe_tree, do_ugrid.uuid)
        return tree

    def _dset_label_text(self, label, tree_node, attr_key):
        """Set the label text of a dataset selector widget.

        Args:
            label (:obj:`QLabel`): The label widget to set the text on.
            tree_node (:obj:`TreeNode`): The tree node representing the selected dataset.
            attr_key (:obj:`str`): The attribute key used to retrieve the dataset information.
        """
        text = tree_util.build_tree_path(tree_node, self._data.info.attrs[attr_key])
        label.setText(text if text else '(none selected)')

    def _dataset_selector_slot(self, label: QWidget, dialog_title: str, storage_dict, key: str, is_scalar: bool):
        """
        Make a slot for selecting a dataset.

        Args:
            label: Label widget to place the dataset's path in for display to the user.
            dialog_title: Title to put on the dialog shown to select a dataset.
            storage_dict: A dict-like object for storing the dataset's UUID in.
            key: Key in storage_dict to store the dataset's UUID under.
            is_scalar: Whether this is a scalar dataset, as opposed to a vector one.
        """
        if is_scalar:
            condition = DatasetSelector.is_scalar_if_dset
        else:
            condition = DatasetSelector.is_vector_if_dset
        tree = self._simulation_grid

        def helper():
            DatasetSelector.select_dataset(
                self, label, dialog_title, tree, condition, storage_dict, key, self.icon_connected
            )

        return helper

    def _setup_parameters_tab(self):
        """Called to set up the parameters tab when the dialog is loaded."""
        self.ui.plane_combo.setCurrentText(self._data.info.attrs['plane'])
        self.ui.source_terms_combo.setCurrentText(self._data.info.attrs['source_terms'])
        self.ui.current_interaction_combo.setCurrentText(self._data.info.attrs['current_interaction'])
        slot = self._dataset_selector_slot(
            self.ui.current_interaction_label,
            'Select current interaction dataset',
            self._data.info.attrs,
            'current_uuid',
            is_scalar=False
        )
        self.ui.current_interaction_button.clicked.connect(slot)
        self._dset_label_text(self.ui.current_interaction_label, self._vector_dataset, 'current_uuid')
        self.ui.friction_combo.setCurrentText(self._data.info.attrs['friction'])
        self.ui.friction_button.clicked.connect(self._on_friction_button)
        if const.TEXT_DARCY_CONSTANT == self._data.info.attrs['friction']:
            self.ui.friction_edit.setText(str(self._data.info.attrs['darcy']))
        elif const.TEXT_MANNINGS_CONSTANT == self._data.info.attrs['friction']:
            self.ui.friction_edit.setText(str(self._data.info.attrs['manning']))
        else:
            self.ui.friction_edit.setText('0.0')
        self.ui.friction_edit.setValidator(self.double_validator)
        if const.TEXT_DARCY_DATASET == self._data.info.attrs['friction']:
            text = tree_util.build_tree_path(self._scalar_dataset, self._data.info.attrs['darcy_uuid'])
            self.ui.friction_label.setText(text)
        elif const.TEXT_MANNINGS_DATASET == self._data.info.attrs['friction']:
            text = tree_util.build_tree_path(self._scalar_dataset, self._data.info.attrs['manning_uuid'])
            self.ui.friction_label.setText(text)
        self.ui.surge_combo.setCurrentText(self._data.info.attrs['surge'])
        slot = self._dataset_selector_slot(
            self.ui.surge_label, 'Select surge fields dataset', self._data.info.attrs, 'surge_uuid', is_scalar=True
        )
        self.ui.surge_button.clicked.connect(slot)
        self._dset_label_text(self.ui.surge_label, self._scalar_dataset, 'surge_uuid')
        self.ui.wind_combo.setCurrentText(self._data.info.attrs['wind'])
        slot = self._dataset_selector_slot(
            self.ui.wind_label, 'Select wind fields dataset', self._data.info.attrs, 'wind_uuid', is_scalar=False
        )
        self.ui.wind_button.clicked.connect(slot)
        self._dset_label_text(self.ui.wind_label, self._vector_dataset, 'wind_uuid')
        self.ui.limit_wave_inflation_check.setChecked(int(self._data.info.attrs['limit_wave_inflation']) != 0)

        self.ui.solver_combo.setCurrentText(self._data.info.attrs['matrix_solver'])
        self.ui.numthreads_edit.setValidator(self.int_validator)
        self.ui.numthreads_edit.setText(str(int(self._data.info.attrs['num_threads'])))

    def _setup_boundary_tab(self):
        """Called to set up the boundary control tab when the dialog is loaded."""
        self.ui.boundary_source_combo.setCurrentText(self._data.info.attrs['boundary_source'])
        self.ui.interpolation_combo.setCurrentText(self._data.info.attrs['interpolation'])
        self.ui.num_frequencies_edit.setValidator(self.int_validator)
        self.ui.num_frequencies_edit.setText(str(int(self._data.info.attrs['num_frequencies'])))
        self.ui.delta_frequency_edit.setValidator(self.double_validator)
        self.ui.delta_frequency_edit.setText(str(self._data.info.attrs['delta_frequency']))
        self.ui.min_frequency_edit.setValidator(self.double_validator)
        self.ui.min_frequency_edit.setText(str(self._data.info.attrs['min_frequency']))
        self.ui.angle_convention_combo.setCurrentText(self._data.info.attrs['angle_convention'])
        self._reftime_label_update()
        self.ui.btn_populate_from_spectra.clicked.connect(self._set_populate_from_spectra)
        self.ui.btn_reftime.clicked.connect(self._set_reftime)
        self.ui.case_data_group.layout().addWidget(self._case_data_table)
        self.ui.side1_combo.addItems([const.CBX_TEXT_ZERO_SPECTRA, const.CBX_TEXT_SPECIFIED_SPECTRA])
        self.ui.side3_combo.addItems([const.CBX_TEXT_ZERO_SPECTRA, const.CBX_TEXT_SPECIFIED_SPECTRA])
        self._case_data_table.table_view.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self._setup_grid_preview()
        wbd.style_splitter(self.ui.splitter_horiz)
        wbd.style_splitter(self.ui.splitter_vert)

    def _setup_grid_preview(self):
        """Sets up the grid preview plot."""
        cogrid = self._grid_data.get('cogrid')
        grid_name = self._grid_data.get('grid_name', '')
        plotter = TwoDRectGridPlot(cogrid, grid_name)
        plotter.generate_grid_preview_plot(self.ui.hlay_grid_preview)
        if cogrid is None:  # If there is no linked CGrid, dim the grid preview plot groupbox.
            self.ui.grid_preview_group.setEnabled(False)

    def _setup_output_tab(self):
        """Called to set up the output control tab when the dialog is loaded."""
        self.ui.limit_observation_check.setChecked(int(self._data.info.attrs['limit_observation_output']) != 0)
        self.ui.rad_stress_check.setChecked(int(self._data.info.attrs['rad_stress']) != 0)
        self.ui.seaswell_check.setChecked(int(self._data.info.attrs['sea_swell']) != 0)
        self.ui.breaking_type_combo.setCurrentText(self._data.info.attrs['breaking_type'])

    def _setup_options_tab(self):
        """Called to set up the options tab when the dialog is loaded."""
        gamma_validator = QxDoubleValidator(parent=self)
        gamma_validator.setRange(0.4, 0.8, 4)

        self.ui.wetdry_check.setChecked(int(self._data.info.attrs['wet_dry']) != 0)
        self.ui.infragravity_check.setChecked(int(self._data.info.attrs['infragravity']) != 0)
        self.ui.diffraction_check.setChecked(int(self._data.info.attrs['diffraction_option']) != 0)
        self.ui.diffraction_edit.setValidator(self.double_validator)
        self.ui.diffraction_edit.setText(str(self._data.info.attrs['diffraction_intensity']))
        self.ui.nonlinear_check.setChecked(int(self._data.info.attrs['nonlinear_wave']) != 0)
        self.ui.runup_check.setChecked(int(self._data.info.attrs['runup']) != 0)
        self.ui.roller_combo.setCurrentIndex(self._data.info.attrs['roller'])
        self.ui.fastmode_check.setChecked(int(self._data.info.attrs['fastmode']) != 0)
        self.ui.forward_reflection_combo.setCurrentText(self._data.info.attrs['forward_reflection'])
        self.ui.forward_reflection_edit.setValidator(self.double_validator)
        self.ui.forward_reflection_edit.setText(str(self._data.info.attrs['forward_reflection_const']))
        self.ui.backward_reflection_combo.setCurrentText(self._data.info.attrs['backward_reflection'])
        self.ui.backward_reflection_edit.setValidator(self.double_validator)
        self.ui.backward_reflection_edit.setText(str(self._data.info.attrs['backward_reflection_const']))
        self.ui.muddy_bed_combo.setCurrentText(self._data.info.attrs['muddy_bed'])
        self.ui.wave_breaking_combo.setCurrentText(self._data.info.attrs['wave_breaking_formula'])
        self.ui.gamma_bj78_edit.setValidator(gamma_validator)
        self.ui.gamma_bj78_edit.setText(str(self._data.info.attrs['gamma_bj78']))
        self.ui.date_format_combo.setCurrentText(self._data.info.attrs['date_format'])

        slot = self._dataset_selector_slot(
            self.ui.forward_reflection_selected_label,
            'Select forward reflection dataset',
            self._data.info.attrs,
            'forward_reflection_uuid',
            is_scalar=True
        )
        self.ui.forward_reflection_button.clicked.connect(slot)
        slot = self._dataset_selector_slot(
            self.ui.backward_reflection_selected_label,
            'Select backward reflection dataset',
            self._data.info.attrs,
            'backward_reflection_uuid',
            is_scalar=True
        )
        self.ui.backward_reflection_button.clicked.connect(slot)
        slot = self._dataset_selector_slot(
            self.ui.muddy_bed_selected_label,
            'Select muddy bed dataset',
            self._data.info.attrs,
            'muddy_bed_uuid',
            is_scalar=True
        )
        self.ui.muddy_bed_button.clicked.connect(slot)
        self._dset_label_text(
            self.ui.forward_reflection_selected_label, self._output_stations_coverage, 'forward_reflection_uuid'
        )
        self._dset_label_text(
            self.ui.backward_reflection_selected_label, self._output_stations_coverage, 'backward_reflection_uuid'
        )
        self._dset_label_text(self.ui.muddy_bed_selected_label, self._output_stations_coverage, 'muddy_bed_uuid')

    def _parameters_tab_connections(self):
        """Handle the connections on the parameters tab based on which options are selected."""
        self.ui.current_interaction_combo.currentTextChanged[str].connect(self._current_dataset_update)
        self.ui.friction_combo.currentTextChanged[str].connect(self._friction_dataset_update)
        self.ui.surge_combo.currentTextChanged[str].connect(self._surge_dataset_update)
        self.ui.wind_combo.currentTextChanged[str].connect(self._wind_dataset_update)
        self.ui.source_terms_combo.currentTextChanged[str].connect(self._wind_group_update)
        self.ui.solver_combo.currentTextChanged[str].connect(self._solver_update)

        # Update on the boundary tab
        self.ui.source_terms_combo.currentTextChanged.connect(self._case_data_table.table_view.filter_model.invalidate)
        self.ui.surge_combo.currentTextChanged.connect(self._case_data_table.table_view.filter_model.invalidate)
        self.ui.wind_combo.currentTextChanged.connect(self._case_data_table.table_view.filter_model.invalidate)

        self._current_dataset_update(self.ui.current_interaction_combo.currentText())
        self._friction_dataset_update(self.ui.friction_combo.currentText())
        self._surge_dataset_update(self.ui.surge_combo.currentText())
        self._wind_dataset_update(self.ui.wind_combo.currentText())
        self._wind_group_update(self.ui.source_terms_combo.currentText())
        self._solver_update(self.ui.solver_combo.currentText())
        self._case_data_table.table_view.filter_model.invalidate()

    def _boundary_tab_connections(self):
        """Handle the connections on the parameters tab based on which options are selected."""
        self.ui.plane_combo.currentTextChanged[str].connect(self._angle_distribution_update)
        self.ui.boundary_source_combo.currentTextChanged[str].connect(self._source_dependencies_update)
        # Spectral coverage selectors
        self.ui.plane_combo.currentIndexChanged.connect(self._plane_side_update)
        self.ui.side1_combo.currentIndexChanged.connect(self._side1_coverage_update)
        self.ui.side3_combo.currentIndexChanged.connect(self._side3_coverage_update)
        self.ui.btn_spectral.clicked.connect(self._on_btn_spectral)
        self.ui.btn_spectral_side3.clicked.connect(self._on_btn_spectral_side3)
        self._intialize_spectral_pickers()

        self._angle_distribution_update(self.ui.plane_combo.currentText())
        self._source_dependencies_update(self.ui.boundary_source_combo.currentText())
        self.ui.side1_combo.setCurrentText(self._data.info.attrs['side1'])
        self.ui.side3_combo.setCurrentText(self._data.info.attrs['side3'])
        # Sides 2 and 4 are always open lateral boundaries if full plane of either type
        self.ui.lbl_side2.setText(const.CBX_TEXT_OPEN_LATERAL_SPECTRA)
        self.ui.lbl_side4.setText(const.CBX_TEXT_OPEN_LATERAL_SPECTRA)
        self._plane_side_update(self.ui.plane_combo.currentIndex())
        self._side1_coverage_update(-1)
        self._side3_coverage_update(-1)

    def _options_tab_connections(self):
        """Handle the connections on the options tab based on which options are selected."""
        self.ui.forward_reflection_combo.currentTextChanged.connect(self._forward_reflection_update)
        self._forward_reflection_update(self.ui.forward_reflection_combo.currentText())
        self.ui.backward_reflection_combo.currentTextChanged.connect(self._backward_reflection_update)
        self._backward_reflection_update(self.ui.backward_reflection_combo.currentText())
        self.ui.muddy_bed_combo.currentTextChanged.connect(self._muddy_bed_update)
        self._muddy_bed_update(self.ui.muddy_bed_combo.currentText())
        self.ui.wave_breaking_combo.currentTextChanged.connect(self._wave_breaking_update)
        self._wave_breaking_update(self.ui.wave_breaking_combo.currentText())
        self.ui.wetdry_check.stateChanged[int].connect(self._use_sea_swell_update)
        self._use_sea_swell_update(self.ui.wetdry_check.checkState())
        self.ui.obs_chk.stateChanged[int].connect(self._use_monitor_update)
        self._use_monitor_update(self.ui.obs_chk.checkState())
        self.ui.nest_chk.stateChanged[int].connect(self._use_nesting_update)
        self._use_nesting_update(self.ui.obs_chk.checkState())
        self.ui.obs_btn.clicked.connect(self._on_btn_monitor)
        self.ui.nest_btn.clicked.connect(self._on_btn_nesting)

    def _intialize_spectral_pickers(self):
        """Initialize the state of the spectral coverage pickers."""
        side_1_uuid = self._data.info.attrs['spectral_uuid']
        side_1_item = tree_util.find_tree_node_by_uuid(self._trimmed_spectral_tree, side_1_uuid)
        if side_1_item:
            self.ui.lbl_spectral_selection.setText(side_1_item.name)
            self._get_spectral_coverage(True)
        side_3_uuid = self._data.info.attrs['spectral2_uuid']
        side_3_item = tree_util.find_tree_node_by_uuid(self._trimmed_spectral_tree, side_3_uuid)
        if side_3_item:
            self.ui.lbl_spectral_side3_selection.setText(side_3_item.name)
            self._get_spectral_coverage(False)

    def _select_spectral_coverage(self, side_1):
        """Display the spectral coverages tree item selector for side 1 or 3.

        Args:
            side_1 (:obj:`bool`): True if selecting a spectral coverage for side 1, False for side 3
        """
        # Display a tree item selector dialog.
        if side_1:
            title = 'Select Spectral Coverage for Side 1'
            attr_name = 'spectral_uuid'
            previous_selection = self._data.info.attrs[attr_name]
            selection_widget = self.ui.lbl_spectral_selection
        else:
            title = 'Select Spectral Coverage for Side 3'
            attr_name = 'spectral2_uuid'
            previous_selection = self._data.info.attrs[attr_name]
            selection_widget = self.ui.lbl_spectral_side3_selection
        selector_dlg = TreeItemSelectorDlg(
            title=title,
            target_type=xmd.CoverageItem,
            pe_tree=self._trimmed_spectral_tree,
            previous_selection=previous_selection,
            parent=self,
            allow_multi_select=False
        )
        if selector_dlg.exec():
            selected_uuid = selector_dlg.get_selected_item_uuid()
            selected_item = tree_util.find_tree_node_by_uuid(self._trimmed_spectral_tree, selected_uuid)
            if selected_item:
                selection_widget.setText(selected_item.name)
            else:
                selected_uuid = ''
                selection_widget.setText('(none selected)')
            self._data.info.attrs[attr_name] = selected_uuid
            self._get_spectral_coverage(side_1)

    def _select_monitor_coverage(self):
        """Select an observation coverage."""
        self._select_any_coverage(
            title='Select Monitoring Cell Coverage',
            attr_name='observation_uuid',
            pe_tree=self._monitor_coverage,
            lbl_widget=self.ui.obs_lbl
        )

    def _select_nesting_coverage(self):
        """Select a nesting coverage."""
        self._select_any_coverage(
            title='Select Nesting Coverage',
            attr_name='nesting_uuid',
            pe_tree=self._nesting_coverage,
            lbl_widget=self.ui.nest_lbl
        )

    def _select_any_coverage(self, title, attr_name, pe_tree, lbl_widget):
        """Select a coverage of any type.

        Args:
            title (str): Title of the selector dialog
            attr_name (str): Name of the attr in the simulation data
            pe_tree (TreeNode): The project explorer tree to sho in the selector dialog
            lbl_widget (QWidget): The selector item's label widget
        """
        selector_dlg = TreeItemSelectorDlg(
            title=title,
            target_type=xmd.CoverageItem,
            pe_tree=pe_tree,
            previous_selection=self._data.info.attrs[attr_name],
            parent=self,
            allow_multi_select=False
        )

        if selector_dlg.exec():
            selected_uuid = selector_dlg.get_selected_item_uuid()
            selected_item = tree_util.find_tree_node_by_uuid(pe_tree, selected_uuid)
            if selected_item:
                lbl_widget.setText(selected_item.name)
            else:
                selected_uuid = ''
                lbl_widget.setText('(none selected)')
            self._data.info.attrs[attr_name] = selected_uuid

    def _get_spectral_coverage(self, side_1):
        """Called when the dialog is initialized or when changing the spectra associated with the sides.

        Args:
            side_1 (:obj:`bool`): True if getting the spectral coverage for side 1, False for side 3
        """
        spec_uuid = self._data.info.attrs['spectral_uuid'] if side_1 else self._data.info.attrs['spectral2_uuid']
        if spec_uuid:
            spec_cov = self._query.item_with_uuid(spec_uuid, generic_coverage=True)
            if side_1:
                self._spec_cov = spec_cov
            else:
                self._spec_cov2 = spec_cov
        else:
            if side_1:
                self._spec_cov = None
            else:
                self._spec_cov2 = None

    def _set_reftime(self):
        """Sets the reftime information for the case times table."""
        orig_reftime = string_to_datetime(str(self._data.info.attrs['reftime']))
        orig_units = self._data.info.attrs['reftime_units']
        reftime_dlg = RefTimeDialog(data=self._data, time_format=self._time_format, parent=self)
        if reftime_dlg.exec():
            reftime = self._reftime_label_update()  # Update the label text

            if self._case_data_table.model.rowCount() < 1:
                return  # Don't bother with the next bit if the table is empty

            # Update the values if the user desires
            reply = QMessageBox.question(
                self, 'SMS', 'Would you like the relative spectral times to be recalculated '
                'according to the new reference time?', QMessageBox.Yes, QMessageBox.No
            )
            if reply == QMessageBox.Yes:
                # Change the time values by the new reftime
                new_units = self._data.info.attrs['reftime_units']

                for i in range(self._case_data_table.model.rowCount(0)):
                    idx = self._case_data_table.model.index(i, 0)
                    time_data = float(self._case_data_table.model.data(idx))
                    # change value
                    time_data = self._convert_time(orig_reftime, orig_units, reftime, new_units, time_data)
                    self._case_data_table.model.setData(idx, time_data)

    @staticmethod
    def _convert_time(orig_timeref, orig_units, new_timeref, new_units, old_value):
        """Convert the old time value to a new time value (based on old and new reftimes).

        Args:
            orig_timeref (:obj:`datetime.datetime`):  The original time reference.
            orig_units (:obj:`str`):  The original time units (days, hours, minutes).
            new_timeref (:obj:`datetime.datetime`):  The new time reference.
            new_units (:obj:`str`):  The original time units (days, hours, minutes).
            old_value (:obj:`float`):  The old time difference value.

        Return:
            (:obj:`float`):  The converted time difference value.
        """
        # Convert the original time offset to seconds
        if orig_units == const.TIME_UNITS_MINUTES:
            old_value = old_value * 60.0
        elif orig_units == const.TIME_UNITS_HOURS:
            old_value = old_value * 3600.0
        elif orig_units == const.TIME_UNITS_DAYS:
            old_value = old_value * 3600.0 * 24.0
        else:
            raise ValueError(f'Invalid time units:  {orig_units}')
        # Calculate the orinal time as a datetime, based on the original time offset
        orig_dt = orig_timeref + datetime.timedelta(seconds=float(old_value))

        # Get a difference between the original time and new time reference
        time_diff_seconds = (orig_dt - new_timeref).total_seconds()
        if new_units == const.TIME_UNITS_MINUTES:
            return time_diff_seconds / 60.0
        elif new_units == const.TIME_UNITS_HOURS:
            return time_diff_seconds / 3600.0
        elif new_units == const.TIME_UNITS_DAYS:
            return time_diff_seconds / (3600.0 * 24.0)
        else:
            raise ValueError(f'Invalid time units:  {new_units}')

    def _set_populate_from_spectra(self):
        """Populate the case times table with the spectra."""
        if not self._spec_cov:
            msg = QMessageBox(QMessageBox.Warning, 'SMS', 'No spectral coverage in simulation.', QMessageBox.Ok, self)
            msg.exec()
            return

        # We want to use the earliest reference date from the spectral grids.
        reftime = None

        # Loop through the points in spectral coverage for side 1
        spectra_times = []
        spec_pts = self._spec_cov.m_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        for spec_pt in spec_pts:
            spec_pt_id = spec_pt.id
            spec_grids = self._spec_cov.GetSpectralGrids(spec_pt_id)
            for spec_grid in spec_grids:
                grid_reftime = julian_to_datetime(spec_grid.m_refTime)
                if reftime is None or grid_reftime < reftime:
                    reftime = grid_reftime
                # only do this once so we only have one file per dataset
                spec_dset = spec_grid.get_dataset(io_util.temp_filename())
                for i in range(spec_dset.num_times):
                    spec_dset.ts_idx = i
                    ts_time = julian_to_datetime(spec_dset.ts_time)
                    spectra_times.append(ts_time)
        # see if we have a second spectral coverage on side 3
        # Spectral coverage, looking for time steps (both spectra if they are there)
        if self._spec_cov2:
            spec_pts = self._spec_cov2.m_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
            for spec_pt in spec_pts:
                spec_pt_id = spec_pt.id
                spec_grids = self._spec_cov2.GetSpectralGrids(spec_pt_id)
                for spec_grid in spec_grids:
                    grid_reftime = julian_to_datetime(spec_grid.m_refTime)
                    if reftime is None or grid_reftime < reftime:
                        reftime = grid_reftime
                    # only do this once so we only have one file per dataset
                    spec_dset = spec_grid.get_dataset(io_util.temp_filename())
                    for i in range(spec_dset.num_times):
                        spec_dset.ts_idx = i
                        ts_time = julian_to_datetime(spec_dset.ts_time)
                        spectra_times.append(ts_time)

        spectra_times = list(set(spectra_times))
        if reftime is None:  # No spectral grids?
            reftime = string_to_datetime(self._data.info.attrs['reftime'])
        self._data.info.attrs['reftime'] = datetime_to_string(reftime)
        self._reftime_label_update()

        # Convert the timesteps to a delta time vs. reftime in seconds
        one_unit = 60.0  # self._data.info.attrs['reftime_units'] == 'minutes'
        if self._data.info.attrs['reftime_units'] == 'hours':
            one_unit = 3600.0
        elif self._data.info.attrs['reftime_units'] == 'days':
            one_unit = 3600.0 * 24.0
        spectra_delta_seconds = [(spec_time - reftime).total_seconds() for spec_time in spectra_times]
        spectra_delta_seconds.sort()

        # Set up case times data
        wind_dir = [0.0] * len(spectra_delta_seconds)
        wind_mag = [0.0] * len(spectra_delta_seconds)
        water_level = [0.0] * len(spectra_delta_seconds)
        times = np.array(spectra_delta_seconds, dtype=np.float64) / one_unit  # Convert to currently selected units
        times = np.around(times, 3)  # Discard excess precision
        case_time_data = simulation_data.case_data_table(times, wind_dir, wind_mag, water_level)
        case_time_dataset = xr.Dataset(data_vars=case_time_data)

        # Update the model which will update the table
        self._case_data_table.model.removeRows(0, self._case_data_table.model.rowCount(0))
        self._case_data_table.model.insertRows(0, len(spectra_delta_seconds))
        self._case_data_table.model.data_frame = case_time_dataset.to_dataframe()

    def _on_friction_button(self):
        """Handles connecting the bottom friction button, depending on whether Darcy-Weisbach or manning is chosen."""
        if const.TEXT_DARCY_DATASET == self.ui.friction_combo.currentText():
            DatasetSelector.select_dataset(
                self, self.ui.friction_label, 'Select bottom friction dataset', self._simulation_grid,
                DatasetSelector.is_scalar_if_dset, self._data.info.attrs, 'darcy_uuid', self.icon_connected
            )
        elif const.TEXT_MANNINGS_DATASET == self.ui.friction_combo.currentText():
            DatasetSelector.select_dataset(
                self, self.ui.friction_label, 'Select bottom friction dataset', self._simulation_grid,
                DatasetSelector.is_scalar_if_dset, self._data.info.attrs, 'manning_uuid', self.icon_connected
            )

    def _current_dataset_update(self, current_current_type):
        """Connections for current interaction combo box changing.

        Args:
            current_current_type (:obj:`str`): The current text on the Current interaction combo box on the Parameters
                tab.
        """
        self.ui.current_interaction_label.setVisible(current_current_type == const.TEXT_USE_DATASET)
        self.ui.current_interaction_button.setVisible(current_current_type == const.TEXT_USE_DATASET)

    def _friction_dataset_update(self, current_friction_type):
        """Connections for bottom friction combo box changing.

        Args:
            current_friction_type (:obj:`str`): The current text on the Bottom friction combo box on the Parameters tab.
        """
        show_dataset = False
        show_constant = False

        if 'constant' in self._previous_friction:
            friction_val = self.ui.friction_edit.text()
            if self._previous_friction == const.TEXT_DARCY_CONSTANT:
                self._data.info.attrs['darcy'] = float(friction_val)
            elif self._previous_friction == const.TEXT_MANNINGS_CONSTANT:
                self._data.info.attrs['manning'] = float(friction_val)

        if current_friction_type == const.TEXT_DARCY_DATASET:
            show_dataset = True
            self.ui.friction_label.setText(
                tree_util.build_tree_path(self._scalar_dataset, self._data.info.attrs['darcy_uuid'])
            )
        elif current_friction_type == const.TEXT_MANNINGS_DATASET:
            show_dataset = True
            self.ui.friction_label.setText(
                tree_util.build_tree_path(self._scalar_dataset, self._data.info.attrs['manning_uuid'])
            )
        elif current_friction_type == const.TEXT_DARCY_CONSTANT:
            show_constant = True
            self.ui.friction_edit.setText(str(self._data.info.attrs['darcy']))
        elif current_friction_type == const.TEXT_MANNINGS_CONSTANT:
            show_constant = True
            self.ui.friction_edit.setText(str(self._data.info.attrs['manning']))

        self.ui.friction_label.setVisible(show_dataset)
        self.ui.friction_button.setVisible(show_dataset)
        self.ui.friction_edit.setVisible(show_constant)
        self._previous_friction = current_friction_type

    def _surge_dataset_update(self, current_surge_type):
        """Connections for surge combo box changing.

        Args:
            current_surge_type (:obj:`str`): The current text on the Surge fields combo box on the Parameters tab.
        """
        self.ui.surge_label.setVisible(current_surge_type == const.TEXT_USE_DATASET)
        self.ui.surge_button.setVisible(current_surge_type == const.TEXT_USE_DATASET)

    def _wind_dataset_update(self, current_wind_type):
        """Connections for surge combo box changing.

        Args:
            current_wind_type (:obj:`str`): The current text on the Wind fields combo box on the Parameters tab.
        """
        self.ui.wind_label.setVisible(current_wind_type == const.TEXT_USE_DATASET)
        self.ui.wind_button.setVisible(current_wind_type == const.TEXT_USE_DATASET)

    def _wind_group_update(self, current_source_terms):
        """Connections for the wind group based on source terms.

        Args:
            current_source_terms (:obj:`str`): The current text on the Source terms combo box on the Parameters tab.
        """
        wind_enabled = current_source_terms == 'Source terms and propagation'
        self.ui.wind_group.setVisible(wind_enabled)
        self.ui.angle_convention_combo.setVisible(wind_enabled)
        self.ui.case_wind_convention_label.setVisible(wind_enabled)

        # These should never be on at the same time.
        self.ui.boundary_source_combo.setVisible(wind_enabled)
        self.ui.boundary_source_label.setVisible(wind_enabled)
        self.ui.spectra_label.setVisible(not wind_enabled)
        self.ui.spectra_nowind_label.setVisible(not wind_enabled)

    def _solver_update(self, current_solver):
        """Connections for solver and num processes based on solver type.

        Args:
            current_solver (:obj:`str`): The current text on the Solver combo box on the Parameters tab.
        """
        self.ui.numthreads_edit.setEnabled(current_solver == 'Gauss-Seidel')

    def _reftime_label_update(self):
        """Sets the reftime label text using datetime format from SMS preferences.

        Returns:
            (:obj:`QDateTime`): The current reference datetime
        """
        reftime = string_to_datetime(self._data.info.attrs['reftime'])
        qreftime = datetime_to_qdatetime(reftime)
        abs_date = qreftime.toString(self._time_format) if self._time_format else datetime_to_string(reftime)
        msg = f'{abs_date}    Units: {self._data.info.attrs["reftime_units"]}'
        self.ui.lbl_reftime.setText(msg)
        return reftime

    def _angle_distribution_update(self, current_plane_type):
        """Connections for angle distribution changing.

        Args:
            current_plane_type (:obj:`str`): The current text on the CMS-Wave plane mode combo box on the Parameters
                tab.
        """
        if current_plane_type == const.CBX_TEXT_HALF_PLANE:
            self.ui.num_frequencies_angle_label.setText('35')
            self.ui.min_frequency_angle_label.setText('-85.0°')
        elif current_plane_type in [const.CBX_TEXT_FULL_PLANE, const.CBX_TEXT_FULL_PLANE_REVERSE]:
            self.ui.num_frequencies_angle_label.setText('72')
            self.ui.min_frequency_angle_label.setText('0.0°')

    def _source_dependencies_update(self, current_source_type):
        """Slot to update widgets when the boundary source combo box changes.

        Args:
            current_source_type (:obj:`str`): The current text on the Source combo box on the Boundary control tab.
        """
        spectra_enabled = current_source_type == 'Spectra (+ Wind)'
        wind_enabled = self.ui.source_terms_combo.currentText() == 'Source terms and propagation'

        self.ui.sides_group.setVisible(spectra_enabled)
        self.ui.btn_populate_from_spectra.setEnabled(spectra_enabled)
        self.ui.wind_only_label.setVisible(not spectra_enabled and wind_enabled)
        self.ui.wind_only_label.setStyleSheet('color: red;')

    def _side1_coverage_update(self, _):
        """Show/hide coverage selector widgets when option changes for side 1.

        Args:
            '_' (:obj:`int`): Unused
        """
        # Show/hide the coverage picker widgets
        enable = self.ui.side1_combo.currentText() == const.CBX_TEXT_SPECIFIED_SPECTRA
        self.ui.btn_spectral.setVisible(enable)
        self.ui.lbl_spectral_selection.setVisible(enable)

    def _side3_coverage_update(self, _):
        """Show/hide coverage selector widgets when option changes for side 3.

        Args:
            '_' (:obj:`int`): Unused
        """
        # Show/hide the coverage picker widgets
        enable = self.ui.side3_combo.currentText() == const.CBX_TEXT_SPECIFIED_SPECTRA
        self.ui.btn_spectral_side3.setVisible(enable)
        self.ui.lbl_spectral_side3_selection.setVisible(enable)

    def _plane_side_update(self, index):
        """Show/hide coverage selector widgets when option changes for plane type.

        Args:
            index (:obj:`int`): Index of the currently selected option
        """
        # If half plane or full plane without reverse spectra, set side 3 type to zero spectrum and disable.
        if index != const.CBX_IDX_FULL_PLANE_REVERSE:
            self.ui.side3_combo.setCurrentIndex(const.CBX_IDX_ZERO_SPECTRA)
            self.ui.side3_combo.setEnabled(False)
        else:
            self.ui.side3_combo.setEnabled(True)
            self.ui.side3_combo.setCurrentIndex(const.CBX_IDX_SPECIFIED_SPECTRA)

    def _on_btn_spectral(self):
        """Slot called when spectral coverage selector button for side 1 is clicked."""
        self._select_spectral_coverage(True)

    def _on_btn_spectral_side3(self):
        """Slot called when spectral coverage selector button for side 3 is clicked."""
        self._select_spectral_coverage(False)

    def _on_btn_monitor(self):
        """Select a monitor coverage."""
        self._select_monitor_coverage()

    def _on_btn_nesting(self):
        """Select a nesting coverage."""
        self._select_nesting_coverage()

    def _forward_reflection_update(self, current_type):
        """Connections for forward reflection combo box changing.

        Args:
            current_type (:obj:`str`): The current text on the forward reflection combo box on the Options tab.
        """
        self.ui.forward_reflection_button.setVisible(current_type == const.TEXT_USE_DATASET)
        self.ui.forward_reflection_selected_label.setVisible(current_type == const.TEXT_USE_DATASET)
        self.ui.forward_reflection_edit.setVisible(current_type == 'Constant')

    def _backward_reflection_update(self, current_type):
        """Connections for backward reflection combo box changing.

        Args:
            current_type (:obj:`str`): The current text on the backward reflection combo box on the Options tab.
        """
        self.ui.backward_reflection_button.setVisible(current_type == const.TEXT_USE_DATASET)
        self.ui.backward_reflection_selected_label.setVisible(current_type == const.TEXT_USE_DATASET)
        self.ui.backward_reflection_edit.setVisible(current_type == 'Constant')

    def _muddy_bed_update(self, current_type):
        """Connections for muddy bed combo box changing.

        Args:
            current_type (:obj:`str`): The current text on the muddy bed combo box on the Options tab.
        """
        self.ui.muddy_bed_button.setVisible(current_type == const.TEXT_USE_DATASET)
        self.ui.muddy_bed_selected_label.setVisible(current_type == const.TEXT_USE_DATASET)

    def _wave_breaking_update(self, current_type):
        """Connections for wave breaking formula combo box changing.

        Args:
            current_type (:obj:`str`): The current text on the wave breaking formula combo box on the Options tab.
        """
        self.ui.gamma_bj78_edit.setVisible(current_type == 'Battjes and Janssen 1978')
        self.ui.gamma_bj78_label.setVisible(current_type == 'Battjes and Janssen 1978')

    def _use_sea_swell_update(self, current_wet_dry):
        """Handles connection for the sea swell checkbox, based on the wet dry checkbox state.

        Args:
            current_wet_dry (:obj:`Qt.CheckState`): The check state of the allow wet dry check box.
        """
        self.ui.seaswell_check.setVisible(current_wet_dry == Qt.Checked)

    def _use_monitor_update(self, state):
        """Handles connection for the use monitor checkbox.

        Args:
            state (Qt.CheckState): The check state of the checkbox.
        """
        self.ui.obs_btn.setVisible(state == Qt.Checked)
        self.ui.obs_lbl.setVisible(state == Qt.Checked)

    def _use_nesting_update(self, state):
        """Handles use for the use nesting checkbox.

        Args:
            state (Qt.CheckState): The check state of the checkbox.
        """
        self.ui.nest_btn.setVisible(state == Qt.Checked)
        self.ui.nest_lbl.setVisible(state == Qt.Checked)

    def _accept_parameters_tab(self):
        """Save the data on the Paramters tab on accept."""
        self._data.info.attrs['plane'] = self.ui.plane_combo.currentText()
        self._data.info.attrs['source_terms'] = self.ui.source_terms_combo.currentText()
        self._data.info.attrs['current_interaction'] = self.ui.current_interaction_combo.currentText()
        self._data.info.attrs['friction'] = self.ui.friction_combo.currentText()
        if const.TEXT_DARCY_CONSTANT == self._data.info.attrs['friction']:
            self._data.info.attrs['darcy'] = float(self.ui.friction_edit.text())
        elif const.TEXT_MANNINGS_CONSTANT == self._data.info.attrs['friction']:
            self._data.info.attrs['manning'] = float(self.ui.friction_edit.text())
        self._data.info.attrs['surge'] = self.ui.surge_combo.currentText()
        self._data.info.attrs['wind'] = self.ui.wind_combo.currentText()
        self._data.info.attrs['limit_wave_inflation'] = 1 if self.ui.limit_wave_inflation_check.isChecked() else 0
        self._data.info.attrs['matrix_solver'] = self.ui.solver_combo.currentText()
        num_threads = int(self.ui.numthreads_edit.text()) if self.ui.solver_combo.currentText() == 'Gauss-Seidel' else 1
        self._data.info.attrs['num_threads'] = num_threads

    def _accept_boundary_tab(self):
        """Save the data on the Boundary control tab on accept."""
        self._data.info.attrs['boundary_source'] = self.ui.boundary_source_combo.currentText()
        self._data.info.attrs['interpolation'] = self.ui.interpolation_combo.currentText()
        self._data.info.attrs['num_frequencies'] = int(self.ui.num_frequencies_edit.text())
        self._data.info.attrs['delta_frequency'] = float(self.ui.delta_frequency_edit.text())
        self._data.info.attrs['min_frequency'] = float(self.ui.min_frequency_edit.text())
        self._data.info.attrs['angle_convention'] = self.ui.angle_convention_combo.currentText()
        self._data.info.attrs['side1'] = self.ui.side1_combo.currentText()
        self._data.info.attrs['side3'] = self.ui.side3_combo.currentText()
        self._data.case_times = self._case_data_table.model.data_frame.to_xarray()

    def _accept_output_tab(self):
        """Save the data on the Output control tab."""
        self._data.info.attrs['limit_observation_output'] = 1 if self.ui.limit_observation_check.isChecked() else 0
        self._data.info.attrs['rad_stress'] = 1 if self.ui.rad_stress_check.isChecked() else 0
        self._data.info.attrs['sea_swell'] = 1 if self.ui.seaswell_check.isChecked() else 0
        self._data.info.attrs['breaking_type'] = self.ui.breaking_type_combo.currentText()

    def _accept_options_tab(self):
        """Save the data on the Options tab."""
        self._data.info.attrs['wet_dry'] = 1 if self.ui.wetdry_check.isChecked() else 0
        self._data.info.attrs['infragravity'] = 1 if self.ui.infragravity_check.isChecked() else 0
        self._data.info.attrs['diffraction_option'] = 1 if self.ui.diffraction_check.isChecked() else 0
        self._data.info.attrs['diffraction_intensity'] = float(self.ui.diffraction_edit.text())
        self._data.info.attrs['nonlinear_wave'] = 1 if self.ui.nonlinear_check.isChecked() else 0
        self._data.info.attrs['runup'] = 1 if self.ui.runup_check.isChecked() else 0
        self._data.info.attrs['roller'] = self.ui.roller_combo.currentIndex()
        self._data.info.attrs['fastmode'] = 1 if self.ui.fastmode_check.isChecked() else 0
        self._data.info.attrs['forward_reflection'] = self.ui.forward_reflection_combo.currentText()
        self._data.info.attrs['forward_reflection_const'] = float(self.ui.forward_reflection_edit.text())
        self._data.info.attrs['backward_reflection'] = self.ui.backward_reflection_combo.currentText()
        self._data.info.attrs['backward_reflection_const'] = float(self.ui.backward_reflection_edit.text())
        self._data.info.attrs['muddy_bed'] = self.ui.muddy_bed_combo.currentText()
        self._data.info.attrs['wave_breaking_formula'] = self.ui.wave_breaking_combo.currentText()
        self._data.info.attrs['gamma_bj78'] = float(self.ui.gamma_bj78_edit.text())
        self._data.info.attrs['date_format'] = self.ui.date_format_combo.currentText()

    def _verify_selected_datasets(self):
        """Verifies if the required datasets are selected for the model control dialog."""
        dataset_keys = [
            'current_uuid',
            'surge_uuid',
            'wind_uuid',
            'forward_reflection_uuid',
            'backward_reflection_uuid',
            'muddy_bed_uuid',
            'darcy_uuid',
            'manning_uuid',
        ]
        for key in dataset_keys:
            text = self._data.info.attrs.get(key)
            if text is None or text == '(none selected)':
                self._data.info.attrs[key] = ''

    @staticmethod
    def icon_connected(tree_node):
        """Gets CGRID2D icon.

        Args:
            tree_node(:obj:`TreeNode`): The tree node.

        Returns:
            (:obj:`str`): Path to the icon.
        """
        return get_tree_icon_from_xms_typename(tree_node)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def showEvent(self, event):  # noqa: N802
        """Restore last position and geometry when showing dialog."""
        super().showEvent(event)
        wbd.restore_splitter_geometry(
            splitter=self.ui.splitter_horiz, package_name='xms.cmswave', dialog_name=f'{self._dlg_name}_horiz'
        )
        wbd.restore_splitter_geometry(
            splitter=self.ui.splitter_vert, package_name='xms.cmswave', dialog_name=f'{self._dlg_name}_vert'
        )

    def accept(self):
        """Handles the accept action."""
        self._accept_parameters_tab()
        self._accept_boundary_tab()
        self._accept_output_tab()
        self._accept_options_tab()
        self._verify_selected_datasets()
        wbd.save_splitter_geometry(
            splitter=self.ui.splitter_horiz, package_name='xms.cmswave', dialog_name=f'{self._dlg_name}_horiz'
        )
        wbd.save_splitter_geometry(
            splitter=self.ui.splitter_vert, package_name='xms.cmswave', dialog_name=f'{self._dlg_name}_vert'
        )
        super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        # We always save the splitter position, even if user rejects.
        wbd.save_splitter_geometry(
            splitter=self.ui.splitter_horiz, package_name='xms.cmswave', dialog_name=f'{self._dlg_name}_horiz'
        )
        wbd.save_splitter_geometry(
            splitter=self.ui.splitter_vert, package_name='xms.cmswave', dialog_name=f'{self._dlg_name}_vert'
        )
        super().reject()
