"""Filter model for the structure properties table."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSortFilterProxyModel, Qt

# 3. Aquaveo modules

# 4. Local modules
from xms.cmswave.data.structures_data import StructuresData


COLUMN_TEXT = [  # Pretty text with newlines for the GUI.
    'ID',  # Should be hidden, but keep parallel with 'COL_*' constants in StructuresData
    'Color and Texture',
    'Name',
    'Type',
    'Use Modification',
    'Modification Type',
    'Modification Value'
]


class StructuresFilterModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)

    def _set_flags_from_toggle(self, model_idx, row_idx, col_idx, flags):
        """Set window flags based on toggle state.

        Args:
            model_idx (:obj:`QModelIndex`): The item's Qt model index
            row_idx (:obj:`int`): Index of the item row
            col_idx (:obj:`int`): Index of the toggle column
            flags (:obj:`Qt.ItemFlags`): The flags for the item

        Returns:
            (:obj:`Qt.ItemFlags`): The updated flags for the item
        """
        checked = model_idx.model().index(row_idx, col_idx).data(Qt.CheckStateRole)
        if checked == Qt.Checked:
            flags |= Qt.ItemIsEnabled
            flags |= Qt.ItemIsEditable
        else:
            flags = flags & (~Qt.ItemIsEnabled)
            flags = flags & (~Qt.ItemIsEditable)
        return flags

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (:obj:`int`): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (:obj:`int`): The role.

        Returns:
            The data.
        """
        if role != Qt.DisplayRole:
            return super().headerData(section, orientation, role)

        if orientation == Qt.Horizontal:
            if section >= len(COLUMN_TEXT):
                return ''  # Watch out for the hidden columns
            return COLUMN_TEXT[section]
        else:
            return section + 1

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        flags = super().flags(index)
        row = index.row()
        col = index.column()

        if col == StructuresData.COL_TOG_MOD:
            flags |= Qt.ItemIsEnabled
            flags |= Qt.ItemIsEditable
        elif col == StructuresData.COL_EDT_MOD:
            flags = self._set_flags_from_toggle(index, row, StructuresData.COL_TOG_MOD, flags)
        elif col == StructuresData.COL_TXT_MOD:
            flags = self._set_flags_from_toggle(index, row, StructuresData.COL_TOG_MOD, flags)
            flags = flags & (~Qt.ItemIsEditable)
        else:
            flags |= Qt.ItemIsEnabled

        if col == StructuresData.COL_COLOR:
            flags = flags & (~Qt.ItemIsEditable)
        return flags
