"""Map locations and attributes of all linked coverages to the CMS-Wave domain."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util

# 4. Local modules
from xms.cmswave.dmi.xms_data import XmsData
from xms.cmswave.mapping.structures_mapper import StructureMapper


class CoverageMapper:
    """Class for mapping coverages to a grid for CMS-Wave."""
    def __init__(self, query, main_file):
        """Constructor."""
        # super().__init__()
        self._logger = logging.getLogger('xms.cmswave')
        self.co_grid = None
        self.grid_wkt = None
        self.component_folder = None
        self.structure_coverage = None
        self.structure_component = None
        self.structure_data = None
        self.structure_mapped_comp_uuid = None
        self.structure_mapped_comp_display_uuid = None
        self.structure_comp_id_to_grid_cell_ids = None
        self.structure_types = None
        self.structure_use_mods = None
        self.structure_mod_vals = None
        self.structure_mapper = None
        self.struct_df = None
        self._sim_uuid = None

        self.mapped_comps = []
        self._query = query
        self._generate_snap = True
        self._sim_uuid = self._query.parent_item_uuid()

        xms_data = XmsData(query)
        # Get the grid
        do_grid = xms_data.do_ugrid
        if do_grid is None:
            return
        self.co_grid = xms_data.cogrid
        self.grid_wkt = do_grid.projection.well_known_text
        self.component_folder = os.path.dirname(os.path.dirname(main_file))

        struct_cov = xms_data.structures_coverage
        if struct_cov:
            self.structure_coverage = struct_cov
            component = xms_data.structures_component
            if component:
                self.structure_component = component
                self.structure_data = self.structure_component.data

            else:  # Better have the component if we have the coverage.
                raise RuntimeError('Unable to get structures coverage attributes.')

    def do_map(self):
        """Creates the snap preview of coverages onto the grid."""
        try:
            self._map_structures()
            self._delete_old_mapping()
        except:  # pragma: no cover  # noqa
            self._logger.exception('Error generating snap.')  # pragma: no cover

    def _map_structures(self):
        """Maps the arcs from the structures coverage to the grid."""
        if self.structure_coverage is None:
            return
        self._logger.info('Mapping structure coverage to grid.')
        comp_path = self.component_folder
        mapper = StructureMapper(self, wkt=self.grid_wkt, comp_path=comp_path)
        mapper.mapped_comp_uuid = self.structure_mapped_comp_uuid
        mapper.mapped_structures_display_uuid = self.structure_mapped_comp_display_uuid
        self.structure_comp_id_to_grid_cell_ids = mapper._poly_to_cells
        do_comp, comp = mapper.do_map()
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()], 'mapped_structures_component'))
        self._logger.info('Finished mapping structures coverage to grid.')

    def _delete_old_mapping(self):
        """Deletes the old mapping if it exists, when creating a new one."""
        sim_item = tree_util.find_tree_node_by_uuid(tree_node=self._query.project_tree, uuid=self._sim_uuid)
        if sim_item is None:
            self._logger.error('Could not find sim item parent to remove old mapped item.')
            return
        child_comps = tree_util.descendants_of_type(tree_root=sim_item, xms_types=['TI_COMPONENT'])
        for child_comp in child_comps:
            self._query.delete_item(child_comp.uuid)
            self._logger.debug('Removing old mapped item.')
