from abc import ABC

from .refine_level import RefineLevel
from ..rectilinear_geometry import Numbering


class QuadGridCellsGetters(ABC):
    """Tracks refinement and cell indices of a quad tree constrained UGrid.
    Interface for read-only data."""

    def __init__(self, instance):
        """Only called by subclasses.
        Args:
            instance: The C++ wrapped instance.
        """
        self._instance = instance

    @property
    def number_of_levels(self):
        """The maximum number of refinement levels."""
        return self._instance.GetNumLevels()

    def get_level(self, i, j, k):
        """Get the refinement level of a cell in refinement grid coordinates.
        Refinement level coordinates is the number of base rows or columns
        times two to maximum refinement levels power.

        Args:
            i: The row of the cell in refinement grid coordinates.
            j: The column of the cell in refinement grid coordinates.
            k: The layer of the cell.

        Returns:
            The refinement level of the cell.
        """
        self._require_valid_refine_coordinates(i, j, k)
        return RefineLevel(instance=self._instance.GetLevel(i, j, k))

    def get_included(self, i, j, k):
        """Determine if a given cell is included in the grid. Uses refinement
        grid coordinates. When included there is an existing cell in the UGrid.
        Otherwise, there is no cell and the index numbering skips that
        location.
        Args:
            i: The row of the cell in refinement grid coordinates.
            j: The column of the cell in refinement grid coordinates.
            k: The layer of the cell.
        Returns:
            True if the cell is included.
        """
        self._require_valid_refine_coordinates(i, j, k)
        return self._instance.GetIncluded(i, j, k)

    def get_cell_index(self, i, j, k):
        """Get the index of a cell in the UGrid. Uses refinement grid
        coordinates.
        Args:
            i: The row of the cell in refinement grid coordinates.
            j: The column of the cell in refinement grid coordinates.
            k: The layer of the cell.
        Returns:
            The index of the cell in the UGrid.
        """
        self._require_valid_refine_coordinates(i, j, k)
        return self._instance.GetCellIdx(i, j, k)

    def has_cell_index(self, index):
        """Determine if a cell index exists in the UGrid.
        Args:
            index: The index of the cell.
        Returns:
            If the given cell index is in the UGrid.
        """
        return self._instance.HasCellIdx(index)

    def get_cell_ijk(self, index):
        """Get the I, J, and K value of a given cell index in the UGrid.
        Args:
            index: The index of the UGrid cell.
        Returns:
            tuple: A tuple of the I, J, and K index of the cell in refinement
                   coordinates.
        """
        ijk = self._instance.GetCellIjk(index)
        if ijk[0] < 0 or ijk[1] < 0 or ijk[2] < 0:
            raise ValueError('Index is out of range.')
        return ijk

    def get_cell_refinement(self, index):
        """Get the cell refinement level for the given UGrid cell.
        Args:
            index: The index of the UGrid cell.
        Returns:
            The number of refinement levels at the given index.
        """
        level = self._instance.GetCellRefinement(index)
        if level is None:
            raise IndexError('Index is out of range.')
        level = level[0]
        return RefineLevel(instance=level)

    @property
    def most_refined(self):
        """The maximum refinement level of any cell in the UGrid."""
        return RefineLevel(instance=self._instance.MostRefined())

    @property
    def number_of_refined_rows(self):
        """The number of rows in refinement coordinates."""
        return self._instance.GetNumRefinedRows()

    @property
    def number_of_refined_columns(self):
        """The number of columns in refinement coordinates."""
        return self._instance.GetNumRefinedCols()

    @property
    def number_of_base_rows(self):
        """The number of rows in the base grid or unrefined grid."""
        return self._instance.GetNumBaseRows()

    @property
    def number_of_base_columns(self):
        """The number of columns in the base grid or unrefined grid."""
        return self._instance.GetNumBaseCols()

    @property
    def number_of_base_layers(self):
        """The number of layers in the base grid."""
        return self._instance.GetNumBaseLays()

    def get_base_ijk(self, index):
        """Get the I, J, K location of the given UGrid cell in the base grid.
        Args:
            index:
        Returns:
            tuple: The I, J, K location of the UGrid cell within the base grid.
        """
        return self._instance.GetBaseIjk(index)

    @property
    def numbering(self):
        """The cell numbering (KIJ or KJI). The order the cells are numbered
        in."""
        return Numbering(self._instance.GetNumbering())

    def get_smoothed_neighbor(self, index, direction):
        """
        Get index of neighbor in given direction, or -1 if cell doesn't exist.
        Args:
            index: The index of the UGrid cell.
            direction: The direction to the neighbor.
        Returns:
            The cell index of the neighbor or -1 if there is no neighbor.
        """
        return self._instance.GetSmoothedNeighbor(index, direction)

    def get_smoothed_neighbors(self, index):
        """
        Get indices of all neighbors of cell, or -1 for cells that don't exist.
        Args:
            index: The index of the UGrid cell.
        Returns:
            An iterable of cell indices of the 12 potential neighbors in
            NeighborDirection order. Returns -1 for neighbors that don't
            exist.
        """
        return self._instance.GetSmoothedNeighbors(index)

    @property
    def get_quad_grid_array(self):
        """A 1-dimensional array that represents cell indices and cell
        refinement. The first row by column by layer elements contain UGrid
        indices of the base grid if unrefined. If the cell is refined then
        the element has a negative value that gives the index of refined
        quad indices toward the end of the array. A refined quad is 5
        elements long. The first element is a negative value giving the
        index of the parent base cell or refined quad. For excluded cells
        the element value is QuadGridCells.EXCLUDED.
        Returns:
            A 1-dimensional array describing the cell indices and refinement.
        """
        return self._instance.GetQuadGridArray()

    def _require_valid_refine_coordinates(self, i, j, k):
        invalid = i < 0 or i >= self.number_of_refined_rows
        invalid = invalid or j < 0 or j >= self.number_of_refined_columns
        invalid = invalid or k < 0 or k >= self.number_of_base_layers
        if invalid:
            raise ValueError('Invalid refined grid coordinates.')
