"""Python wrapping for xms._data_objects.parameters.Arc."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects._data_objects.parameters import Arc as CArc
from xms.data_objects.parameters.spatial.Point import Point as PyPoint


class Arc:
    """The pure Python wrapper for C++ exposed xms._data_objects.parameters.Arc objects."""
    def __init__(self, feature_id=None, start_node=None, end_node=None, vertices=None, instance=None):
        """Construct the wrapper.

        Args:
            feature_id (int): Arc's XMS feature id
            start_node (PyPoint): Arc's start node
            end_node (PyPoint): Arc's end node
            vertices (list of PyPoint): Arc's vertices
            instance (CArc): The C++ Arc object to wrap
        """
        if instance:
            self._instance = instance
        else:
            self._instance = CArc()

        if feature_id is not None:
            self._instance.SetId(feature_id)
        if start_node is not None:
            self.start_node = start_node
        if end_node is not None:
            self.end_node = end_node
        if vertices is not None:
            self.vertices = vertices

    @property
    def id(self):
        """Returns the arc's XMS feature id."""
        return self._instance.GetId()

    @id.setter
    def id(self, arc_id):
        """Sets the arc's XMS feature id.

        Args:
            arc_id (int): Id to assign the arc
        """
        self._instance.SetId(arc_id)

    @property
    def start_node(self):
        """Returns the arc's start node."""
        return PyPoint(instance=self._instance.GetStartNode())

    @start_node.setter
    def start_node(self, node):
        """Sets the arc's start node.

        Args:
            node (PyPoint): The start node
        """
        self._instance.SetStartNode(node._instance)

    @property
    def end_node(self):
        """Returns the arc's start node."""
        return PyPoint(instance=self._instance.GetEndNode())

    @end_node.setter
    def end_node(self, node):
        """Sets the arc's end node.

        Args:
            node (PyPoint): The end node
        """
        self._instance.SetEndNode(node._instance)

    @property
    def vertices(self):
        """Returns a list of the arc's vertices."""
        return [PyPoint(instance=pt) for pt in self._instance.GetVertices()]

    @vertices.setter
    def vertices(self, pts):
        """Sets the arc's vertices.

        Args:
            pts (list of PyPoint): The vertices
        """
        self._instance.SetVertices([pt._instance for pt in pts])

    @property
    def point_ids(self):
        """Returns a list of the arc's vertices and end node IDs (all vertice IDs are negative one)."""
        return [PyPoint(instance=pt) for pt in self._instance.GetPointIds()]

    def get_points(self, location):
        """Returns the arc's points at a specified location.

        Args:
            location (xms.data_objects.parameters.FilterLocation): The point locations to retrieve. Valid values:
                FilterLocation.PT_LOC_CORNER - arc start/end nodes
                FilterLocation.PT_LOC_MID - arc vertices
                FilterLocation.PT_LOC_ALL - arc nodes and vertices
        Returns:
            list of PyPoint: The arc points at the specified location.
        """
        return [PyPoint(instance=pt) for pt in self._instance.GetPoints(location)]
