"""Python wrapping for xms._data_objects.parameters.RectilinearGrid."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects._data_objects.parameters import RectilinearGrid as CRectilinearGrid
from xms.data_objects.parameters.spatial.Point import Point as PyPoint
from xms.data_objects.parameters.spatial.RectilinearCell import RectilinearCell as PyRectilinearCell
from xms.data_objects.parameters.spatial.spatial_vector.SpatialVector import SpatialVector


XMDF_DEFAULT_RECT_GRID_PATH = 'Cart2DModule'  # Matches constant defined in C++ library


class RectilinearGrid(SpatialVector):
    """The pure Python wrapper for C++ exposed xms._data_objects.parameters.RectilinearGrid objects."""
    def __init__(self, *args, **kwargs):
        """Construct the wrapper.

        Args:
            *args:
                filename (str): The path to the H5 file containing the grid data (in internal format)
                h5_group (str, optional): Path in the H5 file to the grid data. XMDF_DEFAULT_RECT_GRID_PATH by
                    default.
            **kwargs:
                'instance': CRectilinearGrid - The C++ RectilinearGrid object to wrap
                See SpatialVector for base class kwargs
        """
        num_args = len(args)
        if num_args > 0:  # Use H5 file constructor
            group_path = XMDF_DEFAULT_RECT_GRID_PATH if num_args == 1 else args[1]
            kwargs['instance'] = CRectilinearGrid(args[0], group_path)
        elif 'instance' not in kwargs:  # Default empty RectilinearGrid
            kwargs['instance'] = CRectilinearGrid()
        super().__init__(**kwargs)

    @property
    def origin(self):
        """Returns the grid origin as a xms.data_objects.parameters.Point."""
        return PyPoint(instance=self._instance.GetOrigin())

    @origin.setter
    def origin(self, pt):
        """Sets the grid origin.

        Args:
            pt (xms.data_objects.parameters.Point): The grid origin
        """
        self._instance.SetOrigin(pt._instance)

    @property
    def angle(self):
        """Returns a the angle of the grid as a float."""
        return self._instance.GetAngle()

    @angle.setter
    def angle(self, grid_angle):
        """Sets the angle of the grid.

        Args:
            grid_angle (float): The angle of the grid
        """
        self._instance.SetAngle(grid_angle)

    @property
    def cells(self):
        """Returns a list of the grid's xms.data_objects.parameters.RectilinearCell."""
        return [PyRectilinearCell(instance=cell) for cell in self._instance.GetCells()]

    @cells.setter
    def cells(self, grid_cells):
        """Sets the cells of the grid.

        Args:
            grid_cells (list of xms.data_objects.parameters.RectilinearCell): The cells of the grid
        """
        self._instance.SetCells([cell._instance for cell in grid_cells])

    @property
    def i_sizes(self):
        """Returns a list of the i-direction sizes."""
        return self._instance.GetISizes()

    @property
    def j_sizes(self):
        """Returns a list of the j-direction sizes."""
        return self._instance.GetJSizes()

    def set_sizes(self, *args):
        """Set the i and j sizes of the grid.

        Args:
            *args:
                i_sizes (list of float): The cell sizes in the i-direction (can be variable sized)
                j_sizes (list of float): The cell sizes in the j-direction (can be variable sized)

                OR

                num_i (int): The number of cells in the i-direction
                i_size (float): The fixed size of cells in the i-direction
                num_j (int): The number of cells in the j-direction
                j_size (float): The fixed size of cells in the j-direction
        """
        num_args = len(args)
        if num_args == 2:  # Variable-sized version
            self._instance.SetSizes(args[0], args[1])
        elif num_args == 4:
            self._instance.SetSizes(args[0], args[1], args[2], args[3])
        else:
            raise ValueError(
                'Must provide either two lists of variable i-sizes and j-sizes, OR the number of i-cells, the fixed '
                'i-size, the number of j-cells, and the fixed j-size.'
            )

    def write_h5(self, filename, group_path=XMDF_DEFAULT_RECT_GRID_PATH, overwrite=True):
        """Write the grid data to an H5 file in internal format.

        Args:
            filename (str or xms.data_objects._data_objects.parameters.FileLocation): Path to the output H5 file. Either
                a string or a FileLocation object retrieved from an xms.api.dmi.Query.
            group_path (str): Group path in the H5 file of the grid data
            overwrite (bool): If True, overwrite any existing data in the file
        """
        self.complete()  # Ensure complete flag is set to True.
        self._instance.InitH5(filename, group_path, overwrite)
