"""Worker thread for the FUNWAVE .txt file importer."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import Signal

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.funwave.file_io.dataset_reader import DatasetReader


class ControlImportWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing a FUNWAVE control input file."""
    processing_finished = Signal()

    def __init__(self, parent, filename, query):
        """Construct the worker.

        Args:
            parent (:obj:`QWidget`): The Qt parent
            filename (:obj:`str`): The input.funwave file to import
            query (:obj:`Query`): The XMS interprocess communicator
        """
        super().__init__(parent=parent, do_work=self.do_work)
        self.control_reader = None
        self._filename = filename
        self._query = query

    def do_work(self):
        """Thread runner that imports the .fvc file."""
        try:

            self.control_reader = DatasetReader(self._filename, self._query)
            self.control_reader.read()
        except Exception:
            logging.getLogger('xms.funwave').exception('Error(s) encountered while reading FUNWAVE control file.')


def import_dataset_with_feedback(single_sim):
    """Driver for the .funtxt import script with a feedback dialog."""
    xms_dlg.ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    if XmEnv.xms_environ_running_tests() != 'TRUE':
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        parent = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, parent)  # Keep the timer in scope
    else:
        parent = None
        main_hwnd = None

    query = Query()
    filename = query.read_file
    worker = ControlImportWorkerThread(parent, filename, query)
    debug = False
    if debug:
        worker.do_work()
        worker.control_reader.send(query)
    else:
        display_text = {
            'title': 'Reading FUNWAVE Dataset',
            'working_prompt': 'Reading FUNWAVE dataset file. Please wait...',
            'error_prompt': 'Error(s) encountered reading dataset file. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered reading dataset file. Review log output for more details.',
            'success_prompt': 'Successfully read FUNWAVE dataset.',
            'note': '',
            'auto_load': 'Auto load data into SMS when operation is complete',
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
        feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        if feedback_dlg.exec():  # and not LogEchoQSignalStream.logged_error:  # Only send if no error.
            worker.control_reader.send(query)  # Send data back to XMS
        if parent is not None:
            win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
